# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

##########################################################################
# Calibre general information
##########################################################################

from collections import defaultdict

from calibre.gui2.actions import InterfaceAction

import calibre_plugins.kindle_collections.messages as msg
from calibre_plugins.kindle_collections.utilities import debug_print

CALIBRE_COLUMNS_SHOW  = [ 'authors', 'publisher', 'series', 'tags', 'author_sort', 'title' ]

device_metadata_available = False
ci = None

##########################################################################

def init(parent):
    global ci
    ci = CalibreInfo(parent)

class CalibreInfo():

    def __init__ (self, parent):
        debug_print('BEGIN Initialize CalibreInfo')
        self.gui = parent.gui
        self.db = self.gui.library_view.model().db
        self.device_model = self.gui.memory_view.model() # Kindle only has one memory location - no cards
        self.library_uuid = self.get_library_uuid()
        self.device_uuid = self.get_device_uuid()
        self.device_path = self.get_device_path()
        self.get_columns()
        debug_print('END Initialize CalibreInfo')

    def get_library_uuid(self):
        debug_print('BEGIN Library uuid')
        try:
            library_uuid = self.gui.library_view.model().db.library_id
        except:
            library_uuid = ''
        debug_print('END Library uuid: %s' % library_uuid)
        return library_uuid
    
    def get_device_uuid(self):
        debug_print('BEGIN Device uuid')
        try:
            device_connected = self.gui.library_view.model().device_connected
            device_uuid = self.gui.device_manager.connected_device.driveinfo['main']['device_store_uuid']
        except:
            device_uuid = ''
        debug_print('END Device uuid: %s' % device_uuid)
        return device_uuid
    
    # Check if Kindle is connected - only debug messages since messages are initialized just after
    def get_device_path(self):
        from calibre.devices.usbms.driver import USBMS
        debug_print('BEGIN Get Device Path')
    
        device_path = ''
        try:
            # If we're in test mode TEST_DEVICE is defined, use the predefined test directory
            #TEST_DEVICE = 'fakeKindleDir2'
            device_path = TEST_DEVICE
            debug_print('RUNNING IN TEST MODE')
        except:
            # Not in test mode, so confirm a device is connected
            device_connected = self.gui.library_view.model().device_connected
            try:
                device_connected = self.gui.library_view.model().device_connected
            except:
                debug_print('No device connected')
                device_connected = None
    
            # If there is a device connected, test if we can retrieve the mount point from Calibre
            if device_connected is not None:
                try:
                    # _main_prefix is not reset when device is ejected so must be sure device_connected above
                    device_path = self.gui.device_manager.connected_device._main_prefix
                    debug_print('Root path of device: %s' % device_path)
                except:
                    debug_print('A device appears to be connected, but device path not defined')
            else:
                debug_print('No device appears to be connected')
    
        debug_print('END Get Device Path')
        return device_path

    def get_custom_columns(self):
        return self.db.field_metadata.custom_field_metadata()

    def get_custom_value(self, calibre_id, label):
        return self.db.get_custom(calibre_id, label=label, index_is_id=True)

    def set_custom_value(self, id, label, value):
        self.db.set_custom(id, value, label, commit=False)

    def set_custom_values(self, id_list, label):
        for id in id_list.keys():
            value = id_list[id]
            self.set_custom_value(id, label, value)
        self.db.commit()

    def clear_custom_column(self, label):
        debug_print('Clearing custom column %s' % label)
        self.db.set_custom_bulk(self.all_ids, None, label=label)

    def is_data_in_column(self, label):
        found_data = False
        for id in self.all_ids:
            value = self.get_custom_value(id, label)
            if value:
                found_data = True
                break
        return found_data

    # Build list of selected built-in column label/names, all custom column label/names and add entry for user categories
    def get_columns(self):
        debug_print('BEGIN Calibre get_columns')

        self.custom_columns = self.db.custom_field_keys()

        # Loop through each field entry and save field label and name
        self.column_labels = defaultdict()
        name_label = defaultdict()
        # Custom field keys start with '#'
        fields = self.db.custom_field_keys() + CALIBRE_COLUMNS_SHOW
        for i in fields:
            name = ''
            try:
                meta = self.db.metadata_for_field(i)
                name = meta['name']
            except:
                pass
            # Author Sort is the one column without a name in Calibre as its an internal column
            if i == 'author_sort':
                name = 'Author Sort' 
            debug_print('Including Calibre column label "%s", name "%s"' % (i, name))
            if not name or name in name_label:
                name = unicode(i)
                debug_print('Duplicate or missing name, reset field %s to name "%s"' % (i, name))
            self.column_labels[i] = name
            name_label[name] = i
    
        # Special entry for user categories
        if 'User Categories' in name_label:
            self.column_labels['user_categories'] = 'Calibre User Categories_'
        else:
            self.column_labels['user_categories'] = 'User Categories'
        debug_print('END Calibre get_columns')

    # Build a list of collections/column names and what books they contain, and save column details per book
    def load_column_book_info(self):

        debug_print('BEGIN Calibre Load Book Info')

        self.active_collections = defaultdict()
        self.lpath_info = defaultdict()

        # Get a list of book ids on the device from the current library
        debug_print('Getting ids')
        query = 'ondevice:True'
        self.ids_on_device = self.db.search_getting_ids(query, None)
        query = ''
        self.all_ids = self.db.search_getting_ids(query, None)
        debug_print('%d ids, %d on device' % (len(self.all_ids), len(self.ids_on_device)))

        # Get the list of book data using those ids in the library
        # Throw away the row value in the (row, book) tuple returned
        debug_print('Getting books')
        calibre_books = []
        calibre_books.append([tup[1] for tup in self.device_model.paths_for_db_ids(self.all_ids)])

        # Create a map from the id to the book data
        debug_print('Mapping id to book')
        self.id_map = defaultdict(set)
        for blist in calibre_books:
            for abook in blist:
                self.id_map[abook.application_id] = abook
        debug_print('%d ids mapped' % len(self.id_map))

        debug_print('Loading book details for each book from Calibre')
        for id in self.all_ids:
            
            mi = self.db.get_metadata(id, index_is_id=True)

            # Save pathname only for books on the device so we know which paths are in calibre
            lpath = ''
            ond = id in self.ids_on_device
            debug_print('CALIBRE: on device: "%s" - %s' % (ond, mi.title))
            if id in self.ids_on_device:
                lpath = self.id_map[id].lpath
                if lpath:
                    # Save the details for books on the device
                    self.lpath_info[lpath] = { 'id': id, 'tags': mi.tags, 'authors': mi.authors, 'title': mi.title, 'author_sort': mi.author_sort }
                else:
                    debug_print('Unexpected error: lpath missing from id %s' % id)

            # Save column values for every book in Calibre because we need to know all Calibre 'collection' names to be able to delete unused names

            # Built-in columns
            for field in CALIBRE_COLUMNS_SHOW:
                data = mi.get(field)
                self.add_path(field, data, lpath)

            # Custom columns
            custom_data = mi.get_all_user_metadata(False)
            for label in custom_data.keys():
                if custom_data[label] and '#value#' in custom_data[label]:
                    self.add_path(label, custom_data[label]['#value#'], lpath)

            # User categories
            if mi.user_categories:
                for category in mi.user_categories.keys():
                    if mi.user_categories[category]:
                        self.add_path('user_categories', category, lpath)

        debug_print('END Calibre Load Book Info')

    # Add the path of one book to a collection/column only if it is on the Kindle (path not none)
    def add_path(self, column, collection, path):
        collections = collection if type(collection) == list else [ collection ]
        for c in collections:
            if c:
                if column in self.active_collections:
                    if c in self.active_collections[column]:
                        if path:
                            self.active_collections[column][c].append(path)
                    else:
                        self.active_collections[column][c] = [ path ] if path else []
                else:
                    self.active_collections[column] = { c: [ path ] } if path else { c: [] }

    def get_personal_doc_tag(self):
        from calibre.ebooks.conversion.config import load_defaults
        prefs = load_defaults('mobi_output')
        return prefs.get('personal_doc', None)

# No printing in this routine as it can be called before the plugin is run
def device_connection_changed_signalled(device_connected):
    global device_metadata_available
    if not device_connected:
        device_metadata_available = False 
    #    debug_print('Kindle Collections received signal connection changed: device not connected and setting device_metadata_available = False')
    #else:
    #    debug_print('Kindle Collections received signal connection changed: device connected, available = %s' % device_metadata_available)

# No printing in this routine as its called before the plugin is run
def device_metadata_available_signalled():
    global device_metadata_available
    device_metadata_available = True
    #debug_print('Kindle Collections received signal metadata available: setting device_metadata_available = True')

