#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, time, shutil, traceback

from calibre import CurrentDir, guess_type
from calibre.ebooks.metadata.opf2 import OPF
from calibre.ebooks.metadata.meta import set_metadata
from calibre.libunzip import extract as zipextract
from calibre.ptempfile import TemporaryDirectory
from calibre.utils.zipfile import ZipFile, ZIP_DEFLATED, ZIP_STORED

from calibre_plugins.modify_epub.container import ExtendedContainer, OPF_NS
from calibre_plugins.modify_epub.jacket import (remove_legacy_jackets, remove_all_jackets,
                                                add_replace_jacket)

ITUNES_FILES = ['iTunesMetadata.plist', 'iTunesArtwork']
BOOKMARKS_FILES = ['META-INF/calibre_bookmarks.txt']
OS_FILES = ['.DS_Store', 'thumbs.db']
ALL_ARTIFACTS = ITUNES_FILES + BOOKMARKS_FILES + OS_FILES


def modify_epub(log, title, epub_path, calibre_opf_path, cover_path, options):
    start_time = time.time()
    kwargs = {
        'title': title,
        'epub_path': epub_path,
        'calibre_opf_path': calibre_opf_path,
        'cover_path': cover_path,
        'options': options,
    }
    #log('Running Modify ePub with parameters:')
    #log(kwargs)

    modifier = BookModifier(log)
    new_book_path = modifier.process_book(title, epub_path, calibre_opf_path,
                                          cover_path, options)
    if new_book_path:
        log('ePub updated in %.2f seconds'%(time.time() - start_time))
    else:
        log('ePub not changed after %.2f seconds'%(time.time() - start_time))
    return new_book_path


class BookModifier(object):

    def __init__(self, log):
        self.log = log

    def process_book(self, title, epub_path, calibre_opf_path, cover_path, options):
        self.log('  Modifying: ', epub_path)
        try:
            self._restore_metadata_from_opf(calibre_opf_path, cover_path)
            self.output_profile = self._get_output_profile()

            # Extract the epub into a temp directory
            with TemporaryDirectory('_modify-epub') as tdir:
                with CurrentDir(tdir):
                    zipextract(epub_path, tdir)

                    # Use our own simplified wrapper around an ePub that will
                    # preserve the file structure and css
                    container = ExtendedContainer(tdir, self.log)
                    is_changed = self._process_book(container, options)
                    if is_changed:
                        container.write(epub_path)

            # If the user is updating metadata, we need to do this as a separate
            # step at the end, because it takes a stream object as input
            if options['update_metadata']:
                self.log('\tUpdating metadata and cover')
                with open(epub_path, 'r+b') as f:
                    set_metadata(f, self.mi, stream_type='epub')
                is_changed = True # Going to "assume" it did something

            # Only return path to the ePub if we have changed it
            if is_changed:
                return epub_path
        except:
            self.log.exception('%s - ERROR: %s' %(title, traceback.format_exc()))
        finally:
            if calibre_opf_path and os.path.exists(calibre_opf_path):
                os.remove(calibre_opf_path)
            if cover_path and os.path.exists(cover_path):
                os.remove(cover_path)

    def _restore_metadata_from_opf(self, calibre_opf_path, cover_path):
        '''
        Create an mi object from our copy of the latest Calibre metadata
        stored in an OPF, so that we can perform functions that update
        the book metadata, such as generating a new jacket.
        '''
        with open(calibre_opf_path, 'rb') as f:
            calibre_opf = OPF(f, os.path.dirname(calibre_opf_path))
        self.mi = calibre_opf.to_book_metadata()

        # Store our link to a copy of the book cover, so that we can perform
        # functions such as replacing the cover image.
        self.cover_path = cover_path
        # Populate our mi object with the cover data
        if cover_path:
            if os.access(cover_path, os.R_OK):
                fmt = cover_path.rpartition('.')[-1]
                data = open(cover_path, 'rb').read()
                self.mi.cover_data = (fmt, data)

    def _get_output_profile(self):
        from calibre.ebooks.conversion.config import load_defaults
        from calibre.customize.ui import output_profiles
        ps = load_defaults('page_setup')
        output_profile_name = 'default'
        if 'output_profile' in ps:
            output_profile_name = ps['output_profile']
        for x in output_profiles():
            if x.short_name == output_profile_name:
                return x
        self.log.warn('Output Profile %s is no longer available, using default'%output_profile_name)
        for x in profiles():
            if x.short_name == 'default':
                setattr(self.opts, attr, x)
                break

    def _process_book(self, container, options):
        is_changed = False

        # FILE OPTIONS
        if options['remove_itunes_files']:
            is_changed |= self._remove_files_if_exist(container, ITUNES_FILES)
        if options['remove_calibre_bookmarks']:
            is_changed |= self._remove_files_if_exist(container, BOOKMARKS_FILES)
        if options['remove_os_artifacts']:
            is_changed |= self._remove_files_if_exist(container, OS_FILES)

        # MANIFEST OPTIONS
        if options['remove_missing_files']:
            is_changed |= self._remove_missing_files(container)
        if options['add_unmanifested_files']:
            is_changed |= self._process_unmanifested_files(container, add=True)
        elif options['remove_unmanifested_files']:
            is_changed |= self._process_unmanifested_files(container, add=False)
        if options['remove_non_dc_elements']:
            is_changed |= self._remove_non_dc_elements(container)

        # JACKET OPTIONS
        if options['remove_legacy_jackets']:
            is_changed |= remove_legacy_jackets(container, self.log)
        if options['remove_all_jackets']:
            is_changed |= remove_all_jackets(container, self.log)
        if options['add_replace_jacket']:
            is_changed |= add_replace_jacket(container, self.log, self.mi, self.output_profile)

        # CONTENT OPTIONS
        if options['zero_xpgt_margins']:
            is_changed |= self._zero_xpgt_margins(container)
        if options['rewrite_css_margins']:
            is_changed |= self._rewrite_css_margins(container)
        if options['remove_embedded_fonts']:
            is_changed |= self._remove_embedded_fonts(container)

        return is_changed

    def _remove_missing_files(self, container):
        self.log('\tLooking for redundant entries in manifest')
        missing_files = set(container.mime_map.keys()) - set(container.name_map.keys())
        dirtied = False
        for name in missing_files:
            self.log('\t  Found entry to remove:', name)
            container.delete_from_manifest(name)
            dirtied = True
        if dirtied:
            container.set(container.opf_name, container.opf)
        return dirtied

    def _process_unmanifested_files(self, container, add=False):
        self.log('\tLooking for unmanifested files')
        all_artifacts = [f.lower() for f in ALL_ARTIFACTS]
        dirtied = False
        for name in list(container.manifest_worthy_names()):
            # Special exclusion for bookmarks, plist files and other OS artifacts
            known_artifact = False
            if name.lower() in all_artifacts:
                known_artifact = True
            if not known_artifact:
                for a in all_artifacts:
                    if name.lower().endswith('/'+a):
                        known_artifact = True
                        break
            if known_artifact:
                continue

            item = container.manifest_item_for_name(name)
            if item is None:
                if add:
                    self.log('\t  Found file to to add:', name)
                    ext = os.path.splitext(name)[1]
                    mt = None   # Let the mime-type be guessed from the extension
                    if ext.lower().startswith('.htm'):
                        # If this is really an xhtml file, need to explicitly declare it
                        raw = container.get(name)
                        if raw.find('xmlns="http://www.w3.org/1999/xhtml"') != -1:
                            mt = guess_type('a.xhtml')[0]
                            self.log('\t Switching mimetype to:', mt)
                    container.add_name_to_manifest(name, mt)
                else:
                    self.log('\t  Found file to to remove:', name)
                    container.delete_name(name)
                dirtied = True
        if dirtied:
            container.set(container.opf_name, container.opf)
        return dirtied

    def _remove_non_dc_elements(self, container):
        self.log('\tLooking for non dc: elements in manifest')
        if not container.opf_name:
            self.log('\t  No opf manifest found')
            return False
        to_remove = []
        metadata = container.opf.xpath('//opf:metadata', namespaces={'opf':OPF_NS})[0]
        for child in metadata:
            try:
                if not child.tag.startswith('{http://purl.org/dc/'):
                    to_remove.append(child)
                    self.log('\t  Removing child:', child.tag)
            except:
                # Dunno how to elegantly handle in lxml parsing
                # text like <!-- stuff --> which blows up when
                # calling the .tag function.
                to_remove.append(child)
                self.log('\t  Removing child of commented out text:', child.text)

        if to_remove:
            for node in to_remove:
                metadata.remove(node)
            container.set(container.opf_name, container.opf)
        return bool(to_remove)

    def _remove_files_if_exist(self, container, files):
        self.log('\tLooking for files to remove:', files)
        files = [f.lower() for f in files]
        for name in list(container.name_map.keys()):
            found = False
            if name.lower() in files:
                found = True
            if not found:
                for f in files:
                    if name.lower().endswith('/'+f):
                        found = True
                        break
            if found:
                self.log('\t  Found file to remove:', name)
                container.delete_from_manifest(name)
                return True
        return False

    def _zero_xpgt_margins(self, container):
        TEMPLATE_MIME_TYPES = ['application/adobe-page-template+xml',
                               'application/vnd.adobe-page-template+xml',
                               'application/vnd.adobe.page-template+xml']
        dirtied = False
        self.log('\tLooking for Adobe page template margins')
        for name in container.name_map:
            mt = container.mime_map.get(name, '')
            data = container.get(name)
            if (mt.lower() in TEMPLATE_MIME_TYPES and
                    hasattr(data, 'xpath')):
                for elem in data.xpath(
                        '//*[@margin-bottom or @margin-top '
                        'or @margin-left or @margin-right]'):
                    for margin in ('left', 'right', 'top', 'bottom'):
                        attr = 'margin-'+margin
                        elem.attrib.pop(attr, None)
                        dirtied = True
                if dirtied:
                    self.log('\t  Removed page margins from:', name)
                    container.set(name, data)
                    break
        return dirtied

    def _rewrite_css_margins(self, container):
        CSS_MIME_TYPES = ['text/css']
        dirtied = False

        def get_user_margins():
            all_margins = ['margin_top','margin_bottom','margin_left','margin_right']
            user_margins = []
            total_prefs_margins = 0.0

            from calibre.ebooks.conversion.config import load_defaults
            ps = load_defaults('page_setup')
            if 'margin_top' in ps:
                for style in all_margins:
                    value = ps[style]
                    user_margins.append([style, value])
                    total_prefs_margins = total_prefs_margins + value
            else:
                for style in all_margins:
                    user_margins.append([style, 5.0])
                    total_prefs_margins = 20.0

            if total_prefs_margins > 0.0:
                margins = ''
                for style, value in user_margins:
                    if value > 0.0:
                        margins = margins+style+':'+str(value)+'pt;'
                return margins
            else:
                return ''

        self.log('\tLooking for book level css margins')

        # check user margin pref

        def modify_margins(match):
            id = match.group('element_id')
            styles = match.group('styles').strip()
            stylelist = styles.split(';')
            retained_styles = []
            for style in stylelist:
                if style.lower().find('margin') != -1:
                    pass
                elif style:
                    retained_styles.append(style)

            user_margins = get_user_margins()
            if match.group('selector') == '@page' and user_margins:
                final_styles = ''
                if len(retained_styles) >= 1:
                    remaining_styles = '; '.join(retained_styles)
                    final_styles = user_margins+remaining_styles
                else:
                    final_styles = user_margins
                return match.group('selector')+' {'+final_styles+' }'

            elif len(retained_styles) >= 1:
                remaining_styles = '; '.join(retained_styles)
                if id:
                    return id+match.group('selector')+' {'+remaining_styles+' }'
                else:
                    return match.group('selector')+' {'+remaining_styles+' }'
            else:
                return ''


        for name in container.name_map:
            mt = container.mime_map.get(name, '')
            data = container.get(name)

            if mt.lower() in CSS_MIME_TYPES:
                at_page_exists = True if data.find('@page') != -1 else False
                import re
                body_page_detect = re.compile(r'(?P<element_id>#\w+\s+)?(?P<selector>\bbody|@page)\b\s*{(?P<styles>[^}]+);?\s*\}', re.IGNORECASE)

                if body_page_detect.findall(data):
                    dirtied = True
                    data = body_page_detect.sub(modify_margins, data)

                user_margins = get_user_margins()
                if not at_page_exists and user_margins:
                    dirtied = True
                    data = data+'\n@page {\n    '+user_margins+'\n  }'

                if dirtied:
                    self.log('\t  Modified book level css margins in:', name)
                    container.set(name, data)

        return dirtied

    def _remove_embedded_fonts(self, container):
        self.log('\tLooking for embedded fonts')
        dirtied = False
        for name in list(container.name_map.keys()):
            if name.lower().endswith('.ttf') or name.lower().endswith('.otf'):
                self.log('\t  Found font to remove:', name)
                container.delete_from_manifest(name)
                dirtied = True
        return dirtied
