#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
from calibre.gui2 import error_dialog, Dispatcher
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.dialogs.message_box import ProceedNotification
from calibre.ptempfile import PersistentTemporaryDirectory, remove_dir

from calibre_plugins.modify_epub.common_utils import set_plugin_icon_resources, get_icon
from calibre_plugins.modify_epub.dialogs import (ModifyEpubDialog, QueueProgressDialog,
                                                 AddBooksProgressDialog)

PLUGIN_ICONS = ['images/modify_epub.png']

class ModifyEpubAction(InterfaceAction):

    name = 'Modify ePub'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = (_('Modify ePub'), None, _('Modify the contents of an ePub without a conversion'), None)
    action_type = 'current'

    def genesis(self):
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.modify_epub)

    def modify_epub(self):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, _('Cannot modify ePub'),
                _('You must select one or more books to perform this action.'), show=True)

        book_ids = set(self.gui.library_view.get_selected_ids())
        db = self.gui.library_view.model().db
        book_epubs = []
        for book_id in book_ids:
            try:
                path_to_epub = db.format_abspath(
                        book_id, 'EPUB', index_is_id=True)
                book_epubs.append((book_id, path_to_epub))
            except:
                pass

        if not book_epubs:
            return error_dialog(self.gui, _('Cannot modify ePub'),
                    _('No ePub available. First convert the book to ePub.'),
                    show=True)

        # Launch dialog asking user to specify what options to modify
        dlg = ModifyEpubDialog(self.gui)
        if dlg.exec_() == dlg.Accepted:
            # Create a temporary directory to copy all the ePubs to while scanning
            tdir = PersistentTemporaryDirectory('_modify_epub', prefix='')
            QueueProgressDialog(self.gui, book_epubs, tdir, dlg.options, self._queue_job, db)

    def _queue_job(self, tdir, options, books_to_modify):
        if not books_to_modify:
            # All failed so cleanup our temp directory
            remove_dir(tdir)
            return

        func = 'arbitrary_n'
        cpus = self.gui.job_manager.server.pool_size
        args = ['calibre_plugins.modify_epub.jobs', 'do_modify_epubs',
                (books_to_modify, options, cpus)]
        desc = 'Modify ePubs'
        job = self.gui.job_manager.run_job(
                self.Dispatcher(self._modify_completed), func, args=args,
                    description=desc)
        job._tdir = tdir
        self.gui.status_bar.show_message('Modifying %d books'%len(books_to_modify))

    def _modify_completed(self, job):
        if job.failed:
            self.gui.job_exception(job, dialog_title=_('Failed to modify ePubs'))
            return
        modified_epubs_map = job.result
        self.gui.status_bar.show_message(_('Modify ePub completed'), 3000)

        update_count = len(modified_epubs_map)
        if update_count == 0:
            msg = _('No ePub files were updated')
            return error_dialog(self.gui, _('Modify ePub failed'), msg,
                                show_copy_button=False, show=True)

        msg = _('<p>Modify ePub modified <b>%d ePub files(s)</b>. '
              'Proceed with updating your library?' % update_count)

        payload = (modified_epubs_map, job._tdir)
        p = ProceedNotification(self._proceed_with_updating_epubs,
                payload, job.details,
                _('Modify log'), _('Modify ePub complete'), msg,
                show_copy_button=False, parent=self.gui,
                cancel_callback=self._cancel_updating_epubs)
        p.show()

    def _proceed_with_updating_epubs(self, payload):
        modified_epubs_map, tdir = payload
        AddBooksProgressDialog(self.gui, modified_epubs_map, tdir)

    def _cancel_updating_epubs(self, payload):
        modified_epubs_map, tdir = payload
        # All failed so cleanup our temp directory
        remove_dir(tdir)

