#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, shutil, traceback
from PyQt4.Qt import (QWidget, QVBoxLayout, QLabel, QCheckBox,
                      QGroupBox, Qt, QDialogButtonBox, QHBoxLayout,
                      QProgressDialog, QString, QTimer)
from calibre.gui2 import gprefs, warning_dialog, error_dialog
from calibre.gui2.convert.metadata import create_opf_file, create_cover_file
from calibre.ptempfile import remove_dir

from calibre_plugins.modify_epub.common_utils import (get_icon, SizePersistedDialog,
                                         ReadOnlyTableWidgetItem, ImageTitleLayout)

JACKET_OPTIONS = [
            ('remove_legacy_jackets', 'Remove legacy metadata jackets', 'Remove jackets generated using versions of Calibre prior to 0.6.50'),
            ('remove_all_jackets',    'Remove all metadata jackets',    'Remove all Calibre jackets, both legacy and current'),
            ('add_replace_jacket',    'Add/replace metadata jacket',    'Add a jacket if not existing, or replace a non-legacy jacket')
            ]

MANIFEST_OPTIONS = [
            ('remove_missing_files',      'Remove missing file entries from manifest',  'Remove entries in the manifest for files listed\nthat do not exist in the ePub'),
            ('add_unmanifested_files',    'Add unmanifested files to manifest',         'Add files to manifest that are in the ePub but\ndo not exist in the .opf manifest\n(excluding iTunes/Calibre bookmarks)'),
            ('remove_unmanifested_files', 'Remove unmanifested files from ePub',        'Remove files from the ePub that do not exist\nin the .opf manifest (excluding iTunes/Calibre bookmarks)\nWill not be applied if the add option is also checked'),
            ('remove_non_dc_elements',    'Remove non dc: metadata from manifest',      'Remove any metadata from the .opf manifest that\nis not in the dc: namespace. Such entries are created by\nediting in Sigil or Calibre. Remove for publishing externally'),
            ]

FILE_OPTIONS = [
            ('remove_itunes_files',       'Remove iTunes files',                        'Removes any iTunesMetadata.plist or artwork files\nadded by viewing the ePub in iTunes'),
            ('remove_calibre_bookmarks',  'Remove Calibre bookmark files',              'Remove any bookmark files added by the Calibre ebook viewer'),
            ('remove_os_artifacts',       'Remove OS artifact files',                   'Removes any OS artifacts like thumbs.db or .DS_Store\nthat are not needed by the ePub'),
            ]

CONTENT_OPTIONS = [
            ('zero_xpgt_margins',         u'Remove book-wide css && xpgt margins',      'Remove any Adobe .xpgt file margins to prevent them\ninterfering with viewing.\nRemove css margins specified on @page or body tags.'),
            ('remove_embedded_fonts',     'Remove embedded fonts',                      'Remove embedded fonts from the manifest and their files\nto reduce ePub size. The css will not be changed'),
            ]

METADATA_OPTIONS = [
            ('update_metadata',           'Update metadata (including cover)',          'Update the manifest with the latest Calibre metadata\nand replace the cover if possible. This will run\nafter all other modify options have completed'),
            ]

ALL_OPTIONS = JACKET_OPTIONS + MANIFEST_OPTIONS + FILE_OPTIONS + CONTENT_OPTIONS + METADATA_OPTIONS

class ModifyEpubDialog(SizePersistedDialog):
    '''
    Configure which options you want applied during the modify process
    '''
    def __init__(self, gui):
        SizePersistedDialog.__init__(self, gui, 'modify epub plugin:options dialog')
        self.setWindowTitle(_('Modify ePub'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        title_layout = ImageTitleLayout(self, 'images/modify_epub.png', _('Modify ePub Options'))
        layout.addLayout(title_layout)

        layout.addSpacing(5)
        main_layout = QHBoxLayout()
        layout.addLayout(main_layout, 1)
        col1_layout = QVBoxLayout()
        main_layout.addLayout(col1_layout)
        col2_layout = QVBoxLayout()
        main_layout.addLayout(col2_layout)

        options = gprefs.get(self.unique_pref_name+':settings', {})

        self._add_groupbox(col1_layout, 'Known Artifacts:', FILE_OPTIONS, options)
        col1_layout.addSpacing(5)
        self._add_groupbox(col1_layout, 'Manifest:', MANIFEST_OPTIONS, options)
        col1_layout.insertStretch(-1)

        self._add_groupbox(col2_layout, 'Calibre Jackets:', JACKET_OPTIONS, options)
        col2_layout.addSpacing(5)
        self._add_groupbox(col2_layout, 'Content:', CONTENT_OPTIONS, options)
        col2_layout.addSpacing(5)
        self._add_groupbox(col2_layout, 'Metadata:', METADATA_OPTIONS, options)
        col2_layout.insertStretch(-1)

        layout.addSpacing(10)
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self._ok_clicked)
        button_box.rejected.connect(self.reject)
        self.select_none_button = button_box.addButton(_(' Clear all '), QDialogButtonBox.ResetRole)
        self.select_none_button.setToolTip(_('Clear all selections'))
        self.select_none_button.clicked.connect(self._select_none_clicked)
        layout.addWidget(button_box)

        # Cause our dialog size to be restored from prefs or created on first usage
        self.resize_dialog()

    def _add_groupbox(self, layout, title, option_info, options):
        groupbox = QGroupBox(title)
        layout.addWidget(groupbox)
        groupbox_layout = QVBoxLayout()
        groupbox.setLayout(groupbox_layout)

        for key, text, tooltip in option_info:
            checkbox = QCheckBox(_(text), self)
            checkbox.setToolTip(_(tooltip))
            checkbox.setCheckState(Qt.Checked if options.get(key, False) else Qt.Unchecked)
            setattr(self, key, checkbox)
            groupbox_layout.addWidget(checkbox)

    def _ok_clicked(self):
        self.options = {}
        for option_name, t, tt in ALL_OPTIONS:
            self.options[option_name] = getattr(self, option_name).checkState() == Qt.Checked
        gprefs.set(self.unique_pref_name+':settings', self.options)

        # Only if the user has checked at least one option will we continue
        for key in self.options:
            if self.options[key]:
                self.accept()
                return
        return error_dialog(self, _('No options selected'),
                            _('You must select at least one option to continue'),
                            show=True, show_copy_button=False)

    def _select_none_clicked(self):
        for option_name, t, tt in ALL_OPTIONS:
            getattr(self, option_name).setCheckState(Qt.Unchecked)


class QueueProgressDialog(QProgressDialog):

    def __init__(self, gui, book_epubs, tdir, options, queue, db):
        QProgressDialog.__init__(self, '', QString(), 0, len(book_epubs), gui)
        self.setWindowTitle(_('Queueing books for modifying ePubs'))
        self.setMinimumWidth(500)
        self.book_epubs, self.tdir, self.options, self.queue, self.db = \
            book_epubs, tdir, options, queue, db
        self.gui = gui
        self.i, self.bad, self.books_to_modify = 0, [], []
        QTimer.singleShot(0, self.do_book)
        self.exec_()

    def do_book(self):
        (book_id, epub_orig) = self.book_epubs[self.i]
        self.i += 1

        try:
            if epub_orig is None or not os.path.exists(epub_orig):
                self.bad.append(book_id)
            else:
                mi, opf_file = create_opf_file(self.db, book_id)
                self.setLabelText(_('Queueing ')+mi.title)
                cover_file = create_cover_file(self.db, book_id)
                cover_file_name = cover_file.name if cover_file else None
                # Copy the book to the temp directory, using book id as filename
                epub_file = os.path.join(self.tdir, '%d.epub'%book_id)
                shutil.copyfile(epub_orig, epub_file)
                self.books_to_modify.append((book_id, mi.title, epub_file,
                                             opf_file.name, cover_file_name))
        except:
            traceback.print_exc()
            self.bad.append(book_id)

        self.setValue(self.i)
        if self.i >= len(self.book_epubs):
            return self.do_queue()
        else:
            QTimer.singleShot(0, self.do_book)

    def do_queue(self):
        self.hide()
        if self.bad != []:
            res = []
            for id in self.bad:
                title = self.db.title(id, True)
                res.append('%s'%title)
            msg = '%s' % '\n'.join(res)
            warning_dialog(self.gui, _('Could not modify ePub for some books'),
                _('Could not modify %d of %d books, because no ePub '
                'source format was found.') % (len(res), len(self.book_epubs)),
                msg).exec_()
        self.gui = None
        self.db = None
        # Queue a job to process these ePub books
        self.queue(self.tdir, self.options, self.books_to_modify)


class AddBooksProgressDialog(QProgressDialog):

    def __init__(self, gui, modified_epubs, tdir):
        self.total_count = len(modified_epubs)
        QProgressDialog.__init__(self, '', QString(), 0, self.total_count, gui)
        self.setWindowTitle('Adding %d modified ePubs...' % self.total_count)
        self.setMinimumWidth(500)
        self.modified_epubs, self.tdir = modified_epubs, tdir
        self.book_ids = list(modified_epubs.keys())
        self.gui = gui
        self.db = self.gui.current_db
        self.i = 0
        QTimer.singleShot(0, self.do_book_check)
        self.exec_()

    def do_book_check(self):
        if self.i >= self.total_count:
            return self.do_close()
        book_id = self.book_ids[self.i]
        epub_path = self.modified_epubs[book_id]
        self.i += 1

        title = self.db.title(book_id, index_is_id=True)
        self.setLabelText(_('Adding')+': '+title)
        # Add the epub back, causing the size information to be updated
        self.db.add_format(book_id, 'EPUB', open(epub_path, 'rb'), index_is_id=True)
        self.setValue(self.i)

        QTimer.singleShot(0, self.do_book_check)

    def do_close(self):
        self.hide()
        remove_dir(self.tdir)
        self.gui.status_bar.show_message(_('ePub files updated'), 3000)
        self.gui = None
        self.db = None

