#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, posixpath, urllib
from lxml import etree

from calibre import guess_type
from calibre.ebooks.epub.fix import InvalidEpub, ParseError
from calibre.ebooks.epub.fix.container import Container, OPF_NS, OCF_NS
from calibre.ebooks.oeb.base import urlnormalize
from calibre.ebooks.metadata.toc import TOC
from calibre.utils.zipfile import ZipFile, ZIP_DEFLATED, ZIP_STORED

NCX_NS = 'http://www.daisy.org/z3986/2005/ncx/'

class ExtendedContainer(Container):
    '''
    Extend the Calibre epub-fix container with additional functions
    that assist with writing updated manifests and toc
    '''
    def __init__(self, path, log):
        Container.__init__(self, path, log)
        self.ncx = self.ncx_name = None
        for name in self.manifest_worthy_names():
            if name.endswith('.ncx'):
                try:
                    self.ncx_name = name
                    self.ncx = self.get(self.ncx_name)
                except ParseError:
                    # This ePub is probably protected with DRM and the NCX is encrypted
                    self.ncx_name = None
                    self.ncx = None
                break

    def href_to_name(self, href, base=''):
        '''
        Overridden to fix a bug in the Calibre function which incorrectly
        splits the href on # when # is part of the filename
        '''
        hash_index = href.find('#')
        period_index = href.find('.')
        if hash_index > 0 and hash_index > period_index:
            href = href.partition('#')[0]
        href = urllib.unquote(href)
        name = href
        if base:
            name = posixpath.join(base, href)
        return name

    def name_to_href(self, name, base):
        '''
        Overridden to ensure that blank href names are correctly
        referenced as "" rather than "."
        '''
        if not base:
            return name
        href = posixpath.relpath(name, base)
        if href == '.':
            href = ''
        return href

    def delete_name(self, name):
        '''
        Overridden to ensure that it will not blow up if called with
        a name that is not in the map
        '''
        if name in self.mime_map:
            self.mime_map.pop(name, None)
        if name in self.name_map:
            path = self.name_map[name]
            os.remove(path)
            self.name_map.pop(name)

    def write(self, path):
        '''
        Overridden to change how the zip file is assembled as found
        issues with the add_dir function as it was written
        '''
        for name in self.dirtied:
            data = self.cache[name]
            raw = data
            if hasattr(data, 'xpath'):
                raw = etree.tostring(data, encoding='utf-8',
                        xml_declaration=True)
            with open(self.name_map[name], 'wb') as f:
                f.write(raw)
        self.dirtied.clear()
        with ZipFile(path, 'w', compression=ZIP_DEFLATED) as zf:
            # Write mimetype
            zf.writestr('mimetype', bytes(guess_type('a.epub')[0]),
                    compression=ZIP_STORED)
            # Write everything else
            exclude_files = ['.DS_Store','mimetype']
            for root, dirs, files in os.walk(self.root):
                for fn in files:
                    if fn in exclude_files:
                        continue
                    absfn = os.path.join(root, fn)
                    zfn = os.path.relpath(absfn,
                            self.root).replace(os.sep, '/')
                    zf.write(absfn, zfn)

    def delete_from_manifest(self, name):
        '''
        Remove this item from the manifest, spine, guide and TOC ncx if it exists
        '''
        self.delete_name(name)
        item = self.manifest_item_for_name(name)
        if item is None:
            return
        manifest = self.opf.xpath('//opf:manifest', namespaces={'opf':OPF_NS})[0]
        self.log('\t  Manifest item removed: %s (%s)'%(item.get('href'), item.get('id')))
        manifest.remove(item)
        self.set(self.opf_name, self.opf)

        # Now remove the item from the spine if it exists
        self.delete_from_spine(item)

        # Remove from the guide if it exists
        self.delete_from_guide(item)

        # Finally remove the item from the TOC
        self.delete_from_toc(item)

    def delete_from_spine(self, item):
        '''
        Given an item, remove it from the spine
        '''
        item_id = item.get('id')
        itemref = self.opf.xpath('//opf:spine/opf:itemref[@idref="%s"]'%item_id,
                namespaces={'opf':OPF_NS})
        if itemref:
            self.log('\t  Spine itemref removed:', item_id)
            spine = itemref[0].getparent()
            spine.remove(itemref[0])
            self.set(self.opf_name, self.opf)

    def delete_from_guide(self, item):
        '''
        Given an item, remove it from the guide
        '''
        item_href = item.get('href')
        reference = self.opf.xpath('//opf:guide/opf:reference[@href="%s"]'%item_href,
                namespaces={'opf':OPF_NS})
        if reference:
            self.log('\t  Guide reference removed: %s'%item_href)
            guide = reference[0].getparent()
            guide.remove(reference[0])
            self.set(self.opf_name, self.opf)

    def delete_from_toc(self, item):
        '''
        Given an item from the manifest, remove any matching entry from
        the TOC ncx file
        '''
        def test_navpoint_for_removal(navpoint):
            src = navpoint.xpath('ncx:content/@src', namespaces={'ncx':NCX_NS})
            if src:
                src = src[0].lower()
                href = item.get('href').lower()
                if src == href or src.startswith(href + '#'):
                    self.log('\t  TOC Navpoint removed of:', src)
                    return True
            return False

        if item is None or self.ncx_name is None:
            return
        dirtied = False
        for navpoint in self.ncx.xpath('//ncx:navPoint', namespaces={'ncx':NCX_NS}):
            if test_navpoint_for_removal(navpoint):
                dirtied = True
                p = navpoint.getparent()
                idx = p.index(navpoint)
                p.remove(navpoint)
                for child in reversed(navpoint):
                    if child.tag == '{%s}navPoint'%NCX_NS:
                        self.log('\t  TOC Navpoint child promoted')
                        p.insert(idx, child)
        if dirtied:
            self._indent(self.ncx)
            self.set(self.ncx_name, self.ncx)

    def _indent(self, elem, level=0):
        i = '\n' + level*'    '
        if len(elem):
            if not elem.text or not elem.text.strip():
                elem.text = i + '    '
            for e in elem:
                self._indent(e, level+1)
                if not e.tail or not e.tail.strip():
                    e.tail = i + '    '
            if not e.tail or not e.tail.strip():
                e.tail = i
        else:
            if level and (not elem.text or not elem.text.strip()):
                elem.text = i
            if level and (not elem.tail or not elem.tail.strip()):
                elem.tail = i

    def generate_unique(self, id=None, href=None):
        '''
        Generate a new unique identifier and/or internal path for use in
        creating a new manifest item, using the provided :param:`id` and/or
        :param:`href` as bases.

        Returns an two-tuple of the new id and path.  If either :param:`id` or
        :param:`href` are `None` then the corresponding item in the return
        tuple will also be `None`.

        Grant: Copied/modified from calibre.ebooks.oeb.base.Manifest
        '''
        if id is not None:
            items = self.opf.xpath('//opf:manifest/opf:item[@id]',
                    namespaces={'opf':OPF_NS})
            ids = set([x.get('id') for x in items])

            base = id
            index = 1
            while id in ids:
                id = base + str(index)
                index += 1
        if href is not None:
            items = self.opf.xpath('//opf:manifest/opf:item[@href]',
                    namespaces={'opf':OPF_NS})
            hrefs = set([x.get('href') for x in items])

            href = urlnormalize(href)
            base, ext = os.path.splitext(href)
            index = 1
            lhrefs = set([x.lower() for x in hrefs])
            while href.lower() in lhrefs:
                href = base + str(index) + ext
                index += 1
        return id, href

    def add_to_manifest(self, id, href, mt=None):
        '''
        Given an id and an href, create an item in the manifest for it
        '''
        manifest = self.opf.xpath('//opf:manifest', namespaces={'opf':OPF_NS})[0]
        item = manifest.makeelement('{%s}item'%OPF_NS, nsmap={'opf':OPF_NS},
                href=href, id=id)
        if not mt:
            mt = guess_type(href)[0]
        if not mt:
            mt = 'application/octest-stream'
        item.set('media-type', mt)
        manifest.append(item)
        self.fix_tail(item)
        self.log('\t  Manifest item added: %s (%s)'%(href, id))

    def add_to_spine(self, id, index=-1):
        '''
        Given an id, add it to the spine, optionally at the specified position
        '''
        spine = self.opf.xpath('//opf:spine', namespaces={'opf':OPF_NS})[0]
        itemref = spine.makeelement('{%s}itemref'%OPF_NS, nsmap={'opf':OPF_NS},
                idref=id)
        if index >= 0:
            spine.insert(index, itemref)
        else:
            spine.append(itemref)
        self.fix_tail(itemref)
        self.log('\t  Spine item inserted: %s at pos: %d'%(id, index))

    def get_spine_itemref_idref(self, index):
        spine = self.opf.xpath('//opf:spine', namespaces={'opf':OPF_NS})[0]
        if index < len(spine):
            return spine[index].get('idref')

