/****************************************************************************
**
** Copyright (C) 2011 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtGui module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "qscreeneinkfb_qws.h"

#ifndef QT_NO_QWS_EINKFB
#include "qwsdisplay_qws.h"
#include <QStringList>
#include <QRegExp>
#include <QColor>
#include <private/qdrawhelper_p.h>
#include <private/qcore_unix_p.h> // overrides QT_OPEN

#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>

#include "qwindowsystem_qws.h"

#include <linux/fb.h>
#include <linux/einkfb.h>


QT_BEGIN_NAMESPACE

extern int qws_client_id;

QEInkFbScreenPrivate::QEInkFbScreenPrivate()
    : fd(-1), updateMode(QEInkFbScreen::updateModePartial), updateWaitTime(300)
{
    dirtyRect = new QRect;
    lazyTimer = new QTimer(this);
    lazyTimer->setSingleShot(true);
    connect(lazyTimer, SIGNAL(timeout()), this, SLOT(doUpdate()));
}

QEInkFbScreenPrivate::~QEInkFbScreenPrivate()
{
    lazyTimer->stop();
    delete lazyTimer;
    delete dirtyRect;
}

void QEInkFbScreenPrivate::setDirty(const QRect &r)
{
    lazyTimer->stop();
    if(dirtyRect->isEmpty()) {
	*dirtyRect = r;
    } else {
	*dirtyRect |= r;
    }
    lazyTimer->start(updateWaitTime);
}

void QEInkFbScreenPrivate::doUpdate()
{
    // for Kindle e-ink display
    update_area_t myarea;

    if(fd == -1)
	return;

    //qWarning("update(%d,%d | %d,%d)", dirtyRect->left(), dirtyRect->top(), dirtyRect->right(), dirtyRect->bottom());
    if(dirtyRect->left() == 0 && dirtyRect->top() == 0 && dirtyRect->width() == startupw && dirtyRect->height() == startuph) {
	ioctl(fd, FBIO_EINK_UPDATE_DISPLAY, updateMode);
    } else {
	myarea.x1 = dirtyRect->left();
	myarea.x2 = dirtyRect->right();
	myarea.y1 = dirtyRect->top();
	myarea.y2 = dirtyRect->bottom();
	myarea.buffer = NULL;
	myarea.which_fx = (fx_type)updateMode;

	ioctl(fd, FBIO_EINK_UPDATE_DISPLAY_AREA, &myarea);
    }
    dirtyRect->setRect(0, 0, 0, 0);
}

// QT_QWS_DEPTH_4
/*!
    \internal

    \class QEInkFbScreen
    \ingroup qws

    \brief The QEInkFbScreen class implements a screen driver for the
    Linux framebuffer.

    Note that this class is only available in \l{Qt for Embedded Linux}.
    Custom screen drivers can be added by subclassing the
    QScreenDriverPlugin class, using the QScreenDriverFactory class to
    dynamically load the driver into the application, but there should
    only be one screen object per application.

    The QEInkFbScreen class provides the cache() function allocating
    off-screen graphics memory, and the complementary uncache()
    function releasing the allocated memory. The latter function will
    first sync the graphics card to ensure the memory isn't still
    being used by a command in the graphics card FIFO queue. The
    deleteEntry() function deletes the given memory block without such
    synchronization.  Given the screen instance and client id, the
    memory can also be released using the clearCache() function, but
    this should only be necessary if a client exits abnormally.

    In addition, when in paletted graphics modes, the set() function
    provides the possibility of setting a specified color index to a
    given RGB value.

    The QEInkFbScreen class also acts as a factory for the
    unaccelerated screen cursor and the unaccelerated raster-based
    implementation of QPaintEngine (\c QRasterPaintEngine);
    accelerated drivers for Linux should derive from this class.

    \sa QScreen, QScreenDriverPlugin, {Running Applications}
*/

// Unaccelerated screen/driver setup. Can be overridden by accelerated
// drivers

/*!
    \fn QEInkFbScreen::QEInkFbScreen(int displayId)

    Constructs a QEInkFbScreen object. The \a displayId argument
    identifies the Qt for Embedded Linux server to connect to.
*/

QEInkFbScreen::QEInkFbScreen(int display_id)
    : QScreen(display_id, LinuxFBClass), d_ptr(new QEInkFbScreenPrivate)
{
#ifdef QT_QWS_CLIENTBLIT
    setSupportsBlitInClients(true);
#endif
}

/*!
    Destroys this QEInkFbScreen object.
*/

QEInkFbScreen::~QEInkFbScreen()
{
}

/*!
    \reimp

    This is called by \l{Qt for Embedded Linux} clients to map in the framebuffer.
    It should be reimplemented by accelerated drivers to map in
    graphics card registers; those drivers should then call this
    function in order to set up offscreen memory management. The
    device is specified in \a displaySpec; e.g. "/dev/fb".

    \sa disconnect()
*/

bool QEInkFbScreen::connect(const QString &displaySpec)
{
    d_ptr->displaySpec = displaySpec;

    const QStringList args = displaySpec.split(QLatin1Char(':'));

#if Q_BYTE_ORDER == Q_BIG_ENDIAN
#ifndef QT_QWS_FRAMEBUFFER_LITTLE_ENDIAN
    if (args.contains(QLatin1String("littleendian")))
#endif
        QScreen::setFrameBufferLittleEndian(true);
#endif

    QString dev = QLatin1String("/dev/fb0");
    foreach(QString d, args) {
	if (d.startsWith(QLatin1Char('/'))) {
	    dev = d;
	    break;
	}
	if (d.startsWith(QLatin1String("waitrefresh="))) {
	    // TODO: parse and set private class accordingly
	}
    }

    if (access(dev.toLatin1().constData(), R_OK|W_OK) == 0)
        d_ptr->fd = QT_OPEN(dev.toLatin1().constData(), O_RDWR);
    if (d_ptr->fd == -1) {
        if (QApplication::type() == QApplication::GuiServer) {
            perror("QScreenEInkFb::connect");
            qCritical("Error opening framebuffer device %s", qPrintable(dev));
            return false;
        }
        if (access(dev.toLatin1().constData(), R_OK) == 0)
            d_ptr->fd = QT_OPEN(dev.toLatin1().constData(), O_RDONLY);
    }

    ::fb_fix_screeninfo finfo;
    ::fb_var_screeninfo vinfo;
    //#######################
    // Shut up Valgrind
    memset(&vinfo, 0, sizeof(vinfo));
    memset(&finfo, 0, sizeof(finfo));
    //#######################

    /* Get fixed screen information */
    if (d_ptr->fd != -1 && ioctl(d_ptr->fd, FBIOGET_FSCREENINFO, &finfo)) {
        perror("QEInkFbScreen::connect");
        qWarning("Error reading fixed information");
        return false;
    }

    if (finfo.type != FB_TYPE_PACKED_PIXELS) {
        qWarning("Error: video type not supported");
        return false;
    }

    /* Get variable screen information */
    if (d_ptr->fd != -1 && ioctl(d_ptr->fd, FBIOGET_VSCREENINFO, &vinfo)) {
        perror("QEInkFbScreen::connect");
        qWarning("Error reading variable information");
        return false;
    }

    grayscale = vinfo.grayscale;
    if (!grayscale) {
        qWarning("Error: only grayscale is supported");
        return false;
    }

    d = vinfo.bits_per_pixel;
    if (d != 4) {
        qWarning("Error: 4BPP is supported for now");
        return false;
    }
    lstep = finfo.line_length;

    int xoff = vinfo.xoffset;
    int yoff = vinfo.yoffset;
    const char* qwssize;
    if((qwssize=::getenv("QWS_SIZE")) && sscanf(qwssize,"%dx%d",&w,&h)==2) {
        if (d_ptr->fd != -1) {
            if ((uint)w > vinfo.xres) w = vinfo.xres;
            if ((uint)h > vinfo.yres) h = vinfo.yres;
        }
        dw=w;
        dh=h;
        int xxoff, yyoff;
        if (sscanf(qwssize, "%*dx%*d+%d+%d", &xxoff, &yyoff) == 2) {
            if (xxoff < 0 || xxoff + w > vinfo.xres)
                xxoff = vinfo.xres - w;
            if (yyoff < 0 || yyoff + h > vinfo.yres)
                yyoff = vinfo.yres - h;
            xoff += xxoff;
            yoff += yyoff;
        } else {
            xoff += (vinfo.xres - w)/2;
            yoff += (vinfo.yres - h)/2;
        }
    } else {
        dw=w=vinfo.xres;
        dh=h=vinfo.yres;
    }

    if (w == 0 || h == 0) {
        qWarning("QScreenEInkFb::connect(): Unable to find screen geometry, "
                 "will use 600x800.");
        dw = w = 600;
        dh = h = 800;
    }

    QScreen::setPixelFormat(QImage::Format_Invalid);

    // Handle display physical size spec.
    QStringList displayArgs = displaySpec.split(QLatin1Char(':'));
    QRegExp mmWidthRx(QLatin1String("mmWidth=?(\\d+)"));
    int dimIdxW = displayArgs.indexOf(mmWidthRx);
    QRegExp mmHeightRx(QLatin1String("mmHeight=?(\\d+)"));
    int dimIdxH = displayArgs.indexOf(mmHeightRx);
    if (dimIdxW >= 0) {
        mmWidthRx.exactMatch(displayArgs.at(dimIdxW));
        physWidth = mmWidthRx.cap(1).toInt();
        if (dimIdxH < 0)
            physHeight = dh*physWidth/dw;
    }
    if (dimIdxH >= 0) {
        mmHeightRx.exactMatch(displayArgs.at(dimIdxH));
        physHeight = mmHeightRx.cap(1).toInt();
        if (dimIdxW < 0)
            physWidth = dw*physHeight/dh;
    }
    if (dimIdxW < 0 && dimIdxH < 0) {
        if (vinfo.width != 0 && vinfo.height != 0
            && vinfo.width != UINT_MAX && vinfo.height != UINT_MAX) {
            physWidth = vinfo.width;
            physHeight = vinfo.height;
        } else {
            const int dpi = 72;
            physWidth = qRound(dw * 25.4 / dpi);
            physHeight = qRound(dh * 25.4 / dpi);
        }
    }

    /* set up palette (QT-representation) */
    screencols = 16;
    int loopc;
    for(loopc=0;loopc<screencols;loopc++) {
        screenclut[loopc]=qRgb(0xff - loopc*0x11, 0xff - loopc*0x11, 0xff - loopc*0x11);
    }

    /* Figure out the size of the screen in bytes */
    size = h * lstep;

    mapsize = finfo.smem_len;

    data = (unsigned char *)-1;
    if (d_ptr->fd != -1)
        data = (unsigned char *)mmap(0, mapsize, PROT_READ | PROT_WRITE,
                                     MAP_SHARED, d_ptr->fd, 0);

    if ((long)data == -1) {
        if (QApplication::type() == QApplication::GuiServer) {
            perror("QEInkFbScreen::connect");
            qWarning("Error: failed to map framebuffer device to memory.");
            return false;
        }
        data = 0;
    }

    return true;
}

/*!
    \reimp

    This unmaps the framebuffer.

    \sa connect()
*/

void QEInkFbScreen::disconnect()
{
    if (data)
        munmap((char*)data,mapsize);
    close(d_ptr->fd);
}

// #define DEBUG_VINFO

/*!
    \reimp

    This is called by the \l{Qt for Embedded Linux} server at startup time.
    It turns off console blinking, sets up the color palette, enables write
    combining on the framebuffer and initialises the off-screen memory
    manager.
*/

bool QEInkFbScreen::initDevice()
{
    // Grab current mode so we can reset it
    fb_var_screeninfo vinfo;
    fb_fix_screeninfo finfo;
    //#######################
    // Shut up Valgrind
    memset(&vinfo, 0, sizeof(vinfo));
    memset(&finfo, 0, sizeof(finfo));
    //#######################

    if (ioctl(d_ptr->fd, FBIOGET_VSCREENINFO, &vinfo)) {
        perror("QEInkFbScreen::initDevice");
        qFatal("Error reading variable information in card init");
        return false;
    }

#ifdef DEBUG_VINFO
    qDebug("Greyscale %d",vinfo.grayscale);
    qDebug("Nonstd %d",vinfo.nonstd);
    qDebug("Red %d %d %d",vinfo.red.offset,vinfo.red.length,
           vinfo.red.msb_right);
    qDebug("Green %d %d %d",vinfo.green.offset,vinfo.green.length,
           vinfo.green.msb_right);
    qDebug("Blue %d %d %d",vinfo.blue.offset,vinfo.blue.length,
           vinfo.blue.msb_right);
    qDebug("Transparent %d %d %d",vinfo.transp.offset,vinfo.transp.length,
           vinfo.transp.msb_right);
#endif

    if (ioctl(d_ptr->fd, FBIOGET_FSCREENINFO, &finfo)) {
        perror("QEInkFbScreen::initDevice");
        qCritical("Error reading fixed information in card init");
        // It's not an /error/ as such, though definitely a bad sign
        // so we return true
        return true;
    }

    d_ptr->startupw=vinfo.xres;
    d_ptr->startuph=vinfo.yres;
    d_ptr->startupd=vinfo.bits_per_pixel;
    grayscale = vinfo.grayscale;

#ifndef QT_NO_QWS_CURSOR
    QScreenCursor::initSoftwareCursor();
#endif
    blank(false);

    return true;
}

/*!
    \reimp

    This is called by the \l{Qt for Embedded Linux} server when it shuts
    down, and should be inherited if you need to do any card-specific cleanup.
    The default version hides the screen cursor and reenables the blinking
    cursor and screen blanking.
*/

void QEInkFbScreen::shutdownDevice()
{
    // nothing to do
}

/*!
    \reimp

    Sets the framebuffer to a new resolution and bit depth. The width is
    in \a nw, the height is in \a nh, and the depth is in \a nd. After
    doing this any currently-existing paint engines will be invalid and the
    screen should be completely redrawn. In a multiple-process
    Embedded Qt situation you must signal all other applications to
    call setMode() to the same mode and redraw.
*/

void QEInkFbScreen::setMode(int nw,int nh,int nd)
{
    // nothing here yet, if ever
    if((nw != w) && (nh != h) && (nd != d)) {
	qFatal("Error: Can't change mode from %dx%d@%d to %dx%d@%d", w, h, d, nw, nh, nd);
    }
}

void QEInkFbScreen::setUpdateMode(int mode)
{
    qWarning("setting Blit mode to %d", mode);
    d_ptr->updateMode = mode;
}

/*!
    \reimp
*/
void QEInkFbScreen::setDirty(const QRect &r)
{
    d_ptr->setDirty(r);
}

void QEInkFbScreen::setPixelFormat()
{
    QImage::Format format = QImage::Format_Invalid;

    switch (d) {
    case 1:
        format = QImage::Format_Mono; //###: LSB???
        break;
    default:
        break;
    }

    QScreen::setPixelFormat(format);
}

static inline void qt_rectfill_gray4r(quint8 *dest, quint8 value,
                                     int x, int y, int width, int height,
                                     int stride)
{
    const int pixelsPerByte = 2;
    dest += y * stride + x / pixelsPerByte;
    const int doAlign = x & 1;
    const int doTail = (width - doAlign) & 1;
    const int width8 = (width - doAlign) / pixelsPerByte;
    const quint8 c = value ^ 15;

    for (int j = 0; j < height; ++j) {
        if (doAlign)
            *dest = (*dest & 0xf0) | (value & 0x0f);
        if (width8)
            qt_memfill<quint8>(dest + doAlign, c, width8);
        if (doTail) {
            quint8 *d = dest + doAlign + width8;
            *d = (*d & 0x0f) | (c & 0xf0);
        }
        dest += stride;
    }
}

static void solidFill_gray4r(QScreen *screen, const QColor &color,
                            const QRegion &region)
{
    quint8 *dest = reinterpret_cast<quint8*>(screen->base());
    const quint8 c = qGray(color.rgba()) >> 4;
    const quint8 c8 = (c << 4) | c;

    const int stride = screen->linestep();
    const QVector<QRect> rects = region.rects();

    for (int i = 0; i < rects.size(); ++i) {
        const QRect r = rects.at(i);
        qt_rectfill_gray4r(dest, c8, r.x(), r.y(), r.width(), r.height(),
                          stride);
    }
}

/*!
    Fills the given \a region of the screen with the specified \a
    color.

    This function is called from the exposeRegion() function; it is
    not intended to be called explicitly.

    Reimplement this function to make use of \l{Adding an Accelerated
    Graphics Driver to Qt for Embedded Linux}{accelerated hardware}. Note that
    this function must be reimplemented if the framebuffer format is
    not supported by \l{Qt for Embedded Linux} (See the
    \l{Qt for Embedded Linux Display Management}{Display Management}
    documentation for more details).

    \sa exposeRegion(), blit(), blank()
*/
// the base class implementation works in device coordinates, so that transformed drivers can use it
void QEInkFbScreen::solidFill(const QColor &color, const QRegion &region)
{
    QWSDisplay::grab();
    solidFill_gray4r(this, color,
                     region.translated(-offset()) & QRect(0, 0, dw, dh));
    QWSDisplay::ungrab();
}


template <typename DST, typename SRC>
static void blit_template(QScreen *screen, const QImage &image,
                          const QPoint &topLeft, const QRegion &region)
{
    DST *dest = reinterpret_cast<DST*>(screen->base());
    const int screenStride = screen->linestep();
    const int imageStride = image.bytesPerLine();

    if (region.rectCount() == 1) {
        const QRect r = region.boundingRect();
        const SRC *src = reinterpret_cast<const SRC*>(image.scanLine(r.y()))
                         + r.x();
        qt_rectconvert<DST, SRC>(dest, src,
                                 r.x() + topLeft.x(), r.y() + topLeft.y(),
                                 r.width(), r.height(),
                                 screenStride, imageStride);
    } else {
        const QVector<QRect> rects = region.rects();

        for (int i = 0; i < rects.size(); ++i) {
            const QRect r = rects.at(i);
            const SRC *src = reinterpret_cast<const SRC*>(image.scanLine(r.y()))
                             + r.x();
            qt_rectconvert<DST, SRC>(dest, src,
                                     r.x() + topLeft.x(), r.y() + topLeft.y(),
                                     r.width(), r.height(),
                                     screenStride, imageStride);
        }
    }
}


struct qgray4r { quint8 dummy; } Q_PACKED;

template <typename SRC>
Q_STATIC_TEMPLATE_FUNCTION inline quint8 qt_convertToGray4r(SRC color);

template <>
inline quint8 qt_convertToGray4r(quint32 color)
{
    return (qGray(color) >> 4) ^ 15;
}

template <>
inline quint8 qt_convertToGray4r(quint16 color)
{
    const int r = (color & 0xf800) >> 11;
    const int g = (color & 0x07e0) >> 6; // only keep 5 bit
    const int b = (color & 0x001f);
    return ((r * 11 + g * 16 + b * 5) >> 6) ^ 15;
}

template <>
inline quint8 qt_convertToGray4r(qrgb444 color)
{
    return qt_convertToGray4r(quint32(color));
}

template <>
inline quint8 qt_convertToGray4r(qargb4444 color)
{
    return qt_convertToGray4r(quint32(color));
}

template <typename SRC>
Q_STATIC_TEMPLATE_FUNCTION inline void qt_rectconvert_gray4r(qgray4r *dest4, const SRC *src,
                                        int x, int y, int width, int height,
                                        int dstStride, int srcStride)
{
    const int pixelsPerByte = 2;
    quint8 *dest8 = reinterpret_cast<quint8*>(dest4)
                    + y * dstStride + x / pixelsPerByte;
    const int doAlign = x & 1;
    const int doTail = (width - doAlign) & 1;
    const int width8 = (width - doAlign) / pixelsPerByte;
    const int count8 = (width8 + 3) / 4;

    srcStride = srcStride / sizeof(SRC) - width;
    dstStride -= (width8 + doAlign);

    for (int i = 0; i < height; ++i) {
        if (doAlign) {
            *dest8 = (*dest8 & 0xf0) | qt_convertToGray4r<SRC>(*src++);
            ++dest8;
        }
        if (count8) {
            int n = count8;
            switch (width8 & 0x03) // duff's device
            {
            case 0: do { *dest8++ = qt_convertToGray4r<SRC>(src[0]) << 4
                                    | qt_convertToGray4r<SRC>(src[1]);
                         src += 2;
            case 3:      *dest8++ = qt_convertToGray4r<SRC>(src[0]) << 4
                                    | qt_convertToGray4r<SRC>(src[1]);
                         src += 2;
            case 2:      *dest8++ = qt_convertToGray4r<SRC>(src[0]) << 4
                                    | qt_convertToGray4r<SRC>(src[1]);
                         src += 2;
            case 1:      *dest8++ = qt_convertToGray4r<SRC>(src[0]) << 4
                                    | qt_convertToGray4r<SRC>(src[1]);
                         src += 2;
            } while (--n > 0);
            }
        }

        if (doTail)
            *dest8 = qt_convertToGray4r<SRC>(*src++) << 4 | (*dest8 & 0x0f);

        dest8 += dstStride;
        src += srcStride;
    }
}

template <>
void qt_rectconvert(qgray4r *dest, const quint32 *src,
                    int x, int y, int width, int height,
                    int dstStride, int srcStride)
{
    qt_rectconvert_gray4r<quint32>(dest, src, x, y, width, height,
                                  dstStride, srcStride);
}

template <>
void qt_rectconvert(qgray4r *dest, const quint16 *src,
                    int x, int y, int width, int height,
                    int dstStride, int srcStride)
{
    qt_rectconvert_gray4r<quint16>(dest, src, x, y, width, height,
                                  dstStride, srcStride);
}

template <>
void qt_rectconvert(qgray4r *dest, const qrgb444 *src,
                    int x, int y, int width, int height,
                    int dstStride, int srcStride)
{
    qt_rectconvert_gray4r<qrgb444>(dest, src, x, y, width, height,
                                  dstStride, srcStride);
}

template <>
void qt_rectconvert(qgray4r *dest, const qargb4444 *src,
                    int x, int y, int width, int height,
                    int dstStride, int srcStride)
{
    qt_rectconvert_gray4r<qargb4444>(dest, src, x, y, width, height,
                                    dstStride, srcStride);
}

static void blit_4r(QScreen *screen, const QImage &image,
                   const QPoint &topLeft, const QRegion &region)
{
    switch (image.format()) {
    case QImage::Format_ARGB32_Premultiplied:
        blit_template<qgray4r, quint32>(screen, image, topLeft, region);
        return;
    case QImage::Format_RGB16:
        blit_template<qgray4r, quint16>(screen, image, topLeft, region);
        return;
    case QImage::Format_RGB444:
        blit_template<qgray4r, qrgb444>(screen, image, topLeft, region);
        return;
    case QImage::Format_ARGB4444_Premultiplied:
        blit_template<qgray4r, qargb4444>(screen, image, topLeft, region);
        return;
    default:
        qCritical("blit_4(): Image format %d not supported!", image.format());
    }
}

/*!
    \fn void QScreen::blit(const QImage &image, const QPoint &topLeft, const QRegion &region)

    Copies the given \a region in the given \a image to the point
    specified by \a topLeft using device coordinates.

    This function is called from the exposeRegion() function; it is
    not intended to be called explicitly.

    Reimplement this function to make use of \l{Adding an Accelerated
    Graphics Driver to Qt for Embedded Linux}{accelerated hardware}. Note that
    this function must be reimplemented if the framebuffer format is
    not supported by \l{Qt for Embedded Linux} (See the
    \l{Qt for Embedded Linux Display Management}{Display Management}
    documentation for more details).

    \sa exposeRegion(), solidFill(), blank()
*/
void QEInkFbScreen::blit(const QImage &img, const QPoint &topLeft, const QRegion &reg)
{
    const QRect bound = (region() & QRect(topLeft, img.size())).boundingRect();
    QWSDisplay::grab();
    blit_4r(this, img, topLeft - offset(),
            (reg & bound).translated(-topLeft));
    QWSDisplay::ungrab();
}

QT_END_NAMESPACE

#endif // QT_NO_QWS_EINKFB
