#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, drMerry'
__docformat__ = 'restructuredtext en'

import os, time, shutil, traceback, re, posixpath, urllib
from PIL import Image
from calibre.gui2 import info_dialog, error_dialog
import calibre_plugins.epub_checker.config as cfg

from calibre import guess_type
from calibre.ebooks.epub.fix import InvalidEpub
from calibre.ebooks.epub.fix.container import OCF_NS, OPF_NS
from calibre.utils.zipfile import ZipFile
from calibre_plugins.epub_checker.check_base import BaseCheck
from calibre_plugins.epub_checker.dialogs import CoverOptionsDialog, ResultsSummaryDialog
from calibre.gui2.dialogs.message_box import MessageBox

class SelectedFileAction(BaseCheck):
    '''
    All checks related to selected files.
    '''
    def perform_check(self, menu_key, pathtoselectedbook):
        if menu_key == 'check_epub_numberoffiles':
            self.check_number_of_files(pathtoselectedbook)
        if menu_key == 'check_image_files':
            self.check_image_files(pathtoselectedbook)
        else:
            return error_dialog(self.gui, _('Epub Checker failed'),
                                _('Unknown menu key for %s in Epub-Checker (selectedfileaction)')%(menu_key),
                                show=False, show_copy_button=False)

    def check_number_of_files(self, path_to_book):
        numberoffiles = 0
        try:
            with ZipFile(path_to_book, 'r') as zf:
                contents = zf.namelist()
                for resource_name in contents:
                    numberoffiles += 1
        except InvalidEpub:
            self.log.error('Invalid epub: ', path_to_book)
            sd = ResultsSummaryDialog(self.gui, 'Cannot modify ePub', ('Invalid Epub-path: %s')%(path_to_book), self.log)
            sd.exec_()
            return False
            #return error_dialog(self.gui, _('Cannot modify ePub'),
            #                _('Invalid Epub-path: %s')%(path_to_book),
            #                show=True)
        except:
            self.log.error('ERROR parsing book: ', path_to_book)
            self.log(traceback.format_exc())
            sd = ResultsSummaryDialog(self.gui, 'Epub Checker failed', ('ERROR parsing book: %s')%(path_to_book), self.log)
            sd.exec_()
            return False
#            return error_dialog(self.gui, _('Epub Checker failed'),
#                                _('ERROR parsing book: %s')%(path_to_book),
#                                show=True, show_copy_button=True)
        sd = ResultsSummaryDialog(self.gui, 'Completed sucsesfull', ('Epub-file on:\n%s.\n\nHas %s files.')%(path_to_book, numberoffiles), self.log)
        sd.exec_()
        return True
#        return info_dialog(self.gui, _('Test Complete'),
#                     _('Epub on:\n%s.\n\nHas %s files.')%(path_to_book, numberoffiles),
#                    show=True, show_copy_button=False)


    def check_image_files(self, path_to_book):
        marked_names = ''
        numbermarked = 0
        numberoffiles = 0
        unmarked_names = ''
        c = cfg.plugin_prefs[cfg.STORE_NAME]
        custom_extensions = c.get(cfg.KEY_CUSTOM_IMAGE_EXTENSION, cfg.DEFAULT_STORE_VALUES[cfg.KEY_CUSTOM_IMAGE_EXTENSION])
        self.log.debug('Testing extensions: ', custom_extensions)
        comma_sepparated_extension_list = ''
        regex_extension_list = ''
        incremental_item = False
        for custom_extension in custom_extensions:
            if incremental_item:
                comma_sepparated_extension_list += ', ' + custom_extension.lower()
                regex_extension_list = regex_extension_list + '|' + custom_extension.lower()
            else:
                comma_sepparated_extension_list = custom_extension.lower()
                regex_extension_list = '^.*(' + custom_extension.lower()
                incremental_item = True

        regex_extension_list += ')$'
        cust_ex_reg = re.compile(regex_extension_list)
        self.log.debug('Testing regex: ', regex_extension_list)
        self.log.debug('Testing list: ', comma_sepparated_extension_list)
        self.log.info('Testing book: ', path_to_book)
        try:
            with ZipFile(path_to_book, 'r') as zf:
                contents = zf.namelist()
                for resource_name in contents:
                    numberoffiles += 1
                    found_match = cust_ex_reg.search(resource_name.lower())
                    if found_match is None:
                        self.log.info('skip: ', resource_name)
                        unmarked_names += '\n- '
                        unmarked_names += resource_name
                    else:
                        numbermarked += 1
                        self.log.debug('FOUND: ', resource_name)
                        marked_names += '\n- ' + resource_name
            returnmessage = 'files: ' + str(numberoffiles) + '\nMarked: ' + str(numbermarked) + '\nUnmarked: ' + str(numberoffiles - numbermarked)
            self.log.info('files: ' + str(numberoffiles) + '\nMarked: ' + str(numbermarked) + '\nUnmarked: ' + str(numberoffiles - numbermarked))
            self.log.debug('\n\nNames:' + marked_names)
            self.log.error('\n\nNonimages:' + unmarked_names)
            sd = ResultsSummaryDialog(self.gui, 'Matches', returnmessage, self.log)
            sd.exec_()
            return True
        except InvalidEpub:
            returnmessage = 'files: ' + nstr(numberoffiles) + '\nPath: ' + path_to_book
            self.log.error('Invalid epub: ', path_to_book)
            sd = ResultsSummaryDialog(self.gui, 'Invalid Ebook', returnmessage, self.log)
            sd.exec_()
            return False
        except:
            self.log.error('ERROR parsing book: ', path_to_book)
            self.log(traceback.format_exc())
            returnmessage = 'files: ' + str(numberoffiles) + '\nPath: ' + path_to_book
            sd = ResultsSummaryDialog(self.gui, 'ERROR', returnmessage, self.log)
            sd.exec_()
            return False    



class FileSizeChecks(BaseCheck):
    '''
    All checks related to working with file sizes.
    '''
    def perform_check(self, menu_key):
        if menu_key == 'check_filesize':
            self.check_covers()
        else:
            return error_dialog(self.gui, _('Epub Checker failed'),
                                _('Unknown menu key for %s in Epub-Checker (filesizechecks)')%(menu_key),
                                show=True, show_copy_button=False)


    def check_covers(self):
        d = CoverOptionsDialog(self.gui)
        d.exec_()
        if d.result() != d.Accepted:
            return

        if d.opt_no_cover.isChecked():
            # This is a simple search
            self.gui.search.set_search_string('cover:False')
            return

        # The other options require iterating through the library data
        check_type = d.check_type
        if d.opt_file_size.isChecked():
            is_file_size_check = True
            min_file_size = d.file_size * 1024
        else:
            is_file_size_check = False
            min_image_width = d.image_width
            min_image_height = d.image_height

        def evaluate_book(book_id, db):
            if not db.has_cover(book_id):
                return False
            cover_path = os.path.join(db.library_path, db.path(book_id, index_is_id=True), 'cover.jpg')
            if not os.path.exists(cover_path)or not os.access(cover_path, os.R_OK):
                self.log.error('Unable to access cover: ', cover_path)
                return False

            mark_book = False
            if is_file_size_check:
                cover_size = os.path.getsize(cover_path)
                if check_type == 'less than' and cover_size < min_file_size:
                    mark_book = True
                elif check_type == 'greater than' and cover_size > min_file_size:
                    mark_book = True
            else:
                try:
                    im = Image.open(cover_path)
                except IOError:
                    self.log('Failed to identify cover:', cover_path)
                else:
                    (cover_width, cover_height) = im.size
                    if check_type == 'less than':
                        if cover_width < min_image_width:
                            mark_book = True
                        elif cover_height < min_image_height:
                            mark_book = True
                    elif check_type == 'greater than':
                        if cover_width > min_image_width and min_image_width != 0:
                            mark_book = True
                        elif cover_height > min_image_height and min_image_height != 0:
                            mark_book = True
                    elif check_type == 'equal to':
                        if min_image_height == 0:
                            if cover_width == min_image_width:
                                mark_book = True
                        elif min_image_width == 0:
                            if cover_height == min_image_height:
                                mark_book = True
                        elif cover_width == min_image_width and cover_height == min_image_height:
                            mark_book = True
                    elif check_type == 'not equal to':
                        if min_image_width != 0:
                            if cover_width != min_image_width:
                                mark_book = True
                        if min_image_height != 0:
                            if cover_height != min_image_height:
                                mark_book = True
            return mark_book

        total_count, result_ids, cancelled = self.check_all_files(evaluate_book,
                                                                  marked_text='cover_check',
                                                                  status_msg_type='books for covers')
        if not cancelled:
            msg = 'Checked %d books, found %d cover matches' % (total_count, len(result_ids))
            self.gui.status_bar.showMessage(msg)
            if len(result_ids) == 0:
                d = ResultsSummaryDialog(self.gui, 'No Matches', 'No matches found', self.log)
                d.exec_()
            elif self.log.plain_text:
                d = ResultsSummaryDialog(self.gui, 'Epub Checker',
                                         '%d matches found, see log for errors'%total_count,
                                         self.log)
                d.exec_()

