#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import os
from functools import partial
from PyQt4.Qt import QMenu, QToolButton, QUrl
from calibre.gui2 import open_url
from calibre.gui2.actions import InterfaceAction
from calibre.utils.config import config_dir

import calibre_plugins.quality_check.config as cfg
from calibre_plugins.quality_check.common_utils import set_plugin_icon_resources, get_icon, \
                                                     create_menu_item
from calibre_plugins.quality_check.check_covers import CoverCheck
from calibre_plugins.quality_check.check_epub import EpubCheck
from calibre_plugins.quality_check.check_metadata import MetadataCheck

DEFAULT_ICON = 'images/quality_check.png'

class QualityCheckAction(InterfaceAction):

    name = 'Quality Check'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Quality Check', None, None, None)
    popup_type = QToolButton.InstantPopup
    action_type = 'current'

    def genesis(self):
        self.menu = QMenu(self.gui)

        # Build the list of plugin icons from the configuration menus
        plugin_icons = set([DEFAULT_ICON])
        for menu_config in cfg.PLUGIN_MENUS.values():
            image_name = menu_config['image']
            if image_name.startswith('images/'):
                plugin_icons.add(image_name)

        icon_resources = self.load_resources(list(plugin_icons))
        set_plugin_icon_resources(self.name, icon_resources)

        self.rebuild_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(DEFAULT_ICON))

    def rebuild_menus(self):
        m = self.menu
        m.clear()
        c = cfg.plugin_prefs[cfg.STORE_NAME]
        hidden_menus = c.get(cfg.KEY_HIDDEN_MENUS, [])
        last_sub_menu = None
        last_group = 0
        parent_menu = m
        for key, menu_config in cfg.PLUGIN_MENUS.iteritems():
            if key in hidden_menus:
                continue
            sub_menu = menu_config['sub_menu']
            if sub_menu:
                if sub_menu != last_sub_menu:
                    parent_menu = m.addMenu(sub_menu)
                    last_sub_menu = sub_menu
            elif last_sub_menu:
                parent_menu = m
                last_sub_menu = None
            group = menu_config['group']
            if group != last_group:
                parent_menu.addSeparator()
            last_group = group
            create_menu_item(self, parent_menu, menu_config['name'], image=menu_config['image'],
                             tooltip=menu_config['tooltip'],
                             triggered=partial(self.perform_check, key, menu_config['cat']))
        m.addSeparator()
        create_menu_item(self, m, _('&Customize plugin')+'...', 'config.png',
                         triggered=self.show_configuration)
        create_menu_item(self, m, _('&Help'), 'help.png',
                         triggered=self.show_help)

    def perform_check(self, menu_key, menu_cat):
        if menu_cat == 'epub':
            check = EpubCheck(self.gui)
        elif menu_cat == 'covers':
            check = CoverCheck(self.gui)
        else:
            check = MetadataCheck(self.gui)
        check.perform_check(menu_key)

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def show_help(self):
        # Extract on demand the help file resource
        def get_help_file_resource():
            # We will write the help file out every time, in case the user upgrades the plugin zip
            # and there is a later help file contained within it.
            HELP_FILE = 'Quality Check Help.htm'
            file_path = os.path.join(config_dir, 'plugins', HELP_FILE)
            file_data = self.load_resources(HELP_FILE)[HELP_FILE]
            with open(file_path,'w') as f:
                f.write(file_data)
            return file_path
        url = 'file:///' + get_help_file_resource()
        open_url(QUrl(url))
