#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from PyQt4.Qt import (QVBoxLayout, QLabel, QRadioButton, QDialogButtonBox,
                      QGroupBox, QGridLayout, QComboBox, QProgressDialog,
                      QString, QTimer, QIcon)
from PyQt4 import QtGui
from calibre.gui2 import gprefs
from calibre.gui2.dialogs.message_box import MessageBox, ViewLog

from calibre_plugins.quality_check.common_utils import SizePersistedDialog, \
                                                       ImageTitleLayout


class QualityProgressDialog(QProgressDialog):

    def __init__(self, gui, book_ids, callback_fn, db, status_msg_type='books'):
        self.total_count = len(book_ids)
        QProgressDialog.__init__(self, '', QString('Cancel'), 0, self.total_count, gui)
        self.setWindowTitle('Checking %d %s...' % (self.total_count, status_msg_type))
        self.setMinimumWidth(500)
        self.book_ids, self.callback_fn, self.db = book_ids, callback_fn, db
        self.gui = gui
        self.i, self.result_ids = 0, []
        QTimer.singleShot(0, self.do_book_check)
        self.exec_()

    def do_book_check(self):
        if self.wasCanceled():
            return self.do_close()
        if self.i >= self.total_count:
            return self.do_close()
        book_id = self.book_ids[self.i]
        self.i += 1

        dtitle = self.db.title(book_id, index_is_id=True)
        self.setLabelText('Checking: '+dtitle)
        if self.callback_fn(book_id, self.db):
            self.result_ids.append(book_id)
        self.setValue(self.i)

        QTimer.singleShot(0, self.do_book_check)

    def do_close(self):
        self.hide()
        self.gui = None


class CompareTypeComboBox(QComboBox):

    def __init__(self, parent, allow_equality=True):
        QComboBox.__init__(self, parent)
        self.addItems(['less than', 'greater than'])
        if allow_equality:
            self.addItems(['equal to', 'not equal to'])

    def select_text(self, selected_text):
        idx = self.findText(selected_text)
        if idx != -1:
            self.setCurrentIndex(idx)
        else:
            self.setCurrentIndex(0)


class CoverOptionsDialog(SizePersistedDialog):

    def __init__(self, parent):
        SizePersistedDialog.__init__(self, parent, 'quality check plugin:cover options dialog')

        self.initialize_controls()

        # Set some default values from last time dialog was used.
        last_opt = gprefs.get(self.unique_pref_name+':last_opt', 0)
        if last_opt == 0:
            self.opt_no_cover.setChecked(True)
        elif last_opt == 1:
            self.opt_file_size.setChecked(True)
        else:
            self.opt_dimensions.setChecked(True)
        size_check_type = gprefs.get(self.unique_pref_name+':last_size_check_type', 'less than')
        self.file_size_check_type.select_text(size_check_type)
        dimensions_check_type = gprefs.get(self.unique_pref_name+':last_dimensions_check_type', 'less than')
        self.dimensions_check_type.select_text(dimensions_check_type)
        last_size = gprefs.get(self.unique_pref_name+':last_size', 20)
        self.file_size_spin.setProperty('value', last_size)
        last_width = gprefs.get(self.unique_pref_name+':last_width', 300)
        self.image_width_spin.setProperty('value', last_width)
        last_height = gprefs.get(self.unique_pref_name+':last_height', 400)
        self.image_height_spin.setProperty('value', last_height)

        # Cause our dialog size to be restored from prefs or created on first usage
        self.resize_dialog()

    def initialize_controls(self):
        self.setWindowTitle('Quality Check')
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        title_layout = ImageTitleLayout(self, 'images/quality_check.png', 'Cover Search Options')
        layout.addLayout(title_layout)

        options_group = QGroupBox('Search library for books where', self)
        layout.addWidget(options_group)
        options_layout = QGridLayout()
        options_group.setLayout(options_layout)
        self.opt_file_size = QRadioButton('Cover file size is', self)
        options_layout.addWidget(self.opt_file_size, 0, 0, 1, 1)
        self.file_size_check_type = CompareTypeComboBox(self, allow_equality=False)
        options_layout.addWidget(self.file_size_check_type, 0, 1, 1, 1)
        self.file_size_spin = QtGui.QSpinBox(self)
        self.file_size_spin.setMinimum(1)
        self.file_size_spin.setMaximum(99000000)
        options_layout.addWidget(self.file_size_spin, 0, 2, 1, 1)
        options_layout.addWidget(QLabel('kb'), 0, 3, 1, 1)

        self.opt_dimensions = QRadioButton('Cover dimensions are', self)
        options_layout.addWidget(self.opt_dimensions, 1, 0, 1, 1)
        self.dimensions_check_type = CompareTypeComboBox(self)
        options_layout.addWidget(self.dimensions_check_type, 1, 1, 1, 1)
        self.image_width_spin = QtGui.QSpinBox(self)
        self.image_width_spin.setMinimum(0)
        self.image_width_spin.setMaximum(99000000)
        options_layout.addWidget(self.image_width_spin, 1, 2, 1, 1)
        options_layout.addWidget(QLabel('width'), 1, 3, 1, 1)
        self.image_height_spin = QtGui.QSpinBox(self)
        self.image_height_spin.setMinimum(0)
        self.image_height_spin.setMaximum(99000000)
        options_layout.addWidget(self.image_height_spin, 2, 2, 1, 1)
        options_layout.addWidget(QLabel('height'), 2, 3, 1, 1)

        self.opt_no_cover = QRadioButton('No cover', self)
        options_layout.addWidget(self.opt_no_cover, 3, 0, 1, 1)

        # Dialog buttons
        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.ok_clicked)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

    def ok_clicked(self):
        if self.opt_no_cover.isChecked():
            gprefs[self.unique_pref_name+':last_opt'] = 0
        elif self.opt_file_size.isChecked():
            gprefs[self.unique_pref_name+':last_opt'] = 1
        else:
            gprefs[self.unique_pref_name+':last_opt'] = 2
        gprefs[self.unique_pref_name+':last_size_check_type'] = \
                            unicode(self.file_size_check_type.currentText()).strip()
        gprefs[self.unique_pref_name+':last_dimensions_check_type'] = \
                            unicode(self.dimensions_check_type.currentText()).strip()
        gprefs[self.unique_pref_name+':last_size'] = self.file_size
        gprefs[self.unique_pref_name+':last_width'] = self.image_width
        gprefs[self.unique_pref_name+':last_height'] = self.image_height
        self.accept()

    @property
    def check_type(self):
        if self.opt_file_size.isChecked():
            return unicode(self.file_size_check_type.currentText()).strip()
        elif self.opt_dimensions.isChecked():
            return unicode(self.dimensions_check_type.currentText()).strip()

    @property
    def file_size(self):
        return int(unicode(self.file_size_spin.value()))

    @property
    def image_width(self):
        return int(unicode(self.image_width_spin.value()))

    @property
    def image_height(self):
        return int(unicode(self.image_height_spin.value()))


class ResultsSummaryDialog(MessageBox): # {{{

    def __init__(self, parent, title, msg, log=None, det_msg=''):
        '''
        A modal popup that summarises the result of Quality Check with
        opportunity to review the log.

        :param log: An HTML or plain text log
        :param title: The title for this popup
        :param msg: The msg to display
        :param det_msg: Detailed message
        '''
        MessageBox.__init__(self, MessageBox.INFO, title, msg,
                det_msg=det_msg, show_copy_button=False,
                parent=parent)
        self.log = log
        self.vlb = self.bb.addButton(_('View log'), self.bb.ActionRole)
        self.vlb.setIcon(QIcon(I('debug.png')))
        self.vlb.clicked.connect(self.show_log)
        self.det_msg_toggle.setVisible(bool(det_msg))
        self.vlb.setVisible(bool(log.plain_text))

    def show_log(self):
        self.log_viewer = ViewLog('Quality Check log', self.log.html,
                parent=self)
