#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import OrderedDict
from PyQt4 import QtGui
from PyQt4.Qt import QWidget, QVBoxLayout, QLabel, QLineEdit, QVariant, \
                     QGroupBox, QGridLayout, QListWidget, QListWidgetItem, \
                     QAbstractItemView, Qt
from calibre.gui2.complete import MultiCompleteLineEdit
from calibre.utils.config import JSONConfig

from calibre_plugins.quality_check.common_utils import get_icon

STORE_NAME = 'options'
KEY_MAX_TAGS = 'maxTags'
KEY_MAX_TAG_EXCLUSIONS = 'maxTagExclusions'
KEY_SEARCH_AUTHOR_COMMAS = 'searchAuthorCommas'
KEY_SEARCH_AUTHOR_NOCOMMAS = 'searchAuthorNoCommas'
KEY_SEARCH_TITLE_SERIES = 'searchTitleSeries'
KEY_HIDDEN_MENUS = 'hiddenMenus'
KEY_CUSTOM_EXTENSION = 'Extensions'

DEFAULT_STORE_VALUES = {
                           KEY_MAX_TAGS: 5,
                           KEY_MAX_TAG_EXCLUSIONS: [],
                           KEY_SEARCH_AUTHOR_COMMAS: 'authors:,',
                           KEY_SEARCH_AUTHOR_NOCOMMAS: 'not authors:,',
                           KEY_SEARCH_TITLE_SERIES: 'title:- or title:"~[0-9]"',
                           KEY_HIDDEN_MENUS: [],
                           KEY_CUSTOM_EXTENSION: []
                          }

PLUGIN_MENUS = OrderedDict([
       ('check_epub_jacket',        {'name': 'Check having any jacket',       'cat':'epub',     'sub_menu': 'Check ePub', 'group': 0, 'image': 'images/check_epub_jacket.png',         'tooltip':'Check for ePub formats which have any Calibre jacket'}),
       ('check_epub_legacy_jacket', {'name': 'Check having legacy jacket',    'cat':'epub',     'sub_menu': 'Check ePub', 'group': 0, 'image': 'images/check_epub_jacket_legacy.png',  'tooltip':'Check for ePub formats which have a Calibre jacket from versions prior to 0.6.51'}),
       ('check_epub_multi_jacket',  {'name': 'Check having multiple jackets', 'cat':'epub',     'sub_menu': 'Check ePub', 'group': 0, 'image': 'images/check_epub_jacket_multi.png',   'tooltip':'Check for ePub formats which have multiple jackets'}),
       ('check_epub_no_jacket',     {'name': 'Check missing jacket',          'cat':'epub',     'sub_menu': 'Check ePub', 'group': 0, 'image': 'images/check_epub_jacket_missing.png', 'tooltip':'Check for ePub formats which do not have a jacket'}),
       ('check_epub_no_container',  {'name': 'Check missing container.xml',   'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_no_container.png',   'tooltip':'Check for ePub formats with a missing container.xml indicating an invalid ePub'}),
       ('check_epub_namespaces',    {'name': 'Check invalid namespaces',      'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_namespaces.png',     'tooltip':'Check for ePub formats with invalid namespaces in the container xml or opf manifest'}),
       ('check_epub_files_missing', {'name': 'Check manifest files missing',  'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_files_missing.png',  'tooltip':'Check for ePub formats with files missing that are listed in their opf manifest'}),
       ('check_epub_unman_files',   {'name': 'Check unmanifested files',      'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_unmanifested.png',   'tooltip':'Check for ePub formats with files that are not listed in their opf manifest'}),
       ('check_epub_itunes',        {'name': 'Check iTunes plist files',      'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_itunes.png',         'tooltip':'Check for ePub formats with a iTunesMetadata.plist file'}),
       ('check_epub_bookmark',      {'name': 'Check Calibre bookmark files',  'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_bookmarks.png',      'tooltip':'Check for ePub formats with a Calibre bookmarks file'}),
       ('check_epub_custom_files',  {'name': 'Check books with custom files', 'cat':'epub',     'sub_menu': 'Check ePub', 'group': 1, 'image': 'images/check_epub_custom.png',         'tooltip':'Check for ePub formats with custom files'}),
       ('check_epub_xpgt',          {'name': 'Check Adobe .xpgt margins',     'cat':'epub',     'sub_menu': 'Check ePub', 'group': 2, 'image': 'images/check_epub_xpgt_margin.png',    'tooltip':'Check for ePub formats with an xpgt file with non-zero margins'}),
       ('check_epub_toc_size',      {'name': 'Check TOC with < 3 entries',    'cat':'epub',     'sub_menu': 'Check ePub', 'group': 2, 'image': 'images/check_epub_toc.png',            'tooltip':'Check for ePub formats with a metadata TOC with less than 3 entries'}),
       ('check_epub_fonts',         {'name': 'Check embedded fonts',          'cat':'epub',     'sub_menu': 'Check ePub', 'group': 2, 'image': 'images/check_epub_fonts.png',          'tooltip':'Check for ePub formats with embedded fonts'}),
       ('check_epub_drm',           {'name': 'Check DRM',                     'cat':'epub',     'sub_menu': 'Check ePub', 'group': 2, 'image': 'images/check_epub_drm.png',            'tooltip':'Check for ePub formats with DRM encryption xml files'}),
       ('check_epub_cover',         {'name': 'Check inline Calibre cover',    'cat':'epub',     'sub_menu': 'Check ePub', 'group': 3, 'image': 'images/check_epub_cover.png',          'tooltip':'Check for ePub formats with a Calibre cover page embedded inline'}),
       ('check_epub_no_cover',      {'name': 'Check no inline Calibre cover', 'cat':'epub',     'sub_menu': 'Check ePub', 'group': 3, 'image': 'images/check_epub_no_cover.png',       'tooltip':'Check for ePub formats with no Calibre cover page embedded inline'}),
       ('check_epub_converted',     {'name': 'Check Calibre conversion',      'cat':'epub',     'sub_menu': 'Check ePub', 'group': 4, 'image': 'images/check_epub_converted.png',      'tooltip':'Check for ePub formats that have been converted by Calibre'}),
       ('check_epub_not_converted', {'name': 'Check not Calibre conversion',  'cat':'epub',     'sub_menu': 'Check ePub', 'group': 4, 'image': 'images/check_epub_not_converted.png',  'tooltip':'Check for ePub formats that have not been converted by Calibre'}),
       ('check_covers',             {'name': 'Check covers...',               'cat':'covers',   'sub_menu': '',           'group': 0, 'image': 'images/check_cover.png',               'tooltip':'Find books with book covers matching your criteria'}),
       ('check_title_sort',         {'name': 'Check title sort',              'cat':'metadata', 'sub_menu': '',           'group': 1, 'image': 'images/check_book.png',                'tooltip':'Find books with an invalid title sort value'}),
       ('check_author_sort',        {'name': 'Check author sort',             'cat':'metadata', 'sub_menu': '',           'group': 1, 'image': 'images/check_book.png',                'tooltip':'Find books with an invalid author sort value'}),
       ('check_isbn',               {'name': 'Check ISBN',                    'cat':'metadata', 'sub_menu': '',           'group': 1, 'image': 'images/check_book.png',                'tooltip':'Find books with an invalid ISBN'}),
       ('check_pubdate',            {'name': 'Check pubdate',                 'cat':'metadata', 'sub_menu': '',           'group': 1, 'image': 'images/check_book.png',                'tooltip':'Find books with an invalid pubdate where it is set to the timestamp date'}),
       ('check_dup_isbn',           {'name': 'Check duplicate ISBN',          'cat':'metadata', 'sub_menu': '',           'group': 2, 'image': 'images/check_dup_isbn.png',            'tooltip':'Find books that have duplicate ISBN values'}),
       ('check_dup_series',         {'name': 'Check duplicate series',        'cat':'metadata', 'sub_menu': '',           'group': 2, 'image': 'series.png',                           'tooltip':'Find books that have duplicate series values'}),
       ('check_excess_tags',        {'name': 'Check excess tags',             'cat':'metadata', 'sub_menu': '',           'group': 3, 'image': 'tags.png',                             'tooltip':'Find books with an excess number of tags'}),
       ('check_html_comments',      {'name': 'Check html comments',           'cat':'metadata', 'sub_menu': '',           'group': 3, 'image': 'images/check_html.png',                'tooltip':'Find books which have comments html with style formatting embedded'}),
       ('check_no_html_comments',   {'name': 'Check no html comments',        'cat':'metadata', 'sub_menu': '',           'group': 3, 'image': 'images/check_nohtml.png',              'tooltip':'Find books which have comments with no html tags at all'}),
       ('check_authors_commas',     {'name': 'Check authors with commas',     'cat':'metadata', 'sub_menu': '',           'group': 4, 'image': 'images/check_comma.png',               'tooltip':'Find authors with commas in their name'}),
       ('check_authors_no_commas',  {'name': 'Check authors missing commas',  'cat':'metadata', 'sub_menu': '',           'group': 4, 'image': 'images/check_nocomma.png',             'tooltip':'Find authors with no commas in their name'}),
       ('check_titles_series',      {'name': 'Check titles with series',      'cat':'metadata', 'sub_menu': '',           'group': 4, 'image': 'images/check_titleseries.png',         'tooltip':'Find titles with possible series info in their name'}),
       ('check_title_case',         {'name': 'Check title case',              'cat':'metadata', 'sub_menu': '',           'group': 4, 'image': 'images/check_titlecase.png',           'tooltip':'Find books which are candidates to apply the titlecase function to'}),
       ('check_fix_book_size',      {'name': 'Check and repair book sizes',   'cat':'metadata', 'sub_menu': '',           'group': 5, 'image': 'images/check_file_size.png',           'tooltip':'Check and update file sizes for your books'}),
       ('cleanup_opf_files',        {'name': 'Cleanup .opf files/folders',    'cat':'metadata', 'sub_menu': '',           'group': 5, 'image': 'images/cleanup_folders.png',           'tooltip':'Delete orphaned opf/jpg files and remove empty folders'}),
       ])

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Quality Check')

# Set defaults
plugin_prefs.defaults[STORE_NAME] = DEFAULT_STORE_VALUES

class VisibleMenuListWidget(QListWidget):
    def __init__(self, parent=None):
        QListWidget.__init__(self, parent)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setSelectionMode(QAbstractItemView.SingleSelection)
        self.populate()

    def populate(self):
        self.clear()
        hidden_prefs = plugin_prefs[STORE_NAME].get(KEY_HIDDEN_MENUS, [])
        for key, value in PLUGIN_MENUS.iteritems():
            name = value['name']
            sub_menu = value['sub_menu']
            if sub_menu:
                name = sub_menu + ' / ' + name
            item = QListWidgetItem(name, self)
            item.setIcon(get_icon(value['image']))
            item.setData(Qt.UserRole, QVariant(key))
            if key in hidden_prefs:
                item.setCheckState(Qt.Unchecked)
            else:
                item.setCheckState(Qt.Checked)
            self.addItem(item)

    def get_hidden_menus(self):
        hidden_menus = []
        for x in xrange(self.count()):
            item = self.item(x)
            if item.checkState() == Qt.Unchecked:
                key = unicode(item.data(Qt.UserRole).toString()).strip()
                hidden_menus.append(key)
        return hidden_menus


class ConfigWidget(QWidget):

    def __init__(self, all_tags):
        QWidget.__init__(self)
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        c = plugin_prefs[STORE_NAME]
        tags_groupbox = QGroupBox('Check excess tags options')
        layout.addWidget(tags_groupbox)
        tags_layout = QGridLayout()
        tags_groupbox.setLayout(tags_layout)

        max_label = QLabel('Maximum tags:', self)
        max_label.setToolTip('Books with more than this value will be displayed')
        tags_layout.addWidget(max_label, 0, 0, 1, 1)
        self.max_tags_spin = QtGui.QSpinBox(self)
        self.max_tags_spin.setMinimum(0)
        self.max_tags_spin.setMaximum(100)
        self.max_tags_spin.setProperty('value', c.get(KEY_MAX_TAGS, 5))
        tags_layout.addWidget(self.max_tags_spin, 0, 1, 1, 1)

        exclude_label = QLabel('Exclude tags:', self)
        exclude_label.setToolTip('Exclude these tags from when counting the tags for each book')
        tags_layout.addWidget(exclude_label, 1, 0, 1, 1)
        self.exclude_tags = MultiCompleteLineEdit(self)
        self.exclude_tags.set_add_separator(True)
        self.exclude_tags.update_items_cache(all_tags)
        self.exclude_tags.setText(', '.join(c.get(KEY_MAX_TAG_EXCLUSIONS, [])))
        tags_layout.addWidget(self.exclude_tags, 1, 1, 1, 2)

        custom_extensions_label = QLabel('extensions to search:', self)
        custom_extensions_label.setToolTip('Search for this type of files in epubs (will search on last part so .db will matcht thumbs.db mbs.db also will')
        tags_layout.addWidget(custom_extensions_label, 2, 0, 1, 1)
        self.custom_extensions_tags = MultiCompleteLineEdit(self)
        self.custom_extensions_tags.set_add_separator(True)
        self.custom_extensions_tags.update_items_cache(all_tags)
        self.custom_extensions_tags.setText(', '.join(c.get(KEY_CUSTOM_EXTENSION, [])))
        tags_layout.addWidget(self.custom_extensions_tags, 2, 1, 1, 2)

        search_groupbox = QGroupBox('Author/Title searches')
        layout.addWidget(search_groupbox)
        search_layout = QGridLayout()
        search_groupbox.setLayout(search_layout)
        row = 0
        for key, text in [(KEY_SEARCH_AUTHOR_COMMAS, 'Authors with commas'),
                          (KEY_SEARCH_AUTHOR_NOCOMMAS, 'Authors without commas'),
                          (KEY_SEARCH_TITLE_SERIES, 'Titles with series info')]:
            label = QLabel(text+':', self)
            lineEdit = QLineEdit(c.get(key, DEFAULT_STORE_VALUES[key]), self)
            setattr(self, '_'+key, lineEdit)
            label.setBuddy(lineEdit)
            search_layout.addWidget(label, row, 0, 1, 1)
            search_layout.addWidget(lineEdit, row, 1, 1, 1)
            row += 1

        menus_groupbox = QGroupBox('Menus Visible')
        layout.addWidget(menus_groupbox)
        menus_layout = QVBoxLayout()
        menus_groupbox.setLayout(menus_layout)
        self.visible_menus_list = VisibleMenuListWidget(self)
        menus_layout.addWidget(self.visible_menus_list)

    def save_settings(self):
        new_prefs = {}
        new_prefs[KEY_MAX_TAGS] = int(unicode(self.max_tags_spin.value()))
        exclude_tag_text = unicode(self.exclude_tags.text()).strip()
        if exclude_tag_text.endswith(','):
            exclude_tag_text = exclude_tag_text[:-1]
        new_prefs[KEY_MAX_TAG_EXCLUSIONS] = [t.strip() for t in exclude_tag_text.split(',')]

        custom_extensions_text = unicode(self.custom_extensions_tags.text()).strip()
        if custom_extensions_text.endswith(','):
            custom_extensions_text = custom_extensions_text[:-1]
        new_prefs[KEY_CUSTOM_EXTENSION] = [t.strip() for t in custom_extensions_text.split(',')]

        for key in (KEY_SEARCH_AUTHOR_COMMAS, KEY_SEARCH_AUTHOR_NOCOMMAS, KEY_SEARCH_TITLE_SERIES):
            new_prefs[key] = unicode(getattr(self, '_'+key).text()).strip()

        new_prefs[KEY_HIDDEN_MENUS] = self.visible_menus_list.get_hidden_menus()

        plugin_prefs[STORE_NAME] = new_prefs
