# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Main menu
#####################################################################

from PyQt4.Qt import QMenu, QToolButton, QIcon

from calibre.gui2.actions import InterfaceAction
from calibre.gui2.device import device_signals

import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.config as cfg
import calibre_plugins.kindle_collections.calibre_info as calibre_info
import calibre_plugins.kindle_collections.create_preview as create_preview
import calibre_plugins.kindle_collections.kindle_view_collections as kindle_view
import calibre_plugins.kindle_collections.kindle_edit_collections as kindle_edit
import calibre_plugins.kindle_collections.kindle_modify_settings as kindle_settings
import calibre_plugins.kindle_collections.kindle_restore_collections as kindle_restore
import calibre_plugins.kindle_collections.import_collections as import_collections
import calibre_plugins.kindle_collections.help as help
import calibre_plugins.kindle_collections.about as about
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME, PLUGIN_VERSION
from calibre_plugins.kindle_collections.utilities import debug_print, get_icon, set_plugin_icon_resources, get_pref

PLUGIN_ICONS = [ 'images/kindle_collections.png' ]

#####################################################################

class InterfacePlugin(InterfaceAction):

    name = PLUGIN_NAME
    action_spec = (_(PLUGIN_NAME), None, PLUGIN_NAME, None)
    action_type = 'current'
    dont_add_to = frozenset(['toolbar'])

    # Run when Calibre starts up - not when selecting the icon
    # Do not put print statements here - they would be seen even before plugin is run
    def genesis(self):

        # Define menu used on toolbar icon
        m = QMenu(self.gui) 
        self.menu = m
        self.qaction.setMenu(m)

        # Load the plugin icons
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(PLUGIN_NAME, icon_resources)

        # Set signals so we know when device metadata is available and good to read
        device_signals.device_metadata_available.connect(calibre_info.device_metadata_available_signalled)
        device_signals.device_connection_changed.connect(calibre_info.device_connection_changed_signalled)

    # Routines to run once calibre/gui is fully configured
    def initialization_complete(self):
        self.rebuild_menus()
        self.set_icon_style()

    # Build the menu list attached to the toolbar plugin icon/right-click menu
    # Do not put print statements here - they would be seen even before plugin is run
    def rebuild_menus(self):
        m = self.menu
        m.clear()

        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))

        m.addAction(get_icon(PLUGIN_ICONS[0]), _('&Create collections on the Kindle from Calibre'), self.create_kindle_collections)
        m.addAction( _('&Preview collections from Calibre without saving'), self.preview_kindle_collections)
        m.addAction(QIcon(I('config.png')), _('C&ustomize collections to create from Calibre...'), self.customize)
        m.addSeparator()
        m.addAction( _('&View report of Kindle collections and books'), self.view_kindle_collections)
        m.addSeparator()
        m.addAction( _('&Edit Kindle collections manually...'), self.edit_kindle_collections)
        m.addAction( _('&Import Kindle collections into Calibre...'), self.import_kindle_collections)
        m.addSeparator()
        m.addAction( _('&Restore previous Kindle collections file...'), self.restore_previous_kindle_collections)
        m.addSeparator()
        m.addAction( _('&Modify Kindle Settings...'), self.modify_kindle_settings)
        m.addSeparator()
        m.addAction(QIcon(I('help.png')), _('&Help'), self.help)
        m.addAction( _('&About'), self.about)

    def set_icon_style(self):
        # Get the current menu style/default
        click_action = get_pref(cfg.MENU_CLICK_STYLE)
        if click_action not in cfg.MENU_OPTIONS:
            click_action = cfg.MENU_DEFAULT

        # Setup the actions for the icon/menu
        if click_action ==  cfg.MENU_DEFAULT:
            self.qaction.setToolTip(PLUGIN_NAME)
            self.popup_type = QToolButton.InstantPopup
        else:
            self.popup_type = QToolButton.MenuButtonPopup
            if click_action ==  cfg.MENU_CREATE:
                self.qaction.triggered.connect(self.create_kindle_collections)
                self.qaction.setToolTip(PLUGIN_NAME + ': Create collections')
            elif click_action ==  cfg.MENU_PREVIEW:
                self.qaction.triggered.connect(self.preview_kindle_collections)
                self.qaction.setToolTip(PLUGIN_NAME + ': Preview')
            elif click_action ==  cfg.MENU_CUSTOMIZE:
                self.qaction.triggered.connect(self.customize)
                self.qaction.setToolTip(PLUGIN_NAME + ': Customize')
            elif click_action ==  cfg.MENU_VIEW:
                self.qaction.triggered.connect(self.view_kindle_collections)
                self.qaction.setToolTip(PLUGIN_NAME + ': View Report')
            elif click_action ==  cfg.MENU_EDIT:
                self.qaction.triggered.connect(self.edit_kindle_collections)
                self.qaction.setToolTip(PLUGIN_NAME + ': Edit collections')
            elif click_action ==  cfg.MENU_IMPORT:
                self.qaction.triggered.connect(self.import_kindle_collections)
                self.qaction.setToolTip(PLUGIN_NAME + ': Import collections')
            elif click_action ==  cfg.MENU_MODIFY:
                self.qaction.triggered.connect(self.modify_kindle_settings)
                self.qaction.setToolTip(PLUGIN_NAME + ': Modify settings')

        self.change_toolbar_popup_type(self.popup_type)

    def change_toolbar_popup_type(self, popup_type):
        self.popup_type = popup_type
        for bar in self.gui.bars_manager.bars:
            if hasattr(bar, 'setup_tool_button'):
                if self.qaction in bar.added_actions:
                    bar.setup_tool_button(bar, self.qaction, self.popup_type)

    # Initialize customization (called from __init__.py Base class)
    def is_configurable(self):
        if not cfg.init(self):
            (info, error, detail) = msg.message.get_all()
            return (error, detail)
        return None

    # Menu item Create
    def create_kindle_collections(self):
        self.show_version()
        create_preview.create(self)

    # Menu item Preview
    def preview_kindle_collections(self):
        self.show_version()
        create_preview.preview(self)

    # Menu item Customize (call __init__.py function to allow Preferences->Plugin->Customize to work too)
    def customize(self):
        self.show_version()
        self.interface_action_base_plugin.do_user_config(self.gui)

    # Menu item View
    def view_kindle_collections(self):
        self.show_version()
        kindle_view.run(self)

    # Menu item Edit
    def edit_kindle_collections(self):
        self.show_version()
        kindle_edit.run(self)

    # Menu item Import
    def import_kindle_collections(self):
        self.show_version()
        import_collections.run(self)

    # Menu item Restore
    def restore_previous_kindle_collections(self):
        self.show_version()
        kindle_restore.run(self)

    # Menu item Modify
    def modify_kindle_settings(self):
        self.show_version()
        kindle_settings.run(self)

    # Menu item Help
    def help(self):
        self.show_version()
        help.run()

    # Menu item About
    def about(self):
        self.show_version()
        about.run(self)

    def show_version(self):
        debug_print(PLUGIN_NAME + ' ' + PLUGIN_VERSION)
