# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# View collection report
#####################################################################

import re
import calibre_plugins.kindle_collections.config as cfg
import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.kindle_device as kindle_device
import calibre_plugins.kindle_collections.calibre_info as calibre_info
import calibre_plugins.kindle_collections.kindle_collections as kindle_collections
import calibre_plugins.kindle_collections.kindle_books as kindle_books
import calibre_plugins.kindle_collections.save as save
import calibre_plugins.kindle_collections.reports as reports
from calibre_plugins.kindle_collections.kindle_sort import KindleSort
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME, PLUGIN_VERSION
from calibre_plugins.kindle_collections.utilities import debug_print, wording

PERIODICAL_BACK_ISSUES = 'Periodicals: Back Issues'

#####################################################################

def run(parent):
    debug_print('BEGIN View Report')

    if not cfg.init(parent):
        msg.message.display()
        debug_print('END View Report - Initialization failed')
        return False

    calibre_info.ci.load_column_book_info()
    kindle_device.init(calibre_info.ci.device_path)
    kindle_collections.init(kindle_device.kdevice.get_collections())
    kindle_paths = kindle_device.kdevice.get_file_paths()

    # Check that books were found on the Kindle
    if len(kindle_paths) < 1:
        error_dialog(gui, _(PLUGIN_NAME), _('No books were found on the Kindle.<P>If you know your books are on the device, try disconnecting/reconnecting your Kindle or restarting Calibre.'), '', show=True)
        return

    kindle_books.init(kindle_paths)
    generate_existing_kindle_collections_report()

    msg.message.display()
    debug_print('END View Report')

def generate_existing_kindle_collections_report(show_collection_count=True):

    msg.message.header('\n=== %s %s View Collections Report\n' % (PLUGIN_NAME, PLUGIN_VERSION))

    if show_collection_count:
        msg.message.info('%s' % save.get_summary())

    if kindle_collections.kc.get_len() <= 0:
        msg.message.report('There are no collections on the Kindle.\n\nUse Create or Edit to save collections to the Kindle.')
    else:
        ksort = KindleSort()

        # Report Sort by Collections
        msg.message.report('\nCollections sorted by Most Recent (Kindle view "Sort by Collections"):\n')
        # Show the collections
        for collection in kindle_collections.kc.get_time_sorted_names():
            msg.message.report('    %s (%d)' % (collection, kindle_books.kbooks.get_visible_book_count(collection)))
        # Show Periodical Back Issues last since we can't tell its update time
        if kindle_books.kbooks.periodical_back_issues:
            msg.message.report('    %s (%d)' % (PERIODICAL_BACK_ISSUES, len(kindle_books.kbooks.periodical_back_issues)))


        # Report Sort by Title
        msg.message.report('\n\nCollections and Books sorted by Title (Kindle view "Sort by Title"):\n')

        # Add author or date to end of book names
        book_info = []
        for (title, info) in kindle_books.kbooks.get_visible_unsorted_titles_and_info():
            title_string = '%-80s %s' % (title, info)
            book_info.append(title_string)

        col_names = ksort.sort_names(kindle_collections.kc.get_unsorted_names() + book_info + [ PERIODICAL_BACK_ISSUES ])

        for name in col_names:
            if not re.match(' Book with no File - ', name):
                if name == PERIODICAL_BACK_ISSUES:
                    title = '%s (%d)' % (PERIODICAL_BACK_ISSUES, len(kindle_books.kbooks.periodical_back_issues))
                    msg.message.report('    %-80s <SPECIAL COLLECTION>' % title)
                else:
                    if kindle_collections.kc.get_collection(name):
                        title = '%s (%d)' % (name, kindle_books.kbooks.get_visible_book_count(name))
                        msg.message.report('    %-80s <COLLECTION>' % title)
                    else:
                        msg.message.report('    %s' % name)


        # Report Sort by Collections with books
        msg.message.report('\n\nCollections with Books sorted by Title:')

        # Pre-sort by date so periodicals with same name are sorted by date
        kindle_books.kbooks.get_sorted_paths(kindle_books.kbooks.get_paths_from_codes(), 'date', True)
        # Sort all paths by title once so listing per collection is faster
        all_sorted_paths = kindle_books.kbooks.get_sorted_paths(kindle_books.kbooks.get_paths_from_codes(), 'title', False)

        # Show the collections
        for collection in kindle_collections.kc.get_sorted_names():
            msg.message.report('\n    %s (%d)' % (collection, kindle_books.kbooks.get_visible_book_count(collection)))
            print_sorted_books(kindle_books.kbooks.get_visible_codes(collection), all_sorted_paths)
                    
        # Show Periodical Back Issues last
        collection = PERIODICAL_BACK_ISSUES
        if kindle_books.kbooks.periodical_back_issues:
            msg.message.report('\n    %s (%d)' % (PERIODICAL_BACK_ISSUES, len(kindle_books.kbooks.periodical_back_issues)))
            codes=[]
            for path in kindle_books.kbooks.periodical_back_issues:
                codes.append(kindle_books.kbooks.path_info[path]['code'])
            print_sorted_books(codes, all_sorted_paths, reverse=False)
                    
        # Show books not in a collection
        all_book_codes  = kindle_books.kbooks.code_paths.keys()
        uncollected_codes = []
        collected_codes = set(kindle_collections.kc.get_all_book_codes())
        uncollected_codes = [ c for c in all_book_codes if c not in collected_codes ]
        if uncollected_codes:
            msg.message.report('\n    Books not in a Collection:')
            print_sorted_books(uncollected_codes, all_sorted_paths)
                    
        # Helpful tips
        msg.message.report('\n\nBooks listed as "Book with no file" indicate books found in a collection that do not have a file on the Kindle filesystem (or are empty) and therefore the title is not known.  Either the book is an Internal Kindle file (like a dictionary) which is okay, or it is a book that has been deleted but left in a collection - you can use Edit Collections to remove it.')

        msg.message.report('\n\nIf you have recently updated collections using this plugin you may need to ' + \
                    'disconnect and Restart the Kindle so that it loads the collections you see above.  ' + \
                    'Otherwise the Kindle will show the old version of the collection information.')

        reports.report_book_details()

def print_sorted_books(codes, all_sorted_paths, reverse=False):
    if reverse:
        all_sorted_paths.reverse()
    for path in all_sorted_paths:
        path_code = kindle_books.kbooks.path_info[path]['code']
        for code in codes:
            if code == path_code:
                title = kindle_books.kbooks.path_info[path]['title']
                info = kindle_books.kbooks.get_book_author_date(path)
                msg.message.report('        %-76s %s' % (title, info))
