# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2011, meme'
__docformat__ = 'restructuredtext en'

#####################################################################
# Customization window
#####################################################################

import os, re
from functools import partial

from PyQt4 import QtGui 
from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel,  \
                     QLineEdit, QFormLayout, QTableWidget, QTableWidgetItem, QFrame, \
                     QAbstractItemView, QComboBox, QMenu, QToolButton, QCheckBox, QGridLayout

from calibre.gui2 import question_dialog, info_dialog

import calibre_plugins.kindle_collections.messages as msg
import calibre_plugins.kindle_collections.config as cfg
import calibre_plugins.kindle_collections.calibre_info as calibre_info
from calibre_plugins.kindle_collections.utilities import debug_print, array_to_csv, csv_to_array, \
         ComboTableWidgetItem, CheckableTableWidgetItem, CheckableBoxWidgetItem, get_icon, get_pref, set_pref
from calibre_plugins.kindle_collections.__init__ import PLUGIN_NAME

#####################################################################

class ConfigWidget(QWidget):

    # Starting point when Customize preferences is selected for the plugin 
    def __init__(self, actual_plugin):
        debug_print('BEGIN ConfigWidget')

        QWidget.__init__(self)
        self.actual_plugin = actual_plugin

        layout = QVBoxLayout(self)
        self.setLayout(layout)

        # Sort the rows in the table based on the real name not the label
        self.table_sorted_indexes = sorted(calibre_info.ci.column_labels.keys(), key=lambda x: calibre_info.ci.column_labels[x].lower())

        # Create and add table containing rows/columns for configuration
        self._table = ConfigTableWidget(self.table_sorted_indexes, cfg.config_table, self)

        # Add the 'menu' at the top right

        words_layout = QHBoxLayout()
        layout.addLayout(words_layout)
        words_layout.addStretch(1)

        # Add Preview link by itself
        preview_label = QLabel('<a href="http://www.foo.com/">Preview without saving collections</a>', self)
        preview_label.setToolTip('Show what would happen if you created collections.<P>Does not modify any files on the Kindle.')
        preview_label.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.LinksAccessibleByKeyboard)
        preview_label.linkActivated.connect(self.preview_activated)
        words_layout.addWidget(preview_label)

        # Add Help link - same line as heading to improve look
        help_label = QLabel(' | <a href="http://www.foo.com/">Help</a>', self)
        help_label.setToolTip('Go to the Mobileread forum posting for this plugin')
        help_label.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.LinksAccessibleByKeyboard)
        help_label.linkActivated.connect(self.help_link_activated)
        words_layout.addWidget(help_label)

        layout.addSpacing(10)

        # Add the intro and reset button above the table

        header_layout = QGridLayout()
        layout.addLayout(header_layout)

        # Add a comment describing the window
        heading_label = QLabel('Configure which Kindle collections to update:', self)
        heading_label.setToolTip('Use this form to customize exactly which collections to create on your Kindle from Calibre.<P>For example, try setting the Action for the "Series" entry to "Create" and then running Preview.<P>When entering words/patterns be aware that<BR>there are some special pattern characters that need to be preceded by "\\"<BR>to use them as normal characters (e.g. "\\[" - without quotes): | [ ] . + ? * ^ $')

        # Add a reset button
        reset_button = QtGui.QToolButton(self)
        reset_button.setToolTip('Reset to defaults')
        reset_button.setIcon(get_icon('clear_left.png'))
        reset_button.clicked.connect(partial(self.reset_to_defaults, self))

        header_layout.addWidget(heading_label, 0, 0, 1, 1)
        header_layout.addWidget(reset_button, 0, 1, 1, 1)

        #Add the table
        
        # Add a horizontal layout containing the table and the buttons next to it
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        # Add the table created by the class
        table_layout.addWidget(self._table)

        layout.addSpacing(10)

        # Add the lower section of settings under the table

        # Define a grid for the settings under the table
        lower_layout = QGridLayout()

        # Ignore all collections edit box
        ignore_all_label = QLabel('Never delete or modify Kindle collections that match these patterns:', self)
        ignore_all_label.setToolTip('These Kindle collections will never be modified or deleted, whether or not they are in Calibre or Preserve Kindle-only is checked.<BR>    e.g.: __Currently Reading, .*Kindle, ~.*')

        ignore_all_layout = QHBoxLayout()
        self._ignore_all_ledit = QLineEdit(array_to_csv(cfg.config_settings['ignore_all']), self)
        ignore_all_layout.addSpacing(20)
        ignore_all_layout.addWidget(self._ignore_all_ledit)

        # Menu style choice
        menu_label = QLabel('Toolbar icon default action:', self)
        menu_label.setToolTip('Choose what the plugin icon on the toolbar does when clicked<P>You can always access the menu by clicking on the small drop-down arrown on the right of the icon.')

        menu_layout = QHBoxLayout()
        menu_default = get_pref(cfg.MENU_CLICK_STYLE)
        if menu_default not in cfg.MENU_OPTIONS:
            menu_default = cfg.MENU_DEFAULT
        self.original_menu_style = menu_default
        self._menu_choices = ComboTableWidgetItem(self, menu_default, cfg.MENU_OPTIONS)
        menu_layout.addSpacing(20)
        menu_layout.addWidget(self._menu_choices)
        menu_layout.addStretch(1)

        # Keep Kindle-only collections checkbox
        self._keep_kindle_only = CheckableBoxWidgetItem(cfg.config_settings['keep_kindle_only'], 'Preserve existing Kindle-only collections','Keep any collections on the Kindle that aren\'t managed by Calibre. (A collection is managed by Calibre if it is selected using the settings you have above.)<P>If you uncheck this, your Kindle-only collections will be deleted (unless their name is in the Never delete box above).')

        # Ignore prefix/suffix when comparing
        self._ignore_prefix_suffix = CheckableBoxWidgetItem(cfg.config_settings['ignore_prefix_suffix'], 'Ignore prefix and suffix when comparing collection names', 'Replace Kindle collections generated from Calibre even if prefix/suffix changes (prefix/suffix must be non-alphanumeric).<BR>Does not affect the never delete/modify list.')

        # Case sensitive checkbox
        self._ignore_case = CheckableBoxWidgetItem(cfg.config_settings['ignore_case'], 'Ignore uppercase / lowercase when matching patterns','If unchecked, all pattern matches will be case-sensitive')

        # Fast reboot - show only if hacks are installed on Kindle (but define it so it can be saved)
        self._fast_reboot = CheckableBoxWidgetItem(cfg.config_settings['fast_reboot'], 'If hacks installed trigger fast reboot on USB unplug','Requires the Fonts or Screensavers hack to be installed on the Kindle.')

        # Space
        space_layout = QHBoxLayout()
        space_layout.addSpacing(50)

        # Populate the grid
        lower_layout.addWidget(self._keep_kindle_only, 0, 0, 1, 1)
        lower_layout.addWidget(self._ignore_prefix_suffix, 1, 0, 1, 1)
        lower_layout.addWidget(self._ignore_case, 2, 0, 1, 1)
        lower_layout.addWidget(self._fast_reboot, 3, 0, 1, 1)

        lower_layout.addLayout(space_layout, 0, 1, 1, 1)

        lower_layout.addWidget(ignore_all_label, 0, 2, 1, 1)
        lower_layout.addLayout(ignore_all_layout, 1, 2, 1, 1)
        lower_layout.addWidget(menu_label, 2, 2, 1, 1)
        lower_layout.addLayout(menu_layout, 3, 2, 1, 1)

        layout.addLayout(lower_layout)

        debug_print('END ConfigWidget')

    # Called by Calibre before save_settings 
    def validate(self):
        debug_print('BEGIN Validate')
        valid = True
        # Only save if we were able to get data to avoid corrupting stored data
        try:
            if not cfg.validate_configuration(self.get_custom_window_table_data(), self.get_custom_window_ignore_all()):
                msg.message.error('Not saving customizations due to invalid patterns.')
                valid = False
        except:
                msg.message.error('Not saving customizations due to unexpected errors.')
                valid = False

        if not valid:
            msg.message.display()

        debug_print('END Validate, status = %s' % valid)
        return valid

    # Invoked when user clicks OK in preferences dialog.  Saves customizations to file store for later use.
    # Already validated by validate() - if the customization pre-checks did not generate an error
    def save_settings(self):
        debug_print('Customization trying to save data')
        collections_data = {}
        try:
            # _table will not be defined if the customization script did not run due to pre-check errors
            collections_data[cfg.STORE_KEY_ROWS] = self._table.get_data()
            collections_data[cfg.STORE_KEY_SETTINGS] =  { 
                                                 'ignore_all': csv_to_array(self.get_custom_window_ignore_all()),
                                                 'keep_kindle_only': self._keep_kindle_only.isChecked(), 
                                                 'ignore_prefix_suffix': self._ignore_prefix_suffix.isChecked(), 
                                                 'ignore_case': self._ignore_case.isChecked(), 
                                                 'fast_reboot': self._fast_reboot.isChecked(), 
                                                 'store_version': cfg.STORE_VERSION
                                                }
            cfg.store.write_store_data(collections_data)
        except:
            debug_print('Customizations not saved - unexpected errors in saving customization.')

        # Save the menu style 
        new_menu_style = cfg.MENU_OPTIONS[self._menu_choices.currentIndex()]
        set_pref(cfg.MENU_CLICK_STYLE, new_menu_style)
        if new_menu_style != self.original_menu_style:
            info_dialog(self.actual_plugin.gui, _(PLUGIN_NAME), _('You need to restart Calibre to see the new toolbar icon style.'), '', show=True)

        debug_print('Customization saved data')

    # Define the help message shown when customize-calibre -l is run
    # Would also be displayed if config_widget is not defined, but it is for this plugin
    def customization_help(self, gui=False):
        return 'This plugin can only be customized using the GUI.'

    # Show the help URL
    def help_link_activated(self, url):
        self.actual_plugin.help()

    # Run preview while currently in customize window
    def preview_activated(self):
        debug_print('Preview Activated within customization window')
        # Run in preview mode
        if self.validate():
            debug_print('BEGIN saving configuration temporarily for preview')
            # Save current settings so we can run with temporary settings - so Cancel in Customization works ok
            current_data = cfg.store.read_store_data()
            # Temporarily update version so preview will run
            old = 'No store version'
            if current_data and cfg.STORE_KEY_SETTINGS in current_data:
                if 'store_version' in current_data[cfg.STORE_KEY_SETTINGS]:
                    old = current_data[cfg.STORE_KEY_SETTINGS]['store_version']
            self.save_settings()
            debug_print('END saving configuration temporarily for preview')
            self.actual_plugin.preview_kindle_collections()
            current_data[cfg.STORE_KEY_SETTINGS]['store_version'] = old
            debug_print('Restoring pre-preview configuration')
            cfg.store.write_store_data(current_data)
        else:
            msg.message.error('Cannot run preview until customization errors are corrected.')
        msg.message.display()

    def reset_to_defaults(self, w):
        if question_dialog(w, _('Question'), '<P>'+
                'Are you sure you want to reset to default values?<P>Any modified settings will be discarded.', show_copy_button=False):
            cfg.load_config_table()
            w._table.populate_table(self.table_sorted_indexes, cfg.config_table)

            cfg.load_config_settings()
            w._ignore_all_ledit.setText(array_to_csv(cfg.config_settings['ignore_all']))
            w._keep_kindle_only.setChecked(cfg.config_settings['keep_kindle_only'])
            w._ignore_prefix_suffix.setChecked(cfg.config_settings['ignore_prefix_suffix'])
            w._ignore_case.setChecked(cfg.config_settings['ignore_case'])
            w._fast_reboot.setChecked(cfg.config_settings['fast_reboot'])

            w._menu_choices.setCurrentIndex(0)

    def get_custom_window_table_data(self):
        return self._table.get_data()

    def get_custom_window_ignore_all(self):
        return unicode(self._ignore_all_ledit.text())


#####################################################################

# Create and populate rows of collection configuration data
class ConfigTableWidget(QTableWidget):

    def __init__(self, row_sorted_indexes, row_data, *args):
        QTableWidget.__init__(self, *args)
        self.revrow = {}
        self.header_labels = self.get_header_labels()
        self.populate_table(row_sorted_indexes, row_data)

    # Extract the column header labels in the right order from the main dictionary
    def get_header_labels(self):
        header_labels = [ '' ] * len(cfg.CUSTOMIZE_COLUMNS)
        for k in cfg.CUSTOMIZE_COLUMNS.keys():
            header_labels[cfg.CUSTOMIZE_COLUMNS[k]['pos']] = cfg.CUSTOMIZE_COLUMNS[k]['title']
        return header_labels

    # Fill the table with the given data
    def populate_table(self, row_sorted_indexes, row_data):
        self.clear()

        # Define the row/columns before putting any data in table
        h = self.horizontalHeader()
        h.setMinimumSectionSize(50)
        self.setRowCount(len(row_sorted_indexes))
        self.setColumnCount(len(self.header_labels))

        # Fill the table, save column label for each row number to  make it easier to lookup when saving
        #debug_print('Populate Config table, rows = %s' % self.rowCount())
        for row, index in enumerate(row_sorted_indexes):
            self.revrow[str(row)] = index
            self.populate_table_row(row, row_data[index])

        # Set style of table, and set selection to first row
        self.setAlternatingRowColors(True)
        self.setSortingEnabled(False)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.selectRow(0)
        self.setMinimumSize(700, 200)

        # Set the column headers - after the data so they are resized correctly
        self.verticalHeader().setDefaultSectionSize(24)
        self.setHorizontalHeaderLabels(self.header_labels)
        self.resizeColumnsToContents()

    # Return the column in the row for the given label
    def pos(self, column):
        return cfg.CUSTOMIZE_COLUMNS[column]['pos']

    def populate_table_row(self, row, data):
        self.blockSignals(True)
        item = ComboTableWidgetItem(self, data['action'], cfg.ACTION_OPTIONS)
        item.setToolTip('<dt>Create</dt><dd>Generate collections on the Kindle using entries from Calibre columns/categories.</dd><dt>Delete</dt> <dd>Remove collections from the Kindle that match entries from the Calibre column.</dd>')
        self.setCellWidget(row, self.pos('action'), item)

        item = QTableWidgetItem(data['column'])
        item.setToolTip('The source column to use from Calibre to get the list of collection names.')
        self.setItem(row, self.pos('column'), item)

        item = QTableWidgetItem(data['prefix'])
        item.setToolTip('Add text to start of collection names for extra sorting control when using Sort by Title.<BR>You should only use symbols to allow changing the prefix later.<BR>The Kindle will sort in this order:  _`^~\'([{$+<=>|<BR>Most other symbols are "invisible" to the Kindle.')
        self.setItem(row, self.pos('prefix'), item)

        item = QTableWidgetItem(data['suffix'])
        item.setToolTip('Add text to the end of collection name.<BR>You should only use symbols to allow changing the suffix later.')
        self.setItem(row, self.pos('suffix'), item)

        item = QTableWidgetItem(data['minimum'])
        item.setToolTip('Set the minimum number of books a collection must contain before <BR>creating, updating, or deleting the collection on the Kindle.<BR>Blank is equivalent to 1.')
        self.setItem(row, self.pos('minimum'), item)

        item = QTableWidgetItem(array_to_csv(data['ignore']))
        item.setToolTip('Identify the entries to ignore when creating collections on the Kindle,<BR>using a comma separate list of words or patterns.<BR>e.g.: IgnoreMe, .*ignorecontaining.*, ignorestarting.*, .*ignoreending')
        self.setItem(row, self.pos('ignore'), item)

        item = QTableWidgetItem(array_to_csv(data['include']))
        item.setToolTip('Identify the entries to include when creating collections on the Kindle.<BR>Leave blank to include all entries except ones marked ignore.')
        self.setItem(row, self.pos('include'), item)

        item = QTableWidgetItem(data['rename_from'])
        item.setToolTip('Advanced feature to allow renaming collection names and add/remove/substitute characters.<BR>Uses the python command re.sub(from_pattern, to_pattern, collection_name).<BR>Rename is applied after ignore/include. e.g.:<BR>from: "^+(.*)$"  to: "\\1" will remove the "+" at the start of any collection name.')
        self.setItem(row, self.pos('rename_from'), item)

        item = QTableWidgetItem(data['rename_to'])
        item.setToolTip('See Rename these patterns...')
        self.setItem(row, self.pos('rename_to'), item)

        item = QTableWidgetItem(data['split_char'])
        item.setToolTip('Split the text in the Calibre field into separate collections using this character.<BR>Split is applied before ignore/include or rename.')
        self.setItem(row, self.pos('split_char'), item)

        self.blockSignals(False)

    # Get all the data from the updated customization window to save for later
    def get_data(self):
        row_data = {}
        for row in range(self.rowCount()):
            row_data[self.revrow[str(row)]] = self.convert_row_to_data(row)
        return row_data

    # Get data from one row in the Customization window to save for later
    def convert_row_to_data(self, row):
        # Blank data since we'll populate it with specific data, leave spaces in prefix
        data = {}
        data['action'] = unicode(self.cellWidget(row, self.pos('action')).currentText()).strip()
        data['prefix'] = unicode(self.item(row, self.pos('prefix')).text())
        data['suffix'] = unicode(self.item(row, self.pos('suffix')).text())
        data['minimum'] = unicode(self.item(row, self.pos('minimum')).text()).strip()
        data['ignore'] = csv_to_array(unicode(self.item(row, self.pos('ignore')).text()))
        data['include'] = csv_to_array(unicode(self.item(row, self.pos('include')).text()))
        data['rename_from'] = unicode(self.item(row, self.pos('rename_from')).text())
        data['rename_to'] = unicode(self.item(row, self.pos('rename_to')).text())
        data['split_char'] = unicode(self.item(row, self.pos('split_char')).text())
        return data
