/*
 * File Name: ipc.c
 */

/*
 * This file is part of hello_world.
 *
 * hello_world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello_world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >
#include <liberipc/eripc.h>

#include "dr.h"

///////////////////////////////////////////////
/// DR update routine ////////
void display_update(int waveform)
{
    // force full update
//    if (waveform == DM_HINT_FULL) updating = 0;
    
//    if (updating > 0) return;
    
//    updating = 1;
    
    char msg[256];
    int take_control = 0; // always return control
 
    if (dm_socket == -1)
    {
      // create and bind socket
      dm_socket = socket(PF_INET, SOCK_DGRAM, 0);
      dm_addr.sin_family = AF_INET;
      dm_addr.sin_port = htons(DMPORT);
      dm_addr.sin_addr.s_addr = inet_addr(DMSERVER);
      memset(dm_addr.sin_zero, '\0', sizeof(dm_addr.sin_zero));
    }
 
    // send display update message
    snprintf(msg, sizeof(msg)-1, "!F,%d,%d,%d\n", getpid(), take_control > 0 ? 1 : 0, waveform);
    sendto(dm_socket, msg, strlen(msg), 0, (struct sockaddr*) &dm_addr, sizeof(dm_addr));
    
//    updating = 0;
}

//// end DR update
///////////////////////////////////////////////

//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct 
{
    eripc_handler_t *handler;
    const gchar     *message_name;
    const gchar     *interface;
    gint            handler_id;
} functionEntry;

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// IPC application
// TODO: set DBUS_APPL_NAME to your application name, use lower-case and digits only
//       or set to PACKAGENAME when this is lower-case and digits only
#define DBUS_APPL_NAME                  "xmahjong"            // lower-case and digits only
#define DBUS_SERVICE                     "com.irexnet." DBUS_APPL_NAME
#define DBUS_PATH                       "/com/irexnet/" DBUS_APPL_NAME
#define DBUS_INTERFACE                   "com.irexnet." DBUS_APPL_NAME

// IPC system control
#define DBUS_SERVICE_SYSTEM_CONTROL     "com.irexnet.sysd"

// IPC popup menu
#define DBUS_SERVICE_POPUP_MENU         "com.irexnet.popupmenu"


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_context_t  *eripcContext = NULL;

//============================================================================
// Local Function Definitions
//============================================================================

static void on_window_activated  ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
static void on_window_deactivated( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
                                 
// Exported DBUS API list
static functionEntry    service_functions[] =
        {
            // message handlers (method calls to this service)
            { on_window_activated,    "activatedWindow",        NULL                        },
            { on_window_deactivated,  "deactivatedWindow",      NULL                        },
            { NULL }
        };


//============================================================================
// Functions Implementation
//============================================================================

#define MENU_NAME "xmahjongg_menu_main"
static void menu_show()
{
  ipc_menu_show_menu(MENU_NAME);
}

void dr_menu_init(void)
{
  ipc_menu_add_menu(MENU_NAME, NULL, NULL, NULL);
  menu_show();
}

//----------------------------------------------------------------------------
// Generic
//----------------------------------------------------------------------------

// initialise
void ipc_set_services (void)
{
    gint          i;
    eripc_error_t retval;

    g_assert(eripcContext == NULL);

	printf("ipc_set_services\n");
	// initialise IPC context
	eripcContext = eripc_init(DBUS_APPL_NAME, "1.0", NULL);
	if (eripcContext == NULL) 
	{
		printf("Failed to initialize eripc context\n");
		exit(1);
	}
	printf("ipc_set_services-end\n");

    // register message/signal handlers
    for (i = 0 ; service_functions[i].handler != NULL ; i++)
    {
        if (service_functions[i].interface)
        {
            // install signal handler
            retval = eripc_set_signal_handler( eripcContext,
                                               service_functions[i].handler,
                                               NULL,
                                               ERIPC_BUS_SESSION,
                                               service_functions[i].interface,
                                               service_functions[i].message_name,
                                               &(service_functions[i].handler_id) );
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                printf( "eripc_set_signal_handler [%s] returns [%d]\n",
                             service_functions[i].message_name,
                             retval );
            }
        }
        else
        {
            // install message handler
            retval = eripc_set_message_handler( eripcContext,
                                                service_functions[i].handler,
                                                NULL,
                                                ERIPC_BUS_SESSION,
                                                DBUS_INTERFACE,
                                                service_functions[i].message_name,
                                                &(service_functions[i].handler_id) );
              
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                printf( "eripc_set_message_handler [%s] returns [%d]\n",
                             service_functions[i].message_name,
                             retval );
            }
        }
    }
}


// un-initialise
void ipc_unset_services (void)
{
    gint i;

    for (i = 0 ; service_functions[i].handler_id != 0 ; i++)
    {
        eripc_unset_handler(eripcContext, service_functions[i].handler_id);
    }    
}


//----------------------------------------------------------------------------
// System control
//----------------------------------------------------------------------------
// report "application started"
void ipc_sys_startup_complete ( int my_xid )
{
    eripc_error_t   result;

    // parameters for ipc message
    const int       my_pid = getpid();
//    const int       my_xid = GDK_WINDOW_XID(pub->mw->window->window);
    const gboolean  is_multidoc = FALSE;        // TODO: set TRUE/FALSE depending on your application
    const gboolean  show_taskbar = TRUE;

    g_assert( eripcContext );

    printf("ipc_sys_startup_complete:%x\n", my_xid);
    // broadcast signal over session bus
    result = eripc_send_signal_varargs( eripcContext,
                                        ERIPC_BUS_SESSION,
                                        DBUS_PATH,
                                        DBUS_SERVICE_SYSTEM_CONTROL,
                                        "startupComplete",
                                        ERIPC_TYPE_STRING, DBUS_APPL_NAME,
                                        ERIPC_TYPE_INT,    my_pid,
                                        ERIPC_TYPE_BOOL,   is_multidoc,
                                        ERIPC_TYPE_STRING, DBUS_SERVICE,
                                        ERIPC_TYPE_INT,    my_xid,
                                        ERIPC_TYPE_BOOL,   show_taskbar,
                                        ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        printf("eripc_send_varargs returns [%d]", result);
    } 
}

// add a menu set
void ipc_menu_add_menu( const char *name,
                        const char *group1,
                        const char *group2,
                        const char *group3 )
{
    eripc_error_t result;

    printf( "entry: name [%s] groups [%s] [%s] [%s]\n",
                       name,     group1, group2, group3 );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, DBUS_SERVICE,
                                 ERIPC_TYPE_STRING, group1,
                                 ERIPC_TYPE_STRING, group2,
                                 ERIPC_TYPE_STRING, group3,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        printf("ERROR:eripc_send_varargs returns [%d]\n", result);
    } 
}

// show the given menu set
void ipc_menu_show_menu( const char *name )
{
    eripc_error_t result;

    printf( "entry: name [%s]", name );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "showMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        printf("eripc_send_varargs returns [%d]", result);
    } 
}

static void on_window_activated( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    gboolean          result      = FALSE; 
    const eripc_arg_t *arg_array  = info->args;

	// TODO: this function is not called for some reason, so menu is not
	//       refreshed when switching back to this application
	printf("on_window_activated\n");
    if (arg_array[0].type == ERIPC_TYPE_INT)
    {
        menu_show();
        
        result = TRUE;
    }

    // return result to caller
    eripc_reply_bool(context, info->message_id, result);
}

/* @brief Called after a window was deactivated (set to the background)
 *
 * Application (callee) may adapt its context and free resources.
 */  
static void on_window_deactivated( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data )
{
    gchar             *msg        = NULL;
    gboolean          result      = FALSE; 
    const eripc_arg_t *arg_array  = info->args;

	printf("on_window_deactivated\n");
    if (arg_array[0].type == ERIPC_TYPE_INT)
    {
        // TODO: Replace implementation

        gint window = arg_array[0].value.i;
        
        result = TRUE;
    }
    
    // return result to caller
    eripc_reply_bool(context, info->message_id, result);
}




