#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from collections import defaultdict, deque
from calibre.ebooks.metadata import authors_to_string
from calibre.gui2 import config, info_dialog

import calibre_plugins.find_duplicates.config as cfg
from calibre_plugins.find_duplicates.algorithms import create_algorithm, \
                    DUPLICATE_SEARCH_FOR_BOOK, DUPLICATE_SEARCH_FOR_AUTHOR, \
                    get_first_author, authors_to_list

class DuplicateFinder(object):
    '''
    Responsible for executing a duplicates search and navigating the results
    '''
    DUPLICATES_MARK = 'duplicates'
    BOOK_EXEMPTION_MARK = 'not_book_duplicate'
    AUTHOR_EXEMPTION_MARK = 'not_author_duplicate'
    DUPLICATE_GROUP_MARK = 'duplicate_group_'

    def __init__(self, gui):
        self.gui = gui
        self.db = gui.library_view.model().db
        self._ignore_clear_signal = False
        self._book_exemptions_map, self._author_exemptions_map = cfg.get_exemption_maps(self.db)
        self._persist_gui_state()
        self.clear_duplicates_mode()

    def is_valid_to_clear_search(self):
        return not self._ignore_clear_signal

    def clear_duplicates_mode(self, clear_search=True, reapply_restriction=True):
        '''
        We call this method when all duplicates have been resolved
        Reset the gui, clear the marked column data and all our duplicate state.
        '''
        self._is_new_search = True
        self._is_showing_duplicate_exemptions = False
        self._is_show_all_duplicates_mode = False
        self._is_duplicate_exemptions_changed = False
        self._books_for_group_map = None
        self._groups_for_book_map = None
        self._authors_for_group_map = None
        self._is_group_changed = False
        self._group_ids_queue = None
        self._algorithm_text = None
        self._duplicate_search_mode = None
        self._current_group_id = None
        self._update_marked_books()
        if clear_search:
            self.gui.search.clear()
        self._restore_previous_gui_state(reapply_restriction)

    def run_duplicate_check(self):
        '''
        Execute a duplicates search using the specified algorithm and display results
        '''
        if not self.is_showing_duplicate_exemptions() and not self.has_results():
            # We are in a safe state to preserve the users current restriction/highlighting
            self._persist_gui_state()
        self.clear_duplicates_mode()

        title_match = cfg.plugin_prefs.get(cfg.KEY_TITLE_MATCH, 'identical')
        author_match  = cfg.plugin_prefs.get(cfg.KEY_AUTHOR_MATCH, 'identical')
        show_all = cfg.plugin_prefs.get(cfg.KEY_SHOW_ALL_GROUPS, True)
        sort_groups_by_title = cfg.plugin_prefs.get(cfg.KEY_SORT_GROUPS_TITLE, True)

        algorithm, self._algorithm_text  = create_algorithm(self.gui,
                        self._book_exemptions_map, self._author_exemptions_map,
                        title_match, author_match)
        self._duplicate_search_mode = algorithm.duplicate_search_mode()
        self._books_for_group_map, self._groups_for_book_map = algorithm.run_duplicate_check(sort_groups_by_title)
        self._group_ids_queue = deque(sorted(self._books_for_group_map.keys()))

        if len(self._group_ids_queue) == 0:
            self.gui.status_bar.showMessage('')
            return info_dialog(self.gui, _('No duplicates'),
                    _('No duplicate groups were found when searching with: <b>%s</b>') % self._algorithm_text,
                    show=True, show_copy_button=False)
        else:
            self._is_show_all_duplicates_mode = show_all
            self.show_next_result()
            return info_dialog(self.gui, _('Find Duplicates'),
                    _('Found %d duplicate groups when searching with: <b>%s</b>') % \
                    (len(self._group_ids_queue), self._algorithm_text),
                    show=True, show_copy_button=False)

    def has_results(self):
        '''
        Returns whether there is any duplicate groups outstanding from
        the last search run in the current session.
        '''
        if self._books_for_group_map:
            return len(self._books_for_group_map) > 0
        return False

    def is_searching_for_authors(self):
        '''
        Returns whether the current algorithm is a search by authors ignoring title
        rather than by books. For use with more contextual messages in the gui.
        '''
        return self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_AUTHOR

    def has_duplicate_exemptions(self):
        '''
        Returns whether we have any duplicate exemptions configured for
        any books or authors.
        '''
        return self.has_book_exemptions() or  self.has_author_exemptions()

    def has_book_exemptions(self):
        '''
        Returns whether we have any duplicate exemptions configured for
        any books.
        '''
        return len(self._book_exemptions_map) > 0

    def has_author_exemptions(self):
        '''
        Returns whether we have any duplicate exemptions configured for
        any authors.
        '''
        return len(self._author_exemptions_map) > 0

    def is_book_in_exemption(self, book_id):
        '''
        Returns whether this book id currently has any duplicate exemption
        pairings. Note that it is possible that the pairing is no longer
        valid due to the paired book having been deleted.
        '''
        if book_id in self._book_exemptions_map:
            return True
        author = get_first_author(self.db, book_id)
        if author and author in self._author_exemptions_map:
            return True
        return False

    def get_exemptions_for_book(self, book_id):
        '''
        Returns the book ids of all the duplicate exemptions for this book
        Returns None if no exemptions
        '''
        book_exemptions = set()
        if book_id in self._book_exemptions_map:
            book_exemptions = self._book_exemptions_map[book_id]

        author_exemptions = set()
        author = get_first_author(self.db, book_id)
        if author and author in self._author_exemptions_map:
            author_exemptions = self._author_exemptions_map[author]

        return book_exemptions, author_exemptions

    def is_showing_duplicate_exemptions(self):
        '''
        Returns whether we are currently displaying all duplicate exemptions
        '''
        return self._is_showing_duplicate_exemptions

    def get_current_duplicate_group_ids(self):
        '''
        Returns the book ids of all the contents in the current duplicate group
        Returns None if no current group
        '''
        if self._current_group_id is not None:
            return self._books_for_group_map[self._current_group_id]
        return None

    def show_next_result(self, forward=True):
        '''
        Navigate/highlight the next or previous result group if any available
        Checks for any merged/deleted books and recomputes all the remaining
        duplicate groups before moving on.
        '''
        if self._is_duplicate_exemptions_changed:
            # Re-run the duplicate search again using the current algorithm and display results
            self.run_duplicate_check()
            return

        self._is_showing_duplicate_exemptions = False
        self._cleanup_deleted_books()

        if len(self._books_for_group_map) == 0:
            msg = _('No further duplicate groups exist for searching with: <b>%s</b>' % self._algorithm_text)
            self.clear_duplicates_mode()
            return info_dialog(self.gui, _('No duplicates'),
                        msg, show=True, show_copy_button=False)

        next_group_id = self._get_next_group_to_display(forward)
        if next_group_id == self._current_group_id:
            # The user has changed direction but not merged the current group - repeat move
            next_group_id = self._get_next_group_to_display(forward)
        self._current_group_id = next_group_id
        self._update_marked_books()
        self._refresh_duplicate_display_mode()
        self._search_for_duplicate_group(self._current_group_id)
        if self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_AUTHOR:
            self._view_authors_in_tag_viewer()
        self._is_new_search = False

    def get_mark_exemption_preview_text(self, all_groups=False):
        '''
        Build a nice summary text for all the duplicate exemptions we will be adding
        The list displayed will be unique to minimise the UI noise
        '''
        # First make sure we cater for any merged/deleted book ids
        self._cleanup_deleted_books()
        if all_groups:
            group_ids = list(self._books_for_group_map.keys())
        else:
            if self._current_group_id is None:
                # Should not happen due to validation elsewhere
                return
            if self._current_group_id not in self._books_for_group_map:
                # The user must have resolved all the merges for this group
                error_dialog(self.gui, _('No duplicates'),
                            _('The current duplicate group no longer exists. '
                              'You cannot perform this action.'),
                            show=True, show_copy_button=False)
                return
            group_ids = [self._current_group_id]
        if len(group_ids) == 0:
            info_dialog(self.gui, _('No duplicates'),
                _('No further duplicate groups exist for searching with: <b>%s</b>'%self._algorithm_text),
                show=True, show_copy_button=False)
            return
        if self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_BOOK:
            return self._build_mark_groups_book_exemption_text(group_ids)
        elif self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_AUTHOR:
            return self._build_mark_groups_author_exemption_text(group_ids)

    def get_remove_exemption_preview_text(self, book_ids):
        '''
        Build a nice summary text for all the duplicate exemptions we will be removing
        The list displayed will be unique to minimise the UI noise.
        Note that this method will not differentiate between book exemptions and
        author exemptions - it will list all available for this set of book ids.
        '''
        lines = []
        displayed = defaultdict(set)
        for book_id in book_ids:
            if book_id in self._book_exemptions_map:
                other_book_ids = self._book_exemptions_map[book_id]
                self._build_book_exemption_text(book_id, other_book_ids, lines, displayed)
        if self._author_exemptions_map:
            # First turn our list of book ids into a list of authors:
            authors = self._get_authors_for_books(book_ids)
            for author in authors:
                if author in self._author_exemptions_map:
                    other_authors = self._author_exemptions_map[author]
                    self._build_author_exemption_text(author, other_authors, lines, displayed)
        return '\n'.join(lines)

    def _build_mark_groups_book_exemption_text(self, group_ids):
        lines = []
        displayed = defaultdict(set)
        for group_id in group_ids:
            book_ids = self._books_for_group_map.get(group_id, [])
            other_book_ids = set(book_ids)
            for book_id in book_ids:
                other_book_ids -= set([book_id])
                if other_book_ids:
                    self._build_book_exemption_text(book_id, other_book_ids, lines, displayed)
        return '\n'.join(lines)

    def _build_book_exemption_text(self, book_id, other_book_ids, lines, displayed):
        title = self.db.title(book_id, index_is_id=True)
        authors = authors_to_list(self.db, book_id)
        authors_text = authors_to_string(authors)
        related_lines = []
        for other_book_id in other_book_ids:
            seen_before = True
            if book_id not in displayed:
                displayed[book_id].add(other_book_id)
                displayed[other_book_id].add(book_id)
                seen_before = False
            elif other_book_id not in displayed[book_id]:
                displayed[book_id].add(other_book_id)
                displayed[other_book_id].add(book_id)
                seen_before = False
            if not seen_before:
                title = self.db.title(other_book_id, index_is_id=True)
                authors = authors_to_list(self.db, other_book_id)
                authors_text = authors_to_string(authors)
                related_lines.append('    %s (%s) (%d)' % (title, authors_text, other_book_id))
        if related_lines:
            lines.append('%s (%s) (%d) is exempted showing as a duplicate of:' % (title, authors_text, book_id))
            lines.extend(related_lines)
            lines.append('')

    def _build_mark_groups_author_exemption_text(self, group_ids):
        lines = []
        displayed = defaultdict(set)
        for group_id in group_ids:
            authors = self._authors_for_group_map.get(group_id, [])
            other_authors = set(authors)
            for author in authors:
                other_authors -= set([author])
                if other_authors:
                    self._build_author_exemption_text(author, other_authors, lines, displayed)
        return '\n'.join(lines)

    def _build_author_exemption_text(self, author, other_authors, lines, displayed):
        related_lines = []
        for other_author in other_authors:
            seen_before = True
            if author not in displayed:
                displayed[author].add(other_author)
                displayed[other_author].add(author)
                seen_before = False
            elif other_author not in displayed[author]:
                displayed[author].add(other_author)
                displayed[other_author].add(author)
                seen_before = False
            if not seen_before:
                related_lines.append('    \'%s\'' % (other_author,))
        if related_lines:
            lines.append('\'%s\' is exempted from showing as a duplicate of:' % (author,))
            lines.extend(related_lines)
            lines.append('')

    def mark_current_group_as_duplicate_exemptions(self):
        '''
        Invoke for the current duplicate group to flag all books it
        contains as not being duplicates of each other within the group.
        Persists these combinations to the config file.
        Moves on to the next duplicate group to display when done.
        If we have marked all groups, clears the search results.
        NOTE: This method relies on get_mark_exemption_preview_text() having been
              called first, to ensure the group is valid and in the case of author
              duplicate searches that the authors_for_group_map is populated
        '''
        # Update our duplicates map
        self._mark_group_ids_as_exemptions([self._current_group_id])
        # Remove the current group from consideration and move to the next group
        self._remove_duplicate_group(self._current_group_id)
        self.show_next_result(forward=True)

    def mark_groups_as_duplicate_exemptions(self):
        '''
        Invoke for all remaining duplicate groups to flag all books they
        contain as not being duplicates of each other within each group.
        Persists these combinations to the config file.
        Clears the search results when done.
        NOTE: This method relies on get_mark_exemption_preview_text() having been
              called first, to ensure the group is valid and in the case of author
              duplicate searches that the authors_for_group_map is populated
        '''
        # Update our duplicates map
        self._mark_group_ids_as_exemptions(self._books_for_group_map.keys())
        # There must be no more duplicate groups so clear the search mode
        self.clear_duplicates_mode()

    def _mark_group_ids_as_exemptions(self, group_ids):
        if self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_BOOK:
            for group_id in group_ids:
                book_ids = self._books_for_group_map.get(group_id, [])
                for book_id in book_ids:
                    for other_book_id in book_ids:
                        if other_book_id != book_id:
                            if book_id not in self._book_exemptions_map:
                                self._book_exemptions_map[book_id] = set()
                            self._book_exemptions_map[book_id].add(other_book_id)
            cfg.set_exemption_map(self.db, cfg.KEY_BOOK_EXEMPTIONS, self._book_exemptions_map)

        elif self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_AUTHOR:
            for group_id in group_ids:
                authors = self._authors_for_group_map.get(group_id, [])
                for author in authors:
                    for other_author in authors:
                        if other_author != author:
                            if author not in self._author_exemptions_map:
                                self._author_exemptions_map[author] = set()
                            self._author_exemptions_map[author].add(other_author)
            cfg.set_exemption_map(self.db, cfg.KEY_AUTHOR_EXEMPTIONS, self._author_exemptions_map)

    def show_all_exemptions(self, for_books=True):
        '''
        Display for the user all the books which have been flagged as a duplicate
        exemption - either the book exemptions or the author exemptions.
        '''
        # Make sure we prune any deleted books from our not duplicates map
        marked = self.BOOK_EXEMPTION_MARK
        mark_author_exemptions = False
        if for_books and self._book_exemptions_map:
            for book_id in list(self._book_exemptions_map.keys()):
                if self.db.data.has_id(book_id):
                    continue
                self._remove_item_from_exemption_map(book_id, self._book_exemptions_map)
        elif not for_books:
            marked = self.AUTHOR_EXEMPTION_MARK
            mark_author_exemptions = True

        self._update_marked_books(mark_author_exemptions)
        self._refresh_exemption_display_mode(marked)
        self.gui.library_view.set_current_row(0)

    def remove_from_book_exemptions(self, book_ids, from_book_id=None):
        '''
        Allow a user to specify that this set of ids should no longer be part
        of any duplicate exemption mappings.
        If from_book_id is specified then only mappings from that book to others
        in the set are removed. This occurs by using the Manage exemptions dialog.
        If from_book_id is not specified, all permutations of mappings between
        this set of books are removed.
        '''
        if from_book_id:
            # We are removing mappings from this book to the other books
            for book_id in book_ids:
                if book_id in self._book_exemptions_map:
                    self._book_exemptions_map[book_id].remove(from_book_id)
                    # Make sure if there are no more exemption mappings for
                    # the book that it gets removed
                    if len(self._book_exemptions_map[book_id]) == 0:
                        del self._book_exemptions_map[book_id]
                self._book_exemptions_map[from_book_id].remove(book_id)
            # Have we removed all the mappings for our source book?
            if len(self._book_exemptions_map[from_book_id]) == 0:
                del self._book_exemptions_map[from_book_id]
        else:
            # We are removing mappings between each of the books
            for book_id in book_ids:
                if book_id in self._book_exemptions_map:
                    self._remove_item_from_exemption_map(book_id, self._book_exemptions_map)

        cfg.set_exemption_map(self.db, cfg.KEY_BOOK_EXEMPTIONS, self._book_exemptions_map)
        self._is_duplicate_exemptions_changed = True
        self._update_marked_books()
        self.gui.search.do_search()

    def remove_from_author_exemptions(self, book_ids=None, authors=None, from_author=None):
        '''
        Allow a user to specify that this set of ids should no longer be part
        of any duplicate exemption mappings.
        If from_author is specified then only mappings from that author to others
        in the set are removed. This occurs by using the Manage exemptions dialog.
        If from_author is not specified, all permutations of mappings between
        this set of author are removed.
        If book_ids are specified, we need to lookup the authors for those books first
        '''
        if from_author:
            # We are removing mappings from this author to the other authors
            for author in authors:
                if author in self._author_exemptions_map:
                    self._author_exemptions_map[author].remove(from_author)
                    # Make sure if there are no more exemption mappings for
                    # the author that it gets removed
                    if len(self._author_exemptions_map[author]) == 0:
                        del self._author_exemptions_map[author]
                self._author_exemptions_map[from_author].remove(author)
            # Have we removed all the mappings for our source author?
            if len(self._author_exemptions_map[from_author]) == 0:
                del self._author_exemptions_map[from_author]
        else:
            # We are removing mappings between each of the authors
            # If only book ids given we need to convert the book ids into a unique set of authors
            if book_ids:
                authors = self._get_authors_for_books(book_ids)
            for author in authors:
                if author in self._author_exemptions_map:
                    self._remove_item_from_exemption_map(author, self._author_exemptions_map)

        cfg.set_exemption_map(self.db, cfg.KEY_AUTHOR_EXEMPTIONS, self._author_exemptions_map)
        self._is_duplicate_exemptions_changed = True
        self._update_marked_books(mark_author_exemptions=True)
        self.gui.search.do_search()

    def _update_marked_books(self, mark_author_exemptions=False):
        '''
        Mark the books using the special 'marked' temp column in Calibre
        Note that we need to store multiple types of marked books at once
        The first is marking all of the duplicate groups
        The second is all duplicate book ids, marked with 'duplicates'
        The third is exemptions marked as 'not_book_duplicate' or 'not_author_duplicate'

        This will allow us to apply a search restriction of 'marked:duplicates'
        at the same time as doing a search of 'marked:xxx' for our subset,
        while also allowing the user to refresh to get updated results

        The only limitation is making sure that we don't overlap the sets by
        using the same substrings like 'duplicates' in the value of marked_text.
        '''
        marked_ids = dict()
        # Build our dictionary of current marked duplicate groups
        if self._books_for_group_map:
            remaining_group_ids = list(sorted(self._books_for_group_map.keys()))
            for index, group_id in enumerate(remaining_group_ids):
                marked_text = '%s%04d' % (self.DUPLICATE_GROUP_MARK, group_id)
                for book_id in self._books_for_group_map[group_id]:
                    if book_id not in marked_ids:
                        marked_ids[book_id] = marked_text
                    else:
                        marked_ids[book_id] = '%s,%s' % (marked_ids[book_id], marked_text)

        # Now add the marks to indicate each book that is in a duplicate group
        if self._groups_for_book_map:
            for book_id in self._groups_for_book_map.keys():
                if book_id not in marked_ids:
                    marked_ids[book_id] = self.DUPLICATES_MARK
                else:
                    # We need to store two bits of text in the one value
                    marked_ids[book_id] = '%s,%s' % (marked_ids[book_id], self.DUPLICATES_MARK)

        # Add the marks for author duplicate exemptions. This is an expensive operation so
        # we only do it when we really have to (i.e. user is showing author exemptions)
        if mark_author_exemptions:
            if self._author_exemptions_map:
                # Rebuild the map of authors to books
                books_for_author_map = self._create_books_for_author_map()
                for author in self._author_exemptions_map.keys():
                    if author in books_for_author_map:
                        for book_id in books_for_author_map[author]:
                            if book_id not in marked_ids:
                                marked_ids[book_id] = self.AUTHOR_EXEMPTION_MARK
                            else:
                                # We need to store two bits of text in the one value
                                marked_ids[book_id] = '%s,%s' % (marked_ids[book_id],
                                                                 self.AUTHOR_EXEMPTION_MARK)
        else:
            # Add the marks for book duplicate exemptions
            if self._book_exemptions_map:
                for book_id in self._book_exemptions_map.keys():
                    if book_id not in marked_ids:
                        marked_ids[book_id] = self.BOOK_EXEMPTION_MARK
                    else:
                        # We need to store two bits of text in the one value
                        marked_ids[book_id] = '%s,%s' % (marked_ids[book_id], self.BOOK_EXEMPTION_MARK)

        # Assign the results to our database
        self.gui.library_view.model().db.set_marked_ids(marked_ids)

    def _get_authors_for_books(self, book_ids):
        authors = set()
        for book_id in book_ids:
            author = get_first_author(self.db, book_id)
            if author:
                authors.add(author)
        return authors

    def _create_books_for_author_map(self):
        books_for_author_map = defaultdict(set)
        for book_id in self.db.all_ids():
            author = get_first_author(self.db, book_id)
            if author:
                books_for_author_map[author].add(book_id)
        # Use this opportunity to purge any author exemptions that we do not have books for
        for author in list(self._author_exemptions_map.keys()):
            if author in books_for_author_map:
                continue
            self._remove_item_from_exemption_map(author, self._author_exemptions_map)
        return books_for_author_map

    def _cleanup_deleted_books(self):
        # First pass is to remove delete/merged books and their associated groups
        book_ids = list(self._groups_for_book_map.keys())
        for book_id in sorted(book_ids):
            if not self.db.data.has_id(book_id):
                # We have a book that has been merged/deleted
                # Remove the book from all of its groups.
                for group_id in self._groups_for_book_map[book_id]:
                    group = self._books_for_group_map[group_id]
                    group.remove(book_id)
                del self._groups_for_book_map[book_id]
                # Ensure it is removed from the exemptions map if present
                if book_id in self._book_exemptions_map:
                    self._remove_item_from_exemption_map(book_id, self._book_exemptions_map)

        # Second pass is through the groups to remove all groups...
        #   with < 2 members if we are viewing a book based duplicate search, or
        #   with < 2 authors if we are viewing and author based duplicate search
        self._authors_for_group_map = defaultdict(set)
        for group_id in list(self._books_for_group_map.keys()):
            if self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_BOOK:
                count = len(self._books_for_group_map[group_id])
            elif self._duplicate_search_mode == DUPLICATE_SEARCH_FOR_AUTHOR:
                authors = set()
                for book_id in self._books_for_group_map[group_id]:
                    author = get_first_author(self.db, book_id)
                    if author not in authors:
                        authors.add(author)
                        self._authors_for_group_map[group_id].add(author)
                count = len(authors)
            if count > 1:
                continue
            if count == 1:
                # There is one book left in this group, so the group can be deleted
                # However we need to cleanup entries for the book too.
                last_book_id = self._books_for_group_map[group_id][0]
                self._groups_for_book_map[last_book_id].remove(group_id)
            del self._books_for_group_map[group_id]
            self._group_ids_queue.remove(group_id)
            if group_id in self._authors_for_group_map:
                del self._authors_for_group_map[group_id]

        # Our final pass is looking for books that can be removed from the maps because
        # they have no groups any more
        for book_id in list(self._groups_for_book_map.keys()):
            if len(self._groups_for_book_map[book_id]) == 0:
                del self._groups_for_book_map[book_id]

        # Set our flag to know whether to force a refresh of our search restriction
        # when we move to the next group, since the name of the restriction will be
        # the same when the marked groups get renumbered
        self._is_group_changed = self._current_group_id not in self._groups_for_book_map

    def _get_next_group_to_display(self, forward):
        if forward:
            next_group_id = self._group_ids_queue.popleft()
            self._group_ids_queue.append(next_group_id)
        else:
            next_group_id = self._group_ids_queue.pop()
            self._group_ids_queue.appendleft(next_group_id)
        return next_group_id

    def _refresh_duplicate_display_mode(self):
        self.gui.library_view.multisort((('marked', True), ('authors', True), ('title', True)),
                                        only_if_different=not self._is_new_search)
        self._apply_highlight_if_different(self._is_show_all_duplicates_mode)
        if self._is_show_all_duplicates_mode:
            restriction = 'marked:%s' % self.DUPLICATES_MARK
            self._apply_restriction_if_different(restriction)

    def _search_for_duplicate_group(self, group_id):
        marked_text = 'marked:%s%04d' % (self.DUPLICATE_GROUP_MARK, group_id)
        if self._is_show_all_duplicates_mode:
            self.gui.search.set_search_string(marked_text)
        else:
            self._apply_restriction_if_different(marked_text)
            # When displaying groups one at a time, we need to move selection
            self.gui.library_view.set_current_row(0)

        remaining_group_ids = list(sorted(self._books_for_group_map.keys()))
        position = remaining_group_ids.index(group_id) + 1
        msg = _('Showing #%d of %d remaining duplicate groups for %s') % \
                    (position, len(remaining_group_ids), self._algorithm_text)
        self.gui.status_bar.showMessage(msg)

    def _refresh_exemption_display_mode(self, marked):
        self._is_showing_duplicate_exemptions = True
        self._apply_highlight_if_different(False)
        restriction = 'marked:%s' % marked
        self._apply_restriction_if_different(restriction)

    def _persist_gui_state(self):
        r = self.gui.search_restriction
        self._restore_restriction = unicode(r.currentText())
        self._restore_restriction_is_text = False
        if self._restore_restriction:
            # How do we know whether this is a named search or a text search?
            # TODO: hacks below will work for 0.7.56 and later, will change it when 0.7.57 released
            special_menu = unicode(r.itemText(1))
            self._restore_restriction_is_text = special_menu == self._restore_restriction
            if self._restore_restriction.startswith('*') and r.currentIndex() == 2:
                self._restore_restriction_is_text = True
                self._restore_restriction = self._restore_restriction[1:]
        self._restore_highlighting_state = config['highlight_search_matches']

    def _restore_previous_gui_state(self, reapply_restriction=True):
        # Restore the user's GUI to it's previous glory
        self._apply_highlight_if_different(self._restore_highlighting_state)
        if reapply_restriction:
            self._apply_restriction_if_different(self._restore_restriction,
                                                 self._restore_restriction_is_text)

    def _apply_highlight_if_different(self, new_state):
        if config['highlight_search_matches'] != new_state:
            config['highlight_search_matches'] = new_state
            self.gui.set_highlight_only_button_icon()

    def _apply_restriction_if_different(self, restriction, is_text_restriction=True):
        prev_ignore = self._ignore_clear_signal
        self._ignore_clear_signal = True
        # TODO: Simplify this in Calibre 0.7.57
        if unicode(self.gui.search_restriction.currentText()) not in [restriction, '*'+restriction]:
            if is_text_restriction:
                self.gui.apply_text_search_restriction(restriction)
            else:
                self.gui.apply_named_search_restriction(restriction)
        self._ignore_clear_signal = prev_ignore

    def _remove_duplicate_group(self, group_id):
        book_ids = self._books_for_group_map[group_id]
        for book_id in book_ids:
            self._groups_for_book_map[book_id].remove(group_id)
        del self._books_for_group_map[group_id]
        self._group_ids_queue.remove(group_id)

    def _remove_item_from_exemption_map(self, item_key, exemptions_map):
        # Double check the key lookup as we may have removed its inverse
        if item_key in exemptions_map:
            for related_value in exemptions_map[item_key]:
                exemptions_map[related_value].remove(item_key)
                # Make sure if there are no more exemption mappings for the other
                # book/author that the whole item gets removed
                if len(exemptions_map[related_value]) == 0:
                    del exemptions_map[related_value]
            del exemptions_map[item_key]

    def _view_authors_in_tag_viewer(self):
        if not self.gui.tags_view.pane_is_visible:
            self.gui.tb_splitter.show_side_pane()
        p = self.gui.tags_view.model().find_category_node('authors')
        if p:
            self.gui.tags_view.model().show_item_at_path(p)
            idx = self.gui.tags_view.model().index_for_path(p)
            self.gui.tags_view.setExpanded(idx, True)

