#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from PyQt4.Qt import QDialogButtonBox, QVBoxLayout, QHBoxLayout, \
                     QLabel, QTextEdit, QVariant, Qt, QGroupBox, \
                     QListWidget, QListWidgetItem, QSizePolicy, \
                     QRadioButton, QTableWidget, QAbstractItemView, \
                     QGridLayout, QButtonGroup, QCheckBox

from calibre.ebooks.metadata import authors_to_string, fmt_sidx
from calibre.utils.date import format_date
from calibre.utils.titlecase import titlecase

import calibre_plugins.find_duplicates.config as cfg
from calibre_plugins.find_duplicates.common_utils import get_icon, SizePersistedDialog, \
                    ImageTitleLayout, ReadOnlyTableWidgetItem, CheckableTableWidgetItem
from calibre_plugins.find_duplicates.algorithms import TITLE_DESCS, AUTHOR_DESCS


class FindDuplicatesDialog(SizePersistedDialog):
    '''
    Dialog to configure search options and perform the search
    '''
    def __init__(self, parent):
        SizePersistedDialog.__init__(self, parent, 'duplicate finder plugin:duplicate dialog')

        self.setWindowTitle(_('Find Duplicates'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        title_layout = ImageTitleLayout(self, 'images/find_duplicates.png', _('Duplicate Search Options'))
        layout.addLayout(title_layout)

        layout.addSpacing(5)
        match_layout = QHBoxLayout()
        layout.addLayout(match_layout)

        title_match_group_box = QGroupBox(_('Title Matching'),self)
        match_layout.addWidget(title_match_group_box)
        title_match_group_box_layout = QVBoxLayout()
        title_match_group_box.setLayout(title_match_group_box_layout)
        self.title_group = QButtonGroup(self)
        self.title_group.buttonClicked[int].connect(self._title_radio_clicked)
        for row, key in enumerate(TITLE_DESCS.keys()):
            rdo = QRadioButton(titlecase(key), self)
            self.title_group.addButton(rdo)
            self.title_group.setId(rdo, row)
            title_match_group_box_layout.addWidget(rdo)

        author_match_group_box = QGroupBox(_('Author Matching'), self)
        match_layout.addWidget(author_match_group_box)
        author_match_group_box_layout = QVBoxLayout()
        author_match_group_box.setLayout(author_match_group_box_layout)
        self.author_group = QButtonGroup(self)
        self.author_group.buttonClicked[int].connect(self._author_radio_clicked)
        for row, key in enumerate(AUTHOR_DESCS.keys()):
            rdo = QRadioButton(titlecase(key), self)
            self.author_group.addButton(rdo)
            self.author_group.setId(rdo, row)
            author_match_group_box_layout.addWidget(rdo)

        self.description = QTextEdit(self)
        self.description.setReadOnly(True)
        layout.addSpacing(5)
        layout.addWidget(self.description)

        layout.addSpacing(5)
        display_group_box = QGroupBox(_('Display your results'), self)
        layout.addWidget(display_group_box)
        display_group_box_layout = QGridLayout()
        display_group_box.setLayout(display_group_box_layout)
        self.show_all_button = QRadioButton(_('Show all groups at once with highlighting'), self)
        self.show_one_button = QRadioButton(_('Show one group at a time'), self)
        display_group_box_layout.addWidget(self.show_all_button, 0, 0, 1, 1)
        display_group_box_layout.addWidget(self.show_one_button, 0, 1, 1, 1)
        self.sort_numdups_checkbox = QCheckBox(_('Sort groups by number of duplicates'))
        self.sort_numdups_checkbox.setToolTip(_('When unchecked, will sort by an approximation of the title\n'
                                                'or by author if title is being ignored'))
        display_group_box_layout.addWidget(self.sort_numdups_checkbox, 1, 0, 1, 2)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self._ok_clicked)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        self.title_match = cfg.plugin_prefs.get(cfg.KEY_TITLE_MATCH, 'identical')
        self.author_match  = cfg.plugin_prefs.get(cfg.KEY_AUTHOR_MATCH, 'identical')
        title_idx = TITLE_DESCS.keys().index(self.title_match)
        self.title_group.button(title_idx).setChecked(True)
        author_idx = AUTHOR_DESCS.keys().index(self.author_match)
        self.author_group.button(author_idx).setChecked(True)
        self._update_description()

        show_all_groups = cfg.plugin_prefs.get(cfg.KEY_SHOW_ALL_GROUPS, True)
        self.show_all_button.setChecked(show_all_groups)
        self.show_one_button.setChecked(not show_all_groups)
        sort_groups_by_title = cfg.plugin_prefs.get(cfg.KEY_SORT_GROUPS_TITLE, True)
        self.sort_numdups_checkbox.setChecked(not sort_groups_by_title)

        # Cause our dialog size to be restored from prefs or created on first usage
        self.resize_dialog()

    def _title_radio_clicked(self, idx):
        self.title_match = TITLE_DESCS.keys()[idx]
        self._update_description()

    def _author_radio_clicked(self, idx):
        self.author_match = AUTHOR_DESCS.keys()[idx]
        self._update_description()

    def _update_description(self):
        if self.title_match == 'ignore' and self.author_match == 'ignore':
            # Special case scenario
            desc = _('<b>Book duplicate search</b><br/>'
                 '- Find groups of books which have an identical ISBN.<br/>'
                 '- Marking a group as exempt will prevent those specific books '
                 'from appearing together in future duplicate book searches.')
        else:
            desc = TITLE_DESCS[self.title_match] % AUTHOR_DESCS[self.author_match]
        self.description.setText(desc)

    def _ok_clicked(self):
        cfg.plugin_prefs[cfg.KEY_TITLE_MATCH] = self.title_match
        cfg.plugin_prefs[cfg.KEY_AUTHOR_MATCH] = self.author_match
        self.show_all_groups = self.show_all_button.isChecked()
        cfg.plugin_prefs[cfg.KEY_SHOW_ALL_GROUPS] = self.show_all_groups
        self.sort_groups_by_title = not self.sort_numdups_checkbox.isChecked()
        cfg.plugin_prefs[cfg.KEY_SORT_GROUPS_TITLE] = self.sort_groups_by_title
        self.accept()


class BookExemptionsTableWidget(QTableWidget):

    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)

    def populate(self, books):
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(books))
        header_labels = ['Remove', 'Title', 'Author', 'Series', 'Tags', 'Date']
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(24)

        for row, book in enumerate(books):
            self._populate_table_row(row, book)

        self.setSortingEnabled(False)
        self.resizeColumnToContents(0)
        self.resizeColumnToContents(1)
        self.resizeColumnToContents(2)
        self.resizeColumnToContents(3)
        self.resizeColumnToContents(5)
        if len(books) > 0:
            self.selectRow(0)

    def _populate_table_row(self, row, book):
        if row == 0:
            self.setItem(row, 0, ReadOnlyTableWidgetItem(''))
        else:
            self.setItem(row, 0, CheckableTableWidgetItem(False))

        title_widget = ReadOnlyTableWidgetItem(book.title)
        title_widget.setData(Qt.UserRole, QVariant(book.id))
        self.setItem(row, 1, title_widget)

        display_authors = authors_to_string(book.authors)
        self.setItem(row, 2, ReadOnlyTableWidgetItem(display_authors))

        display_series = ''
        if book.series:
            display_series = '%s [%s]' % (book.series, fmt_sidx(book.series_index))
        self.setItem(row, 3, ReadOnlyTableWidgetItem(display_series))

        display_tags = ''
        if book.tags:
            display_tags = ', '.join(book.tags)
        self.setItem(row, 4, ReadOnlyTableWidgetItem(display_tags))

        display_timestamp = format_date(book.timestamp, format=None)
        self.setItem(row, 5, ReadOnlyTableWidgetItem(display_timestamp))

    def get_checked_book_ids(self):
        ids = []
        for row in range(self.rowCount()):
            if row:
                if self.item(row, 0).get_boolean_value():
                    ids.append(self.item(row, 1).data(Qt.UserRole).toPyObject())
        return ids

class AuthorExemptionsTableWidget(QTableWidget):

    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)

    def populate(self, authors):
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(authors))
        header_labels = ['Remove', 'Author']
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.horizontalHeader().setStretchLastSection(True)
        self.verticalHeader().setDefaultSectionSize(24)

        for row, author in enumerate(authors):
            self._populate_table_row(row, author)

        self.setSortingEnabled(False)
        self.resizeColumnToContents(0)
        if len(authors) > 0:
            self.selectRow(0)

    def _populate_table_row(self, row, author):
        if row == 0:
            self.setItem(row, 0, ReadOnlyTableWidgetItem(''))
        else:
            self.setItem(row, 0, CheckableTableWidgetItem(False))
        self.setItem(row, 1, ReadOnlyTableWidgetItem(author))

    def get_checked_authors(self):
        authors = []
        for row in range(self.rowCount()):
            if row:
                if self.item(row, 0).get_boolean_value():
                    authors.append(unicode(self.item(row, 1).text()))
        return authors


class ManageExemptionsDialog(SizePersistedDialog):
    '''
    Dialog to configure search options and perform the search
    '''
    def __init__(self, parent, db, book_id, book_exemptions, author_exemptions):
        SizePersistedDialog.__init__(self, parent, 'duplicate finder plugin:exemptions dialog')

        self.setWindowTitle(_('Find Duplicates'))
        layout = QVBoxLayout(self)
        self.setLayout(layout)
        title_layout = ImageTitleLayout(self, 'images/find_duplicates.png', 'Manage Exemptions')
        layout.addLayout(title_layout)

        layout.addSpacing(10)
        help_label1 = QLabel('The first book below will never appear as a duplicate '
                            'with the following books.<br/>'
                            'To allow duplicate consideration, tick the remove checkbox '
                            'and click ok.', self)
        layout.addWidget(help_label1)

        self._exempt_books_table = BookExemptionsTableWidget(self)
        layout.addWidget(self._exempt_books_table)

        layout.addSpacing(10)
        help_label2 = QLabel('The first author below will never appear as a duplicate '
                            'with the following authors.<br/>'
                            'To allow duplicate consideration, tick the remove checkbox '
                            'and click ok.', self)
        layout.addWidget(help_label2)

        self._exempt_authors_table = AuthorExemptionsTableWidget(self)
        layout.addWidget(self._exempt_authors_table)
        layout.addSpacing(10)

        books = self._get_books(db, book_id, book_exemptions)
        self._exempt_books_table.populate(books)
        authors = self._get_authors(db, book_id, author_exemptions)
        self._exempt_authors_table.populate(authors)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.accept)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)

        # Cause our dialog size to be restored from prefs or created on first usage
        self.resize_dialog()

    def get_checked_book_ids(self):
        return self._exempt_books_table.get_checked_book_ids()

    def get_checked_authors(self):
        return self._exempt_authors_table.get_checked_authors()

    def get_author(self):
        return self._author

    def _get_books(self, db, book_id, book_exemptions):
        book_ids = list([book_id])
        book_ids.extend(list(book_exemptions))
        books = [db.get_metadata(book_id, index_is_id=True, get_user_categories=False)
                 for book_id in book_ids]
        return books

    def _get_authors(self, db, book_id, author_exemptions):
        book_authors = db.authors(book_id, index_is_id=True)
        self._author = None
        if not book_authors:
            return []
        self._author = [a.strip().replace('|',',') for a in book_authors.split(',')][0]
        authors = list([self._author])
        authors.extend(sorted(list(author_exemptions)))
        return authors
