#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

from functools import partial
from PyQt4.Qt import QMenu, QToolButton, QApplication, Qt
from calibre import prints
from calibre.gui2 import info_dialog, question_dialog, error_dialog
from calibre.gui2.actions import InterfaceAction

import calibre_plugins.find_duplicates.config as cfg
from calibre_plugins.find_duplicates.common_utils import set_plugin_icon_resources, get_icon, \
                                                     create_menu_item
from calibre_plugins.find_duplicates.dialogs import FindDuplicatesDialog, ManageExemptionsDialog
from calibre_plugins.find_duplicates.duplicates import DuplicateFinder

PLUGIN_ICONS = ['images/find_duplicates.png',
                'images/next_result.png', 'images/previous_result.png']

class FindDuplicatesAction(InterfaceAction):

    name = 'Find Duplicates'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = (_('Find Duplicates'), None, None, None)
    popup_type = QToolButton.MenuButtonPopup
    action_type = 'current'

    def genesis(self):
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        self.rebuild_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.toolbar_button_clicked)
        self.menu.aboutToShow.connect(self.about_to_show_menu)

    def initialization_complete(self):
        # Delay instantiating our finder as we require access to the library view
        self.duplicate_finder = DuplicateFinder(self.gui)
        self.update_actions_enabled()
        self.gui.clear_button.clicked.connect(self.user_has_cleared_search)
        self.gui.search_restriction.currentIndexChanged[int].connect(self.user_has_changed_restriction)

    def library_changed(self, db):
        # We need to reset our duplicate finder after switching libraries
        self.duplicate_finder = DuplicateFinder(self.gui)
        self.update_actions_enabled()

    def rebuild_menus(self):
        c = cfg.plugin_prefs[cfg.STORE_OPTIONS]
        m = self.menu
        m.clear()
        create_menu_item(self, m, _('&Find duplicates...'), image=PLUGIN_ICONS[0],
                         shortcut=c[cfg.KEY_SHORTCUT_FIND_DUPLICATES],
                         triggered=self.find_duplicates)
        m.addSeparator()
        self.next_group_action = create_menu_item(self, m, _('&Next result'), image='images/next_result.png',
                                tooltip=_('Display the next duplicate result group'),
                                shortcut=c[cfg.KEY_SHORTCUT_NEXT],
                                triggered=partial(self.show_next_result, forward=True))
        self.previous_group_action = create_menu_item(self, m, _('&Previous result'), image='images/previous_result.png',
                                tooltip=_('Display the previous duplicate result group'),
                                shortcut=c[cfg.KEY_SHORTCUT_PREVIOUS],
                                triggered=partial(self.show_next_result, forward=False))
        m.addSeparator()
        self.mark_group_exempt_action = create_menu_item(self, m, _('&Mark current group as exempt'),
                                tooltip=_('Mark the current group as not duplicates and exempt from future consideration'),
                                shortcut=c[cfg.KEY_SHORTCUT_GROUP_NOT_DUPLICATE],
                                triggered=partial(self.mark_groups_as_duplicate_exemptions, all_groups=False))
        self.mark_all_groups_exempt_action = create_menu_item(self, m,
                                _('Mark &all groups as exempt'),
                                tooltip=_('Mark all remaining duplicate groups as exempt from future consideration'),
                                triggered=partial(self.mark_groups_as_duplicate_exemptions, all_groups=True))
        m.addSeparator()
        self.show_book_exempt_action = create_menu_item(self, m,
                                _('&Show all book duplicate exemptions'),
                                tooltip=_('Show all books that have book duplicate exemption pairings'),
                                triggered=partial(self.show_all_exemptions, for_books=True))
        self.show_author_exempt_action = create_menu_item(self, m,
                                _('&Show all author duplicate exemptions'),
                                tooltip=_('Show all books that have author duplicate exemption pairings'),
                                triggered=partial(self.show_all_exemptions, for_books=False))
        self.manage_exemptions_action = create_menu_item(self, m,
                                _('&Manage exemptions for this book'),
                                tooltip=_('Show duplicate exemptions for this book to enable removal'),
                                triggered=self.manage_exemptions_for_book)
        self.remove_exemptions_action = create_menu_item(self, m,
                                _('&Remove selected exemptions'),
                                tooltip=_('Remove any duplicate book/author exemptions for the selected books'),
                                triggered=self.remove_from_duplicate_exemptions)
        m.addSeparator()
        self.clear_duplicate_mode_action = create_menu_item(self, m,
                                _('&Clear duplicate results'), image='clear_left.png',
                                shortcut=c.get(cfg.KEY_SHORTCUT_CLEAR_RESULTS),
                                tooltip=_('Exit duplicate search mode'),
                                triggered=self.clear_duplicate_results)
        m.addSeparator()
        create_menu_item(self, m, _('&Customize plugin')+'...', 'config.png',
                         triggered=self.show_configuration)

    def about_to_show_menu(self):
        self.update_actions_enabled()
        # As we are showing a menu we can refine the enabled state of the
        # actions that are based on the selected rows
        has_duplicate_exemptions = self.duplicate_finder.has_duplicate_exemptions()
        if has_duplicate_exemptions:
            book_ids = self.gui.library_view.get_selected_ids()
            remove_enabled = len(book_ids) > 0
            manage_enabled = len(book_ids) == 1
            if manage_enabled:
                manage_enabled = self.duplicate_finder.is_book_in_exemption(book_ids[0])
            for book_id in book_ids:
                if not self.duplicate_finder.is_book_in_exemption(book_id):
                    remove_enabled = False
                    break
            self.manage_exemptions_action.setEnabled(manage_enabled)
            self.remove_exemptions_action.setEnabled(remove_enabled)

    def update_actions_enabled(self):
        has_results = self.duplicate_finder.has_results()
        self.next_group_action.setEnabled(has_results)
        self.previous_group_action.setEnabled(has_results)
        self.mark_group_exempt_action.setEnabled(has_results)
        self.mark_all_groups_exempt_action.setEnabled(has_results)
        is_showing_exemptions = self.duplicate_finder.is_showing_duplicate_exemptions()
        self.clear_duplicate_mode_action.setEnabled(has_results or is_showing_exemptions)

        # As some actions could be via shortcut keys we need them enabled
        # regardless of row selections
        has_duplicate_exemptions = self.duplicate_finder.has_duplicate_exemptions()
        self.show_book_exempt_action.setEnabled(self.duplicate_finder.has_book_exemptions())
        self.show_author_exempt_action.setEnabled(self.duplicate_finder.has_author_exemptions())
        self.manage_exemptions_action.setEnabled(has_duplicate_exemptions)
        self.remove_exemptions_action.setEnabled(has_duplicate_exemptions)

    def find_duplicates(self):
        d = FindDuplicatesDialog(self.gui)
        if d.exec_() == d.Accepted:
            self.duplicate_finder.run_duplicate_check()
            self.update_actions_enabled()

    def toolbar_button_clicked(self):
        if not self.duplicate_finder.has_results():
            return self.find_duplicates()
        # If the user control-clicks on this button/menu, reverse the direction of search
        forward = True
        mods = QApplication.keyboardModifiers()
        if mods & Qt.ControlModifier or mods & Qt.ShiftModifier:
            forward = False
        self.show_next_result(forward)

    def show_next_result(self, forward=True):
        self.duplicate_finder.show_next_result(forward)
        self.update_actions_enabled()

    def mark_groups_as_duplicate_exemptions(self, all_groups):
        summary_text = self.duplicate_finder.get_mark_exemption_preview_text(all_groups)
        if summary_text:
            # Ensure that the selection is moved onto the current duplicate group
            duplicate_ids = self.duplicate_finder.get_current_duplicate_group_ids()
            self.gui.library_view.select_rows(duplicate_ids)
            exemption_type = 'books'
            if self.duplicate_finder.is_searching_for_authors():
                exemption_type = 'authors'
            if not question_dialog(self.gui, _('Are you sure?'),
                   _('This action will ensure that each of the %s in the group '
                     'are exempt from appearing together again in future.<p>'
                     'Are you sure you want to continue?')%exemption_type,
                     det_msg=summary_text, show_copy_button=True):
                return
            if all_groups:
                self.duplicate_finder.mark_groups_as_duplicate_exemptions()
            else:
                self.duplicate_finder.mark_current_group_as_duplicate_exemptions()
        else:
            info_dialog(self.gui, _('No duplicates in group'),
                        _('There are no duplicates remaining in this group.'),
                        show=True, show_copy_button=False)
        self.update_actions_enabled()

    def show_all_exemptions(self, for_books=True):
        self.duplicate_finder.show_all_exemptions(for_books)
        self.update_actions_enabled()

    def manage_exemptions_for_book(self):
        row = self.gui.library_view.currentIndex()
        if not row.isValid():
            return error_dialog(self.gui, _('Cannot manage exemptions'),
                    _('No book selected'), show=True)
        book_id = self.gui.library_view.model().id(row)
        book_exemptions, author_exemptions = self.duplicate_finder.get_exemptions_for_book(book_id)
        if not book_exemptions and not author_exemptions:
            return info_dialog(self.gui, _('Cannot manage exemptions'),
                    _('This book has no duplicate exemptions'), show=True)

        d = ManageExemptionsDialog(self.gui, self.gui.library_view.model().db,
                                   book_id, book_exemptions, author_exemptions)
        d.exec_()
        if d.result() == d.Accepted:
            exempt_book_ids = d.get_checked_book_ids()
            if exempt_book_ids:
                self.duplicate_finder.remove_from_book_exemptions(
                                            exempt_book_ids, from_book_id=book_id)
            exempt_authors = d.get_checked_authors()
            if exempt_authors:
                self.duplicate_finder.remove_from_author_exemptions(
                                            authors=exempt_authors, from_author=d.get_author())

        self.update_actions_enabled()

    def remove_from_duplicate_exemptions(self):
        book_ids = self.gui.library_view.get_selected_ids()
        if len(book_ids) < 1:
            return error_dialog(self.gui, _('Invalid selection'),
                    _('You must select at least one book.'), show=True)
        summary_text = self.duplicate_finder.get_remove_exemption_preview_text(book_ids)
        if summary_text:
            if not question_dialog(self.gui, _('Are you sure?'),
                   _('This action will remove all of the duplicate exemptions '
                     'listed below. This will allow them to potentially appear '
                     'as duplicates together in a future duplicate search.<p>'
                     'Are you sure you want to continue?'),
                     det_msg=summary_text, show_copy_button=True):
                return
            self.duplicate_finder.remove_from_book_exemptions(book_ids)
            self.duplicate_finder.remove_from_author_exemptions(book_ids)
        self.update_actions_enabled()

    def clear_duplicate_results(self, clear_search=True, reapply_restriction=True):
        if not self.clear_duplicate_mode_action.isEnabled():
            return
        self.duplicate_finder.clear_duplicates_mode(clear_search, reapply_restriction)
        self.update_actions_enabled()

    def user_has_cleared_search(self):
        if self.duplicate_finder.is_valid_to_clear_search():
            self.clear_duplicate_results(clear_search=False)

    def user_has_changed_restriction(self, idx):
        if self.duplicate_finder.is_valid_to_clear_search():
            self.clear_duplicate_results(clear_search=False, reapply_restriction=False)

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
