#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import os, shutil, traceback

from PyQt4 import QtGui, QtCore
from PyQt4.Qt import (QWidget, QVBoxLayout, QLabel, QLineEdit, QIcon,
                     QGroupBox, QGridLayout, QHBoxLayout, QPushButton,
                     QTableWidget, QAbstractItemView, Qt, QInputDialog)
from calibre.gui2.device import device_signals
from calibre.gui2 import question_dialog, error_dialog
from calibre.utils.config import JSONConfig, config_dir

from calibre_plugins.book_sync.common_utils import (get_icon,
                          ReadOnlyTextIconWidgetItem, CheckableTableWidgetItem,
                          NoWheelComboBox, get_library_uuid, ReadOnlyTableWidgetItem)

STORE_OPTIONS = 'BookSync'
KEY_ADD_SHORTCUT = 'addToListShortcut'
KEY_REMOVE_SHORTCUT = 'removeFromListShortcut'
KEY_VIEW_SHORTCUT = 'viewShortcut'
KEY_SYNC_SHORTCUT = 'syncShortcut'
KEY_ACTIVE_DEVICE = 'activeDevice'

STORE_DEVICES = 'Devices'
# Devices store consists of:
# 'Devices': { 'dev_uuid': {'type':xxx', 'uuid':'xxx', 'name:'xxx', 'location_code':'main',
#                           'active':True, 'autosync':True, 'collections':False } ,
# For iTunes
#              'iTunes':   {'type':'iTunes', 'uuid':iTunes', 'name':'iTunes', 'location_code':'',
#                           'active':True, 'autosync':True, 'collections':False}, ...}

STORE_LIBRARY_LISTS = 'SyncLists'
# Lists store consists of:
# 'SyncLists': {'library_uuid': {'device_uuid': [id1, id2, id3], ...}, ...}

DEFAULT_STORE_VALUES = {
    KEY_ADD_SHORTCUT: '',
    KEY_REMOVE_SHORTCUT: '',
    KEY_VIEW_SHORTCUT: '',
    KEY_SYNC_SHORTCUT: '',
    KEY_ACTIVE_DEVICE: None
}
DEFAULT_DEVICES_VALUES = {}
DEFAULT_LISTS_VALUES = {}

# For legacy users of this plugin I have renamed the config file
old_json_path = os.path.join(config_dir,'plugins/book_sync.json')
if os.path.exists(old_json_path):
    new_json_path = os.path.join(config_dir,'plugins/Book Sync.json')
    shutil.move(old_json_path, new_json_path)

# This is where all preferences for this plugin will be stored
plugin_prefs = JSONConfig('plugins/Book Sync')

# Set defaults
plugin_prefs.defaults[STORE_OPTIONS] = DEFAULT_STORE_VALUES
plugin_prefs.defaults[STORE_DEVICES] = DEFAULT_DEVICES_VALUES
plugin_prefs.defaults[STORE_LIBRARY_LISTS] = DEFAULT_LISTS_VALUES


def get_library_config(db):
    library_id = get_library_uuid(db)
    libraries = plugin_prefs[STORE_LIBRARY_LISTS]
    library = libraries.get(library_id, {})
    return library

def get_active_device_uuids():
    devices = plugin_prefs[STORE_DEVICES]
    device_uuids = []
    for device_uuid, device_info in devices.iteritems():
        if device_info['active']:
            device_uuids.append(device_uuid)
    return device_uuids

def get_book_list_for_device(db, device_uuid):
    library = get_library_config(db)
    calibre_ids = library.get(device_uuid, [])
    valid_ids = []
    for calibre_id in calibre_ids:
        if db.data.has_id(calibre_id):
            valid_ids.append(calibre_id)
    if len(valid_ids) != len(calibre_ids):
        set_book_list_for_device(db, device_uuid, valid_ids)
    return valid_ids


def set_book_list_for_device(db, device_uuid, book_ids):
    library_id = get_library_uuid(db)
    libraries = plugin_prefs[STORE_LIBRARY_LISTS]
    library = get_library_config(db)
    library[device_uuid] = book_ids
    libraries[library_id] = library
    plugin_prefs[STORE_LIBRARY_LISTS] = libraries


class BoolColumnComboBox(NoWheelComboBox):

    def __init__(self, parent, selected=True):
        NoWheelComboBox.__init__(self, parent)
        self.populate_combo(selected)

    def populate_combo(self, selected):
        self.clear()
        self.addItem(QIcon(I('ok.png')), 'Y')
        self.addItem(QIcon(I('list_remove.png')), 'N')
        if selected:
            self.setCurrentIndex(0)
        else:
            self.setCurrentIndex(1)


class DevicesTableWidget(QTableWidget):

    def __init__(self, parent):
        QTableWidget.__init__(self, parent)
        self.setSortingEnabled(False)
        self.setAlternatingRowColors(True)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.setMinimumSize(350, 0)

    def populate_table(self, devices, connected_device_info):
        self.clear()
        self.setRowCount(len(devices))
        header_labels = ['Menu', 'Name', 'Location', 'Status', 'Autosync', 'Collections']
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(32)
        self.horizontalHeader().setStretchLastSection(False)
        self.setIconSize(QtCore.QSize(32, 32))

        for row, uuid in enumerate(devices.keys()):
            self.populate_table_row(row, uuid, devices[uuid], connected_device_info)

        self.resizeColumnsToContents()
        self.setMinimumColumnWidth(1, 100)

    def setMinimumColumnWidth(self, col, minimum):
        if self.columnWidth(col) < minimum:
            self.setColumnWidth(col, minimum)

    def populate_table_row(self, row, uuid, device_config, connected_device_info):
        device_type = device_config['type']
        device_uuid = device_config['uuid']
        if device_type == 'Folder Device':
            device_icon = 'devices/folder.png'
        elif 'iTunes' in device_type:
            device_icon = 'devices/itunes.png'
        else:
            device_icon = 'reader.png'
        is_connected = False
        if connected_device_info is not None:
            if device_type == connected_device_info[0]:
                drive_info = connected_device_info[4]
                if not drive_info:
                    is_connected = True
                else:
                    for connected_info in drive_info.values():
                        if connected_info['device_store_uuid'] == device_uuid:
                            is_connected = True
                            break
        connected_icon = 'images/device_connected.png' if is_connected else None

        name_widget = ReadOnlyTextIconWidgetItem(device_config['name'], get_icon(device_icon))
        name_widget.setData(Qt.UserRole, (device_config, is_connected))
        self.setItem(row, 0, CheckableTableWidgetItem(device_config['active']))
        self.setItem(row, 1, name_widget)
        self.setItem(row, 2, ReadOnlyTableWidgetItem(device_config['location_code']))
        self.setItem(row, 3, ReadOnlyTextIconWidgetItem('', get_icon(connected_icon)))
        self.setCellWidget(row, 4, BoolColumnComboBox(self, device_config['autosync']))
        is_kindle = device_type == 'Amazon Kindle'
        if is_kindle:
            self.setCellWidget(row, 5, BoolColumnComboBox(self, device_config.get('collections', False)))

    def get_data(self):
        devices = {}
        for row in range(self.rowCount()):
            (device_config, is_connected) = self.item(row, 1).data(Qt.UserRole).toPyObject()
            device_config['active'] = self.item(row, 0).get_boolean_value()
            device_config['autosync'] = unicode(self.cellWidget(row, 4).currentText()).strip() == 'Y'
            w = self.cellWidget(row, 5)
            if w:
                device_config['collections'] = unicode(w.currentText()).strip() == 'Y'
            else:
                device_config['collections'] = False
            devices[device_config['uuid']] = device_config
        return devices

    def get_selected_device_info(self):
        if self.currentRow() >= 0:
            (device_config, is_connected) = self.item(self.currentRow(), 1).data(Qt.UserRole).toPyObject()
            return (device_config, is_connected)
        return None, None

    def set_current_row_device_name(self, device_name):
        if self.currentRow() >= 0:
            widget = self.item(self.currentRow(), 1)
            (device_config, is_connected) = widget.data(Qt.UserRole).toPyObject()
            device_config['name'] = device_name
            widget.setData(Qt.UserRole, (device_config, is_connected))
            widget.setText(device_name)

    def delete_selected_row(self):
        if self.currentRow() >= 0:
            self.removeRow(self.currentRow())


class ConfigWidget(QWidget):

    def __init__(self, plugin_action):
        QWidget.__init__(self)
        self.gui = plugin_action.gui
        self._connected_device_info = plugin_action.connected_device_info
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        devices_group_box = QGroupBox('Devices:', self)
        layout.addWidget(devices_group_box)
        devices_group_box_layout = QVBoxLayout()
        devices_group_box.setLayout(devices_group_box_layout)

        self.devices_table = DevicesTableWidget(self)
        devices_group_box_layout.addWidget(self.devices_table)

        buttons_layout = QHBoxLayout()
        devices_group_box_layout.addLayout(buttons_layout)

        self.add_device_btn = QPushButton('Add connected device', self)
        self.add_device_btn.setToolTip(
                'If you do not have a device connected currently, either plug one\n'
                'in now or exit the dialog and connect to folder/iTunes first')
        self.add_device_btn.setIcon(QIcon(I('plus.png')))
        self.add_device_btn.clicked.connect(self._add_device_clicked)
        buttons_layout.addWidget(self.add_device_btn, 1)
        self.rename_device_btn = QtGui.QToolButton(self)
        self.rename_device_btn.setIcon(get_icon('images/rename.png'))
        self.rename_device_btn.setToolTip('Rename the currently connected device')
        self.rename_device_btn.clicked.connect(self._rename_device_clicked)
        buttons_layout.addWidget(self.rename_device_btn)
        self.delete_device_btn = QtGui.QToolButton(self)
        self.delete_device_btn.setIcon(QIcon(I('trash.png')))
        self.delete_device_btn.setToolTip('Delete this device from the device list')
        self.delete_device_btn.clicked.connect(self._delete_device_clicked)
        buttons_layout.addWidget(self.delete_device_btn)

        other_group_box = QGroupBox('Keyboard shortcuts:', self)
        layout.addWidget(other_group_box)
        other_group_box_layout = QGridLayout()
        other_group_box.setLayout(other_group_box_layout)

        c = plugin_prefs[STORE_OPTIONS]
        for key, text in [(KEY_ADD_SHORTCUT, '&Add to default sync list'),
                          (KEY_REMOVE_SHORTCUT, '&Remove from default sync list'),
                          (KEY_VIEW_SHORTCUT, '&View default sync list'),
                          (KEY_SYNC_SHORTCUT, '&Sync now')]:
            label = QLabel(text+':', self)
            lineEdit = QLineEdit(c.get(key, ''), self)
            setattr(self, '_'+key, lineEdit)
            label.setBuddy(lineEdit)
            other_group_box_layout.addWidget(label, other_group_box_layout.rowCount(), 0, 1, 1)
            other_group_box_layout.addWidget(lineEdit, other_group_box_layout.rowCount()-1, 1, 1, 1)

        self._update_from_connection_status(first_time=True)

        device_signals.device_connection_changed.connect(self._on_device_connection_changed)
        device_signals.device_metadata_available.connect(self._on_device_metadata_available)

    def __exit__(self):
        device_signals.device_connection_changed.disconnect()
        device_signals.device_metadata_available.disconnect()

    def save_settings(self):
        device_prefs = self.devices_table.get_data()
        plugin_prefs[STORE_DEVICES] = device_prefs

        new_prefs = {}
        for key in [KEY_ADD_SHORTCUT, KEY_REMOVE_SHORTCUT, KEY_VIEW_SHORTCUT, KEY_SYNC_SHORTCUT]:
            new_prefs[key] = unicode(getattr(self, '_'+key).text()).strip()

        active_device = plugin_prefs[STORE_OPTIONS].get(KEY_ACTIVE_DEVICE, None)
        if active_device:
            # Check that the device still exists and is enabled
            first_active = None
            active_device_exists = False
            for device_config in device_prefs.itervalues():
                if device_config['active']:
                    if not first_active:
                        first_active = device_config['uuid']
                    if device_config['uuid'] == active_device:
                        active_device_exists = True
                        break
            if not active_device_exists:
                active_device = first_active
        if active_device is None and device_prefs:
            active_device = device_prefs.iterkeys().next()
        new_prefs[KEY_ACTIVE_DEVICE] = active_device
        plugin_prefs[STORE_OPTIONS] = new_prefs

    def _on_device_connection_changed(self, is_connected):
        if not is_connected:
            self._connected_device_info = None
            self._update_from_connection_status()

    def _on_device_metadata_available(self):
        self._connected_device_info = self.gui.device_manager.get_current_device_information().get('info', None)
        self._update_from_connection_status()

    def _add_device_clicked(self):
        devices = self.devices_table.get_data()
        drive_info = self._connected_device_info[4]
        if not drive_info:
            # this is an iTunes type device - use the gui name as the uuid
            new_device = {}
            new_device['type'] = self._connected_device_info[0]
            new_device['active'] = True
            new_device['autosync'] = True
            new_device['kindle_col'] = False
            new_device['uuid'] = new_device['type']
            new_device['name'] = new_device['type']
            new_device['location_code'] = ''
            devices[new_device['uuid']] = new_device
        else:
            for location_info in drive_info.values():
                new_device = {}
                new_device['type'] = self._connected_device_info[0]
                new_device['active'] = True
                new_device['autosync'] = True
                new_device['kindle_col'] = False
                new_device['uuid'] = location_info['device_store_uuid']
                new_device['name'] = location_info['device_name']
                new_device['location_code'] = location_info['location_code']
                devices[new_device['uuid']] = new_device
        self.devices_table.populate_table(devices, self._connected_device_info)
        self._update_from_connection_status(update_table=False)

    def _rename_device_clicked(self):
        (device_info, is_connected) = self.devices_table.get_selected_device_info()
        if not device_info:
            return error_dialog(self, 'Rename failed', 'You must select a device first',
                                show=True, show_copy_button=False)
        if not is_connected:
            return error_dialog(self, 'Rename failed',
                                'You can only rename a device that is currently connected',
                                show=True, show_copy_button=False)

        old_name = device_info['name']
        new_device_name, ok = QInputDialog.getText(self, 'Rename device',
                    'Enter a new display name for this device:', text=old_name)
        if not ok:
            # Operation cancelled
            return
        new_device_name = unicode(new_device_name).strip()
        if new_device_name == old_name:
            return
        try:
            self.gui.device_manager.set_driveinfo_name(device_info['location_code'], new_device_name)
            self.devices_table.set_current_row_device_name(new_device_name)
        except:
            return error_dialog(self, 'Rename failed', 'An error occured while renaming.',
                                det_msg=traceback.format_exc(), show=True)

    def _delete_device_clicked(self):
        (device_info, is_connected) = self.devices_table.get_selected_device_info()
        if not device_info:
            return error_dialog(self, 'Delete failed', 'You must select a device first',
                                show=True, show_copy_button=False)
        name = device_info['name']
        if not question_dialog(self, _('Are you sure?'), '<p>'+
                'You are about to remove the <b>%s</b> device from this list. '%name +
                'Any existing sync lists for this device will also be deleted.<br>'
                'Are you sure you want to continue?'):
            return
        self.devices_table.delete_selected_row()
        self._update_from_connection_status(update_table=False)
        # Delete all book lists for this device
        libraries = plugin_prefs[STORE_LIBRARY_LISTS]
        for library_id, device_lists in libraries.iteritems():
            if device_info['uuid'] in device_lists:
                del device_lists[device_info['uuid']]
                libraries[library_id] = device_lists
        plugin_prefs[STORE_LIBRARY_LISTS] = libraries

    def _update_from_connection_status(self, first_time=False, update_table=True):
        if first_time:
            devices = plugin_prefs[STORE_DEVICES]
        else:
            devices = self.devices_table.get_data()

        if self._connected_device_info is None:
            self.add_device_btn.setEnabled(False)
            self.rename_device_btn.setEnabled(False)
        else:
            # Check to see whether we are connected to a device we already know about
            is_new_device = True
            can_rename = False
            drive_info = self._connected_device_info[4]
            if drive_info:
                # This is a non iTunes device that we can check to see if we have the UUID for
                device_uuid = drive_info['main']['device_store_uuid']
                if device_uuid in devices:
                    is_new_device = False
                    can_rename = True
            else:
                # This is a device without drive info like iTunes
                device_type = self._connected_device_info[0]
                if device_type in devices:
                    is_new_device = False

            self.add_device_btn.setEnabled(is_new_device)
            self.rename_device_btn.setEnabled(can_rename)
        if update_table:
            self.devices_table.populate_table(devices, self._connected_device_info)


