#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import threading, re
from functools import partial
from PyQt4.Qt import QMenu, QToolButton
from calibre import prints
from calibre.constants import DEBUG
from calibre.ebooks.metadata import authors_to_string
from calibre.gui2 import error_dialog, info_dialog
from calibre.gui2.actions import InterfaceAction
from calibre.gui2.device import device_signals

import calibre_plugins.book_sync.config as cfg
from calibre_plugins.book_sync.common_utils import set_plugin_icon_resources, get_icon, \
                                                   create_menu_item

PLUGIN_ICONS = ['images/book_sync.png', 'images/view_list.png',
                'images/rename.png',    'images/device.png',
                'images/device_connected.png']

class BookSyncAction(InterfaceAction):

    name = 'Book Sync'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Book Sync', None, None, None)
    popup_type = QToolButton.InstantPopup
    action_type = 'current'

    def genesis(self):
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)
        self.menus_lock = threading.RLock()
        self.sync_lock = threading.RLock()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.menu.aboutToShow.connect(self.about_to_show_menu)

    def initialization_complete(self):
        self.connected_device_info = None
        self.rebuild_menus()
        # Subscribe to device connection events
        device_signals.device_connection_changed.connect(self._on_device_connection_changed)
        device_signals.device_metadata_available.connect(self._on_device_metadata_available)

    def library_changed(self, db):
        # We need to reset our menus after switching libraries
        self.rebuild_menus()

    def rebuild_menus(self):
        with self.menus_lock:
            c = cfg.plugin_prefs[cfg.STORE_OPTIONS]
            m = self.menu
            m.clear()
            (device_names, current_device_uuid, current_device_name) = self._get_device_menu_names()
            self.add_action = create_menu_item(self, m, 'Add to %s sync list'%current_device_name,
                                 image='plus.png',
                                 shortcut=c[cfg.KEY_ADD_SHORTCUT],
                                 triggered=partial(self.add_to_sync_list, None))
            if len(device_names) > 1:
                self.add_sub_menu = m.addMenu(get_icon('plus.png'), 'Add to sync list for')
                self.add_sub_menu.setStatusTip('Add to sync list for the specified device')
                for (device_name, device_uuid) in device_names:
                    create_menu_item(self, self.add_sub_menu, device_name,
                                     tooltip='Add to the sync list for "%s"'%device_name,
                                     triggered=partial(self.add_to_sync_list, device_uuid))
                self.add_all_action = create_menu_item(self, m, 'Add to all sync lists',
                                     image='plus.png',
                                     triggered=self.add_to_all_sync_lists)
            m.addSeparator()
            self.remove_action = create_menu_item(self, m, 'Remove from %s sync list'%current_device_name,
                                 image='minus.png',
                                 shortcut=c.get(cfg.KEY_REMOVE_SHORTCUT, ''),
                                 triggered=partial(self.remove_from_sync_list, None))
            if len(device_names) > 1:
                self.remove_sub_menu = m.addMenu(get_icon('minus.png'), 'Remove from sync list for')
                self.remove_sub_menu.setStatusTip('Remove from sync list for the specified device')
                for (device_name, device_uuid) in device_names:
                    create_menu_item(self, self.remove_sub_menu, device_name,
                                     tooltip='Remove from the sync list for "%s"'%device_name,
                                     triggered=partial(self.remove_from_sync_list, device_uuid))
                self.remove_all_action = create_menu_item(self, m, 'Remove from all sync lists',
                                     image='minus.png',
                                     triggered=self.remove_from_all_sync_lists)
            m.addSeparator()
            self.view_list_action = create_menu_item(self, m, 'View %s sync list'%current_device_name,
                                 image='images/view_list.png',
                                 shortcut=c[cfg.KEY_VIEW_SHORTCUT],
                                 triggered=partial(self.view_sync_list, None))
            if len(device_names) > 1:
                self.view_sub_menu = m.addMenu(get_icon('images/view_list.png'), 'View sync list for')
                self.view_sub_menu.setStatusTip('View sync list for the specified device')
                for device_name, device_uuid in device_names:
                    create_menu_item(self, self.view_sub_menu, device_name,
                                     tooltip='View the sync list for "%s"'%device_name,
                                     triggered=partial(self.view_sync_list, device_uuid))
            m.addSeparator()
            if len(device_names) > 1:
                self.device_sub_menu = m.addMenu(get_icon('images/device.png'), 'Set default device list')
                self.device_sub_menu.setStatusTip('Switch the device sync list to use as the current default')
                for (device_name, device_uuid) in device_names:
                    is_checked = device_uuid == current_device_uuid
                    create_menu_item(self, self.device_sub_menu, device_name, is_checked=is_checked,
                                     tooltip='Set your default list to "%s"'%device_name,
                                     triggered=partial(self.switch_current_device, device_uuid))
                m.addSeparator()
            self.sync_now_action = create_menu_item(self, m, 'Sync Now', 'images/book_sync.png',
                                     shortcut=c.get(cfg.KEY_SYNC_SHORTCUT, ''),
                                     triggered=partial(self.sync_now, current_device_uuid))
            m.addSeparator()
            create_menu_item(self, m, _('&Customize plugin')+'...', 'config.png',
                             triggered=self.show_configuration)

            if not current_device_uuid:
                self.add_action.setVisible(False)
                self.remove_action.setVisible(False)
                self.view_list_action.setVisible(False)

    def about_to_show_menu(self):
        self.sync_now_action.setEnabled(False)
        if self.gui.device_manager.is_device_connected:
            enabled = self._is_books_for_connected_device()
            self.sync_now_action.setEnabled(enabled)

    def add_to_sync_list(self, device_uuid=None):
        if device_uuid is None:
            device_uuid = cfg.plugin_prefs[cfg.STORE_OPTIONS].get(cfg.KEY_ACTIVE_DEVICE, None)
        if device_uuid is None:
            return error_dialog(self.gui, 'Cannot add to sync list',
                                'No device identifier specified', show=True)
        device = cfg.plugin_prefs[cfg.STORE_DEVICES].get(device_uuid, None)
        if device is None:
            return error_dialog(self.gui, 'Cannot add to sync list',
                                'No device found for UUID: %s'%device_uuid, show=True)

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        selected_ids = set(self.gui.library_view.get_selected_ids())
        db = self.gui.library_view.model().db
        with self.sync_lock:
            ids = set(cfg.get_book_list_for_device(db, device_uuid))
            initial_count = len(ids)

            ids |= selected_ids

            diff_count = len(ids) - initial_count
            if diff_count == 0:
                return error_dialog(self.gui, 'Cannot add to sync list',
                                    'The selected book(s) already exist on this list', show=True)
            cfg.set_book_list_for_device(db, device_uuid, list(ids))
            message = 'Added %d books to your %s sync list' % (diff_count, device['name'])
            self.gui.status_bar.showMessage(message, 3000)

    def add_to_all_sync_lists(self):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        selected_ids = set(self.gui.library_view.get_selected_ids())
        db = self.gui.library_view.model().db
        with self.sync_lock:
            device_uuids = cfg.get_active_device_uuids()
            updated_device_lists = 0
            for device_uuid in device_uuids:
                ids = set(cfg.get_book_list_for_device(db, device_uuid))
                initial_count = len(ids)

                ids |= selected_ids

                diff_count = len(ids) - initial_count
                if diff_count > 0:
                    updated_device_lists += 1
                    cfg.set_book_list_for_device(db, device_uuid, list(ids))

            if updated_device_lists == 0:
                return error_dialog(self.gui, 'Cannot add to sync lists',
                                    'The selected book(s) already exist on all lists', show=True)
            message = 'Updated %d sync lists' % updated_device_lists
            self.gui.status_bar.showMessage(message, 3000)

    def remove_from_sync_list(self, device_uuid=None):
        if device_uuid is None:
            device_uuid = cfg.plugin_prefs[cfg.STORE_OPTIONS].get(cfg.KEY_ACTIVE_DEVICE, None)
        if device_uuid is None:
            return error_dialog(self.gui, 'Cannot remove from sync list',
                                'No device identifier specified', show=True)
        device = cfg.plugin_prefs[cfg.STORE_DEVICES].get(device_uuid, None)
        if device is None:
            return error_dialog(self.gui, 'Cannot remove from sync list',
                                'No device found for UUID: %s'%device_uuid, show=True)

        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        selected_ids = set(self.gui.library_view.get_selected_ids())
        db = self.gui.library_view.model().db
        with self.sync_lock:
            ids = set(cfg.get_book_list_for_device(db, device_uuid))
            initial_count = len(ids)

            ids -= selected_ids

            diff_count = initial_count - len(ids)
            if diff_count == 0:
                return error_dialog(self.gui, 'Cannot remove from sync list',
                                    'The selected book(s) already exist on this list', show=True)
            cfg.set_book_list_for_device(db, device_uuid, list(ids))
            message = 'Removed %d books from your %s sync list' % (diff_count, device['name'])
            self.gui.status_bar.showMessage(message, 3000)
        if unicode(self.gui.search.text()).startswith('marked:sync_list'):
            self.view_sync_list(device_uuid)

    def remove_from_all_sync_lists(self):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        selected_ids = set(self.gui.library_view.get_selected_ids())
        db = self.gui.library_view.model().db
        with self.sync_lock:
            device_uuids = cfg.get_active_device_uuids()
            updated_device_lists = 0
            for device_uuid in device_uuids:
                ids = set(cfg.get_book_list_for_device(db, device_uuid))
                initial_count = len(ids)

                ids -= selected_ids

                diff_count = initial_count - len(ids)
                if diff_count > 0:
                    updated_device_lists += 1
                    cfg.set_book_list_for_device(db, device_uuid, list(ids))

            if updated_device_lists == 0:
                return error_dialog(self.gui, 'Cannot remove from sync lists',
                                    'The selected book(s) does not exist on any lists', show=True)
            message = 'Updated %d sync lists' % updated_device_lists
            self.gui.status_bar.showMessage(message, 3000)
        if unicode(self.gui.search.text()).startswith('marked:sync_list'):
            self.view_sync_list()

    def view_sync_list(self, device_uuid=None):
        if device_uuid is None:
            device_uuid = cfg.plugin_prefs[cfg.STORE_OPTIONS].get(cfg.KEY_ACTIVE_DEVICE, None)
        if device_uuid is None:
            return error_dialog(self.gui, 'Cannot view sync list',
                                'No device identifier specified', show=True)
        device = cfg.plugin_prefs[cfg.STORE_DEVICES].get(device_uuid, None)
        if device is None:
            return error_dialog(self.gui, 'Cannot view sync list',
                                'No device found for UUID: %s'%device_uuid, show=True)
        db = self.gui.library_view.model().db
        ids = cfg.get_book_list_for_device(db, device_uuid)

        # Try to come up with a save name for appending to marked text
        safe_name = device['name'].lower().replace(' ', '_')
        safe_name = re.sub('([^a-z0-9_])', '', safe_name)
        marked_ids = dict.fromkeys(ids, 'sync_list_'+safe_name)
        db.set_marked_ids(marked_ids)
        # Search to display the list contents
        self.gui.search.set_search_string('marked:sync_list_'+safe_name)

    def switch_current_device(self, device_uuid):
        c = cfg.plugin_prefs[cfg.STORE_OPTIONS]
        c[cfg.KEY_ACTIVE_DEVICE] = device_uuid
        cfg.plugin_prefs[cfg.STORE_OPTIONS] = c
        self.rebuild_menus()

    def sync_now(self, device_uuid):
        c = cfg.plugin_prefs[cfg.STORE_DEVICES]
        device = cfg.plugin_prefs[cfg.STORE_DEVICES].get(device_uuid, None)
        if device is None:
            return error_dialog(self.gui, 'Cannot sync to device',
                                'No device found for UUID: %s'%device_uuid, show=True)
        db = self.gui.library_view.model().db
        ids = cfg.get_book_list_for_device(db, device_uuid)
        # Will only be able to sync books that have a format
        # Any that do not we will keep in our list and not attempt to sync
        no_format_ids = []
        for id in ids:
            dbfmts = self.gui.library_view.model().db.formats(id, index_is_id=True)
            if not dbfmts:
                no_format_ids.append(id)
        ids = set(ids) - set(no_format_ids)
        if DEBUG and no_format_ids:
            prints('BOOKSYNC: Skipping %d books with no formats'%len(no_format_ids))
        if ids:
            loc = None
            if device['location_code'] == 'A':
                loc = 'carda'
            elif device['location_code'] == 'B':
                loc = 'cardb'
            message = 'BOOKSYNC: Syncing %d books to: %s (location:%s)'%(len(ids), device_uuid, loc)
            self.gui.status_bar.showMessage(message)
            if DEBUG:
                prints(message)
            self.gui.sync_to_device(on_card=loc, delete_from_library=False, send_ids=ids)
            cfg.set_book_list_for_device(db, device_uuid, no_format_ids)
            create_collections = device.get('collections', False)
            if create_collections:
                self.create_kindle_collections()

    def create_kindle_collections(self):
        # Check for the Kindle Collections plugin being installed
        if DEBUG:
            prints('BOOKSYNC: Attempting to recreate Kindle collections')
        plugin = self.gui.iactions.get('Kindle Collections', None)
        if not plugin:
            return info_dialog(self.gui, 'Kindle Collections Failed',
                               'You must have the Kindle Collections plugin installed '
                               'in order to recreate collections after a sync.',
                               show=True)
        else:
            plugin.create_kindle_collections()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)

    def _convert_calibre_ids_to_books(self, db, ids):
        books = []
        for calibre_id in ids:
            mi = db.get_metadata(calibre_id, index_is_id=True)
            book = {}
            book['calibre_id'] = mi.id
            book['title'] = mi.title
            book['author'] = authors_to_string(mi.authors)
            book['author_sort'] = mi.author_sort
            book['series'] = mi.series
            if mi.series:
                book['series_index'] = mi.series_index
            else:
                book['series_index'] = 0
            books.append(book)
        return books

    def _get_device_menu_names(self):
        c = cfg.plugin_prefs[cfg.STORE_OPTIONS]
        current_device_uuid = c.get(cfg.KEY_ACTIVE_DEVICE, None)
        current_device_name = 'default'
        c = cfg.plugin_prefs[cfg.STORE_DEVICES]
        names = []

        for device in c.itervalues():
            if device['active']:
                uuid = device['uuid']
                device_name = device['name']
                names.append((device_name, uuid))
                if uuid == current_device_uuid:
                    current_device_name = device_name
        names = sorted(names, key=lambda k: k[0])
        if current_device_name == 'default':
            # Did not find our current uuid
            current_device_uuid = None
        return (names, current_device_uuid, current_device_name)

    def _on_device_connection_changed(self, is_connected):
        if not is_connected:
            if DEBUG:
                prints('BOOKSYNC: Device disconnected')
            self.connected_device_info = None
            self.rebuild_menus()

    def _on_device_metadata_available(self):
        self.connected_device_info = self.gui.device_manager.get_current_device_information().get('info', None)
        drive_info = self.connected_device_info[4]
        if DEBUG:
            prints('BOOKSYNC: Metadata available:',drive_info)
        self.rebuild_menus()
        # Lookup to see whether we should sync immediately to this device.
        uuids_to_sync = self._get_connected_uuids_to_sync()
        if uuids_to_sync:
            with self.sync_lock:
                for device_uuid in uuids_to_sync:
                    self.sync_now(device_uuid)

    def _is_specific_device_connected(self, device_uuid):
        if not self.connected_device_info:
            return False
        if device_uuid == self.connected_device_info[0]:
            return True
        for location_info in self.connected_device_info[4].itervalues():
            if location_info['device_store_uuid'] == device_uuid:
                return True
        return False

    def _is_books_for_connected_device(self):
        c = cfg.plugin_prefs[cfg.STORE_DEVICES]
        db = self.gui.library_view.model().db
        if self.connected_device_info:
            if not self.connected_device_info[4]:
                # Will use the device type as the UUID
                device_uuid = self.connected_device_info[0]
                ids = cfg.get_book_list_for_device(db, device_uuid)
                return len(ids) > 0
            else:
                for location_info in self.connected_device_info[4].itervalues():
                    device_uuid = location_info['device_store_uuid']
                    ids = cfg.get_book_list_for_device(db, device_uuid)
                    if len(ids) > 0:
                        return True
        return False

    def _get_connected_uuids_to_sync(self):
        c = cfg.plugin_prefs[cfg.STORE_DEVICES]
        device_uuids = []
        if self.connected_device_info:
            if not self.connected_device_info[4]:
                # Will use the device type as the UUID
                uuid = self.connected_device_info[0]
                self._add_device_to_list_if_should_sync(device_uuids, uuid)
            else:
                for location_info in self.connected_device_info[4].itervalues():
                    uuid = location_info['device_store_uuid']
                    self._add_device_to_list_if_should_sync(device_uuids, uuid)
        return device_uuids

    def _add_device_to_list_if_should_sync(self, device_uuids, uuid):
        c = cfg.plugin_prefs[cfg.STORE_DEVICES]
        device = c.get(uuid, None)
        if device:
            if device['active'] and device['autosync']:
                if DEBUG:
                    prints('BOOKSYNC: Device found to sync to:', device['name'], device['uuid'])
                device_uuids.append(uuid)
            elif DEBUG:
                if not device['active']:
                    prints('BOOKSYNC: Not syncing to device as not active')
                else:
                    prints('BOOKSYNC: Not syncing to device as autosync is false')
