#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Grant Drake <grant.drake@gmail.com>'
__docformat__ = 'restructuredtext en'

import traceback
from PyQt4.Qt import QApplication, Qt, QThread, pyqtSignal
from calibre.gui2.actions import InterfaceAction

from calibre_plugins.plugin_updater.common_utils import set_plugin_icon_resources, get_icon
from calibre_plugins.plugin_updater.dialogs import (PluginUpdaterDialog, get_plugin_updates_available,
                                                    FILTER_ALL, FILTER_UPDATE_AVAILABLE)

PLUGIN_ICONS = ['images/plugin_updater.png', 'images/plugin_deprecated.png',
                'images/plugin_upgrade_ok.png', 'images/plugin_upgrade_valid.png',
                'images/plugin_upgrade_invalid.png', 'images/plugin_disabled_ok.png',
                'images/plugin_disabled_valid.png', 'images/plugin_disabled_invalid.png',
                'images/plugin_new_valid.png', 'images/plugin_new_invalid.png',
                'images/plugin_updater_updates.png', 'images/mobileread.png']


class CheckForUpdates(QThread):

    update_found = pyqtSignal(object)
    INTERVAL = 4*60*60 # Every four hours

    def __init__(self, parent):
        QThread.__init__(self, parent)

    def run(self):
        while True:
            try:
                update_plugins = get_plugin_updates_available()
                if get_plugin_updates_available():
                    self.update_found.emit(update_plugins)
            except:
                traceback.print_exc()
            self.sleep(self.INTERVAL)


class PluginUpdaterAction(InterfaceAction):

    name = 'Plugin Updater'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = (_('Plugin Updater'), None, None, None)
    action_type = 'current'

    def genesis(self):
        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(self.name, icon_resources)

        # Assign our menu to this action and an icon
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.check_for_plugin_updates)

    def initialization_complete(self):
        if not self.gui.opts.no_update_check:
            self.update_checker = CheckForUpdates(self)
            self.update_checker.update_found.connect(self.update_found,
                    type=Qt.QueuedConnection)
            self.update_checker.start()

    def check_for_plugin_updates(self):
        # Get the user to choose a plugin to install
        initial_filter = FILTER_UPDATE_AVAILABLE
        mods = QApplication.keyboardModifiers()
        if mods & Qt.ControlModifier or mods & Qt.ShiftModifier:
            initial_filter = FILTER_ALL

        d = PluginUpdaterDialog(self.gui, initial_filter=initial_filter)
        d.update_found.connect(self.update_found)
        d.exec_()

    def update_found(self, updates):
        # Change the plugin icon to indicate there are updates available
        if updates:
            self.qaction.setText(_('Plugin Updater')+'*')
            self.qaction.setIcon(get_icon('images/plugin_updater_updates.png'))
            self.qaction.setToolTip(_('Updates are available for your plugins'))
        else:
            self.qaction.setText(_('Plugin Updater'))
            self.qaction.setIcon(get_icon('images/plugin_updater.png'))
            self.qaction.setToolTip(_('Plugin Updater'))
