#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__   = 'GPL v3'
__copyright__ = '2011, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

from functools import partial
from PyQt4.Qt import Qt, QApplication, QMenu, QToolButton
from calibre.gui2 import error_dialog
from calibre.gui2.actions import InterfaceAction

import calibre_plugins.temp_marker.config as cfg
from calibre_plugins.temp_marker.common_utils import set_plugin_icon_resources, get_icon, \
                                                     create_menu_item

PLUGIN_ICONS = ['images/temp_marker.png']

class TempMarkerAction(InterfaceAction):

    name = 'Temp Marker'
    # Create our top-level menu/toolbar action (text, icon_path, tooltip, keyboard shortcut)
    action_spec = ('Temp Marker', None, None, None)
    popup_type = QToolButton.MenuButtonPopup
    action_type = 'current'

    def genesis(self):
        self.marked_ids = set()
        self.menu = QMenu(self.gui)

        # Read the plugin icons and store for potential sharing with the config widget
        icon_resources = self.load_resources(PLUGIN_ICONS)
        set_plugin_icon_resources(icon_resources)

        self.rebuild_menus()

        # Assign our menu to this action and an icon
        self.qaction.setMenu(self.menu)
        self.qaction.setIcon(get_icon(PLUGIN_ICONS[0]))
        self.qaction.triggered.connect(self.toolbar_triggered)

    def rebuild_menus(self):
        m = self.menu
        m.clear()
        c = cfg.plugin_prefs[cfg.STORE_NAME]
        create_menu_item(self, m, 'Mark selected', 'plus.png',
                         shortcut=c[cfg.KEY_MARK_SHORTCUT],
                         triggered=partial(self.mark_selected, add=True))
        create_menu_item(self, m, 'Show marked', 'search.png',
                         shortcut=c[cfg.KEY_SHOW_SHORTCUT],
                         triggered=self.show_marked)
        m.addSeparator()
        create_menu_item(self, m, 'Unmark selected', 'minus.png',
                         shortcut=c[cfg.KEY_UNMARK_SHORTCUT],
                         triggered=partial(self.mark_selected, add=False))
        create_menu_item(self, m, 'Clear all marked', 'clear_left.png',
                         shortcut=c[cfg.KEY_CLEAR_SHORTCUT],
                         triggered=self.clear_all_marked)
        m.addSeparator()
        create_menu_item(self, m, _('&Customize plugin')+'...', 'config.png',
                         triggered=self.show_configuration)

    def toolbar_triggered(self):
        mods = QApplication.keyboardModifiers()
        if mods & Qt.ControlModifier or mods & Qt.ShiftModifier:
            self.show_marked()
        else:
            self.mark_selected(add=True)

    def mark_selected(self, add=True):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return error_dialog(self.gui, 'No rows selected',
                                'You must select one of more books to perform this action.',
                                show=True)
        ids = set(self.gui.library_view.get_selected_ids())
        if add:
            self.marked_ids.update(ids)
        else:
            self.marked_ids.difference_update(ids)
        self.gui.library_view.model().db.set_marked_ids(self.marked_ids)

    def show_marked(self):
        self.gui.search.set_search_string('marked:true')

    def clear_all_marked(self):
        self.marked_ids.clear()
        self.gui.library_view.model().db.set_marked_ids([])
        if unicode(self.gui.search.text()).startswith('marked:'):
            self.gui.search.clear()

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
