//////////////////////////////////////////////////////////////////////
// sysinfo: Show System and Process information for DR ereaders
//
// Copyright (C) 2011  Iñigo Serna <inigoserna@gmail.com>
// Released under GPL v3+
//
// To compile:
// $ valac --vapidir=. --pkg gtk+-2.0 --pkg=gdk-x11-2.0 --pkg liberipc --pkg posix SysInfo.vala irex.vala -o SysInfo
//
// Time-stamp: <2011-03-26 01:03:24 inigo>
//////////////////////////////////////////////////////////////////////


using Gdk;
using Gtk;
using Pango;
using irex;


//////////////////////////////////////////////////////////////////////
///// Memory, CPU, Load, Disk, Process
namespace Resources {

    public struct Memory {
        public int used;
        public int free;
        public int total;
    }


    public struct CPU {
        public int usr;
        public int sys;
        public int idle;
        public int io;
    }


    public struct Load {
        public double avg_1m;
        public double avg_5m;
        public double avg_15m;
    }


    public class Disk {
        public string dev;
        public string size;
        public string used;
        public string available;
        public string percent;
        public string mountpoint;

        public Disk(string dev, string size,string used, string available,
                    string percent, string mountpoint) {
            this.dev = dev;
            this.size = size;
            this.used = used;
            this.available = available;
            this.percent = percent;
            this.mountpoint = mountpoint;
        }
    }


    public class Process {
        public int pid;
        public int ppid;
        public int mem;
        public int cpu;
        public string state;
        public string cmd;

        public Process(int pid, int ppid, int mem, int cpu, string state, string cmd) {
            this.pid = pid;
            this.ppid = ppid;
            this.mem = mem;
            this.cpu = cpu;
            this.state = state;
            this.cmd = cmd;
        }
    }


    ///// Utils
    private string open(string filename) {
        string buf;
        try {
            FileUtils.get_contents (filename, out buf);
        } catch (FileError e) {
            stderr.printf ("%s\n", e.message);
        }
        return buf;
    }


    private string popen(string cmd) {
        string strout;
        string strerr;
        int res;
        try {
            GLib.Process.spawn_command_line_sync(cmd, out strout, out strerr, out res);
        } catch (GLib.Error e) {
            error ("%s", e.message);
        }
        return strout;
    }


    public void get_system_info(out string now, out string name, out string cpu,
                                out double bogomips, out string kernel, out string uptime) {
        // Datetime
        // next does not exist in GLib < 2.26
        // now = new DateTime.now_local().format("%A, %d %B %Y    %H:%M:%S");
        now = popen("date +'%A, %d %B %Y    %H:%M:%S'").split("\n")[0];
        // CPU info
        string buf = open("/proc/cpuinfo");
        MatchInfo mi = null;
        Regex regex_name = /^\s*Hardware\s+:\s+(.*)$/;
        Regex regex_cpu = /^\s*Processor\s+:\s+(.*)$/;
        Regex regex_bogomips = /^\s*BogoMIPS\s+:\s+([\d.]+)\s*$/;
        foreach(string line in buf.split("\n")) {
            if (regex_name.match(line, 0, out mi)) {
                name = mi.fetch(1);
            }
            else if (regex_cpu.match(line, 0, out mi)) {
                cpu = mi.fetch(1);
            }
            else if (regex_bogomips.match(line, 0, out mi)) {
                bogomips = double.parse(mi.fetch(1));
            }
        }
        // Kernel version
        kernel = open("/proc/version").split(")")[0] + ")";
        // Uptime
        double h, m, s;
        s = double.parse(open("/proc/uptime").split(" ")[0]);
        m = Math.floor(s/60);
        h = Math.floor(m/60);
        m = m % 60;
        s = s % 60;
        if (h == 0)
            uptime = "%d mins %2d secs".printf((int) m, (int) s);
        else
            uptime = "%d hours %2d mins %2d secs".printf((int) h, (int) m, (int) s);
    }


    public void get_top_info(out Memory mem, out CPU cpu, out Load load,
                             out List<Process> processes) {
        string strout = popen("top -b -n 1 -d 0");
        MatchInfo mi = null;
        Regex regex_m = /^Mem:\s+(\d+)K used,\s+(\d+)K free,\s+(\d+)K shrd,\s+(\d+)K buff,\s+(\d+)K cached\s*$/;
        Regex regex_c = /^CPU:\s+(\d+)% usr\s+(\d+)% sys\s+(\d+)% nice\s+(\d+)% idle\s+(\d+)% io\s+(\d+)% irq\s+(\d+)% softirq\s*$/;
        Regex regex_p = /^\s+(\d+)\s+(\d+)\s+(\w+)\s+([\w <]+)\s+(\d+)\s+(\d+)%\s+(\d+)%\s+(.*)\s*$/;
        Regex regex_l = /^Load average:\s+([\d.]+)\s+([\d.]+)\s+([\d.]+)\s*$/;
        foreach(string line in strout.split("\n")) {
            if (regex_m.match(line, 0, out mi)) {
                mem.used = int.parse(mi.fetch(1));
                mem.free = int.parse(mi.fetch(2));
                mem.total = int.parse(mi.fetch(1)) + int.parse(mi.fetch(2));
            }
            else if (regex_c.match(line, 0, out mi)) {
                cpu.usr = int.parse(mi.fetch(1));
                cpu.sys = int.parse(mi.fetch(2));
                cpu.idle = int.parse(mi.fetch(4));
                cpu.io = int.parse(mi.fetch(5));
            }
            else if (regex_l.match(line, 0, out mi)) {
                load.avg_1m = double.parse(mi.fetch(1));
                load.avg_5m = double.parse(mi.fetch(2));
                load.avg_15m = double.parse(mi.fetch(3));
            }
            else if (regex_p.match(line, 0, out mi)) {
                processes.append(new Process(int.parse(mi.fetch(1)), int.parse(mi.fetch(2)),
                                             int.parse(mi.fetch(6)), int.parse(mi.fetch(7)),
                                             mi.fetch(4), mi.fetch(8)));
            }
        }
    }


    public void get_disks(out List<Disk> disks) {
        string strout = popen("df -h");
        MatchInfo mi = null;
        Regex regex = /^\s*([\w\d\/]+)\s+([\w\d.]+)\s+([\w\d.]+)\s+([\w\d.]+)\s+([\d%]+)\s+([\w\d\/]+)\s*$/;
        foreach(string line in strout.split("\n")) {
            if (regex.match(line, 0, out mi)) {
                disks.append(new Disk(mi.fetch(1), mi.fetch(2), mi.fetch(3),
                                      mi.fetch(4), mi.fetch(5), mi.fetch(6)));
            }
        }
    }
}


//////////////////////////////////////////////////////////////////////
///// SysInfo
public class SysInfo : Gtk.Window, irex.Application {

    ///// Variables
    private int timeout = 60000;
    private uint timer_id;
    private bool is_fullscreen = false;
    private Gtk.Label label_sysinfo;
    private Gtk.ListStore model_disks;
    private Gtk.TreeView view_disks;
    private Gtk.ListStore model_procs;
    private Gtk.TreeView view_procs;
    private Gtk.MessageDialog dialog;
    irex.IPC ipc;
    irex.MenuManager menu_manager;
    irex.Menu menu_main;

    ///// Constants
    private const string DEFAULT_STYLE = """
style 'basic_style' {
  GtkRange::slider_width = 5
  GtkTreeView::horizontal-separator = 3
  GtkTreeView::vertical-separator = 0
}
class 'GtkWidget' style 'basic_style'
""";
    private const string SYSINFO_TMPL = """

<span size='large'>%s</span>
%s


<span size='x-large'><b>Hardware</b></span>

    <u>Processor:</u> %s, %.2f BogoMIPS
    <u>Kernel:</u> %s


<span size='x-large'><b>Resources</b></span>

    <u>Uptime:</u> %s
    <u>Battery:</u> %d%% (%s)
    <u>Load Average:</u>    1 min: %.2f    5 min: %.2f    15 min: %.2f
    <u>CPU:</u>    %d%% user    %d%% sys    %d%% idle    %d%% io
    <u>Memory:</u>    %d KB total    %d KB free    %d KB used


<span size='x-large'><b>Storage</b></span>
""";


    ///// Constructor
    construct {
        ipc = new irex.IPC("SysInfo", "1.0", this);
        menu_manager = new irex.MenuManager(ipc);
        menu_main = new irex.Menu(menu_manager, "menumain", "SysInfo");
        var group2 = menu_main.addGroup("grouprefresh", "Refresh...");
        group2.addItem("refresh_30s", "Refresh 30 sec", "no-icon");
        group2.addItem("refresh_1m", "Refresh 1 min", "no-icon");
        group2.addItem("refresh_5m", "Refresh 5 min", "no-icon");
        group2.addItem("refresh_15m", "Refresh 15 min", "no-icon");
        var group1 = menu_main.addGroup("groupmain", "Main Buttons");
        group1.addItem("close", "Close", "close");
        group1.addItem("fullscreen", "Fullscreen", "mode_full_screen");
        menu_main.realise();
        menu_main.show();
    }


    public SysInfo() {
        this.ui_build();
        ipc.send_startup_complete(); // inform ipc we are ready
    }


    ///// UI
    private void ui_build() {
        this.title = "System Information";
        this.set_default_size(800, 1024);
        this.set_border_width(20);
        this.position = WindowPosition.CENTER;
        this.delete_event.connect((w) => { Gtk.main_quit(); return false; });
        this.destroy.connect(Gtk.main_quit);
        var vbox = new Gtk.VBox(false, 0);
        this.add(vbox);

        // header: title and quit button
        var hbox = new Gtk.HBox(false, 10);
        var evbox = new Gtk.EventBox();
        var color = Gdk.Color();
        Gdk.Color.parse("black", out color);
        evbox.modify_bg(StateType.NORMAL, color);
        var title = new Gtk.Label("");
        title.set_markup("<span size='xx-large' color='white' weight='bold'>System Information</span>");
        evbox.add(title);
        hbox.pack_start(evbox, true, true, 0);
        var evbox_quit = new Gtk.EventBox();
        evbox_quit.set_events(EventMask.BUTTON_PRESS_MASK);
        evbox_quit.button_press_event.connect((evbox) => { Gtk.main_quit(); return false; });
        var img = new Gtk.Image.from_file("_sysinfo/quit.png");
        evbox_quit.add(img);
        hbox.pack_end(evbox_quit, false, false, 0);
        vbox.pack_start(hbox, false, false, 10);

        // notebook
        var nb = new Gtk.Notebook();
        vbox.pack_start(nb, true, true, 10);
        var cr_left = new Gtk.CellRendererText();
        cr_left.set("scale-set", true, "scale", 0.8, "xpad", 5, "xalign", 0.0,
                       "ellipsize-set", true, "ellipsize", Pango.EllipsizeMode.END);
        var cr_right = new Gtk.CellRendererText();
        cr_right.set("scale-set", true, "scale", 0.8, "xpad", 5, "xalign", 1.0,
                        "ellipsize-set", true, "ellipsize", Pango.EllipsizeMode.END);

        // page 1: summary
        var vbox2 = new Gtk.VBox(false, 0);
        nb.append_page(vbox2, new Gtk.Label(" Summary "));
        //         resources
        label_sysinfo = new Gtk.Label("");
        vbox2.pack_start(label_sysinfo, false, false, 0);
        //         disks
        model_disks = new Gtk.ListStore(6, typeof(string), typeof(string), typeof(string),
                                        typeof(string), typeof(string), typeof(string));
        view_disks = new Gtk.TreeView();
        view_disks.set("rules-hint", true, "model", model_disks);
        string[] colsname1 = { "Device", "Size", "Used", "Available", "Use%", "Mounted on" };
        for(int i=0; i<6; i++) {
            var col = new Gtk.TreeViewColumn.with_attributes(colsname1[i],
                                                             (i==0 || i==5) ? cr_left : cr_right,
                                                             "text", i);
            col.set("expand", true);
            view_disks.append_column(col);
        }
        vbox2.pack_start(view_disks, false, false, 0);

        // page 2: processes
        model_procs = new Gtk.ListStore(6, typeof(int), typeof(int),
                                        typeof(string), typeof(string),
                                        typeof(string), typeof(string));
        view_procs = new Gtk.TreeView();
        view_procs.button_press_event.connect(this.cb_tv_clicked);
        view_procs.set("rules-hint", true, "model", model_procs);

        string[] colsname2 = { " PID ", " PPID", "Mem", "CPU", "State", "Command" };
        for(int i=0; i<6; i++) {
            var col = new Gtk.TreeViewColumn.with_attributes(colsname2[i],
                                                             (i==5) ? cr_left : cr_right,
                                                             "text", i);
            col.set_sort_column_id(i);
            view_procs.append_column(col);
        }
        var scrollwin = new Gtk.ScrolledWindow (null, null);
        scrollwin.set_policy(PolicyType.AUTOMATIC, PolicyType.AUTOMATIC);
        scrollwin.add(view_procs);
        nb.append_page(scrollwin, new Gtk.Label(" Processes "));

        // end UI stuff
        if (this.is_fullscreen)
            this.fullscreen();
        else
            this.unfullscreen();
        dialog = new Gtk.MessageDialog(this,
                                       Gtk.DialogFlags.DESTROY_WITH_PARENT | Gtk.DialogFlags.MODAL,
                                       Gtk.MessageType.QUESTION, Gtk.ButtonsType.YES_NO, "");
        dialog.hide();
        this.show_all();
    }


    ///// DR-specific stuff
    public Gtk.Window getMainWindow() {
        return this;
    }

    public void onMenuClick(string item, string group, string menu, string state) {
        stdout.printf("onMenuClick: %s %s %s %s\n", item, group, menu, state);
        if (group=="SysInfo_groupmain") {
            if (item=="close") {
                Gtk.main_quit();
            }
            if (item=="fullscreen") {
                is_fullscreen = !is_fullscreen;
                if (is_fullscreen) {
                    menu_manager.set_item_state("fullscreen", group, "selected");
                    this.fullscreen();
                } else {
                    menu_manager.set_item_state("fullscreen", group, "normal");
                    this.unfullscreen();
                }
            }
        }
        else if (group=="SysInfo_grouprefresh") {
            string[] freqs = {"refresh_30s", "refresh_1m", "refresh_5m", "refresh_15m"};
            foreach(string freq in freqs) {
                menu_manager.set_item_state(freq, group, "normal");
            }
            if (item=="refresh_30s") {
                timeout = 30000;
            } else if (item=="refresh_1m") {
                timeout = 60000;
            } else if (item=="refresh_5m") {
                timeout = 300000;
            } else if (item=="refresh_15m") {
                timeout = 900000;
            } else {
                return;
            }
            menu_manager.set_item_state(item, group, "selected");
            GLib.Source.remove(timer_id);
            timer_id = GLib.Timeout.add(timeout, ui_refresh);
        }
    }

    public void onWindowChange(int xid, bool activated) {
        stdout.printf("onWindowChange: %d, %d\n", xid, (int) activated);
        if (activated)
            menu_main.show();
    }

    public bool onFileOpen(string filename, out int xid, out string error) { return true; }
    public bool onFileClose(string filename) { return true; }
    public void onPrepareUnmount(string device) {}
    public void onUnmounted(string device) {}
    public void onMounted(string device) {}
    public void onPrepareHibernate() {}
    public void onChangedLocale(string locale) {}
    public void onChangedOrientation(string orientation) {}


    ///// calllbacks
    private bool cb_tv_clicked(Gdk.EventButton ev) {
        Gtk.TreeIter iter;
        Gtk.TreePath path;
        Gtk.TreeViewColumn col;
        double cx, cy;
        if (!view_procs.get_path_at_pos((int) ev.x, (int) ev.y, out path, out col, out cx, out cy))
            return false;
        if (!model_procs.get_iter_from_string(out iter, path.to_string()))
            return false;
        GLib.Value val1, val2;
        model_procs.get_value(iter, 0, out val1);
        model_procs.get_value(iter, 5, out val2);
        int pid = val1.get_int();
        string cmd = val2.get_string();
        string buf = @"<big><b>Kill process $pid?</b></big>\n\n<b>Command:</b> $cmd\n\n<big>Are you sure?</big>";
        dialog.set_markup(buf);
        if (dialog.run() == Gtk.ResponseType.YES) {
            try {
                GLib.Process.spawn_command_line_sync("kill -9 %d".printf(pid));
            } catch (GLib.Error e) {
                error ("%s", e.message);
            }
            this.ui_refresh();
        }
        dialog.hide();
        return false;
    }


    private int sort_percent(int i1, int i2) {
        if (i1 < i2)
            return -1;
        else if (i1 > i2)
            return 1;
        else
            return 0;
    }


    private int sort_percent2(Gtk.TreeModel model, Gtk.TreeIter iter1, Gtk.TreeIter iter2) {
        GLib.Value el1, el2;
        model.get_value(iter1, 2, out el1);
        model.get_value(iter2, 2, out el2);
        return sort_percent(int.parse(el1.get_string()[0:-1]), int.parse(el2.get_string()[0:-1]));
    }


    private void populate_model_disks(List<Resources.Disk> disks) {
        Gtk.TreeIter iter;
        model_disks.clear();
        foreach(Resources.Disk d in disks) {
            model_disks.append(out iter);
            // (dev, size, used, available, percent, mountpoint)
            model_disks.set(iter, 0, d.dev, 1, d.size, 2, d.used, 3, d.available,
                            4, d.percent, 5, d.mountpoint);
        }
    }


    private void populate_model_procs(List<Resources.Process> processes) {
        Gtk.TreeIter iter;
        model_procs.clear();
        foreach(Resources.Process p in processes) {
            model_procs.append(out iter);
            // (pid, ppid, mem, cpu, stat, cmd)
            model_procs.set(iter, 0, p.pid, 1, p.ppid,
                            2, p.mem.to_string()+"%", 3, p.cpu.to_string()+"%",
                            4, p.state, 5, p.cmd);
        }
        model_procs.set_sort_column_id(0, SortType.ASCENDING);
        model_procs.set_sort_func(2, sort_percent2);
        model_procs.set_sort_func(3, sort_percent2);
    }


    public bool ui_refresh() {
        string now, name, cpuinfo, kernel, uptime, bat_state;
        int bat_level;
        double bogomips;
        Resources.Memory mem;
        Resources.CPU cpu;
        Resources.Load load;
        List<Resources.Process> processes;
        List<Resources.Disk> disks;

        Resources.get_system_info(out now, out name, out cpuinfo, out bogomips,
                                  out kernel, out uptime);
        Resources.get_top_info(out mem, out cpu, out load, out processes);
        Resources.get_disks(out disks);
        ipc.get_battery_state(out bat_level, out bat_state);
        label_sysinfo.set_markup(SYSINFO_TMPL.printf(name, now, cpuinfo, bogomips,
                                                     kernel, uptime, bat_level, bat_state,
                                                     load.avg_1m, load.avg_5m, load.avg_15m,
                                                     cpu.usr, cpu.sys, cpu.idle, cpu.io,
                                                     mem.total, mem.free, mem.used));
        populate_model_disks(disks);
        populate_model_procs(processes);
        return true;
    }


    public void run() {
        menu_manager.set_item_state("fullscreen", "SysInfo_groupmain",
                                    is_fullscreen ? "selected" : "normal");
        menu_manager.set_item_state("refresh_1m", "SysInfo_grouprefresh", "selected");
        ui_refresh();
        timer_id = GLib.Timeout.add(timeout, ui_refresh);
        Gtk.main();
    }


    ///// Main entry point
    public static int main(string[] args) {
        Gtk.init(ref args);
        Gtk.rc_parse_string(DEFAULT_STYLE);
        var app = new SysInfo();
        app.run();
        return 0;
    }
}


//////////////////////////////////////////////////////////////////////
