#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

#History:
# Version 1.0 Initial release
# Version 1.1 Added support for multiple row selections to launching multiple searches
# Version 1.2 Added support for encoding for passing foreign language names to Amazon etc
# Version 1.3 Support Calibre 0.7.34 feature of plugin images retained within the zip file
# Version 1.4 Add a preferences dialog to replace tweaks file, make menus dynamic, ensure author FN LN,
#             Split into separate code files with proxy architecture (thanks to chaley!)
# Version 1.5 Use the Calibre template processor to support other metadata fields
#             Support HTTP POST websites like FictFact
#             Archive import export use standard zip extension for ease of uploading to forums
#             Ensure config window is parented to prevent multiple Calibre windows in taskbar

import zipimport
from calibre import prints
from calibre.customize import InterfaceActionBase

# A proxy for a plugin. This proxy will not allocate any resources or import
# any modules other than InterfaceActionBase until it is actually used.
#
# The proxy must be customized by supplying:
# - the name of the python file containing the base and action classes.
# - the class names of the base and action classes
# - the plugin identifying information (must be here, not in the base class
#   the proxy will use).

class ActionSearchTheInternet(InterfaceActionBase):
    name                = 'Search The Internet'
    description         = 'Search various websites for the selected book(s)'
    supported_platforms = ['windows', 'osx', 'linux'] # Platforms this plugin will run on
    author              = 'Grant Drake'
    version             = (1, 5, 1)

    python_code_file_name = 'search_the_internet_code'
    base_plugin_name      = 'STIP_Base'
    action_plugin_name    = 'STIP_Action'

    def __init__(self, plugin_path):
        InterfaceActionBase.__init__(self, plugin_path)
        prints('STI_plugin_path=', plugin_path)
        self.plugin_path = plugin_path
        self._base_plugin = None
        self._base_module = None

    def get_base_module(self):
        if self._base_module is None:
            import os
            path = self.plugin_path
            prints('STI_about to zipimport: path=', path)
            prints('Normpath=', os.path.normpath(path))
            importer = zipimport.zipimporter(self.plugin_path)
            self._base_module = importer.load_module(self.python_code_file_name)
        return self._base_module

    def get_base_plugin(self):
        if self._base_plugin is None:
            m = self.get_base_module()
            self._base_plugin = m.__dict__[self.base_plugin_name](self)
        return self._base_plugin

    def config_widget(self):
        return self.get_base_plugin().config_widget()

    def save_settings(self, w):
        self.get_base_plugin().save_settings(w)

    def customization_help(self):
        return self.get_base_plugin().customization_help()

    def load_actual_plugin(self, gui):
        m = self.get_base_module()
        action_plugin = m.__dict__[self.action_plugin_name](gui, self.site_customization)
        self.get_base_plugin().set_action_plugin(action_plugin)
        return action_plugin


# For testing, run from command line with this:
# calibre-debug -e search_the_internet_plugin.py
if __name__ == '__main__':
    from PyQt4.Qt import QApplication
    from calibre.gui2.preferences import test_widget
    app = QApplication([])
    test_widget('Advanced', 'Plugins')
