#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

from functools import partial
import os, shutil, urllib, webbrowser
from urllib import quote_plus
from zipfile import ZipFile

from PyQt4.Qt import Qt, QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, \
                     QLineEdit, QFormLayout, QTableWidget, QTableWidgetItem, \
                     QAbstractItemView, QComboBox, QVariant, QAction, QMenu, \
                     QToolButton, QIcon, QPixmap, QDialog, QDialogButtonBox, \
                     QGridLayout, QGroupBox, QRadioButton, QUrl, QKeySequence, \
                     QFileDialog
from PyQt4 import QtGui, QtCore

from calibre.constants import iswindows
from calibre.ebooks.metadata.book.base import Metadata
from calibre.gui2 import error_dialog, question_dialog, info_dialog, choose_files, open_local_file, open_url, FileDialog
from calibre.gui2.actions import InterfaceAction
from calibre.library.save_to_disk import SafeFormat
from calibre.utils.config import config_dir, tweaks, JSONConfig

template_formatter = SafeFormat()

STIP_COL_NAMES = ['active', 'menuText', 'subMenu', 'shortcut', 'openGroup', 'image', 'url', 'encoding', 'method']
STIP_DEFAULT_MENU_SET = [
        (False, 'Audible for Author',             '', '', False, 'stip_audible.png',   'http://www.audible.com/search?advsearchKeywords=&searchTitle=&searchAuthor={author}&field_language=English','utf-8', 'GET'),
        (False, 'Audible for Book',               '', '', False, 'stip_audible.png',   'http://www.audible.com/search?advsearchKeywords=&searchTitle={title}&searchAuthor={author}&field_language=English','utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (True,  'Amazon.com for Book',            '', '', False, 'stip_amazon.png',    'http://www.amazon.com/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'Amazon.co.uk for Book',          '', '', False, 'stip_amazon.png',    'http://www.amazon.co.uk/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Barnes and Noble for Author',    '', '', False, 'stip_bn.png',        'http://productsearch.barnesandnoble.com/search/results.aspx?store=book&ATH={author}', 'utf-8', 'GET'),
        (False, 'Barnes and Noble for Book',      '', '', False, 'stip_bn.png',        'http://productsearch.barnesandnoble.com/search/results.aspx?store=book&ATH={author}&TTL={title}', 'utf-8', 'GET'),
        (False, 'Barnes and Noble for ISBN',      '', '', False, 'stip_bn.png',        'http://search.barnesandnoble.com/books/product.aspx?EAN={isbn}', 'utf-8', 'GET'),
        (False, 'Barnes and Noble for Title',     '', '', False, 'stip_bn.png',        'http://productsearch.barnesandnoble.com/search/results.aspx?store=book&TTL={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Books-by-ISBN for ISBN',         '', '', False, 'stip_isbn.png',      'http://books-by-isbn.com/cgi-bin/isbn-lookup.pl?isbn={isbn}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Classify for Author',            '', '', False, 'stip_classify.png',  'http://classify.oclc.org/classify2/ClassifyDemo?search-author-txt={author}', 'utf-8', 'GET'),
        (False, 'Classify for Book',              '', '', False, 'stip_classify.png',  'http://classify.oclc.org/classify2/ClassifyDemo?search-title-txt={title}&search-author-txt={author}', 'utf-8', 'GET'),
        (False, 'Classify for ISBN',              '', '', False, 'stip_classify.png',  'http://classify.oclc.org/classify2/ClassifyDemo?search-standnum-txt={isbn}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Demonoid for Author',            '', '', False, 'stip_demonoid.png',  'http://www.demonoid.me/files/?category=11&subcategory=All&quality=All&seeded=0&external=2&query={author}', 'utf-8', 'GET'),
        (False, 'Demonoid for Book',              '', '', False, 'stip_demonoid.png',  'http://www.demonoid.me/files/?category=11&subcategory=All&quality=All&seeded=0&external=2&query={author}+{title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'EBooks for Author',              '', '', False, 'stip_ebooks.png',    'http://w.ebooks.com/SearchApp/SearchResults.net?term={author}&RestrictBy=author', 'utf-8', 'GET'),
        (False, 'EBooks for Book',                '', '', False, 'stip_ebooks.png',    'http://w.ebooks.com/SearchApp/SearchResults.net?term={author}+{title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Ellora\'s Cave for Title',       '', '', False, 'stip_ellora.png',    'http://www.jasminejade.com/searchadv.aspx?IsSubmit=true&SearchTerm={title}&ProductTypeID=5&ShowPics=1', 'utf-8', 'GET'),
        (True,  '', '', '', False, '', '', '', 'GET'),
        (True,  'FantasticFiction for Author',    '', '', False, 'stip_ff.png',        'http://www.fantasticfiction.co.uk/search/?searchfor=author&keywords={author}', 'utf-8', 'GET'),
        (True,  'FantasticFiction for Title',     '', '', False, 'stip_ff.png',        'http://www.fantasticfiction.co.uk/search/?searchfor=book&keywords={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'FictFact for Author',            '', '', False, 'stip_fictfact.png',  'http://www.fictfact.com/search/?q={author}', 'utf-8', 'GET'),
        (False, 'FictFact for Book',              '', '', False, 'stip_fictfact.png',  'http://www.fictfact.com/search/?q={author}+{title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'FictionDB for Author',           '', '', False, 'stip_fictiondb.png', 'http://www.fictiondb.com/search/searchresults.htm?styp=1&srchtxt={author}', 'utf-8', 'POST'),
        (False, 'FictionDB for Book',             '', '', False, 'stip_fictiondb.png', 'http://www.fictiondb.com/search/searchresults.htm?styp=6&author={author}&title={title}&srchtxt=multi&sgcode=0&tpcode=0&imprint=0&pubgroup=0&genretype=--&rating=-&myrating=-&status=-', 'utf-8', 'POST'),
        (False, 'FictionDB for ISBN',             '', '', False, 'stip_fictiondb.png', 'http://www.fictiondb.com/search/searchresults.htm?styp=4&srchtxt={isbn}', 'utf-8', 'POST'),
        (False, 'FictionDB for Title',            '', '', False, 'stip_fictiondb.png', 'http://www.fictiondb.com/search/searchresults.htm?styp=2&srchtxt={title}', 'utf-8', 'POST'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Goodreads for Author',           '', '', False, 'stip_goodreads.png', 'http://www.goodreads.com/search/search?q={author}&search_type=books', 'utf-8', 'GET'),
        (False, 'Goodreads for Book',             '', '', False, 'stip_goodreads.png', 'http://www.goodreads.com/search/search?q={author}+{title}&search_type=books', 'utf-8', 'GET'),
        (False, 'Goodreads for ISBN',             '', '', False, 'stip_goodreads.png', 'http://www.goodreads.com/search/search?q={isbn}&search_type=books', 'utf-8', 'GET'),
        (False, 'Goodreads for Title',            '', '', False, 'stip_goodreads.png', 'http://www.goodreads.com/search/search?q={title}&search_type=books', 'utf-8', 'GET'),
        (True,  '', '', '', False, '', '', '', 'GET'),
        (True,  'Google images for Book',         '', '', False, 'stip_google.png',    'http://www.google.com/images?q=%22{author}%22+%22{title}%22', 'utf-8', 'GET'),
        (True,  'Google.com for Book',            '', '', False, 'stip_google.png',    'http://www.google.com/#sclient=psy&q=%22{author}%22+%22{title}%22', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'isfdb for Author',               '', '', False, 'stip_isfdb.png',     'http://www.isfdb.org/cgi-bin/se.cgi?type=Name&arg={author}', 'latin-1', 'GET'),
        (False, 'isfdb for Book',                 '', '', False, 'stip_isfdb.png',     'http://www.isfdb.org/cgi-bin/edit/tp_search.cgi?TERM_1={title}&USE_1=title&OPERATOR_1=AND&TERM_2={author}&USE_2=author&OPERATOR_2=AND' , 'latin-1', 'GET'),
        (False, 'isfdb for ISBN',                 '', '', False, 'stip_isfdb.png',     'http://www.isfdb.org/cgi-bin/se.cgi?type=ISBN&arg={isbn}', 'latin-1', 'GET'),
        (False, 'isfdb for Title',                '', '', False, 'stip_isfdb.png',     'http://www.isfdb.org/cgi-bin/se.cgi?type=Fiction+Titles&arg={title}', 'latin-1', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Inkmesh for Author',             '', '', False, 'stip_inkmesh.png',   'http://www.inkmesh.com/search/?qs={author}&btnE=Find+Ebooks','utf-8', 'GET'),
        (False, 'Inkmesh for Book',               '', '', False, 'stip_inkmesh.png',   'http://www.inkmesh.com/search/?qs={title}+by+{author}&btnE=Find+Ebooks','utf-8', 'GET'),
        (False, 'Inkmesh for Title',              '', '', False, 'stip_inkmesh.png',   'http://www.inkmesh.com/search/?qs={title}&btnE=Find+Ebooks','utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Kobo for Author',                '', '', False, 'stip_kobo.png',      'http://www.kobobooks.com/search/search.html?q={author}&f=author','utf-8', 'GET'),
        (False, 'Kobo for Book',                  '', '', False, 'stip_kobo.png',      'http://www.kobobooks.com/search/search.html?q={author}+{title}&f=author','utf-8', 'GET'),
        (False, 'Kobo for Title',                 '', '', False, 'stip_kobo.png',      'http://www.kobobooks.com/search/search.html?q={title}','utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Library of Congress for Author', '', '', False, 'stip_loc.png',       'http://catalog.loc.gov/cgi-bin/Pwebrecon.cgi?DB=local&Search_Arg={author}&Search_Code=NAME%40&CNT=100&hist=1&type=quick', 'utf-8', 'GET'),
        (False, 'Library of Congress for ISBN',   '', '', False, 'stip_loc.png',       'http://catalog.loc.gov/cgi-bin/Pwebrecon.cgi?DB=local&Search_Arg={isbn}&Search_Code=STNO^*&CNT=100&hist=1&type=quick', 'utf-8', 'GET'),
        (False, 'Library of Congress for Title',  '', '', False, 'stip_loc.png',       'http://catalog.loc.gov/cgi-bin/Pwebrecon.cgi?DB=local&Search_Arg={title}&Search_Code=TKEY^*&CNT=100&hist=1&type=quick', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'LibraryThing for Author',        '', '', False, 'stip_lthing.png',    'http://www.librarything.com/search.php?search={author}', 'utf-8', 'GET'),
        (False, 'LibraryThing for Book',          '', '', False, 'stip_lthing.png',    'http://www.librarything.com/search.php?search={title}+{author}', 'utf-8', 'GET'),
        (False, 'LibraryThing for ISBN',          '', '', False, 'stip_lthing.png',    'http://www.librarything.com/search.php?search={isbn}', 'utf-8', 'GET'),
        (False, 'LibraryThing for Title',         '', '', False, 'stip_lthing.png',    'http://www.librarything.com/search.php?search={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Literature-Map like Author',     '', '', False, 'stip_litmap.png',    'http://www.literature-map.com/{author}.html', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Lovereading like Author',        '', '', False, 'stip_loveread.png',  'http://www.lovereading.co.uk/authorrec/{author:re(\\+, )}', 'utf-8', 'GET'),
        (False, 'Lovereading for Author',         '', '', False, 'stip_loveread.png',  'http://www.lovereading.co.uk/search.php?author={author}&format=All+Formats&advsearch=1', 'utf-8', 'GET'),
        (False, 'Lovereading for Book',           '', '', False, 'stip_loveread.png',  'http://www.lovereading.co.uk/search.php?author={author}&title={title}&format=All+Formats&advsearch=1', 'utf-8', 'GET'),
        (False, 'Lovereading for ISBN',           '', '', False, 'stip_loveread.png',  'http://www.lovereading.co.uk/search.php?isbn={isbn}&format=All+Formats&advsearch=1', 'utf-8', 'GET'),
        (False, 'Lovereading for Title',          '', '', False, 'stip_loveread.png',  'http://www.lovereading.co.uk/search.php?title={title}&format=All+Formats&advsearch=1', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'ManyBooks for Author',           '', '', False, 'stip_manybooks.png', 'http://manybooks.net/search.php?search={author}', 'utf-8', 'GET'),
        (False, 'ManyBooks for Title',            '', '', False, 'stip_manybooks.png', 'http://manybooks.net/search.php?search={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Mobipocket for Author',          '', '', False, 'stip_mobi.png',      'http://www.mobipocket.com/en/eBooks/searchebooks.asp?Language=EN&searchType=Author&lang=EN&searchStr={author}', 'utf-8', 'GET'),
        (False, 'Mobipocket for Book',            '', '', False, 'stip_mobi.png',      'http://www.mobipocket.com/en/eBooks/searchebooks.asp?Language=EN&searchType=All&lang=EN&searchStr={title}+{author}', 'utf-8', 'GET'),
        (False, 'Mobipocket for ISBN',            '', '', False, 'stip_mobi.png',      'http://www.mobipocket.com/en/eBooks/searchebooks.asp?Language=EN&searchType=Publisher&lang=EN&searchStr={isbn}', 'utf-8', 'GET'),
        (False, 'Mobipocket for Title',           '', '', False, 'stip_mobi.png',      'http://www.mobipocket.com/en/eBooks/searchebooks.asp?Language=EN&searchType=Title&lang=EN&searchStr={tittle}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'NYTimes for Author',             '', '', False, 'stip_nyt.png',       'http://query.nytimes.com/search/sitesearch?query={author}&more=date_all',' utf-8'),
        (False, 'NYTimes for Book',               '', '', False, 'stip_nyt.png',       'http://query.nytimes.com/search/sitesearch?query={author}+{title}&more=date_all',' utf-8'),
        (False, 'NYTimes for Title',              '', '', False, 'stip_nyt.png',       'http://query.nytimes.com/search/sitesearch?query={title}&more=date_all',' utf-8'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'SimilarAuthors like Author',     '', '', False, 'stip_simauth.png',   'http://www.similarauthors.com/search.php?author={author}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Sony for Author',                '', '', False, 'stip_sony.png',      'http://ebookstore.sony.com/search?keyword={author}', 'utf-8', 'GET'),
        (False, 'Sony for Book',                  '', '', False, 'stip_sony.png',      'http://ebookstore.sony.com/search?keyword={author}+{title}', 'utf-8', 'GET'),
        (False, 'Sony for ISBN',                  '', '', False, 'stip_sony.png',      'http://ebookstore.sony.com/search?keyword={isbn}', 'utf-8', 'GET'),
        (False, 'Sony for Title',                 '', '', False, 'stip_sony.png',      'http://ebookstore.sony.com/search?keyword={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Waterstones for Author',         '', '', False, 'stip_wstones.png',   'http://www.waterstones.com/waterstonesweb/advancedSearch.do?buttonClicked=1&title=&author={author}', 'utf-8', 'GET'),
        (False, 'Waterstones for Book',           '', '', False, 'stip_wstones.png',   'http://www.waterstones.com/waterstonesweb/advancedSearch.do?buttonClicked=1&title={title}&author={author}', 'utf-8', 'GET'),
        (False, 'Waterstones for ISBN',           '', '', False, 'stip_wstones.png',   'http://www.waterstones.com/waterstonesweb/advancedSearch.do?buttonClicked=2&isbn={isbn}', 'utf-8', 'GET'),
        (False, 'Waterstones for Title',          '', '', False, 'stip_wstones.png',   'http://www.waterstones.com/waterstonesweb/advancedSearch.do?buttonClicked=1&title={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'WhatShouldIReadNext for ISBN',   '', '', False, 'stip_wsirn.png',     'http://www.whatshouldireadnext.com/wsirn.php?isbn={isbn}', 'utf-8', 'GET'),
        (True,  '', '', '', False,  '', '', ''),
        (True,  'Wikipedia for Author',           '', '', False, 'stip_wikipedia.png', 'http://en.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}', 'utf-8', 'GET'),
        (False, 'Wikipedia for Book',             '', '', False, 'stip_wikipedia.png', 'http://en.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}+{title}', 'utf-8', 'GET'),
        (False, 'Wikipedia for Title',            '', '', False, 'stip_wikipedia.png', 'http://en.wikipedia.org/w/index.php?title=Special%3ASearch&search={title}', 'utf-8', 'GET'),
        (False, '', '', '', False, '', '', '', 'GET'),
        (False, 'Amazon.ca for Book',             '', '', False, 'stip_amazon.png',    'http://www.amazon.ca/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'Amazon.cn for Book',             '', '', False, 'stip_amazon.png',    'http://www.amazon.cn/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'Amazon.co.jp for Book',          '', '', False, 'stip_amazon.png',    'http://www.amazon.co.jp/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'Amazon.de for Book',             '', '', False, 'stip_amazon.png',    'http://www.amazon.de/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'Amazon.it for Book',             '', '', False, 'stip_amazon.png',    'http://www.amazon.it/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'Amazon.fr for Book',             '', '', False, 'stip_amazon.png',    'http://www.amazon.fr/s/ref=nb_sb_noss?url=search-alias%3Dstripbooks&field-keywords={author}+{title}', 'latin-1', 'GET'),
        (False, 'bol.de for Author',              '', '', False, 'stip_bol.png',       'http://www.bol.de/shop/buecher/suche/?sa={author}&forward=weiter&sswg=BUCH', 'latin-1', 'GET'),
        (False, 'bol.de for Book',                '', '', False, 'stip_bol.png',       'http://www.bol.de/shop/buecher/suche/?st={title}&sa={author}&forward=weiter&sswg=BUCH', 'latin-1', 'GET'),
        (False, 'bol.de for Title',               '', '', False, 'stip_bol.png',       'http://www.bol.de/shop/buecher/suche/?st={title}&forward=weiter&sswg=BUCH', 'latin-1', 'GET'),
        (False, 'Chapitre for Title',             '', '', False, 'stip_chapitre.png',  'http://www.chapitre.com/CHAPITRE/fr/search/Default.aspx?optSearch=BOOKS&titre={title}', 'utf-8', 'GET'),
        (False, 'Chapters.ca for Author',         '', '', False, 'stip_chapters.png',  'http://www.chapters.indigo.ca/home/search/?keywords={author}', 'utf-8', 'GET'),
        (False, 'Chapters.ca for Book',           '', '', False, 'stip_chapters.png',  'http://www.chapters.indigo.ca/home/search/?keywords={author}+{title}', 'utf-8', 'GET'),
        (False, 'Chapters.ca for ISBN',           '', '', False, 'stip_chapters.png',  'http://www.chapters.indigo.ca/home/search/?keywords={isbn}', 'utf-8', 'GET'),
        (False, 'Chapters.ca for Title',          '', '', False, 'stip_chapters.png',  'http://www.chapters.indigo.ca/home/search/?keywords={title}', 'utf-8', 'GET'),
        (False, 'Fnac for Author',                '', '', False, 'stip_fnac.png',      'http://recherche.fnac.com/Search/SearchResult.aspx?SCat=2&Search={author}', 'utf-8', 'GET'),
        (False, 'Fnac for Book',                  '', '', False, 'stip_fnac.png',      'http://recherche.fnac.com/Search/SearchResult.aspx?SCat=2&Search={author}+{title}', 'utf-8', 'GET'),
        (False, 'Fnac for Title',                 '', '', False, 'stip_fnac.png',      'http://recherche.fnac.com/Search/SearchResult.aspx?SCat=2&Search={title}', 'utf-8', 'GET'),
        (False, 'Google.de for Book',             '', '', False, 'stip_google.png',    'http://www.google.de/#sclient=psy&q=%22{author}%22+%22{title}%22', 'utf-8', 'GET'),
        (False, 'Google.es for Book',             '', '', False, 'stip_google.png',    'http://www.google.es/#sclient=psy&q=%22{author}%22+%22{title}%22', 'utf-8', 'GET'),
        (False, 'Google.fr for Book',             '', '', False, 'stip_google.png',    'http://www.google.fr/#sclient=psy&q=%22{author}%22+%22{title}%22', 'utf-8', 'GET'),
        (False, 'Google.it for Book',             '', '', False, 'stip_google.png',    'http://www.google.it/#sclient=psy&q=%22{author}%22+%22{title}%22', 'utf-8', 'GET'),
        (False, 'libri.de for Author',            '', '', False, 'stip_libri.png',     'http://www.libri.de/shop/action/advancedSearch?action=search&nodeId=-1&binderType=Alle&languageCode=DE&person={author}', 'utf-8', 'GET'),
        (False, 'libri.de for Book',              '', '', False, 'stip_libri.png',     'http://www.libri.de/shop/action/advancedSearch?action=search&nodeId=-1&binderType=Alle&languageCode=DE&title={title}&person={author}', 'utf-8', 'GET'),
        (False, 'libri.de for Title',             '', '', False, 'stip_libri.png',     'http://www.libri.de/shop/action/advancedSearch?action=search&nodeId=-1&binderType=Alle&languageCode=DE&title={title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.de for Author',        '', '', False, 'stip_wikipedia.png', 'http://de.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}', 'utf-8', 'GET'),
        (False, 'Wikipedia.de for Book',          '', '', False, 'stip_wikipedia.png', 'http://de.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}+{title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.de for Title',         '', '', False, 'stip_wikipedia.png', 'http://de.wikipedia.org/w/index.php?title=Special%3ASearch&search={title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.es for Author',        '', '', False, 'stip_wikipedia.png', 'http://es.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}', 'utf-8', 'GET'),
        (False, 'Wikipedia.es for Book',          '', '', False, 'stip_wikipedia.png', 'http://es.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}+{title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.es for Title',         '', '', False, 'stip_wikipedia.png', 'http://es.wikipedia.org/w/index.php?title=Special%3ASearch&search={title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.fr for Author',        '', '', False, 'stip_wikipedia.png', 'http://fr.wikipedia.org/w/index.php?title=Sp%E9cial%3ARecherche&search={author}', 'utf-8', 'GET'),
        (False, 'Wikipedia.fr for Book',          '', '', False, 'stip_wikipedia.png', 'http://fr.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}+{title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.fr for Title',         '', '', False, 'stip_wikipedia.png', 'http://fr.wikipedia.org/w/index.php?title=Sp%E9cial%3ARecherche&search={title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.it for Author',        '', '', False, 'stip_wikipedia.png', 'http://it.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}', 'utf-8', 'GET'),
        (False, 'Wikipedia.it for Book',          '', '', False, 'stip_wikipedia.png', 'http://it.wikipedia.org/w/index.php?title=Special%3ASearch&search={author}+{title}', 'utf-8', 'GET'),
        (False, 'Wikipedia.it for Title',         '', '', False, 'stip_wikipedia.png', 'http://it.wikipedia.org/w/index.php?title=Special%3ASearch&search={title}', 'utf-8', 'GET')]

STIP_STORE_MENUS_NAME = 'SearchMenus'
STIP_MENUS_KEY = 'Menus'
STIP_OPEN_GROUP_KEY = 'OpenGroupShortcut'
STIP_COL_WIDTH_KEY = 'UrlColWidth'
STIP_DEFAULT_MENU_STORE = {
    STIP_MENUS_KEY: None,
    STIP_OPEN_GROUP_KEY: '',
    STIP_COL_WIDTH_KEY: -1
}

STIP_STORE_TEST_NAME = 'TestData'
STIP_TEST_VALUES_KEY = 'Values'
STIP_TEST_LAST_BOOK_KEY = 'LastBookIndex'
STIP_DEFAULT_TEST_STORE = {
    STIP_TEST_LAST_BOOK_KEY: 0,
    STIP_TEST_VALUES_KEY: [{ 'display': 'English Book 1',   'title': 'To Kill a Mockingbird ',          'author': 'Harper Lee',         'publisher': 'Harper Perennial Modern Classics', 'isbn': '9780061120084'},
                           { 'display': 'English Book 2',   'title': 'Hyperion',                        'author': 'Dan Simmons',        'publisher': 'Gollancz',                         'isbn': '9780575081147'},
                           { 'display': 'English Book 3',   'title': 'Les Misérables',                  'author': 'Victor Hugo',        'publisher': 'Barnes & Noble Classics',          'isbn': '9781593080662'},
                           { 'display': 'French Book',      'title': 'De l\'inconvénient d\'être né',   'author': 'E. M. Cioran',       'publisher': 'French & European Pubns',          'isbn': '9780785928089'},
                           { 'display': 'German Book',      'title': 'Schändung',                       'author': 'Jussi Adler-Olsen',  'publisher': 'dtv',                              'isbn': '9783423247870'}]
}

STIP_INTERNET_ICON = 'internet.png'
STIP_OPEN_GROUP_ICON = 'open_group.png'
STIP_MOVE_TO_TOP_ICON = 'move_to_top.png'
STIP_IMAGE_ADD_ICON = 'image_add.png'
STIP_IMPORT_ICON = 'import.png'
STIP_EXPORT_ICON = 'export.png'
STIP_PLUGIN_ICONS = [STIP_INTERNET_ICON, STIP_OPEN_GROUP_ICON, STIP_MOVE_TO_TOP_ICON, STIP_IMAGE_ADD_ICON, STIP_IMPORT_ICON, STIP_EXPORT_ICON]

# Global definition of instance of our configuration store for this plugin
search_internet_config = None
# Global definition of our plugin resources
plugin_icon_resources = {}

try:
    _fromUtf8 = QtCore.QString.fromUtf8
except AttributeError:
    _fromUtf8 = lambda s: s

def config_store():
    # Retrieve singleton instance of configuration data via JSON file
    global search_internet_config
    if search_internet_config is None:
        # Lets migrate any existing config file to a new name
        old_json_path = os.path.join(config_dir,'plugins/search_the_internet.json')
        if os.path.exists(old_json_path):
            new_json_path = os.path.join(config_dir,'plugins/Search The Internet.json')
            shutil.move(old_json_path, new_json_path)
        search_internet_config = JSONConfig('plugins/Search The Internet')
    return search_internet_config

def get_default_icon_names():
    # Build a distinct set of icon names to pass to load_resources, including our top level icon
    icon_names = STIP_PLUGIN_ICONS
    for id, val in enumerate(STIP_DEFAULT_MENU_SET):
        icon = val[5]
        if icon is not None and icon not in icon_names:
            icon_names.append(icon)
    return icon_names

def get_icon(icon_name):
    global plugin_icon_resources
    if icon_name:
        if icon_name in plugin_icon_resources:
            pixmap = QPixmap()
            pixmap.loadFromData(plugin_icon_resources[icon_name])
            return QIcon(pixmap)
        else:
            return QIcon(I(icon_name))
    else:
        return QIcon()

def set_plugin_icon_resources(resources):
    global plugin_icon_resources
    plugin_icon_resources = resources

def get_menus_as_dictionary(config_menus=None):
    # Menu items wil be stored in a config dictionary in the JSON configuration file
    # However if no menus defined (like first time user) we build a default dictionary set.
    if config_menus is None:
        # No menu items are defines so populate with the default set of menu items
        config_menus = [dict(zip(STIP_COL_NAMES, tup)) for tup in STIP_DEFAULT_MENU_SET]
    return config_menus

def fix_legacy_url(url):
    # Will fix a corrupt version of the LoveReading.co.uk url from v1.4
    if url == 'http://www.lovereading.co.uk/authorrec/{author_spaced)/gd}':
        url = 'http://www.lovereading.co.uk/authorrec/{author:re(\\+, )}'
    # This is a fix added to v1.5 to ensure that any url's that used to use the old
    # approach of xxx_spaced tokens instead uses the template processor function
    url = url.replace('_spaced}', ':re(\\+, )}')
    return url

class MenuTableWidget(QTableWidget): # {{{
    COMBO_IMAGE_ADD = 'Add New Image...'

    def __init__(self, data_items, *args):
        QTableWidget.__init__(self, *args)
        self.populate_table(data_items)
        self.cellChanged.connect(self.cell_changed)

    def url_column_width(self):
        if self.columnCount() > 5:
            return self.columnWidth(6)
        else:
            c = config_store().get(STIP_STORE_MENUS_NAME, STIP_DEFAULT_MENU_STORE)
            return c.get(STIP_COL_WIDTH_KEY, -1)

    def populate_table(self, data_items):
        self.image_names = self.read_image_combo_names()
        last_url_column_width = self.url_column_width()
        self.clear()
        self.setAlternatingRowColors(True)
        self.setRowCount(len(data_items))
        header_labels = ['', 'Title', 'Submenu', 'Shortcut', 'Open Group', 'Image', 'Url', 'Encoding', 'Method']
        self.setColumnCount(len(header_labels))
        self.setHorizontalHeaderLabels(header_labels)
        self.verticalHeader().setDefaultSectionSize(24)

        for row, data in enumerate(data_items):
            self.populate_table_row(row, data)

        self.resizeColumnsToContents()
        # Special sizing for the URL column as it tends to dominate the dialog
        if last_url_column_width != -1:
            self.setColumnWidth(6, last_url_column_width)
        self.setSortingEnabled(False)
        self.setMinimumSize(800, 0)
        self.setSelectionBehavior(QAbstractItemView.SelectRows)
        self.selectRow(0)

    def populate_table_row(self, row, data):
        self.blockSignals(True)
        icon_name = data['image']
        menu_text = data['menuText']
        self.setItem(row, 0, CheckableTableWidgetItem(data['active']))
        self.setItem(row, 1, TextIconWidgetItem(menu_text, get_icon(icon_name)))
        self.setItem(row, 2, QTableWidgetItem(data['subMenu']))
        if menu_text:
            self.set_editable_cells_in_row(row, shortcut=data['shortcut'], open_group=data['openGroup'],
                        image=icon_name, url=fix_legacy_url(data['url']), encoding=data['encoding'],
                        method=data.get('method', 'GET'))
        else:
            # Make all the later column cells non-editable
            self.set_noneditable_cells_in_row(row)
        self.blockSignals(False)

    def append_data(self, data_items):
        for data in reversed(data_items):
            row = self.currentRow() + 1
            self.insertRow(row)
            self.populate_table_row(row, data)

    def get_data(self):
        data_items = []
        for row in range(self.rowCount()):
            data_items.append(self.convert_row_to_data(row))
        # Remove any blank separator row items from the end as unneeded.
        while len(data_items) > 0 and len(data_items[-1]['menuText']) == 0:
            data_items.pop()
        return data_items

    def get_selected_data(self):
        data_items = []
        for row in self.selectionModel().selectedRows():
            data_items.append(self.convert_row_to_data(row.row()))
        return data_items

    def get_selected_urls_to_test(self):
        rows = self.selectionModel().selectedRows()
        for row in rows:
            url = unicode(self.item(row.row(), 6).text()).strip()
            if url:
                encoding = unicode(self.cellWidget(row.row(), 7).currentText()).strip()
                method = unicode(self.cellWidget(row.row(), 8).currentText()).strip()
                yield url, encoding, method

    def convert_row_to_data(self, row):
        data = self.create_blank_row_data()
        data['active'] = self.item(row, 0).checkState() == Qt.Checked
        data['menuText'] = unicode(self.item(row, 1).text()).strip()
        data['subMenu'] = unicode(self.item(row, 2).text()).strip()
        if data['menuText']:
            data['shortcut'] = unicode(self.item(row, 3).text()).strip()
            data['openGroup'] = self.item(row, 4).checkState() == Qt.Checked
            data['image'] = unicode(self.cellWidget(row, 5).currentText()).strip()
            data['url'] = unicode(self.item(row, 6).text()).strip()
            data['encoding'] = unicode(self.cellWidget(row, 7).currentText()).strip()
            data['method'] = unicode(self.cellWidget(row, 8).currentText()).strip()
        return data

    def cell_changed(self, row, col):
        if col == 1:
            menu_text = unicode(self.item(row, col).text()).strip()
            if menu_text:
                # Make sure that the other columns in this row are enabled if not already.
                if not self.item(row, 3).flags() & Qt.ItemIsEditable:
                    # We need to make later columns in this row editable
                    self.set_editable_cells_in_row(row)
            else:
                # Blank menu text so treat it as a separator row
                self.set_noneditable_cells_in_row(row)

    def set_editable_cells_in_row(self, row, shortcut='', open_group=False, image='', url='', encoding='utf-8', method='GET'):
        self.setItem(row, 3, QTableWidgetItem(shortcut))
        self.setItem(row, 4, CheckableTableWidgetItem(open_group))
        image_combo = ImageComboBox(self, self.image_names, image)
        image_combo.currentIndexChanged.connect(partial(self.image_combo_index_changed, image_combo, row))
        self.setCellWidget(row, 5, image_combo)
        self.setItem(row, 6, QTableWidgetItem(url))
        self.setCellWidget(row, 7, EncodingComboBox(self, encoding))
        self.setCellWidget(row, 8, MethodComboBox(self, method))

    def set_noneditable_cells_in_row(self, row):
        for col in range(3,9):
            if self.cellWidget(row, col):
                self.removeCellWidget(row, col)
            item = QTableWidgetItem()
            item.setFlags(Qt.ItemIsSelectable|Qt.ItemIsEnabled)
            self.setItem(row, col, item)
        self.item(row, 1).setIcon(QIcon())

    def create_blank_row_data(self):
        data = {}
        data['active'] = True
        data['menuText'] = ''
        data['subMenu'] = ''
        data['shortcut'] = ''
        data['openGroup'] = False
        data['image'] = ''
        data['url'] = ''
        data['encoding'] = ''
        data['method'] = ''
        return data

    def display_add_new_image_dialog(self, select_in_combo=False, combo=None):
        add_image_dialog = ImageDialog(self, self.resources_dir, self.image_names)
        add_image_dialog.exec_()
        if add_image_dialog.result() == QDialog.Rejected:
            # User cancelled the add operation or an error - set to previous value
            if select_in_combo and combo:
                prevIndex = combo.itemData(0).toPyObject()
                combo.blockSignals(True)
                combo.setCurrentIndex(prevIndex)
                combo.blockSignals(False)
            return
        # User has added a new image so we need to repopulate every combo with new sorted list
        self.image_names = self.read_image_combo_names()
        for update_row in range(self.rowCount()):
            cellCombo = self.cellWidget(update_row, 5)
            if cellCombo:
                cellCombo.blockSignals(True)
                cellCombo.populate_combo(self.image_names, cellCombo.currentText())
                cellCombo.blockSignals(False)
        # Now select the newly added item in this row if required
        if select_in_combo and combo:
            idx = combo.findText(add_image_dialog.image_name)
            combo.blockSignals(True)
            combo.setCurrentIndex(idx)
            combo.blockSignals(False)

    def image_combo_index_changed(self, combo, row):
        if combo.currentText() == self.COMBO_IMAGE_ADD:
            # Special item in the combo for choosing a new image to add to Calibre
            self.display_add_new_image_dialog(select_in_combo=True, combo=combo)
        # Regardless of new or existing item, update image on the title column
        title_item = self.item(row, 1)
        title_item.setIcon(combo.itemIcon(combo.currentIndex()))
        # Store the current index as item data in index 0 in case user cancels dialog in future
        combo.setItemData(0, QVariant(combo.currentIndex()))

    def add_row(self):
        self.setFocus()
        # We will insert a blank row below the currently selected row
        row = self.currentRow() + 1
        self.insertRow(row)
        self.populate_table_row(row, self.create_blank_row_data())
        self.select_and_scroll_to_row(row)

    def delete_rows(self):
        self.setFocus()
        rows = self.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        message = '<p>Are you sure you want to delete this menu item?'
        if len(rows) > 1:
            message = '<p>Are you sure you want to delete the selected %d menu items?'%len(rows)
        if not question_dialog(self, _('Are you sure?'), message, show_copy_button=False):
            return
        first_sel_row = self.currentRow()
        for selrow in reversed(rows):
            self.removeRow(selrow.row())
        if first_sel_row < self.rowCount():
            self.select_and_scroll_to_row(first_sel_row)
        elif self.rowCount() > 0:
            self.select_and_scroll_to_row(first_sel_row - 1)

    def move_rows_up(self):
        self.setFocus()
        rows = self.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        first_sel_row = rows[0].row()
        if first_sel_row <= 0:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selrows = []
        for row in rows:
            selrows.append(row.row())
        selrows.sort()
        for selrow in selrows:
            self.swap_row_widgets(selrow - 1, selrow + 1)
        scroll_to_row = first_sel_row - 1
        if scroll_to_row > 0:
            scroll_to_row = scroll_to_row - 1
        self.scrollToItem(self.item(scroll_to_row, 0))

    def move_rows_down(self):
        self.setFocus()
        rows = self.selectionModel().selectedRows()
        if len(rows) == 0:
            return
        last_sel_row = rows[-1].row()
        if last_sel_row == self.rowCount() - 1:
            return
        # Workaround for strange selection bug in Qt which "alters" the selection
        # in certain circumstances which meant move down only worked properly "once"
        selrows = []
        for row in rows:
            selrows.append(row.row())
        selrows.sort()
        for selrow in reversed(selrows):
            self.swap_row_widgets(selrow + 2, selrow)
        scroll_to_row = last_sel_row + 1
        if scroll_to_row < self.rowCount() - 1:
            scroll_to_row = scroll_to_row + 1
        self.scrollToItem(self.item(scroll_to_row, 0))

    def swap_row_widgets(self, src_row, dest_row):
        self.blockSignals(True)
        self.insertRow(dest_row)
        for col in range(0,3):
            self.setItem(dest_row, col, self.takeItem(src_row, col))
        menu_text = unicode(self.item(dest_row, 1).text()).strip()
        if menu_text:
            for col in range(3,9):
                if col == 5:
                    # Image column has a combobox we have to recreate as cannot move widget (Qt crap)
                    icon_name = self.cellWidget(src_row, col).currentText()
                    image_combo = ImageComboBox(self, self.image_names, icon_name)
                    image_combo.currentIndexChanged.connect(partial(self.image_combo_index_changed, image_combo, dest_row))
                    self.setCellWidget(dest_row, col, image_combo)
                elif col == 7:
                    # Encoding column has a combo box we also have to recreate
                    encoding = self.cellWidget(src_row, col).currentText()
                    self.setCellWidget(dest_row, col, EncodingComboBox(self, encoding))
                elif col == 8:
                    # Method column has a combo box we also have to recreate
                    method = self.cellWidget(src_row, col).currentText()
                    self.setCellWidget(dest_row, col, MethodComboBox(self, method))
                else:
                    # Any other column we transfer the TableWidgetItem
                    self.setItem(dest_row, col, self.takeItem(src_row, col))
        else:
            # This is a separator row
            self.set_noneditable_cells_in_row(dest_row)
        self.removeRow(src_row)
        self.blockSignals(False)

    def select_and_scroll_to_row(self, row):
        self.selectRow(row)
        self.scrollToItem(self.currentItem())

    def read_image_combo_names(self):
        # Read all of the images that are contained in the zip file
        image_names = get_default_icon_names()
        # Remove all the images that do not have the stip_ prefix
        image_names = filter(lambda x: x.startswith('stip_'), image_names)
        # Now read any images from the config\resources\images directory if any
        self.resources_dir = os.path.join(config_dir, 'resources/images')
        if iswindows:
            self.resources_dir = os.path.normpath(self.resources_dir)

        if os.path.exists(self.resources_dir):
            # Get the names of any .png images in this directory
            for f in os.listdir(self.resources_dir):
                if f.lower().endswith('.png'):
                    image_names.append(os.path.basename(f))

        image_names.sort()
        # Add a blank item at the beginning of the list, and a blank then special 'Add" item at end
        image_names.insert(0, '')
        image_names.append('')
        image_names.append(self.COMBO_IMAGE_ADD)
        return image_names

    def move_active_to_top(self):
        # Select all of the inactive items and move them to the bottom of the list
        if self.rowCount() == 0:
            return
        self.setUpdatesEnabled(False)
        last_row = self.rowCount()
        row = 0
        for count in range(last_row):
            active = self.item(row, 0).checkState() == Qt.Checked
            if active:
                # Move on to the next row
                row = row + 1
            else:
                # Move this row to the bottom of the grid
                self.swap_row_widgets(row, last_row)
        self.setUpdatesEnabled(True)
# }}}

class NoWheelComboBox(QComboBox): # {{{

    def wheelEvent (self, event):
        # Disable the mouse wheel on top of the combo box changing selection as plays havoc in a grid
        event.ignore()
# }}}

class ImageComboBox(NoWheelComboBox): # {{{

    def __init__(self, parent, image_names, selected_text):
        NoWheelComboBox.__init__(self, parent)
        self.populate_combo(image_names, selected_text)

    def populate_combo(self, image_names, selected_text):
        self.clear()
        for i, image in enumerate(image_names):
            self.insertItem(i, get_icon(image), image)
        idx = self.findText(selected_text)
        self.setCurrentIndex(idx)
        self.setItemData(0, QVariant(idx))
# }}}

class EncodingComboBox(NoWheelComboBox): # {{{

    def __init__(self, parent, selected_text):
        NoWheelComboBox.__init__(self, parent)
        self.populate_combo(selected_text)

    def populate_combo(self, selected_text):
        self.addItems(['utf-8', 'latin-1'])
        idx = self.findText(selected_text)
        self.setCurrentIndex(idx)
# }}}

class MethodComboBox(NoWheelComboBox): # {{{

    def __init__(self, parent, selected_text):
        NoWheelComboBox.__init__(self, parent)
        self.populate_combo(selected_text)

    def populate_combo(self, selected_text):
        self.addItems(['GET', 'POST'])
        idx = self.findText(selected_text)
        self.setCurrentIndex(idx)
# }}}

class CheckableTableWidgetItem(QTableWidgetItem): # {{{

    def __init__(self, checked):
        QTableWidgetItem.__init__(self, '')
        self.setFlags(Qt.ItemFlags(Qt.ItemIsSelectable | Qt.ItemIsUserCheckable | Qt.ItemIsEnabled ))
        if checked:
            self.setCheckState(Qt.Checked)
        else:
            self.setCheckState(Qt.Unchecked)
# }}}

class TextIconWidgetItem(QTableWidgetItem): # {{{

    def __init__(self, text, icon):
        QTableWidgetItem.__init__(self, text)
        self.setIcon(icon)
# }}}

class PickTestBookDialog(QDialog): # {{{

    def __init__(self, parent=None):
        QDialog.__init__(self, parent)
        self.setWindowTitle('Select test data')
        layout = QVBoxLayout(self)
        self.setLayout(layout)

        c = config_store().get(STIP_STORE_TEST_NAME, STIP_DEFAULT_TEST_STORE)
        selected_idx = c.get(STIP_TEST_LAST_BOOK_KEY)
        self.data_items = c.get(STIP_TEST_VALUES_KEY)

        combo_layout = QHBoxLayout()
        lbl_choose = QLabel('&Select test book:', self)
        lbl_choose.setMinimumSize(100, 0)
        combo_layout.addWidget(lbl_choose, 0, Qt.AlignLeft)
        self._book_combo = TestDataComboBox(self, self.data_items)
        self._book_combo.currentIndexChanged.connect(self.combo_index_changed)
        lbl_choose.setBuddy(self._book_combo)
        self._book_combo.setMinimumSize(200, 0)
        combo_layout.addWidget(self._book_combo, 1, Qt.AlignLeft)
        layout.addLayout(combo_layout)

        group_box = QGroupBox(self)
        f = QFormLayout()
        self._title_edit = QLineEdit('')
        f.addRow(QLabel('Title:'), self._title_edit)
        self._author_edit = QLineEdit('')
        f.addRow(QLabel('Author:'), self._author_edit)
        self._publisher_edit = QLineEdit('')
        f.addRow(QLabel('Publisher:'), self._publisher_edit)
        self._isbn_edit = QLineEdit('')
        f.addRow(QLabel('ISBN:'), self._isbn_edit)
        group_box.setLayout(f)
        layout.addWidget(group_box)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.ok_clicked)
        button_box.rejected.connect(self.reject)
        layout.addWidget(button_box)
        self.resize(self.sizeHint())
        self._book_combo.setCurrentIndex(selected_idx)
        # Force the display of the currently selected item in case index changed event not fired
        self.combo_index_changed()

    def ok_clicked(self):
        # Persist the test data and selected index into the JSON file
        test_data = {}
        test_data[STIP_TEST_LAST_BOOK_KEY] = self._book_combo.currentIndex()
        data_row = self.data_items[self._book_combo.currentIndex()]
        data_row['author'] = unicode(self._author_edit.text()).strip()
        data_row['title'] = unicode(self._title_edit.text()).strip()
        data_row['publisher'] = unicode(self._publisher_edit.text()).strip()
        data_row['isbn'] = unicode(self._isbn_edit.text()).strip()
        test_data[STIP_TEST_VALUES_KEY] = self.data_items
        c = config_store()
        c.set(STIP_STORE_TEST_NAME, test_data)
        self.accept()

    def combo_index_changed(self):
        # Update the dialog contents with metadata for the selected item
        selected_idx = self._book_combo.currentIndex()
        data_item = self.data_items[selected_idx]
        self._author_edit.setText(_fromUtf8(data_item['author']))
        self._title_edit.setText(_fromUtf8(data_item['title']))
        self._publisher_edit.setText(_fromUtf8(data_item['publisher']))
        self._isbn_edit.setText(_fromUtf8(data_item['isbn']))
# }}}

class TestDataComboBox(QComboBox): # {{{

    def __init__(self, parent, data_items):
        QComboBox.__init__(self, parent)
        self.populate_combo(data_items)

    def populate_combo(self, data_items):
        self.clear()
        for i, data in enumerate(data_items):
            self.insertItem(i, data['display'])
# }}}

class ImageDialog(QDialog): # {{{

    def __init__(self, parent=None, resources_dir='', image_names=[]):
        QDialog.__init__(self, parent)
        self.resources_dir = resources_dir
        self.image_names = image_names
        self.setWindowTitle('Add New Image')
        v = QVBoxLayout(self)

        group_box = QGroupBox('&Select image source', self)
        v.addWidget(group_box)
        grid = QGridLayout()
        self._radio_web = QRadioButton('From &web domain favicon', self)
        self._radio_web.setChecked(True)
        self._web_domain_edit = QLineEdit(self)
        self._radio_web.setFocusProxy(self._web_domain_edit)
        grid.addWidget(self._radio_web, 0, 0)
        grid.addWidget(self._web_domain_edit, 0, 1)
        grid.addWidget(QLabel('e.g. www.amazon.com'), 0, 2)
        self._radio_file = QRadioButton('From .png &file', self)
        self._input_file_edit = QLineEdit(self)
        self._input_file_edit.setMinimumSize(200, 0)
        self._radio_file.setFocusProxy(self._input_file_edit)
        pick_button = QPushButton('...', self)
        pick_button.setMaximumSize(24, 20)
        pick_button.clicked.connect(self.pick_file_to_import)
        grid.addWidget(self._radio_file, 1, 0)
        grid.addWidget(self._input_file_edit, 1, 1)
        grid.addWidget(pick_button, 1, 2)
        group_box.setLayout(grid)

        save_layout = QHBoxLayout()
        lbl_filename = QLabel('&Save as filename:', self)
        lbl_filename.setMinimumSize(155, 0)
        self._save_as_edit = QLineEdit('', self)
        self._save_as_edit.setMinimumSize(200, 0)
        lbl_filename.setBuddy(self._save_as_edit)
        lbl_ext = QLabel('.png', self)
        save_layout.addWidget(lbl_filename, 0, Qt.AlignLeft)
        save_layout.addWidget(self._save_as_edit, 0, Qt.AlignLeft)
        save_layout.addWidget(lbl_ext, 1, Qt.AlignLeft)
        v.addLayout(save_layout)

        button_box = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        button_box.accepted.connect(self.ok_clicked)
        button_box.rejected.connect(self.reject)
        v.addWidget(button_box)
        self.resize(self.sizeHint())
        self._web_domain_edit.setFocus()
        self.new_image_name = None

    @property
    def image_name(self):
        return self.new_image_name

    def pick_file_to_import(self):
        images = choose_files(None, 'menu icon dialog', 'Select a .png file for the menu icon',
                             filters=[('PNG Image Files', ['png'])], all_files=False, select_only_single_file=True)
        if not images:
            return
        f = images[0]
        if not f.lower().endswith('.png'):
            return error_dialog(self, 'Cannot select image',
                    'Source image must be a .png file.', show=True)
        self._input_file_edit.setText(f)
        self._save_as_edit.setText(os.path.splitext(os.path.basename(f))[0])

    def ok_clicked(self):
        # Validate all the inputs
        save_name = unicode(self._save_as_edit.text()).strip()
        if not save_name:
            return error_dialog(self, 'Cannot import image',
                    'You must specify a filename to save as.', show=True)
        self.new_image_name = os.path.splitext(save_name)[0] + '.png'
        if save_name.find('\\') > -1 or save_name.find('/') > -1:
            return error_dialog(self, 'Cannot import image',
                    'The save as filename should consist of a filename only.', show=True)
        if not os.path.exists(self.resources_dir):
            os.makedirs(self.resources_dir)
        dest_path = os.path.join(self.resources_dir, self.new_image_name)
        if save_name in self.image_names or os.path.exists(dest_path):
            if not question_dialog(self, _('Are you sure?'), '<p>'+
                    'An image with this name already exists - overwrite it?',
                    show_copy_button=False):
                return

        if self._radio_web.isChecked():
            domain = unicode(self._web_domain_edit.text()).strip()
            if not domain:
                return error_dialog(self, 'Cannot import image',
                        'You must specify a web domain url', show=True)
            url = 'http://www.google.com/s2/favicons?domain=' + domain
            urllib.urlretrieve(url, dest_path)
            return self.accept()
        else:
            source_file_path = unicode(self._input_file_edit.text()).strip()
            if not source_file_path:
                return error_dialog(self, 'Cannot import image',
                        'You must specify a source file.', show=True)
            if not source_file_path.lower().endswith('.png'):
                return error_dialog(self, 'Cannot import image',
                        'Source image must be a .png file.', show=True)
            if not os.path.exists(source_file_path):
                return error_dialog(self, 'Cannot import image',
                        'Source image does not exist!', show=True)
            shutil.copyfile(source_file_path, dest_path)
            return self.accept()
# }}}

# The interface action plugin class. An instance of this class is created by the
# proxy upon demand by calibre.
#
# The instance can get a reference to the proxy using the attribute
# self.interface_action_base_plugin, set by calibre when this class is
# instantiated. If a reference to the real base plugin is required, use
# self.interface_action_base_plugin.get_base_plugin()

class STIP_Action(InterfaceAction): # {{{

    name = 'Search The Internet'
    action_spec = ('Search the internet', None, None, None)
    popup_type = QToolButton.InstantPopup
    action_type = 'current'

    open_group_data = []

    def genesis(self):
        m = QMenu(self.gui)
        self.search_internet_menu = m
        c = config_store().get(STIP_STORE_MENUS_NAME, STIP_DEFAULT_MENU_STORE)
        data_items = get_menus_as_dictionary(c.get(STIP_MENUS_KEY))
        open_group_shortcut = c.get(STIP_OPEN_GROUP_KEY)

        # Read the icons listed in the default menu configuration and required for menus
        set_plugin_icon_resources(self.load_resources(get_default_icon_names()))

        self.rebuild_menus(data_items, open_group_shortcut)

        self.qaction.setMenu(m)
        m.setIcon(get_icon(STIP_INTERNET_ICON))

    def rebuild_menus(self, data_items, open_group_shortcut):
        m = self.search_internet_menu
        m.clear()
        sub_menus = {}
        self.open_group_data = []
        for data in data_items:
            active = data['active']
            if active:
                menu_text = data['menuText']
                sub_menu = data['subMenu']
                shortcut = data['shortcut']
                open_group = data['openGroup']
                icon_name = data['image']
                tokenised_url = fix_legacy_url(data['url'])
                encoding = data['encoding']
                method = data.get('method', 'GET')
                self.create_menu_item(m, sub_menus, menu_text, sub_menu, shortcut, icon_name, tokenised_url, encoding, method)
                if open_group and menu_text:
                    self.open_group_data.append((tokenised_url, encoding))
        m.addSeparator()
        if len(self.open_group_data) > 0:
            self.create_menu_item(m, sub_menus, 'Open Group', shortcut=open_group_shortcut, icon_name=STIP_OPEN_GROUP_ICON, open_group=True)
        m.addAction(QIcon(I('config.png')), _('&Customize plugin')+'...', self.show_configuration)

    def create_menu_item(self, m, sub_menus, menu_text, sub_menu='', shortcut='', icon_name='', tokenised_url='', encoding='', method='', open_group=False):
        parent_menu = m
        if sub_menu:
            # Create the sub-menu if it does not exist
            if sub_menu not in sub_menus:
                ac = self.create_action(spec=(sub_menu, None, None, None),
                    attr=sub_menu)
                parent_menu.addAction(ac)
                sm = QMenu()
                ac.setIcon(get_icon(icon_name))
                ac.setMenu(sm)
                sub_menus[sub_menu] = sm
            # Now set our menu variable so the parent menu item will be the sub-menu
            parent_menu = sub_menus[sub_menu]

        if not menu_text:
            ac = parent_menu.addSeparator()
        else:
            ac = self.create_action(spec=(menu_text, None, None, _(shortcut)),
                attr=menu_text)
            parent_menu.addAction(ac)
            ac.triggered.connect(partial(self.search_web_link, tokenised_url, encoding, method, open_group))
            ac.setIcon(get_icon(icon_name))
        return ac

    def search_web_link(self, tokenised_url, encoding, method, open_group=False):
        rows = self.gui.library_view.selectionModel().selectedRows()
        if not rows or len(rows) == 0:
            return
        self.db = self.gui.library_view.model().db
        for row in rows:
            if open_group:
                # User clicked the Open Group menu action, loop through all in group to open
                for url, enc in self.open_group_data:
                    if url:
                        self.search_web_for_book(row.row(), url, enc, method)
            else:
                self.search_web_for_book(row.row(), tokenised_url, encoding, method)

    def search_web_for_book(self, row, tokenised_url, encoding, method):
        mi = self.db.get_metadata(row)
        if not encoding:
            encoding = 'utf-8'
        self.open_tokenised_url(tokenised_url, encoding, method, mi)

    def open_tokenised_url(self, tokenised_url, encoding, method, mi):
        if not tokenised_url:
            return error_dialog(self.gui, 'Cannot open link',
                                'This menu item has not been configured with a url.', show=True)
        # Get all the values from the book's metadata
        vals = mi.all_non_none_fields()
        # Will only use the first author for the lookup if there are multiple
        vals['author'] = self.convert_author_to_search_text(mi.authors[0], encoding)
        vals['authors'] = vals['author'] # for name compatibility
        # rebuild the values dict, converting values to internet safe versions
        fixed_vals = {}
        for k in vals:
            fixed_vals[k] = unicode(vals[k]) # convert non-string types
            if k not in ['author', 'authors']:
                fixed_vals[k] = self.convert_to_search_text(fixed_vals[k], encoding)

        url = template_formatter.safe_format(tokenised_url, fixed_vals, 'STI template error', mi)
        if method == 'POST':
            # We will write to a temporary file to do a form submission
            url = self.create_local_file_for_post(url)
            if url is None:
                return
        # Use the default web browser
        webbrowser.open(url)

    def create_local_file_for_post(self, url):
        if url.find('?') == -1:
            error_dialog(self.gui, 'Invalid URL', 'You cannot use POST for this url '+
                                'as you have not specified any arguments after a ?', show=True)
            return None
        url_parts = url.partition('?')
        args = url_parts[2].split('&')
        input_values = {}
        for arg in args:
            pair = arg.split('=')
            if len(pair) != 2:
                error_dialog(self.gui, 'Invalid URL', 'You cannot use POST for this url '+
                                    'as it does not have name=value for ' + pair, show=True)
                return None
            # When submitting data via post, we do not want the querystring encoding
            input_values[pair[0]] = pair[1].replace('+',' ')
        js_submit = 'function mySubmit() { var frm=document.getElementById("form"); frm.submit(); }'
        # Set up file content elements
        input_field = '<input type="hidden" name="{0}" value="{1}" />'
        base_file_contents = """
<form id="form" action="{1}" method="post">
    <p>This page should disappear automatically once loaded.</p>
    <p>If your browser does not have javascript enabled, click on the button below.</p>
    <input id="button" type="submit" value="search" />
    {2}
</form>
<script type="text/javascript">
    {0}
    window.onLoad = mySubmit();
</script>
        """
        # Build input fields
        input_fields = ""
        for key, value in input_values.items():
            input_fields += input_field.format(key, value)
        # Write out to a temp file
        temp_file = self.interface_action_base_plugin.temporary_file('_post.html')
        temp_file.write(base_file_contents.format(js_submit, url_parts[0], input_fields))
        temp_file.close()
        return os.path.abspath(temp_file.name)

    def convert_to_search_text(self, text, encoding):
        # First we strip characters we will definitely not want to pass through.
        # Periods from author initials etc do not need to be supplied
        text = text.replace('.', '')
        # Now encode the text using Python function with chosen encoding
        text = quote_plus(text.encode(encoding))
        # If we ended up with double spaces as plus signs (++) replace them
        text = text.replace('++','+')
        return text

    def convert_author_to_search_text(self, author, encoding):
        # We want to convert the author name to FN LN format if it is stored LN, FN
        # We do this because some websites (Kobo) have crappy search engines that
        # will not match Adams+Douglas but will match Douglas+Adams
        # Not really sure of the best way of determining if the user is using LN, FN
        # Approach will be to check the tweak and see if a comma is in the name

        # Comma separated author will be pipe delimited in Calibre database
        fn_ln_author = author
        if author.find(',') > -1:
            # This might be because of a FN LN,Jr - check the tweak
            sort_copy_method = tweaks['author_sort_copy_method']
            if sort_copy_method == 'invert':
                # Calibre default. Hence "probably" using FN LN format.
                fn_ln_author = author
            else:
                # We will assume that we need to switch the names from LN,FN to FN LN
                parts = author.split(',')
                surname = parts.pop(0)
                parts.append(surname)
                fn_ln_author = ' '.join(parts).strip()
        return self.convert_to_search_text(fn_ln_author, encoding)

    def show_help(self):
        # Extract on demand the help file resource
        def get_help_file_resource():
            # We will write the help file out every time, in case the user upgrades the plugin zip
            # and there is a later help file contained within it.
            HELP_FILE = 'Search The Internet Help.htm'
            file_path = os.path.join(config_dir, 'plugins', HELP_FILE)
            # In version 1.5 I have renamed the help file, so delete the old one if it exists
            legacy_file_path = os.path.join(config_dir, 'plugins', 'search_the_internet_help.htm')
            if os.path.exists(legacy_file_path) and os.access(legacy_file_path, os.W_OK):
                os.remove(legacy_file_path)
            file_data = self.load_resources(HELP_FILE)[HELP_FILE]
            with open(file_path,'w') as f:
                f.write(file_data)
            return file_path
        url = 'file:///' + get_help_file_resource()
        open_url(QUrl(url))

    def show_configuration(self):
        self.interface_action_base_plugin.do_user_config(self.gui)
# }}}


# The following class is the real implementation of the InterfaceActionBase
# class. The proxy class will create an instance of this class when necessary,
# passing to it a reference to the proxy via __init__ in case it needs to call
# one of the methods in the proxy, such as load_resources.
#
# When the 'real' user interface action plugin instance is requested by calibre,
# the proxy will create the instance and pass it to this class using
# set_action_plugin.

class STIP_Base(object): # {{{
    def __init__(self, proxy):
        self.proxy = proxy

    # Instance variable for our plugin action, so we can invoke methods on it
    plugin_action = None

    def set_action_plugin(self, action_plugin):
        # Stores a reference to the action plugin created by the
        # InterfaceActionBase proxy
        self.plugin_action = action_plugin

    def config_widget(self):
        # Defining this method tells Calibre to display a custom UI dialog in preferences
        c = config_store().get(STIP_STORE_MENUS_NAME, STIP_DEFAULT_MENU_STORE)
        data_items = get_menus_as_dictionary(c.get(STIP_MENUS_KEY))
        open_group_shortcut = c.get(STIP_OPEN_GROUP_KEY)

        w = QWidget()
        layout = QVBoxLayout(w)
        w.setLayout(layout)

        heading_layout = QHBoxLayout()
        layout.addLayout(heading_layout)
        heading_label = QLabel('&Select and configure the menu items to display:', w)
        heading_layout.addWidget(heading_label)
        # Add hyperlink to a help file at the right. We will replace the correct name when it is clicked.
        help_label = QLabel('<a href="http://www.foo.com/">Help</a>', w)
        help_label.setTextInteractionFlags(Qt.LinksAccessibleByMouse | Qt.LinksAccessibleByKeyboard)
        help_label.setAlignment(Qt.AlignRight)
        help_label.linkActivated.connect(self.help_link_activated)
        heading_layout.addWidget(help_label)

        # Add a horizontal layout containing the table and the buttons next to it
        table_layout = QHBoxLayout()
        layout.addLayout(table_layout)

        # Create a table the user can edit the data values in
        w._table = MenuTableWidget(data_items, w)
        heading_label.setBuddy(w._table)
        table_layout.addWidget(w._table)

        # Add a vertical layout containing the the buttons to move up/down etc.
        button_layout = QtGui.QVBoxLayout()
        table_layout.addLayout(button_layout)
        move_up_button = QtGui.QToolButton(w)
        move_up_button.setToolTip('Move row up')
        move_up_button.setIcon(QIcon(I('arrow-up.png')))
        button_layout.addWidget(move_up_button)
        spacerItem = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem)

        add_button = QtGui.QToolButton(w)
        add_button.setToolTip('Add menu item row')
        add_button.setIcon(QIcon(I('plus.png')))
        button_layout.addWidget(add_button)
        spacerItem2 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem2)

        delete_button = QtGui.QToolButton(w)
        delete_button.setToolTip('Delete menu item row')
        delete_button.setIcon(QIcon(I('minus.png')))
        button_layout.addWidget(delete_button)
        spacerItem1 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem1)

        reset_button = QtGui.QToolButton(w)
        reset_button.setToolTip('Reset to defaults')
        reset_button.setIcon(QIcon(I('clear_left.png')))
        button_layout.addWidget(reset_button)
        spacerItem3 = QtGui.QSpacerItem(20, 40, QtGui.QSizePolicy.Minimum, QtGui.QSizePolicy.Expanding)
        button_layout.addItem(spacerItem3)

        move_down_button = QtGui.QToolButton(w)
        move_down_button.setToolTip('Move row down')
        move_down_button.setIcon(QIcon(I('arrow-down.png')))
        button_layout.addWidget(move_down_button)

        move_up_button.clicked.connect(w._table.move_rows_up)
        move_down_button.clicked.connect(w._table.move_rows_down)
        add_button.clicked.connect(w._table.add_row)
        delete_button.clicked.connect(w._table.delete_rows)
        reset_button.clicked.connect(partial(self.reset_to_defaults, w))

        # Define a bottom layout of additional options to configure
        layout.addSpacing(10)
        open_group_layout = QFormLayout()
        layout.addLayout(open_group_layout)
        open_group_label = QLabel('Open &Group keyboard shortcut:', w)
        w._open_group_ledit = QLineEdit(open_group_shortcut, w)
        w._open_group_ledit.setMaximumSize(100, 20)
        open_group_label.setBuddy(w._open_group_ledit)
        open_group_layout.addRow(open_group_label, w._open_group_ledit)

        # Define a context menu for the table widget
        self.create_context_menu(w)

        return w

    def save_settings(self, w):
        # Invoked when user clicks ok in preferences dialog. Persist new configuration data.
        search_menus = {}
        search_menus[STIP_MENUS_KEY] = w._table.get_data()
        search_menus[STIP_OPEN_GROUP_KEY] = unicode(w._open_group_ledit.text()).strip()
        search_menus[STIP_COL_WIDTH_KEY] = w._table.url_column_width()
        c = config_store()
        c.set(STIP_STORE_MENUS_NAME, search_menus)
        # Force the rebuilding of our menu actions so keyboard shortcuts take effect immediately
        if self.plugin_action:
            self.plugin_action.rebuild_menus(search_menus[STIP_MENUS_KEY],
                                             search_menus[STIP_OPEN_GROUP_KEY])
        else:
            return info_dialog(w, 'Restart Calibre',
                    'You must restart Calibre before using this plugin for the first time.', show=True)

    def customization_help(self):
        # Placeholder method when using gui configuration. Not actually invoked by Calibre code
        # when the config_widget function is defined on this class
        return 'This plugin can only be customized using the GUI'

    def create_context_menu(self, parent):
        table = parent._table
        table.setContextMenuPolicy(Qt.ActionsContextMenu)
        act_add_image = QAction(get_icon(STIP_IMAGE_ADD_ICON), '&Add image...', table)
        act_add_image.triggered.connect(table.display_add_new_image_dialog)
        table.addAction(act_add_image)
        act_open = QAction(get_icon('document_open.png'), '&Open images folder', table)
        act_open.triggered.connect(partial(self.open_images_folder, table, table.resources_dir))
        table.addAction(act_open)
        sep1 = QAction(table)
        sep1.setSeparator(True)
        table.addAction(sep1)
        act_move = QAction(get_icon(STIP_MOVE_TO_TOP_ICON), '&Move active to top', table)
        act_move.triggered.connect(table.move_active_to_top)
        table.addAction(act_move)
        sep2 = QAction(table)
        sep2.setSeparator(True)
        table.addAction(sep2)
        act_test1 = QAction(get_icon(STIP_INTERNET_ICON), '&Test url', table)
        act_test1.setShortcut(QKeySequence(_('Ctrl+T')))
        act_test1.triggered.connect(partial(self.test_search, parent))
        table.addAction(act_test1)
        act_test2 = QAction(get_icon(STIP_INTERNET_ICON), 'Test url &using...', table)
        act_test2.setShortcut(QKeySequence(_('Ctrl+Shift+T')))
        act_test2.triggered.connect(partial(self.test_search_via_dialog, parent))
        table.addAction(act_test2)
        sep3 = QAction(table)
        sep3.setSeparator(True)
        table.addAction(sep3)
        act_import = QAction(get_icon(STIP_IMPORT_ICON), '&Import...', table)
        act_import.triggered.connect(partial(self.import_menus, parent))
        table.addAction(act_import)
        act_export = QAction(get_icon(STIP_EXPORT_ICON), '&Export...', table)
        act_export.triggered.connect(partial(self.export_menus, parent))
        table.addAction(act_export)

    def help_link_activated(self, url):
        self.plugin_action.show_help()

    def reset_to_defaults(self, w):
        if not question_dialog(w, _('Are you sure?'), '<p>'+
                'Are you sure you want to reset to the plugin default menu?<br>' +
                'Any modified configuration and custom menu items will be discarded.',
                show_copy_button=False):
            return
        w._table.populate_table(get_menus_as_dictionary())
        w._open_group_ledit.setText(STIP_DEFAULT_MENU_STORE[STIP_OPEN_GROUP_KEY])

    def open_images_folder(self, parent, path):
        if not os.path.exists(path):
            if not question_dialog(parent, _('Are you sure?'), '<p>'+
                    'Folder does not yet exist. Do you want to create it?<br>%s' % path,
                    show_copy_button=False):
                return
            os.makedirs(path)
        open_local_file(path)

    def test_search_via_dialog(self, parent):
        dialog = PickTestBookDialog(parent)
        dialog.exec_()
        if dialog.result() == QDialog.Rejected:
            return
        # Go ahead an display the webpage using the last selected test book
        self.test_search(parent)

    def test_search(self, parent):
        # Check we are not on a separator row
        test_rows = list(parent._table.get_selected_urls_to_test())
        if len(test_rows) == 0:
            return error_dialog(parent, 'Cannot test',
                                'You must select a menu item with a url to test it.', show=True)
        for tokenised_url, encoding, method in test_rows:
            c = config_store().get(STIP_STORE_TEST_NAME, STIP_DEFAULT_TEST_STORE)
            selected_idx = c.get(STIP_TEST_LAST_BOOK_KEY)
            test_data_items = c.get(STIP_TEST_VALUES_KEY)
            test_data_item = test_data_items[selected_idx]
            mi = Metadata(test_data_item['title'], [test_data_item['author']])
            mi.publisher = test_data_item['publisher']
            mi.isbn = test_data_item['isbn']
            self.plugin_action.open_tokenised_url(tokenised_url, encoding, method, mi)

    def import_menus(self, parent):
        table = parent._table
        archive_path = self.pick_archive_name_to_import(parent)
        if not archive_path:
            return
        # Write the whole file contents into the resources\images directory
        if not os.path.exists(table.resources_dir):
            os.makedirs(table.resources_dir)
        with ZipFile(archive_path, 'r') as zf:
            contents = zf.namelist()
            if 'stip_menus.json' not in contents:
                return error_dialog(parent, 'Import Failed',
                                    'This is not a valid STIP export archive', show=True)
            for resource in contents:
                fs = os.path.join(table.resources_dir,resource)
                with open(fs,'wb') as f:
                    f.write(zf.read(resource))
        json_path = os.path.join(table.resources_dir,'stip_menus.json')
        try:
            # Read the .JSON file to add to the menus then delete it.
            archive_config = JSONConfig('resources/images/stip_menus')
            menus_config = archive_config.get(STIP_STORE_MENUS_NAME).get(STIP_MENUS_KEY)
            # Now insert the menus into the table
            table.append_data(menus_config)
            info_dialog(parent, 'Import completed', '%d menu items added' % len(menus_config),
                        show=True, show_copy_button=False)
        finally:
            if os.path.exists(json_path):
                os.remove(json_path)

    def export_menus(self, parent):
        table = parent._table
        data_items = table.get_selected_data()
        if len(data_items) == 0:
            return error_dialog(parent, 'Cannot export',
                                'No menu items selected to export.', show=True)
        archive_path = self.pick_archive_name_to_export(table)
        if not archive_path:
            return
        # Build our unique list of images that need to be exported
        image_names = {}
        for data in data_items:
            image_name = data['image']
            if image_name and image_name not in image_names:
                image_path = os.path.join(table.resources_dir, image_name)
                if os.path.exists(image_path):
                    image_names[image_name] = image_path
        # Write our menu items out to a json file
        if not os.path.exists(table.resources_dir):
            os.makedirs(table.resources_dir)
        archive_config = JSONConfig('resources/images/stip_menus')
        export_menus = {}
        export_menus[STIP_MENUS_KEY] = data_items
        archive_config.set(STIP_STORE_MENUS_NAME, export_menus)
        json_path = os.path.join(table.resources_dir,'stip_menus.json')

        try:
            # Create the zip file archive
            with ZipFile(archive_path, 'w') as archive_zip:
                archive_zip.write(json_path, os.path.basename(json_path))
                # Add any images referred to in those menu items that are local resources
                for image_name, image_path in image_names.iteritems():
                    archive_zip.write(image_path, os.path.basename(image_path))
            info_dialog(parent, 'Export completed', '%d menu items exported to<br>%s' % (len(data_items), archive_path),
                        show=True, show_copy_button=False)
        finally:
            if os.path.exists(json_path):
                os.remove(json_path)

    def pick_archive_name_to_import(self, parent=None):
        archives = choose_files(parent, 'stip archive dialog', 'Select a menu file archive to import',
                             filters=[('STIP Files', ['stip','zip'])], all_files=False, select_only_single_file=True)
        if not archives:
            return
        f = archives[0]
        return f

    def pick_archive_name_to_export(self, parent=None):
        fd = FileDialog(name='stip archive dialog', title='Save archive as', filters=[('STIP Files', ['zip'])],
                        parent=parent, add_all_files_filter=False, mode=QFileDialog.AnyFile)
        fd.setParent(None)
        if not fd.accepted:
            return None
        return fd.get_files()[0]
# }}}


# For testing, run from command line with this:
# calibre-debug -e search_the_internet_plugin.py
if __name__ == '__main__': # {{{
    from PyQt4.Qt import QApplication
    from calibre.gui2.preferences import test_widget
    app = QApplication([])
    test_widget('Advanced', 'Plugins')
# }}}