#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

CONFIG_LABEL_SELECT_MAIN_TITLE = _('Set Book List')
CONFIG_LABEL_GENERAL = _('The profile configuration determines how books for the book list are chosen and how thay will appear on your reader.')
CONFIG_LABEL_CHOOSE_PROFILE = _('Choose Profile:')
CONFIG_LABEL_PROFILE = _('Profile:')
CONFIG_LABEL_PROFILES = _('Profiles:')
CONFIG_LABEL_DETAILS = _('Details:')
CONFIG_LABEL_SELECTION = _('Choose books from Device/Card A/Card B book list selections.')
CONFIG_LABEL_COLLECTIONS = _('Choose books from collections: ')
CONFIG_LABEL_CHOOSE_ONE = _('Choose one book from each source.')
CONFIG_LABEL_CHOOSE_NEWEST = _('Choose books by newest first.')
CONFIG_LABEL_CHOOSE_OLDEST = _('Choose books by oldest first.')
CONFIG_LABEL_CHOOSE_RANDOM = _('Choose books randomly.')
CONFIG_LABEL_ORDER_DATE = _('Order book list by date.')
CONFIG_LABEL_ORDER_AUTHOR = _('Order book list by author.')
CONFIG_LABEL_ORDER_TITLE = _('Order book list by title.')
CONFIG_LABEL_CHOOSE_GROUP = _('Choose Books From:')
CONFIG_LABEL_SELECTION_OPTION = _('Device/Card A/Card B book list selections')
CONFIG_LABEL_COLLECTIONS_OPTION = _('Collections:')
CONFIG_LABEL_STRATEGY_GROUP = _('Choose Books By:')
CONFIG_LABEL_ONE_OPTION = _('One book from each source')
CONFIG_LABEL_NEWEST_OPTION = _('Newest first')
CONFIG_LABEL_OLDEST_OPTION = _('Oldest first')
CONFIG_LABEL_RANDOM_OPTION = _('Randomly')
CONFIG_LABEL_ORDER_GROUP = _('Order Book List By:')
CONFIG_LABEL_DATE_OPTION = _('Date')
CONFIG_LABEL_AUTHOR_OPTION = _('Author')
CONFIG_LABEL_TITLE_OPTION = _('Title')
CONFIG_LABEL_ADD = _('Add')
CONFIG_LABEL_DELETE = _('Delete')
CONFIG_LABEL_UP = _('Up')
CONFIG_LABEL_DOWN = _('Down')
DEFAULT_PROFILE_NAME = _('New Profile')
DEFAULT_COLLECTION_NAME = _('New Collection')

from PyQt4.Qt import QDialog
import ManageSonyX50BookList

class ManageBookListView(QDialog):
    def __init__(self, parent):
        from PyQt4 import QtCore
        from PyQt4.Qt import QGridLayout, QLabel, QDialogButtonBox
        
        QDialog.__init__(self, parent)
        
        self.setWindowTitle(CONFIG_LABEL_SELECT_MAIN_TITLE)
        
        self._layout = QGridLayout(self)
        self.setLayout(self._layout)
        
        self.setBookListText = QLabel(CONFIG_LABEL_GENERAL)
        self.setBookListText.setWordWrap(True)
        self._layout.addWidget(self.setBookListText, 0, 0, 1, 2)

        self.profileLabel = QLabel()
        self._layout.addWidget(self.profileLabel, 1, 0, 1, 1, QtCore.Qt.AlignLeft)
                
        self.infoLabel = QLabel('<b>' + CONFIG_LABEL_DETAILS + '</b>')
        self._layout.addWidget(self.infoLabel, 2, 0, 1, 2, QtCore.Qt.AlignLeft | QtCore.Qt.AlignTop)

        self.profileDetails = QLabel()
        self.profileDetails.setWordWrap(True)
        self._layout.addWidget(self.profileDetails, 3, 0, 1, 2)
        
        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        self._layout.addWidget(buttonBox, 4, 0, 1, 2)
        
        self._layout.setColumnStretch(1, 1)
        self._layout.setColumnMinimumWidth(1, 200)
    
    def getProfileName(self):
        from PyQt4.Qt import QComboBox
        
        if isinstance(self.profileValue, QComboBox):
            profileName = str(self.profileValue.itemText(self.profileValue.currentIndex()))
        else:
            profileName = str(self.profileValue.text())
            
        return profileName
    
class ManageBookListController():
    model = None
    view = None
    
    def __init__(self, model, view):
        from PyQt4 import QtCore
        from PyQt4.Qt import QComboBox, QLabel
        
        self.model = model
        self.view = view
        
        profileNames = self.model.config.keys()
        
        if len(profileNames) > 1:
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_CHOOSE_PROFILE + '</b>')
            self.view.profileValue = QComboBox()
            
            for index, profileName in enumerate(profileNames):
                self.view.profileValue.addItem(profileName)
        else:
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_PROFILE + '</b>')
            self.view.profileValue = QLabel(profileNames[0])
            
        self.view._layout.addWidget(self.view.profileValue, 1, 1, 1, 1, QtCore.Qt.AlignLeft)
            
        self.view.connect(self.view.profileValue, QtCore.SIGNAL('activated(QString)'), self.onProfileActivated)
        self.setProfile(profileNames[0])
    
    def onProfileActivated(self, text):
        self.setProfile(str(text))
    
    def setProfile(self, profileName):
        profile = self.model.getProfile(profileName)
        booksValue = self.model.getBooksValue(profile)
        strategyValue = self.model.getStrategyValue(profile)
        orderValue = self.model.getOrderValue(profile)
        
        details = '<ul>'
        if ManageSonyX50BookList.CONFIG_BOOKS_VALUE_SELECTION in booksValue:
            details = details + '<li>' + CONFIG_LABEL_SELECTION + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_COLLECTIONS
            for index, value in enumerate(booksValue):
                if index is not 0:
                    details = details + ', '
                details = details + '"' + value + '"'
            details = details + '</li>'
            
        if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_ONE + '</li>'
            
        if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_NEWEST in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_NEWEST + '</li>'
        elif ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_OLDEST in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_OLDEST + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_RANDOM + '</li>'
            
        if ManageSonyX50BookList.CONFIG_ORDER_VALUE_DATE in orderValue:
            details = details + '<li>' + CONFIG_LABEL_ORDER_DATE + '</li>'
        elif ManageSonyX50BookList.CONFIG_ORDER_VALUE_AUTHOR in orderValue:
            details = details + '<li>' + CONFIG_LABEL_ORDER_AUTHOR + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_ORDER_TITLE + '</li>'
            
        details = details + '</ul>'
        
        self.view.profileDetails.setText(details)
        self.view.resize(self.view.sizeHint())
 
class ManageBookListConfigView(QDialog):
    controller = None
    
    def __init__(self):
        from PyQt4 import QtCore
        from PyQt4.Qt import QDialog, QLabel, QListWidget, QPushButton, QGroupBox, QRadioButton, \
            QGridLayout, QCheckBox, QVBoxLayout
        
        QDialog.__init__(self)
        
        self._layout = QGridLayout(self)
        self.setLayout(self._layout)
        
        self.setBookListText = QLabel(CONFIG_LABEL_GENERAL)
        self.setBookListText.setWordWrap(True)
        self._layout.addWidget(self.setBookListText, 0, 0, 1, 4)

        self.profileLabel = QLabel('<b>' + CONFIG_LABEL_PROFILES + '</b>')
        self._layout.addWidget(self.profileLabel, 1, 0, 1, 1, QtCore.Qt.AlignLeft)
        
        self.profileList = QListWidget()
        self._layout.addWidget(self.profileList, 2, 0, 2, 2)
        
        self.profileAddButton = QPushButton(CONFIG_LABEL_ADD)
        self._layout.addWidget(self.profileAddButton, 4, 0, 1, 1)
        
        self.profileDeleteButton = QPushButton(CONFIG_LABEL_DELETE)
        self._layout.addWidget(self.profileDeleteButton, 4, 1, 1, 1)
        
        self.chooseGroup = QGroupBox(CONFIG_LABEL_CHOOSE_GROUP)
        self.selectionOption = QRadioButton(CONFIG_LABEL_SELECTION_OPTION)
        self.collectionsOption = QRadioButton(CONFIG_LABEL_COLLECTIONS_OPTION)
        self.collectionsList = QListWidget()
        self.collectionAddButton = QPushButton(CONFIG_LABEL_ADD)
        self.collectionDeleteButton = QPushButton(CONFIG_LABEL_DELETE)
        self.collectionUpButton = QPushButton(CONFIG_LABEL_UP)
        self.collectionDownButton = QPushButton(CONFIG_LABEL_DOWN)
        self.chooseLayout = QGridLayout(self.chooseGroup)
        self.chooseLayout.addWidget(self.selectionOption, 0, 0, 1, 2)
        self.chooseLayout.addWidget(self.collectionsOption, 1, 0, 1, 2)
        self.chooseLayout.addWidget(self.collectionsList, 2, 0, 2, 2)
        self.chooseLayout.addWidget(self.collectionUpButton, 2, 2, 1, 1, QtCore.Qt.AlignBottom)
        self.chooseLayout.addWidget(self.collectionDownButton, 3, 2, 1, 1, QtCore.Qt.AlignTop)
        self.chooseLayout.addWidget(self.collectionAddButton, 4, 0, 1, 1)
        self.chooseLayout.addWidget(self.collectionDeleteButton, 4, 1, 1, 1)
        self.chooseGroup.setLayout(self.chooseLayout)
        self._layout.addWidget(self.chooseGroup, 2, 2, 1, 2)
        
        self.strategyGroup = QGroupBox(CONFIG_LABEL_STRATEGY_GROUP)
        self.oneOption = QCheckBox(CONFIG_LABEL_ONE_OPTION)
        self.newestOption = QRadioButton(CONFIG_LABEL_NEWEST_OPTION)
        self.oldestOption = QRadioButton(CONFIG_LABEL_OLDEST_OPTION)
        self.randomOption = QRadioButton(CONFIG_LABEL_RANDOM_OPTION)
        self.strategyLayout = QVBoxLayout(self.strategyGroup)
        self.strategyLayout.addWidget(self.oneOption)
        self.strategyLayout.addWidget(self.newestOption)
        self.strategyLayout.addWidget(self.oldestOption)
        self.strategyLayout.addWidget(self.randomOption)
        self.strategyGroup.setLayout(self.strategyLayout)
        self._layout.addWidget(self.strategyGroup, 3, 2, 2, 1)
        
        self.orderGroup = QGroupBox(CONFIG_LABEL_ORDER_GROUP)
        self.dateOption = QRadioButton(CONFIG_LABEL_DATE_OPTION)
        self.authorOption = QRadioButton(CONFIG_LABEL_AUTHOR_OPTION)
        self.titleOption = QRadioButton(CONFIG_LABEL_TITLE_OPTION)
        self.orderLayout = QVBoxLayout(self.orderGroup)
        self.orderLayout.addWidget(self.dateOption)
        self.orderLayout.addWidget(self.authorOption)
        self.orderLayout.addWidget(self.titleOption)
        self.orderGroup.setLayout(self.orderLayout)
        self._layout.addWidget(self.orderGroup, 3, 3, 2, 1)
        
        self._layout.setColumnStretch(0, 1)
        self._layout.setColumnStretch(1, 1)
        
    def setController(self, controller):
        self.controller = controller
             
class ManageBookListConfigController():
    model = None
    view = None
    
    def __init__(self, model, view):
        from PyQt4 import QtCore
        
        self.model = model
        self.view = view
        
        profileNames = self.model.config.keys()
        
        for index, profileName in enumerate(profileNames):
            self.view.profileList.addItem(profileName)
            item = self.view.profileList.item(self.view.profileList.count() - 1)
            item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        
        self.view.connect(self.view.profileList, QtCore.SIGNAL('currentItemChanged(QListWidgetItem *, QListWidgetItem *)'), self.onProfileCurrentItemChanged)
        self.view.connect(self.view.profileList, QtCore.SIGNAL('itemDoubleClicked(QListWidgetItem *)'), self.onProfileItemDoubleClicked)
        self.view.connect(self.view.profileList, QtCore.SIGNAL('itemChanged(QListWidgetItem *)'), self.onProfileItemChanged)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('currentItemChanged(QListWidgetItem *, QListWidgetItem *)'), self.onCollectionsCurrentItemChanged)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('itemDoubleClicked(QListWidgetItem *)'), self.onCollectionsItemDoubleClicked)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('itemChanged(QListWidgetItem *)'), self.onCollectionsItemChanged)
        self.view.connect(self.view.selectionOption, QtCore.SIGNAL('toggled(bool)'), self.onSelectionButtonToggled)
        self.view.connect(self.view.collectionsOption, QtCore.SIGNAL('toggled(bool)'), self.onCollectionsButtonToggled)
        self.view.connect(self.view.oneOption, QtCore.SIGNAL('toggled(bool)'), self.onOneButtonToggled)
        self.view.connect(self.view.newestOption, QtCore.SIGNAL('toggled(bool)'), self.onNewestButtonToggled)
        self.view.connect(self.view.oldestOption, QtCore.SIGNAL('toggled(bool)'), self.onOldestButtonToggled)
        self.view.connect(self.view.randomOption, QtCore.SIGNAL('toggled(bool)'), self.onRandomButtonToggled)
        self.view.connect(self.view.dateOption, QtCore.SIGNAL('toggled(bool)'), self.onDateButtonToggled)
        self.view.connect(self.view.authorOption, QtCore.SIGNAL('toggled(bool)'), self.onAuthorButtonToggled)
        self.view.connect(self.view.titleOption, QtCore.SIGNAL('toggled(bool)'), self.onTitleButtonToggled)
        self.view.connect(self.view.collectionDeleteButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionDeleteButtonClicked)
        self.view.connect(self.view.collectionAddButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionAddButtonClicked)
        self.view.connect(self.view.profileDeleteButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileDeleteButtonClicked)
        self.view.connect(self.view.profileAddButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileAddButtonClicked)
        self.view.connect(self.view.collectionUpButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionUpButtonClicked)
        self.view.connect(self.view.collectionDownButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionDownButtonClicked)
        
        self.view.profileList.setCurrentRow(0)
            
    def getConfigString(self):
        import json
        
        return json.dumps(self.model.config)
    
    def getCurrentProfile(self):
        profileName = str(self.view.profileList.item(self.view.profileList.currentRow()).text())
        
        return self.model.getProfile(profileName)
    
    def getCurrentCollection(self):
        return str(self.view.collectionsList.item(self.view.collectionsList.currentRow()).text())
    
    def setCurrentProfile(self, profileName):
        profile = self.model.getProfile(profileName)
        booksValue = self.model.getBooksValue(profile)
        strategyValue = self.model.getStrategyValue(profile)
        orderValue = self.model.getOrderValue(profile)

        if ManageSonyX50BookList.CONFIG_BOOKS_VALUE_SELECTION in booksValue:
            self.view.selectionOption.setChecked(True)
        else:
            if self.view.collectionsOption.isChecked():
                self.onCollectionsButtonToggled(True)
            else:
                self.view.collectionsOption.setChecked(True)
        
        self.view.oneOption.setChecked(ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE in strategyValue)
        
        if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_NEWEST in strategyValue:
            self.view.newestOption.setChecked(True)
        elif ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_OLDEST in strategyValue:
            self.view.oldestOption.setChecked(True)
        else:
            self.view.randomOption.setChecked(True)
        
        if ManageSonyX50BookList.CONFIG_ORDER_VALUE_DATE in orderValue:
            self.view.dateOption.setChecked(True)
        elif ManageSonyX50BookList.CONFIG_ORDER_VALUE_AUTHOR in orderValue:
            self.view.authorOption.setChecked(True)
        else:
            self.view.titleOption.setChecked(True)
    
    def onProfileCurrentItemChanged(self, newItem, previousItem):
        if newItem is not None:
            self.model.setProfile(str(newItem.text()))
            self.setCurrentProfile(str(newItem.text()))
            self.view.profileDeleteButton.setEnabled(self.view.profileList.count() > 1)
    
    def onProfileItemDoubleClicked(self, item):
        self.view.profileList.editItem(item)
    
    def onCollectionsCurrentItemChanged(self, newItem, previousItem):
        if newItem is not None:
            self.view.collectionDeleteButton.setEnabled(self.view.collectionsList.count() > 1)
            self.view.collectionUpButton.setEnabled(self.view.collectionsList.count() > 1 and self.view.collectionsList.currentRow() > 0)
            self.view.collectionDownButton.setEnabled(self.view.collectionsList.count() > 1 and self.view.collectionsList.currentRow() < self.view.collectionsList.count() - 1)
    
    def onCollectionsItemDoubleClicked(self, item):
        self.view.collectionsList.editItem(item)
    
    def onSelectionButtonToggled(self, on):
        if on:
            profile = self.getCurrentProfile()
            profile[ManageSonyX50BookList.CONFIG_BOOKS_KEY] = ManageSonyX50BookList.CONFIG_BOOKS_VALUE_SELECTION
            
            self.view.collectionAddButton.setDisabled(True)
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
    
    def onCollectionsButtonToggled(self, on):
        from PyQt4 import QtCore
        
        if on:
            autoEdit = False
            profile = self.getCurrentProfile()
            booksValue = self.model.getBooksValue(profile)
            
            if (ManageSonyX50BookList.CONFIG_BOOKS_VALUE_SELECTION in booksValue) or (len(booksValue) is 0):
                profile[ManageSonyX50BookList.CONFIG_BOOKS_KEY] = [DEFAULT_COLLECTION_NAME]
                booksValue = self.model.getBooksValue(profile)
                autoEdit = True
            
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
            
            for index, collection in enumerate(booksValue):
                self.view.collectionsList.addItem(collection)
                item = self.view.collectionsList.item(self.view.collectionsList.count() - 1)
                item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
                
            if autoEdit:
                self.view.collectionsList.editItem(self.view.collectionsList.item(0))
                
            self.view.collectionAddButton.setEnabled(True)
    
    def onOneButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE not in profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY]:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY].insert(0, ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE)
        else:
            if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE in profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY]:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY].remove(ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE)
    
    def onNewestButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE in profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY]:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY] = [ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE, ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_NEWEST]
            else:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY] = [ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_NEWEST]
    
    def onOldestButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE in profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY]:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY] = [ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE, ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_OLDEST]
            else:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY] = [ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_OLDEST]
    
    def onRandomButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE in profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY]:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY] = [ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_ONE, ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_RANDOM]
            else:
                profile[ManageSonyX50BookList.CONFIG_STRATEGY_KEY] = [ManageSonyX50BookList.CONFIG_STRATEGY_VALUE_RANDOM]
    
    def onDateButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[ManageSonyX50BookList.CONFIG_ORDER_KEY] = ManageSonyX50BookList.CONFIG_ORDER_VALUE_DATE
    
    def onAuthorButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[ManageSonyX50BookList.CONFIG_ORDER_KEY] = ManageSonyX50BookList.CONFIG_ORDER_VALUE_AUTHOR
    
    def onTitleButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[ManageSonyX50BookList.CONFIG_ORDER_KEY] = ManageSonyX50BookList.CONFIG_ORDER_VALUE_TITLE
    
    def onCollectionDeleteButtonClicked(self):
        profile = self.getCurrentProfile()
        currentRow = self.view.collectionsList.currentRow()
        previousCollection = self.view.collectionsList.takeItem(currentRow)
        newCollection = self.view.collectionsList.item(self.view.collectionsList.currentRow())
        self.onCollectionsCurrentItemChanged(newCollection, previousCollection)
        
        booksValue = self.model.getBooksValue(profile)
        booksValue.remove(str(previousCollection.text()))
        
        del previousCollection
    
    def onProfileDeleteButtonClicked(self):
        profile = self.getCurrentProfile()
        currentRow = self.view.profileList.currentRow()
        previousProfile = self.view.profileList.takeItem(currentRow)
        newProfile = self.view.profileList.item(self.view.profileList.currentRow())
        self.onProfileCurrentItemChanged(newProfile, previousProfile)
        
        del self.model.config[str(previousProfile.text())]
        
        del previousProfile
    
    def onCollectionUpButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        collectionItem = self.view.collectionsList.takeItem(currentRow)
        self.view.collectionsList.insertItem(currentRow - 1, collectionItem)
        self.view.collectionsList.setCurrentRow(currentRow - 1)
        
        collection = self.getCurrentCollection()
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        index = booksValue.index(collection)
        booksValue.pop(index)
        booksValue.insert(index - 1, collection)
    
    def onCollectionDownButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        collectionItem = self.view.collectionsList.takeItem(currentRow)
        self.view.collectionsList.insertItem(currentRow + 1, collectionItem)
        self.view.collectionsList.setCurrentRow(currentRow + 1)
        
        collection = self.getCurrentCollection()
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        index = booksValue.index(collection)
        booksValue.pop(index)
        booksValue.insert(index + 1, collection)
    
    def onCollectionAddButtonClicked(self):
        from PyQt4 import QtCore
        
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        name = self.getUniqueName(DEFAULT_COLLECTION_NAME, booksValue)
        booksValue.append(name)
        
        self.view.collectionsList.addItem(name)
        lastRow = self.view.collectionsList.count() - 1
        self.view.collectionsList.setCurrentRow(lastRow)
        item = self.view.collectionsList.item(lastRow)
        item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        self.view.collectionsList.editItem(item)
    
    def onProfileAddButtonClicked(self):
        from PyQt4 import QtCore
        
        name = self.getUniqueName(DEFAULT_PROFILE_NAME, self.model.config.keys())
        profile = {}
        self.model.normalize(name, profile)
        self.model.config[name] = profile
        
        self.view.profileList.addItem(name)
        lastRow = self.view.profileList.count() - 1
        self.view.profileList.setCurrentRow(lastRow)
        item = self.view.profileList.item(lastRow)
        item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        self.view.profileList.editItem(item)
    
    def onCollectionsItemChanged(self, item):
        from PyQt4.Qt import QString
        
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        row = self.view.collectionsList.row(item)
        oldName = booksValue[row]
        newName = str(item.text()).strip()
        
        if newName != str(item.text()): # Ignore added leading and trailing spaces
            item.setText(QString(newName))
        
        if newName is None or len(newName) is 0:
            newName = oldName
            item.setText(QString(oldName))
        
        if newName != oldName:
            uniqueName = self.getUniqueName(newName, booksValue)
            booksValue[row] = uniqueName
            item.setText(QString(uniqueName))
    
    def onProfileItemChanged(self, item):
        import copy
        from PyQt4.Qt import QString
        
        profile = self.model.currentProfile
        oldName = self.model.getNameValue(profile)
        newName = str(item.text()).strip()
        
        if newName != str(item.text()): # Ignore added leading and trailing spaces
            item.setText(QString(newName))
        
        if newName is None or len(newName) is 0:
            newName = oldName
            item.setText(QString(oldName))
        
        if newName != oldName:
            uniqueName = self.getUniqueName(newName, self.model.config.keys())
            newProfile = copy.copy(profile)
            self.model.normalize(uniqueName, newProfile)
            self.model.config[uniqueName] = newProfile
            self.model.setProfile(uniqueName)
            del self.model.config[oldName]
            item.setText(QString(uniqueName))
    
    def getUniqueName(self, name, existingNames):
        namePrefix = name
        tryName = name
        index = 1
        
        while tryName in existingNames:
            tryName = namePrefix + ':' + str(index)
            index = index + 1
        
        return tryName
