#!/usr/bin/env python
# vim:fileencoding=UTF-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

# This plugin file is the one that will be loaded by Calibre when you first install the plugin
# or when Calibre starts up. We should not add imports in this top level namespace as we
# need to ensure calibre command line based tools will still work (which do not require Qt).
# The class below acts as a proxy for the real implementation and ensures that the
# "real" implementation classes in the ManageSonyX50BookList_plugin_code.py file are only loaded on
# demand allowing more freedom in how the classes within are structured.

#History:
# Version 1.0.0 Initial release
# Version 1.0.1 Implement proxy wrapper to ensure modues only get loaded when required
# Version 1.0.2 Fix regression introduced in 1.0.1 which resulted in JSON error when no prior
#               plugin configuration exists.
# Version 1.0.3 Restructure code; add timestamp check on read/write of cache to make sure no
#               background updates have happened; change order of processing so that read/write
#               operation is not interrupted by user interaction (try to avoid stale data scenario);
#               fix date sort bug


# Utility class to load modules from a zip file and dynamically instantiate objects of a specified
# class from a loaded module.
class ModuleClassLoader(object):
	loadedModules = {}
	
	def getModule(self, zipPath, moduleName):
	    import zipimport
	    
	    module = None
	    moduleKey = self.getModuleKey(zipPath, moduleName)
	    
	    if moduleKey in self.loadedModules.keys():
	        module = self.loadedModules[moduleKey]
	    
	    if module is None:
	        importer = zipimport.zipimporter(zipPath)
	        self.loadedModules[moduleKey] = importer.load_module(moduleName)
	        module = self.loadedModules[moduleKey]
	        
	    return module
	
	def getClassInstance(self, module, className, *args):
	    return module.__dict__[className](*args)
	
	def getModuleKey(self, zipPath, moduleName):
	    return zipPath + '#' + moduleName
    
# A proxy for the ManageSonyX50BookList plugin. This proxy will not allocate any resources or import
# any modules other than InterfaceActionBase until it is actually used.

PLUGIN_NAME = _('Manage Sony x50 Reader Book List')

from calibre.customize import InterfaceActionBase

class ManageSonyX50BookList(InterfaceActionBase):
    name = PLUGIN_NAME
    author = 'Karl Weckworth'
    description = _('Set the Sony x50 Reader book list explicitly or based on collections.')
    version = (1, 0, 3)
    supported_platforms = ['windows', 'osx', 'linux']
    can_be_disabled = True

    python_code_file_name = 'ManageSonyX50BookList'
    base_plugin_name = 'ManageSonyX50BookList'
    action_plugin_name = 'ManageSonyX50BookListAction'
    
    loader = ModuleClassLoader()

    def __init__(self, plugin_path):
        InterfaceActionBase.__init__(self, plugin_path)
        self.plugin_path = plugin_path
        self._base_plugin = None
        self._base_module = None

    def get_base_module(self):
        if self._base_module is None:
            self._base_module = self.loader.getModule(self.plugin_path, self.python_code_file_name)
        return self._base_module

    def get_base_plugin(self):
        module = self.get_base_module()
        if self._base_plugin is None:
            self._base_plugin = self.loader.getClassInstance(module, self.base_plugin_name, self)
        return self._base_plugin

    def config_widget(self):
        return self.get_base_plugin().config_widget()

    def save_settings(self, w):
        self.get_base_plugin().save_settings(w)

    def customization_help(self):
        return self.get_base_plugin().customization_help()

    def load_actual_plugin(self, gui):
        module = self.get_base_module()
        return self.loader.getClassInstance(module, self.action_plugin_name, gui, self.site_customization, self)

# For testing, run from command line with this:
# calibre-debug -e ManageSonyX50BookList_plugin.py
if __name__ == '__main__':
    from PyQt4.Qt import QApplication
    from calibre.gui2.preferences import test_widget
    app = QApplication([])
    test_widget('Advanced', 'Plugins')
