#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

# This file contains the actual implementation of the classes required to
# implement the ManageSonyX50BookList plugin, with any supporting classes it requires

SONY_VENDOR_ID = [0x054c]
SONY_X50_PRODUCT_ID = [0x031e]

CONFIG_NAME_KEY = 'name'
CONFIG_BOOKS_KEY = 'books'
CONFIG_BOOKS_VALUE_SELECTION = 'selection'
CONFIG_STRATEGY_KEY = 'strategy'
CONFIG_STRATEGY_VALUE_ONE = 'one'
CONFIG_STRATEGY_VALUE_NEWEST = 'newest'
CONFIG_STRATEGY_VALUE_OLDEST = 'oldest'
CONFIG_STRATEGY_VALUE_RANDOM = 'random'
CONFIG_ORDER_KEY = 'order'
CONFIG_ORDER_VALUE_DATE = 'date'
CONFIG_ORDER_VALUE_AUTHOR = 'author'
CONFIG_ORDER_VALUE_TITLE = 'title'

DEFAULT_CONFIG_BOOKS = CONFIG_BOOKS_VALUE_SELECTION
DEFAULT_CONFIG_STRATEGY = CONFIG_STRATEGY_VALUE_NEWEST
DEFAULT_CONFIG_ORDER = CONFIG_ORDER_VALUE_DATE

DEFAULT_CONFIG_PROFILE_KEY = _('Default')
DEFAULT_PLUGIN_CUSTOMIZATION = '{"' + DEFAULT_CONFIG_PROFILE_KEY + '": {}}'

MAIN_INDEX = 0
CARD_A_INDEX = 1
CARD_B_INDEX = 2
DEVICE_KEY_PREFIXES = ['Main:', 'A:', 'B:']
DEVICE_KEY_INDICES = {'Main': MAIN_INDEX, 'A': CARD_A_INDEX, 'B': CARD_B_INDEX}
SONY_MEDIA_XML = 'database/cache/media.xml'
SONY_CACHE_XML = 'Sony Reader/database/cache.xml'
SONY_CACHE_FILE = [SONY_MEDIA_XML, SONY_CACHE_XML, SONY_CACHE_XML]
READER_DRIVE_LABEL = [_('Main'), _('Card A'), _('Card B')]

MAX_BOOK_LIST_LENGTH = 3

def getPluginCustomization(self):
    from calibre.customize.ui import plugin_customization
    
    pluginCustomization = plugin_customization(self)
    
    if pluginCustomization is None or len(pluginCustomization.strip()) is 0:
        pluginCustomization = DEFAULT_PLUGIN_CUSTOMIZATION
    
    return pluginCustomization.strip()

class ManageSonyX50BookList(object):
    def __init__(self, proxy):
        self.proxy = proxy
        self.name = self.proxy.name

    def __enter__(self, *args):
        self.proxy.__enter__(*args)

    def __exit__(self, *args):
        self.proxy.__exit__(*args)

    def customization_help(self, gui=False):
        return '{<profile_name>: {name: <profile_name>, books: (*selection | [<collection_name>]), strategy: [(one | (*newest | oldest | random))], order: (*date | author | title)}, ...}'
    
    def config_widget(self):
        import json
        
        self.proxy.loader.getModule(self.proxy.plugin_path, 'ManageSonyX50BookListView')
        from ManageSonyX50BookListView import ManageBookListConfigView, ManageBookListConfigController
        
        config = Config(json.loads(getPluginCustomization(self)))
        configWidget = ManageBookListConfigView()
        controller = ManageBookListConfigController(config, configWidget)
        configWidget.setController(controller) # Hang on to controller reference since widget is externally controlled
        
        return configWidget
    
    def save_settings(self, configWidget):
        from calibre.customize.ui import customize_plugin

        configString = configWidget.controller.getConfigString()
        customize_plugin(self, configString)
        
from calibre.gui2.actions import InterfaceAction

class ManageSonyX50BookListAction(InterfaceAction):
    action_type = 'current'
    action_spec = (_('Book List'), 'books_in_series.png', _('Manage Sony x50 reader book list'), None)
    dont_add_to = frozenset(['toolbar', 'context-menu', 'context-menu-device'])
    
    def __init__(self, gui, site_customization, proxy):
        InterfaceAction.__init__(self, gui, site_customization)
        self.proxy = proxy
        self.name = self.proxy.name

    def genesis(self):
        self.qaction.triggered.connect(self.setSonyBookList)

    def setSonyBookList(self, *args):
        import json
        
        devInfo = SonyX50Reader(self.gui, self.proxy.name)
        devInfo.detect()
                
        if devInfo.isConnected:
            config = Config(json.loads(getPluginCustomization(self)))
            self.chooseProfile(config, devInfo)
            
    def chooseProfile(self, config, devInfo):
        self.proxy.loader.getModule(self.plugin_path, 'ManageSonyX50BookListView')
        from ManageSonyX50BookListView import ManageBookListView, ManageBookListController

        dialog = ManageBookListView(self.gui)
        controller = ManageBookListController(config, dialog)
        
        if dialog.exec_() == dialog.Accepted:
            config.setProfile(dialog.getProfileName())
            self.handleSetBookList(config, devInfo)
            
    def handleSetBookList(self, config, devInfo):
        from calibre.gui2 import warning_dialog
        
        if self.gui.job_manager.has_device_jobs():
            warning_dialog(self.gui, self.proxy.name, _('A background device job is running. Wait until that completes and try again.'), show=True)
        else:
            cache = SonyX50ReaderCache(self.gui, devInfo, self.proxy.name)
            cache.load()
            
            if cache.cacheDOM is not None:
                self.handleCache(config, devInfo, cache)
    
    def handleCache(self, config, devInfo, cache):
        bookList = BookList(config, devInfo)
        bookList.build(cache.cacheDOM)
            
        if bookList.books is not None:
            self.handleBooklist(config, cache, bookList)
    
    def handleBooklist(self, config, cache, bookList):
        bookSelection = BookSelection(self.gui, self.proxy.name)
        
        if CONFIG_BOOKS_VALUE_SELECTION in config.getBooksValue():
            bookSelection.getSelection(bookList.pathIdMap)
        
        if bookSelection.books is not None:
            self.handleProfile(config, cache, bookList, bookSelection)
        
    def handleProfile(self, config, cache, bookList, bookSelection):
        shortBookList = ShortBookList(config, bookList, bookSelection)
        shortBookList.extract()
        
        if shortBookList.books is not None:
            self.handleShortBookList(cache, shortBookList)
    
    def handleShortBookList(self, cache, shortBookList):
        from calibre.gui2 import info_dialog
        
        if len(shortBookList.books) is 0:
            info_dialog(self.gui, self.proxy.name, 'No books satisfied the selection criteria for this profile.', show=True)
        
        else:
            cache.update(shortBookList.books)
            info_dialog(self.gui, self.proxy.name, self.formatConfirmation(shortBookList), show=True)
    
    def formatConfirmation(self, shortBookList):
        confirmation = _('The Sony Reader book list was successfully updated with the following books:') + '<ul>'
        
        for index, book in enumerate(shortBookList.books):
            bookLabel = _('{0} by {1}').format(book['title'], book['author'])
            confirmation = confirmation + '<li>' + bookLabel + '</li>'
        
        confirmation = confirmation + '</ul>'
        
        return confirmation
                
class Config():
    config = None
    currentProfile = None
    
    def __init__(self, config):
        self.config = config
        
        for name in self.config.keys():
            self.normalize(name, self.config[name])
    
    def normalize(self, profileName, profile):
        keys = profile.keys()
        
        profile[CONFIG_NAME_KEY] = profileName
        
        if CONFIG_BOOKS_KEY not in keys:
            profile[CONFIG_BOOKS_KEY] = DEFAULT_CONFIG_BOOKS
        
        if CONFIG_STRATEGY_KEY not in keys:
            profile[CONFIG_STRATEGY_KEY] = [DEFAULT_CONFIG_STRATEGY]
        
        if CONFIG_STRATEGY_VALUE_NEWEST not in profile[CONFIG_STRATEGY_KEY] and \
           CONFIG_STRATEGY_VALUE_OLDEST not in profile[CONFIG_STRATEGY_KEY] and \
           CONFIG_STRATEGY_VALUE_RANDOM not in profile[CONFIG_STRATEGY_KEY]:
            profile[CONFIG_STRATEGY_KEY].append(DEFAULT_CONFIG_STRATEGY)
        
        if CONFIG_ORDER_KEY not in keys:
            profile[CONFIG_ORDER_KEY] = DEFAULT_CONFIG_ORDER
    
    def setProfile(self, profileName):
        self.currentProfile = self.config[profileName]
    
    def getProfile(self, profileName):
        return self.config[profileName]
    
    def getNameValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_NAME_KEY]
    
    def getBooksValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_BOOKS_KEY]
    
    def getStrategyValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_STRATEGY_KEY]
    
    def getOrderValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_ORDER_KEY]
            
class SonyX50Reader():
    gui = None
    
    def __init__(self, gui, pluginName):
        self.gui = gui
        self.pluginName = pluginName
        self.initState()
    
    def initState(self):
        self.isConnected = True
        self.rootPath = [None, None, None] # [main, card A, card B]
        self.vendorId = None
        self.productId = None
    
    def detect(self):
        from calibre.gui2 import error_dialog
        
        self.initState()
        
        # Confirm Sony is connected
        if self.gui.device_manager.connected_device is not None:
            self.rootPath[MAIN_INDEX] = self.gui.device_manager.connected_device._main_prefix
            self.rootPath[CARD_A_INDEX] = self.gui.device_manager.connected_device._card_a_prefix
            self.rootPath[CARD_B_INDEX] = self.gui.device_manager.connected_device._card_b_prefix
            self.vendorId = self.gui.device_manager.connected_device.VENDOR_ID
            self.productId = self.gui.device_manager.connected_device.PRODUCT_ID
            
        else:
            error_dialog(self.gui, self.pluginName, _('Sony reader not detected.'), show=True)
            self.isConnected = False
            
            # DEBUG - comment out lines above and uncomment those below to access test data
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.empty/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.2/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.main/'
            #self.rootPath[CARD_A_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.a/'
            #self.rootPath[CARD_B_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.b/'
            #self.vendorId = SONY_VENDOR_ID
            #self.productId = SONY_X50_PRODUCT_ID
            
        if self.isConnected and ((self.vendorId != SONY_VENDOR_ID) or (self.productId != SONY_X50_PRODUCT_ID)):
            error_dialog(self.gui, self.pluginName, _('Reader does not appear to be a Sony x50 model.'), show=True)
            self.isConnected = False
        
class SonyX50ReaderCache():
    gui = None
    devInfo = None
    
    def __init__(self, gui, devInfo, pluginName):
        self.gui = gui
        self.devInfo = devInfo
        self.pluginName = pluginName
        self.initState()
    
    def initState(self):
        self.cacheDOM = [None, None, None] # [main, card a, card b]
    
    def load(self):
        from calibre.gui2 import error_dialog
        
        self.initState()
        
        if self.devInfo.rootPath[MAIN_INDEX] is not None:
            self.cacheDOM[MAIN_INDEX] = self.loadCache(MAIN_INDEX)
            
        if self.devInfo.rootPath[CARD_A_INDEX] is not None:
            self.cacheDOM[CARD_A_INDEX] = self.loadCache(CARD_A_INDEX)
            
        if self.devInfo.rootPath[CARD_B_INDEX] is not None:
            self.cacheDOM[CARD_B_INDEX] = self.loadCache(CARD_B_INDEX)
        
        if (self.cacheDOM[MAIN_INDEX] is None) and (self.cacheDOM[CARD_A_INDEX] is None) and (self.cacheDOM[CARD_B_INDEX] is None):
            error_dialog(self.gui, self.pluginName, _('The SONY reader does not have any books loaded.'), show=True)
            self.cacheDOM = None
    
    def update(self, bookList):
        import os, time, copy
        
        updatedDOMs = [None, None, None] # [main, card a, card b]
        
        books = copy.copy(bookList)
        books.reverse()
        baseTime = round(time.time())
        adjustment = self.getTimeAdjustment()
        
        for index, book in enumerate(books):
            path = book['path']
            modTime = baseTime + (index * 2) # With one second intervals, I sometimes see duplicate times
            os.utime(path, (modTime, modTime))
            
            book['date'] = time.strftime('%a, %d %b %Y %H:%M:%S GMT', time.gmtime(os.path.getmtime(path)))
            book['tz'] = str(int(adjustment / 60))
            
            deviceIndex = DEVICE_KEY_INDICES[book['id'].split(':')[0]]
            cacheDOM = self.cacheDOM[deviceIndex]
            updatedDOMs[deviceIndex] = cacheDOM
            
            self.updateCache(book)
        
        for index, updatedDOM in enumerate(updatedDOMs):
            if updatedDOM is not None:
                self.writeCache(index, updatedDOM)
        
    def getTimeAdjustment(self):
        import time
        
        return time.timezone
    
    def loadCache(self, deviceIndex):
        import os
        from lxml import etree
        from calibre.constants import DEBUG
        from calibre.ebooks.chardet import xml_to_unicode
        from calibre.gui2 import error_dialog
        
        cache = {}
        path = os.path.join(self.devInfo.rootPath[deviceIndex], SONY_CACHE_FILE[deviceIndex])
        
        if not os.path.exists(path):
            message = _('The SONY XML cache ({0}) is missing. Any books on that drive will not be detected.')
            error_dialog(self.gui, self.pluginName, message.format(READER_DRIVE_LABEL[deviceIndex]), show=True)
            cache = None

        if cache is not None:
            cache['mtime'] = os.path.getmtime(path)
            raw = None
            with open(path, 'rb') as f:
                raw = f.read()
            
            parser = etree.XMLParser(recover=True)
            roots = etree.fromstring(xml_to_unicode(raw, strip_encoding_pats=True, 
                                         assume_utf8=True,
                                         verbose=DEBUG)[0],
                                     parser=parser)
                                 
            if roots is None:
                cache = None
            else:
                cache['roots'] = roots
        
        if cache is not None:
            cache['books'] = roots.xpath('//*[local-name()="text"]')
            
            if not cache['books']:
                cache = None
                
        if cache is not None:
            cache['collections'] = roots.xpath('//*[local-name()="playlist"]')
                  
        return cache
    
    def updateCache(self, book):
        DOMBook = book['DOMRecord']
        DOMBook.set('date', book['date'])
        DOMBook.set('tz', book['tz'])
    
    def writeCache(self, deviceIndex, cacheDOM):
        import os
        from lxml import etree
        from calibre.gui2 import error_dialog
        
        path = os.path.join(self.devInfo.rootPath[deviceIndex], SONY_CACHE_FILE[deviceIndex])
        raw = etree.tostring(cacheDOM['roots'], encoding='UTF-8', xml_declaration=True)
        raw = raw.replace("<?xml version='1.0' encoding='UTF-8'?>", '<?xml version="1.0" encoding="UTF-8"?>')
        
        # Check that cache is still consistent
        if os.path.getmtime(path) != cacheDOM['mtime']:
            message = _('The SONY XML cache ({0}) has been updated outside of this plugin. To avoid corruption, the final write will be bypassed. Please invoke the action again.')
            error_dialog(self.gui, self.pluginName, message.format(READER_DRIVE_LABEL[deviceIndex]), show=True)
        else:
            with open(path, 'wb') as f:
                f.write(raw)

class BookList():
    config = None
    devInfo = None
    
    def __init__(self, config, devInfo):
        self.config = config
        self.devInfo = devInfo
        self.initState()
    
    def initState(self):
        self.books = {}
        self.collections = {}
        self.pathIdMap = {}
        
    def build(self, cacheDOM):
        self.initState()
        
        if cacheDOM[MAIN_INDEX] is not None:
            self.buildBookList(cacheDOM[MAIN_INDEX], MAIN_INDEX)
            
        if cacheDOM[CARD_A_INDEX] is not None:
            self.buildBookList(cacheDOM[CARD_A_INDEX], CARD_A_INDEX)
            
        if cacheDOM[CARD_B_INDEX] is not None:
            self.buildBookList(cacheDOM[CARD_B_INDEX], CARD_B_INDEX)
    
    def buildBookList(self, cacheDOM, deviceIndex):
        import os, calendar
        from calibre.devices.prs505.sony_cache import strptime
        
        for book in cacheDOM['books']:
            id = DEVICE_KEY_PREFIXES[deviceIndex] + str(book.get('id'))
            self.books[id] = {}
            self.books[id]['DOMRecord'] = book
            self.books[id]['id'] = id
            self.books[id]['author'] = book.get('author')
            self.books[id]['date'] = book.get('date')
            #self.books[id]['dateSorter'] = calendar.timegm(time.strptime(self.books[id]['date'], '%a, %d %b %Y %H:%M:%S GMT'))
            self.books[id]['dateSorter'] = calendar.timegm(strptime(self.books[id]['date']))
            self.books[id]['tz'] = book.get('tz')
            self.books[id]['title'] = book.get('title')
            self.books[id]['titleSorter'] = book.get('titleSorter')
            
            path = os.path.normpath(os.path.join(self.devInfo.rootPath[deviceIndex], book.get('path')))
            self.books[id]['path'] = path
            self.pathIdMap[path] = id
        
        names = self.collections.keys()
        for collection in cacheDOM['collections']:
            title = collection.get('title')
            
            if title not in names:
                self.collections[title] = []
            
            for item in collection.xpath('*[local-name()="item"]'):
               self.collections[title].append(DEVICE_KEY_PREFIXES[deviceIndex] + str(item.get('id')))

class BookSelection():
    gui = None
    
    def __init__(self, gui, pluginName):
        self.gui = gui
        self.pluginName = pluginName
        self.initState()
    
    def initState(self):
        self.books = []
    
    def getSelection(self, pathIdMap):
        import os
        from calibre.gui2 import warning_dialog
        
        self.initState()
        
        if self.gui.current_view() is self.gui.library_view:
            warning_dialog(self.gui, self.pluginName, _('You must select books from the device book lists.'), show=True)
            self.books = None
            
        if self.books is not None:
            rows = self.gui.current_view().selectionModel().selectedRows();
        
            if len(rows) is 0:
                warning_dialog(self.gui, self.pluginName, _('No books are selected.'), show=True)
                self.books = None
           
        if self.books is not None:
            paths = self.gui.current_view().model().paths(rows)
        
            for path in paths:
               self.books.append(pathIdMap[os.path.normpath(path)])

class ShortBookList():
    config = None
    bookList = None
    bookSelection = None
    
    def __init__(self, config, bookList, bookSelection):
        self.config = config
        self.bookList = bookList
        self.bookSelection = bookSelection
        self.initState()
    
    def initState(self):
        self.books = []
    
    def extract(self):
        self.initState()
        
        collectionNames = self.bookList.collections.keys()
        idSets = []
        if len(self.bookSelection.books) is not 0:
            idSets.append(self.bookSelection.books)
        
        else:
            for index, collectionName in enumerate(self.config.getBooksValue()):
                if collectionName in collectionNames:
                    idSets.append(self.bookList.collections[collectionName])
                
        bookSets = []
        for index, idSet in enumerate(idSets):
            bookSet = []
            for index2, bookId in enumerate(idSet):
                bookSet.append(self.bookList.books[bookId])
            
            if len(bookSet) is not 0:
                bookSets.append(bookSet)
        
        if len(bookSets) is not 0:
            self.handleBookSets(bookSets)
    
    def handleBookSets(self, bookSets):
        maxRequired = MAX_BOOK_LIST_LENGTH
            
        for index, bookSet in enumerate(bookSets):
            bookSet.sort(key=lambda book: book['dateSorter'])
            self.books.extend(self.chooseBooks(bookSet, maxRequired))
            maxRequired = MAX_BOOK_LIST_LENGTH - len(self.books)
            
            if len(self.books) is MAX_BOOK_LIST_LENGTH:
                break
        
        self.sortBooks()
    
    def chooseBooks(self, bookSet, maxRequired):
        import random
        
        strategy = self.config.getStrategyValue()
        
        if CONFIG_STRATEGY_VALUE_ONE in strategy:
            maxLen = 1
        else:
            maxLen = maxRequired
        
        books = []
            
        # Select newest first
        if CONFIG_STRATEGY_VALUE_NEWEST in strategy:
            for index, book in enumerate(bookSet):
                book = bookSet[len(bookSet) - index - 1]
                
                if (book not in self.books):
                    books.append(book)
                    
                    if len(books) is maxLen:
                        break
        
        # Select oldest first
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategy:
            for index, book in enumerate(bookSet):
                book = bookSet[index]
                
                if book not in self.books:
                    books.append(book)
                    
                    if len(books) is maxLen:
                        break
        
        # Select randomly
        else:
            while len(books) is not maxLen:
                book = bookSet[random.randint(0, len(bookSet) - 1)]
                
                if (book in books) or (book in self.books):
                    bookSet.remove(book)
                else:
                    books.append(book)
                
                if len(bookSet) is 0:
                    break
        
        return books
    
    def sortBooks(self):
        order = self.config.getOrderValue()
        
        if order == CONFIG_ORDER_VALUE_DATE:
            self.books.sort(key=lambda book: book['dateSorter'], reverse=True)
        elif order == CONFIG_ORDER_VALUE_AUTHOR:
            self.books.sort(key=lambda book: book['author'], reverse=False)
        else:
            self.books.sort(key=lambda book: book['titleSorter'], reverse=False)
