#!/usr/bin/env python
# vim:fileencoding=utf-8:ts=4:sw=4:sta:et:sts=4:ai

__license__   = 'GPL v3'
__copyright__ = '2010, Kovid Goyal <kovid@kovidgoyal.net>'
__docformat__ = 'restructuredtext en'

# This file contains the actual implementation of the classes required to
# implement the ManageSonyX50BookList plugin, with any supporting classes it requires

PLUGIN_NAME = _('Manage Sony x50 Reader Book List')

SONY_VENDOR_ID = [0x054c]
SONY_X50_PRODUCT_ID = [0x031e]

CONFIG_NAME_KEY = 'name'
CONFIG_BOOKS_KEY = 'books'
CONFIG_BOOKS_VALUE_SELECTION = 'selection'
CONFIG_STRATEGY_KEY = 'strategy'
CONFIG_STRATEGY_VALUE_ONE = 'one'
CONFIG_STRATEGY_VALUE_NEWEST = 'newest'
CONFIG_STRATEGY_VALUE_OLDEST = 'oldest'
CONFIG_STRATEGY_VALUE_RANDOM = 'random'
CONFIG_ORDER_KEY = 'order'
CONFIG_ORDER_VALUE_DATE = 'date'
CONFIG_ORDER_VALUE_AUTHOR = 'author'
CONFIG_ORDER_VALUE_TITLE = 'title'

DEFAULT_CONFIG_BOOKS = CONFIG_BOOKS_VALUE_SELECTION
DEFAULT_CONFIG_STRATEGY = CONFIG_STRATEGY_VALUE_NEWEST
DEFAULT_CONFIG_ORDER = CONFIG_ORDER_VALUE_DATE

DEFAULT_CONFIG_PROFILE_KEY = _('Default')
DEFAULT_PLUGIN_CUSTOMIZATION = '{"' + DEFAULT_CONFIG_PROFILE_KEY + '": {}}'

MAIN_INDEX = 0
CARD_A_INDEX = 1
CARD_B_INDEX = 2
DEVICE_KEY_PREFIXES = ['Main:', 'A:', 'B:']
DEVICE_KEY_INDICES = {'Main': MAIN_INDEX, 'A': CARD_A_INDEX, 'B': CARD_B_INDEX}
SONY_MEDIA_XML = 'database/cache/media.xml'
SONY_CACHE_XML = 'Sony Reader/database/cache.xml'
SONY_CACHE_FILE = [SONY_MEDIA_XML, SONY_CACHE_XML, SONY_CACHE_XML]
READER_DRIVE_LABEL = [_('Main'), _('Card A'), _('Card B')]

MAX_BOOK_LIST_LENGTH = 3

def getPluginCustomization(self):
    from calibre.customize.ui import plugin_customization
    
    pluginCustomization = plugin_customization(self)
    
    if pluginCustomization is None or len(pluginCustomization.strip()) is 0:
        pluginCustomization = DEFAULT_PLUGIN_CUSTOMIZATION
    
    return pluginCustomization.strip()

from calibre.gui2.actions import InterfaceAction

class ManageSonyX50BookListAction(InterfaceAction):
    name = _('Manage Sony x50 Reader Book List')
    action_type = 'current'
    action_spec = (_('Book List'), 'books_in_series.png', _('Manage Sony x50 reader book list'), None)
    dont_add_to = frozenset(['toolbar', 'context-menu', 'context-menu-device'])
    #dont_add_to = frozenset(['context-menu', 'context-menu-device'])
    
    def genesis(self):
        self.qaction.triggered.connect(self.set_sony_book_list)

    def set_sony_book_list(self, *args):
        import json
        from calibre.gui2 import warning_dialog
        
        if self.gui.job_manager.has_device_jobs():
            warning_dialog(self.gui, PLUGIN_NAME, _('A background device job is running. Wait until that completes and try again.'), show=True)
        else:
            devInfo = SonyX50Reader(self.gui)
            devInfo.detect()
                    
            if devInfo.isConnected:
                config = Config(json.loads(getPluginCustomization(self)))
                self.handle_set_bookList(config, devInfo)
            
    def handle_set_bookList(self, config, devInfo):
        cache = SonyX50ReaderCache(self.gui, devInfo)
        cache.load()
        
        if cache.cacheDOM is not None:
            self.handle_cache(config, devInfo, cache)
    
    def handle_cache(self, config, devInfo, cache):
        bookList = BookList(config, devInfo)
        bookList.build(cache.cacheDOM)
            
        if bookList.books is not None:
            self.handle_booklist(config, cache, bookList)
    
    def handle_booklist(self, config, cache, bookList):
        dialog = ManageBookListView(self.gui)
        controller = ManageBookListController(config, dialog)
        
        if dialog.exec_() == dialog.Accepted:
            config.setProfile(dialog.getProfileName())
            bookSelection = BookSelection(self.gui)
            
            if CONFIG_BOOKS_VALUE_SELECTION in config.getBooksValue():
                bookSelection.get_selection(bookList.pathIdMap)
            
            if bookSelection.books is not None:
                self.handle_profile(config, cache, bookList, bookSelection)
        
    def handle_profile(self, config, cache, bookList, bookSelection):
        shortBookList = ShortBookList(config, bookList, bookSelection)
        shortBookList.extract()
        
        if shortBookList.books is not None:
            self.handle_shortBookList(cache, shortBookList)
    
    def handle_shortBookList(self, cache, shortBookList):
        from calibre.gui2 import info_dialog
        
        if len(shortBookList.books) is 0:
            info_dialog(self.gui, PLUGIN_NAME, 'No books satisfied the selection criteria for this profile.', show=True)
        
        else:
            cache.update(shortBookList.books)
            info_dialog(self.gui, PLUGIN_NAME, self.format_confirmation(shortBookList), show=True)
    
    def format_confirmation(self, shortBookList):
        confirmation = _('The Sony Reader book list was successfully updated with the following books:') + '<ul>'
        
        for index, book in enumerate(shortBookList.books):
            bookLabel = _('{0} by {1}').format(book['title'], book['author'])
            confirmation = confirmation + '<li>' + bookLabel + '</li>'
        
        confirmation = confirmation + '</ul>'
        
        return confirmation
                
class ManageSonyX50BookList(object):
    name = PLUGIN_NAME
    
    def __init__(self, proxy):
        self.proxy = proxy

    def __enter__(self, *args):
        self.proxy.__enter__(*args)

    def __exit__(self, *args):
        self.proxy.__exit__(*args)
    
    def customization_help(self, gui=False):
        return '{<profile_name>: {name: <profile_name>, books: (*selection | [<collection_name>]), strategy: [(one | (*newest | oldest | random))], order: (*date | author | title)}, ...}'
    
    def config_widget(self):
        import json
        
        config = Config(json.loads(getPluginCustomization(self)))
        configWidget = ManageBookListConfigView()
        controller = ManageBookListConfigController(config, configWidget)
        configWidget.setController(controller) # Hang on to controller reference since widget is externally controlled
        
        return configWidget
    
    def save_settings(self, configWidget):
        from calibre.customize.ui import customize_plugin

        configString = configWidget.controller.getConfigString()
        customize_plugin(self, configString)
        
class Config():
    config = None
    currentProfile = None
    
    def __init__(self, config):
        self.config = config
        
        for name in self.config.keys():
            self.normalize_config_profile(name, self.config[name])
    
    def normalize_config_profile(self, profileName, profile):
        keys = profile.keys()
        
        profile[CONFIG_NAME_KEY] = profileName
        
        if CONFIG_BOOKS_KEY not in keys:
            profile[CONFIG_BOOKS_KEY] = DEFAULT_CONFIG_BOOKS
        
        if CONFIG_STRATEGY_KEY not in keys:
            profile[CONFIG_STRATEGY_KEY] = [DEFAULT_CONFIG_STRATEGY]
        
        if CONFIG_STRATEGY_VALUE_NEWEST not in profile[CONFIG_STRATEGY_KEY] and \
           CONFIG_STRATEGY_VALUE_OLDEST not in profile[CONFIG_STRATEGY_KEY] and \
           CONFIG_STRATEGY_VALUE_RANDOM not in profile[CONFIG_STRATEGY_KEY]:
            profile[CONFIG_STRATEGY_KEY].append(DEFAULT_CONFIG_STRATEGY)
        
        if CONFIG_ORDER_KEY not in keys:
            profile[CONFIG_ORDER_KEY] = DEFAULT_CONFIG_ORDER
    
    def setProfile(self, profileName):
        self.currentProfile = self.config[profileName]
    
    def getProfile(self, profileName):
        return self.config[profileName]
    
    def getNameValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_NAME_KEY]
    
    def getBooksValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_BOOKS_KEY]
    
    def getStrategyValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_STRATEGY_KEY]
    
    def getOrderValue(self, profile=None):
        if profile is None:
            useProfile = self.currentProfile
        else:
            useProfile = profile
        return useProfile[CONFIG_ORDER_KEY]
            
class SonyX50Reader():
    gui = None
    
    def __init__(self, gui):
        self.gui = gui
        self.init_state()
    
    def init_state(self):
        self.isConnected = True
        self.rootPath = [None, None, None] # [main, card A, card B]
        self.vendorId = None
        self.productId = None
    
    def detect(self):
        from calibre.gui2 import error_dialog
        
        self.init_state()
        
        # Confirm Sony is connected
        if self.gui.device_manager.connected_device is not None:
            self.rootPath[MAIN_INDEX] = self.gui.device_manager.connected_device._main_prefix
            self.rootPath[CARD_A_INDEX] = self.gui.device_manager.connected_device._card_a_prefix
            self.rootPath[CARD_B_INDEX] = self.gui.device_manager.connected_device._card_b_prefix
            self.vendorId = self.gui.device_manager.connected_device.VENDOR_ID
            self.productId = self.gui.device_manager.connected_device.PRODUCT_ID
            
        else:
            error_dialog(self.gui, PLUGIN_NAME, _('Sony reader not detected.'), show=True)
            self.isConnected = False
            
            # DEBUG - comment out lines above and uncomment those below to access test data
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.empty/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.2/'
            #self.rootPath[MAIN_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.main/'
            #self.rootPath[CARD_A_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.a/'
            #self.rootPath[CARD_B_INDEX] = 'C:/Users/Karl/workspace-calibre/Sony Reader Test Data/Sony.b/'
            #self.vendorId = SONY_VENDOR_ID
            #self.productId = SONY_X50_PRODUCT_ID
            
        if self.isConnected and ((self.vendorId != SONY_VENDOR_ID) or (self.productId != SONY_X50_PRODUCT_ID)):
            error_dialog(self.gui, PLUGIN_NAME, _('Reader does not appear to be a Sony x50 model.'), show=True)
            self.isConnected = False
        
class SonyX50ReaderCache():
    gui = None
    devInfo = None
    
    def __init__(self, gui, devInfo):
        self.gui = gui
        self.devInfo = devInfo
        self.init_state()
    
    def init_state(self):
        self.cacheDOM = [None, None, None] # [main, card a, card b]
    
    def load(self):
        from calibre.gui2 import error_dialog
        
        self.init_state()
        
        if self.devInfo.rootPath[MAIN_INDEX] is not None:
            self.cacheDOM[MAIN_INDEX] = self.load_cache(MAIN_INDEX)
            
        if self.devInfo.rootPath[CARD_A_INDEX] is not None:
            self.cacheDOM[CARD_A_INDEX] = self.load_cache(CARD_A_INDEX)
            
        if self.devInfo.rootPath[CARD_B_INDEX] is not None:
            self.cacheDOM[CARD_B_INDEX] = self.load_cache(CARD_B_INDEX)
        
        if (self.cacheDOM[MAIN_INDEX] is None) and (self.cacheDOM[CARD_A_INDEX] is None) and (self.cacheDOM[CARD_B_INDEX] is None):
            error_dialog(self.gui, PLUGIN_NAME, _('The SONY reader does not have any books loaded.'), show=True)
            self.cacheDOM = None
    
    def update(self, bookList):
        import os, time, copy
        
        updatedDOMs = [None, None, None] # [main, card a, card b]
        
        books = copy.copy(bookList)
        books.reverse()
        baseTime = round(time.time())
        adjustment = self.getTimeAdjustment()
        
        for index, book in enumerate(books):
            path = book['path']
            modTime = baseTime + (index * 2) # With one second intervals, I sometimes see duplicate times
            os.utime(path, (modTime, modTime))
            
            book['date'] = time.strftime('%a, %d %b %Y %H:%M:%S GMT', time.gmtime(os.path.getmtime(path)))
            book['tz'] = str(int(adjustment / 60))
            
            deviceIndex = DEVICE_KEY_INDICES[book['id'].split(':')[0]]
            cacheDOM = self.cacheDOM[deviceIndex]
            updatedDOMs[deviceIndex] = cacheDOM
            
            self.update_cache(book)
        
        for index, updatedDOM in enumerate(updatedDOMs):
            if updatedDOM is not None:
                self.write_cache(index, updatedDOM)
        
    def getTimeAdjustment(self):
        import time
        
        return time.timezone
    
    def load_cache(self, deviceIndex):
        import os
        from lxml import etree
        from calibre.constants import DEBUG
        from calibre.ebooks.chardet import xml_to_unicode
        from calibre.gui2 import error_dialog
        
        cache = {}
        path = os.path.join(self.devInfo.rootPath[deviceIndex], SONY_CACHE_FILE[deviceIndex])
        
        if not os.path.exists(path):
            message = _('The SONY XML cache ({0}) is missing. Any books on that drive will not be detected.')
            error_dialog(self.gui, PLUGIN_NAME, message.format(READER_DRIVE_LABEL[deviceIndex]), show=True)
            cache = None

        if cache is not None:
            raw = None
            with open(path, 'rb') as f:
                raw = f.read()
            
            parser = etree.XMLParser(recover=True)
            roots = etree.fromstring(xml_to_unicode(raw, strip_encoding_pats=True, 
                                         assume_utf8=True,
                                         verbose=DEBUG)[0],
                                     parser=parser)
                                 
            if roots is None:
                cache = None
            else:
                cache['roots'] = roots
        
        if cache is not None:
            cache['books'] = roots.xpath('//*[local-name()="text"]')
            
            if not cache['books']:
                cache = None
                
        if cache is not None:
            cache['collections'] = roots.xpath('//*[local-name()="playlist"]')
                  
        return cache
    
    def update_cache(self, book):
        DOMBook = book['DOMRecord']
        DOMBook.set('date', book['date'])
        DOMBook.set('tz', book['tz'])
    
    def write_cache(self, deviceIndex, cacheDOM):
        import os
        from lxml import etree
        
        path = os.path.join(self.devInfo.rootPath[deviceIndex], SONY_CACHE_FILE[deviceIndex])
        raw = etree.tostring(cacheDOM['roots'], encoding='UTF-8', xml_declaration=True)
        raw = raw.replace("<?xml version='1.0' encoding='UTF-8'?>", '<?xml version="1.0" encoding="UTF-8"?>')
        
        with open(path, 'wb') as f:
            f.write(raw)

class BookList():
    config = None
    devInfo = None
    
    def __init__(self, config, devInfo):
        self.config = config
        self.devInfo = devInfo
        self.init_state()
    
    def init_state(self):
        self.books = {}
        self.collections = {}
        self.pathIdMap = {}
        
    def build(self, cacheDOM):
        self.init_state()
        
        if cacheDOM[MAIN_INDEX] is not None:
            self.build_bookList(cacheDOM[MAIN_INDEX], MAIN_INDEX)
            
        if cacheDOM[CARD_A_INDEX] is not None:
            self.build_bookList(cacheDOM[CARD_A_INDEX], CARD_A_INDEX)
            
        if cacheDOM[CARD_B_INDEX] is not None:
            self.build_bookList(cacheDOM[CARD_B_INDEX], CARD_B_INDEX)
    
    def build_bookList(self, cacheDOM, deviceIndex):
        import os
        
        for book in cacheDOM['books']:
            id = DEVICE_KEY_PREFIXES[deviceIndex] + str(book.get('id'))
            self.books[id] = {}
            self.books[id]['DOMRecord'] = book
            self.books[id]['id'] = id
            self.books[id]['author'] = book.get('author')
            self.books[id]['date'] = book.get('date')
            self.books[id]['tz'] = book.get('tz')
            self.books[id]['title'] = book.get('title')
            self.books[id]['titleSorter'] = book.get('titleSorter')
            
            path = os.path.normpath(os.path.join(self.devInfo.rootPath[deviceIndex], book.get('path')))
            self.books[id]['path'] = path
            self.pathIdMap[path] = id
        
        names = self.collections.keys()
        for collection in cacheDOM['collections']:
            title = collection.get('title')
            
            if title not in names:
                self.collections[title] = []
            
            for item in collection.xpath('*[local-name()="item"]'):
               self.collections[title].append(DEVICE_KEY_PREFIXES[deviceIndex] + str(item.get('id')))

class BookSelection():
    gui = None
    
    def __init__(self, gui):
        self.gui = gui
        self.init_state()
    
    def init_state(self):
        self.books = []
    
    def get_selection(self, pathIdMap):
        import os
        from calibre.gui2 import warning_dialog
        
        self.init_state()
        
        if self.gui.current_view() is self.gui.library_view:
            warning_dialog(self.gui, PLUGIN_NAME, _('You must select books from the device book lists.'), show=True)
            self.books = None
            
        if self.books is not None:
            rows = self.gui.current_view().selectionModel().selectedRows();
        
            if len(rows) is 0:
                warning_dialog(self.gui, PLUGIN_NAME, _('No books are selected.'), show=True)
                self.books = None
           
        if self.books is not None:
            paths = self.gui.current_view().model().paths(rows)
        
            for path in paths:
               self.books.append(pathIdMap[os.path.normpath(path)])

class ShortBookList():
    config = None
    bookList = None
    bookSelection = None
    
    def __init__(self, config, bookList, bookSelection):
        self.config = config
        self.bookList = bookList
        self.bookSelection = bookSelection
        self.init_state()
    
    def init_state(self):
        self.books = []
    
    def extract(self):
        self.init_state()
        
        collectionNames = self.bookList.collections.keys()
        idSets = []
        if len(self.bookSelection.books) is not 0:
            idSets.append(self.bookSelection.books)
        
        else:
            for index, collectionName in enumerate(self.config.getBooksValue()):
                if collectionName in collectionNames:
                    idSets.append(self.bookList.collections[collectionName])
                
        bookSets = []
        for index, idSet in enumerate(idSets):
            bookSet = []
            for index2, bookId in enumerate(idSet):
                bookSet.append(self.bookList.books[bookId])
            
            if len(bookSet) is not 0:
                bookSets.append(bookSet)
        
        if len(bookSets) is not 0:
            self.handle_bookSets(bookSets)
    
    def handle_bookSets(self, bookSets):
        maxRequired = MAX_BOOK_LIST_LENGTH
            
        for index, bookSet in enumerate(bookSets):
            bookSet.sort(key=lambda book: book['date'])
            self.books.extend(self.choose_books(bookSet, maxRequired))
            maxRequired = MAX_BOOK_LIST_LENGTH - len(self.books)
            
            if len(self.books) is MAX_BOOK_LIST_LENGTH:
                break
        
        self.sort_books()
    
    def choose_books(self, bookSet, maxRequired):
        import random
        
        strategy = self.config.getStrategyValue()
        
        if CONFIG_STRATEGY_VALUE_ONE in strategy:
            maxLen = 1
        else:
            maxLen = maxRequired
        
        books = []
            
        # Select newest first
        if CONFIG_STRATEGY_VALUE_NEWEST in strategy:
            for index, book in enumerate(bookSet):
                book = bookSet[len(bookSet) - index - 1]
                
                if (book not in self.books):
                    books.append(book)
                    
                    if len(books) is maxLen:
                        break
        
        # Select oldest first
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategy:
            for index, book in enumerate(bookSet):
                book = bookSet[index]
                
                if book not in self.books:
                    books.append(book)
                    
                    if len(books) is maxLen:
                        break
        
        # Select randomly
        else:
            while len(books) is not maxLen:
                book = bookSet[random.randint(0, len(bookSet) - 1)]
                
                if (book in books) or (book in self.books):
                    bookSet.remove(book)
                else:
                    books.append(book)
                
                if len(bookSet) is 0:
                    break
        
        return books
    
    def sort_books(self):
        order = self.config.getOrderValue()
        
        if order == CONFIG_ORDER_VALUE_DATE:
            self.books.sort(key=lambda book: book['date'], reverse=True)
        elif order == CONFIG_ORDER_VALUE_AUTHOR:
            self.books.sort(key=lambda book: book['author'], reverse=False)
        else:
            self.books.sort(key=lambda book: book['titleSorter'], reverse=False)

#==========================
# I should really break out the UI portion of this plugin into a separate file, but I need to
# figure out how Python packaging works first.  It's on my "To Do" list...
#==========================

CONFIG_LABEL_SELECT_MAIN_TITLE = _('Set Book List')
CONFIG_LABEL_GENERAL = _('The profile configuration determines how books for the book list are chosen and how thay will appear on your reader.')
CONFIG_LABEL_CHOOSE_PROFILE = _('Choose Profile:')
CONFIG_LABEL_PROFILE = _('Profile:')
CONFIG_LABEL_PROFILES = _('Profiles:')
CONFIG_LABEL_DETAILS = _('Details:')
CONFIG_LABEL_SELECTION = _('Choose books from Device/Card A/Card B book list selections.')
CONFIG_LABEL_COLLECTIONS = _('Choose books from collections: ')
CONFIG_LABEL_CHOOSE_ONE = _('Choose one book from each source.')
CONFIG_LABEL_CHOOSE_NEWEST = _('Choose books by newest first.')
CONFIG_LABEL_CHOOSE_OLDEST = _('Choose books by oldest first.')
CONFIG_LABEL_CHOOSE_RANDOM = _('Choose books randomly.')
CONFIG_LABEL_ORDER_DATE = _('Order book list by date.')
CONFIG_LABEL_ORDER_AUTHOR = _('Order book list by author.')
CONFIG_LABEL_ORDER_TITLE = _('Order book list by title.')
CONFIG_LABEL_CHOOSE_GROUP = _('Choose Books From:')
CONFIG_LABEL_SELECTION_OPTION = _('Device/Card A/Card B book list selections')
CONFIG_LABEL_COLLECTIONS_OPTION = _('Collections:')
CONFIG_LABEL_STRATEGY_GROUP = _('Choose Books By:')
CONFIG_LABEL_ONE_OPTION = _('One book from each source')
CONFIG_LABEL_NEWEST_OPTION = _('Newest first')
CONFIG_LABEL_OLDEST_OPTION = _('Oldest first')
CONFIG_LABEL_RANDOM_OPTION = _('Randomly')
CONFIG_LABEL_ORDER_GROUP = _('Order Book List By:')
CONFIG_LABEL_DATE_OPTION = _('Date')
CONFIG_LABEL_AUTHOR_OPTION = _('Author')
CONFIG_LABEL_TITLE_OPTION = _('Title')
CONFIG_LABEL_ADD = _('Add')
CONFIG_LABEL_DELETE = _('Delete')
CONFIG_LABEL_UP = _('Up')
CONFIG_LABEL_DOWN = _('Down')
DEFAULT_PROFILE_NAME = _('New Profile')
DEFAULT_COLLECTION_NAME = _('New Collection')

from PyQt4 import Qt, QtCore
from PyQt4.Qt import QDialog, QGridLayout, QLabel, QComboBox, QDialogButtonBox, QToolButton, \
    QListWidget, QGroupBox, QVBoxLayout, QRadioButton, QCheckBox, QPushButton, QListWidgetItem, \
    QString

class ManageBookListView(QDialog):
    def __init__(self, parent):
        QDialog.__init__(self, parent)
        
        self.setWindowTitle(CONFIG_LABEL_SELECT_MAIN_TITLE)
        
        self._layout = QGridLayout(self)
        self.setLayout(self._layout)
        
        self.setBookListText = QLabel(CONFIG_LABEL_GENERAL)
        self.setBookListText.setWordWrap(True)
        self._layout.addWidget(self.setBookListText, 0, 0, 1, 2)

        self.profileLabel = QLabel()
        self._layout.addWidget(self.profileLabel, 1, 0, 1, 1, QtCore.Qt.AlignLeft)
                
        self.infoLabel = QLabel('<b>' + CONFIG_LABEL_DETAILS + '</b>')
        self._layout.addWidget(self.infoLabel, 2, 0, 1, 2, QtCore.Qt.AlignLeft | QtCore.Qt.AlignTop)

        self.profileDetails = QLabel()
        self.profileDetails.setWordWrap(True)
        self._layout.addWidget(self.profileDetails, 3, 0, 1, 2)
        
        buttonBox = QDialogButtonBox(QDialogButtonBox.Ok | QDialogButtonBox.Cancel)
        buttonBox.accepted.connect(self.accept)
        buttonBox.rejected.connect(self.reject)
        self._layout.addWidget(buttonBox, 4, 0, 1, 2)
        
        self._layout.setColumnStretch(1, 1)
        self._layout.setColumnMinimumWidth(1, 200)
    
    def getProfileName(self):
        if isinstance(self.profileValue, QComboBox):
            profileName = str(self.profileValue.itemText(self.profileValue.currentIndex()))
        else:
            profileName = str(self.profileValue.text())
            
        return profileName
    
class ManageBookListController():
    model = None
    view = None
    
    def __init__(self, model, view):
        self.model = model
        self.view = view
        
        profileNames = self.model.config.keys()
        
        if len(profileNames) > 1:
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_CHOOSE_PROFILE + '</b>')
            self.view.profileValue = QComboBox()
            
            for index, profileName in enumerate(profileNames):
                self.view.profileValue.addItem(profileName)
        else:
            self.view.profileLabel.setText('<b>' + CONFIG_LABEL_PROFILE + '</b>')
            self.view.profileValue = QLabel(profileNames[0])
            
        self.view._layout.addWidget(self.view.profileValue, 1, 1, 1, 1, QtCore.Qt.AlignLeft)
            
        self.view.connect(self.view.profileValue, QtCore.SIGNAL('activated(QString)'), self.onProfileActivated)
        self.setProfile(profileNames[0])
    
    def onProfileActivated(self, text):
        self.setProfile(str(text))
    
    def setProfile(self, profileName):
        profile = self.model.getProfile(profileName)
        booksValue = self.model.getBooksValue(profile)
        strategyValue = self.model.getStrategyValue(profile)
        orderValue = self.model.getOrderValue(profile)
        
        details = '<ul>'
        if CONFIG_BOOKS_VALUE_SELECTION in booksValue:
            details = details + '<li>' + CONFIG_LABEL_SELECTION + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_COLLECTIONS
            for index, value in enumerate(booksValue):
                if index is not 0:
                    details = details + ', '
                details = details + '"' + value + '"'
            details = details + '</li>'
            
        if CONFIG_STRATEGY_VALUE_ONE in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_ONE + '</li>'
            
        if CONFIG_STRATEGY_VALUE_NEWEST in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_NEWEST + '</li>'
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategyValue:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_OLDEST + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_CHOOSE_RANDOM + '</li>'
            
        if CONFIG_ORDER_VALUE_DATE in orderValue:
            details = details + '<li>' + CONFIG_LABEL_ORDER_DATE + '</li>'
        elif CONFIG_ORDER_VALUE_AUTHOR in orderValue:
            details = details + '<li>' + CONFIG_LABEL_ORDER_AUTHOR + '</li>'
        else:
            details = details + '<li>' + CONFIG_LABEL_ORDER_TITLE + '</li>'
            
        details = details + '</ul>'
        
        self.view.profileDetails.setText(details)
        self.view.resize(self.view.sizeHint())
 
class ManageBookListConfigView(QDialog):
    controller = None
    
    def __init__(self):
        QDialog.__init__(self)
        
        self._layout = QGridLayout(self)
        self.setLayout(self._layout)
        
        self.setBookListText = QLabel(CONFIG_LABEL_GENERAL)
        self.setBookListText.setWordWrap(True)
        self._layout.addWidget(self.setBookListText, 0, 0, 1, 4)

        self.profileLabel = QLabel('<b>' + CONFIG_LABEL_PROFILES + '</b>')
        self._layout.addWidget(self.profileLabel, 1, 0, 1, 1, QtCore.Qt.AlignLeft)
        
        self.profileList = QListWidget()
        self._layout.addWidget(self.profileList, 2, 0, 2, 2)
        
        self.profileAddButton = QPushButton(CONFIG_LABEL_ADD)
        self._layout.addWidget(self.profileAddButton, 4, 0, 1, 1)
        
        self.profileDeleteButton = QPushButton(CONFIG_LABEL_DELETE)
        self._layout.addWidget(self.profileDeleteButton, 4, 1, 1, 1)
        
        self.chooseGroup = QGroupBox(CONFIG_LABEL_CHOOSE_GROUP)
        self.selectionOption = QRadioButton(CONFIG_LABEL_SELECTION_OPTION)
        self.collectionsOption = QRadioButton(CONFIG_LABEL_COLLECTIONS_OPTION)
        self.collectionsList = QListWidget()
        self.collectionAddButton = QPushButton(CONFIG_LABEL_ADD)
        self.collectionDeleteButton = QPushButton(CONFIG_LABEL_DELETE)
        self.collectionUpButton = QPushButton(CONFIG_LABEL_UP)
        self.collectionDownButton = QPushButton(CONFIG_LABEL_DOWN)
        self.chooseLayout = QGridLayout(self.chooseGroup)
        self.chooseLayout.addWidget(self.selectionOption, 0, 0, 1, 2)
        self.chooseLayout.addWidget(self.collectionsOption, 1, 0, 1, 2)
        self.chooseLayout.addWidget(self.collectionsList, 2, 0, 2, 2)
        self.chooseLayout.addWidget(self.collectionUpButton, 2, 2, 1, 1, QtCore.Qt.AlignBottom)
        self.chooseLayout.addWidget(self.collectionDownButton, 3, 2, 1, 1, QtCore.Qt.AlignTop)
        self.chooseLayout.addWidget(self.collectionAddButton, 4, 0, 1, 1)
        self.chooseLayout.addWidget(self.collectionDeleteButton, 4, 1, 1, 1)
        self.chooseGroup.setLayout(self.chooseLayout)
        self._layout.addWidget(self.chooseGroup, 2, 2, 1, 2)
        
        self.strategyGroup = QGroupBox(CONFIG_LABEL_STRATEGY_GROUP)
        self.oneOption = QCheckBox(CONFIG_LABEL_ONE_OPTION)
        self.newestOption = QRadioButton(CONFIG_LABEL_NEWEST_OPTION)
        self.oldestOption = QRadioButton(CONFIG_LABEL_OLDEST_OPTION)
        self.randomOption = QRadioButton(CONFIG_LABEL_RANDOM_OPTION)
        self.strategyLayout = QVBoxLayout(self.strategyGroup)
        self.strategyLayout.addWidget(self.oneOption)
        self.strategyLayout.addWidget(self.newestOption)
        self.strategyLayout.addWidget(self.oldestOption)
        self.strategyLayout.addWidget(self.randomOption)
        self.strategyGroup.setLayout(self.strategyLayout)
        self._layout.addWidget(self.strategyGroup, 3, 2, 2, 1)
        
        self.orderGroup = QGroupBox(CONFIG_LABEL_ORDER_GROUP)
        self.dateOption = QRadioButton(CONFIG_LABEL_DATE_OPTION)
        self.authorOption = QRadioButton(CONFIG_LABEL_AUTHOR_OPTION)
        self.titleOption = QRadioButton(CONFIG_LABEL_TITLE_OPTION)
        self.orderLayout = QVBoxLayout(self.orderGroup)
        self.orderLayout.addWidget(self.dateOption)
        self.orderLayout.addWidget(self.authorOption)
        self.orderLayout.addWidget(self.titleOption)
        self.orderGroup.setLayout(self.orderLayout)
        self._layout.addWidget(self.orderGroup, 3, 3, 2, 1)
        
        self._layout.setColumnStretch(0, 1)
        self._layout.setColumnStretch(1, 1)
        
    def setController(self, controller):
        self.controller = controller
             
class ManageBookListConfigController():
    model = None
    view = None
    
    def __init__(self, model, view):
        self.model = model
        self.view = view
        
        profileNames = self.model.config.keys()
        
        for index, profileName in enumerate(profileNames):
            self.view.profileList.addItem(profileName)
            item = self.view.profileList.item(self.view.profileList.count() - 1)
            item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        
        self.view.connect(self.view.profileList, QtCore.SIGNAL('currentItemChanged(QListWidgetItem *, QListWidgetItem *)'), self.onProfileCurrentItemChanged)
        self.view.connect(self.view.profileList, QtCore.SIGNAL('itemDoubleClicked(QListWidgetItem *)'), self.onProfileItemDoubleClicked)
        self.view.connect(self.view.profileList, QtCore.SIGNAL('itemChanged(QListWidgetItem *)'), self.onProfileItemChanged)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('currentItemChanged(QListWidgetItem *, QListWidgetItem *)'), self.onCollectionsCurrentItemChanged)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('itemDoubleClicked(QListWidgetItem *)'), self.onCollectionsItemDoubleClicked)
        self.view.connect(self.view.collectionsList, QtCore.SIGNAL('itemChanged(QListWidgetItem *)'), self.onCollectionsItemChanged)
        self.view.connect(self.view.selectionOption, QtCore.SIGNAL('toggled(bool)'), self.onSelectionButtonToggled)
        self.view.connect(self.view.collectionsOption, QtCore.SIGNAL('toggled(bool)'), self.onCollectionsButtonToggled)
        self.view.connect(self.view.oneOption, QtCore.SIGNAL('toggled(bool)'), self.onOneButtonToggled)
        self.view.connect(self.view.newestOption, QtCore.SIGNAL('toggled(bool)'), self.onNewestButtonToggled)
        self.view.connect(self.view.oldestOption, QtCore.SIGNAL('toggled(bool)'), self.onOldestButtonToggled)
        self.view.connect(self.view.randomOption, QtCore.SIGNAL('toggled(bool)'), self.onRandomButtonToggled)
        self.view.connect(self.view.dateOption, QtCore.SIGNAL('toggled(bool)'), self.onDateButtonToggled)
        self.view.connect(self.view.authorOption, QtCore.SIGNAL('toggled(bool)'), self.onAuthorButtonToggled)
        self.view.connect(self.view.titleOption, QtCore.SIGNAL('toggled(bool)'), self.onTitleButtonToggled)
        self.view.connect(self.view.collectionDeleteButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionDeleteButtonClicked)
        self.view.connect(self.view.collectionAddButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionAddButtonClicked)
        self.view.connect(self.view.profileDeleteButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileDeleteButtonClicked)
        self.view.connect(self.view.profileAddButton, QtCore.SIGNAL('clicked(bool)'), self.onProfileAddButtonClicked)
        self.view.connect(self.view.collectionUpButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionUpButtonClicked)
        self.view.connect(self.view.collectionDownButton, QtCore.SIGNAL('clicked(bool)'), self.onCollectionDownButtonClicked)
        
        self.view.profileList.setCurrentRow(0)
            
    def getConfigString(self):
        import json
        
        return json.dumps(self.model.config)
    
    def getCurrentProfile(self):
        profileName = str(self.view.profileList.item(self.view.profileList.currentRow()).text())
        
        return self.model.getProfile(profileName)
    
    def getCurrentCollection(self):
        return str(self.view.collectionsList.item(self.view.collectionsList.currentRow()).text())
    
    def setCurrentProfile(self, profileName):
        profile = self.model.getProfile(profileName)
        booksValue = self.model.getBooksValue(profile)
        strategyValue = self.model.getStrategyValue(profile)
        orderValue = self.model.getOrderValue(profile)

        if CONFIG_BOOKS_VALUE_SELECTION in booksValue:
            self.view.selectionOption.setChecked(True)
        else:
            if self.view.collectionsOption.isChecked():
                self.onCollectionsButtonToggled(True)
            else:
                self.view.collectionsOption.setChecked(True)
        
        self.view.oneOption.setChecked(CONFIG_STRATEGY_VALUE_ONE in strategyValue)
        
        if CONFIG_STRATEGY_VALUE_NEWEST in strategyValue:
            self.view.newestOption.setChecked(True)
        elif CONFIG_STRATEGY_VALUE_OLDEST in strategyValue:
            self.view.oldestOption.setChecked(True)
        else:
            self.view.randomOption.setChecked(True)
        
        if CONFIG_ORDER_VALUE_DATE in orderValue:
            self.view.dateOption.setChecked(True)
        elif CONFIG_ORDER_VALUE_AUTHOR in orderValue:
            self.view.authorOption.setChecked(True)
        else:
            self.view.titleOption.setChecked(True)
    
    def onProfileCurrentItemChanged(self, newItem, previousItem):
        if newItem is not None:
            self.model.setProfile(str(newItem.text()))
            self.setCurrentProfile(str(newItem.text()))
            self.view.profileDeleteButton.setEnabled(self.view.profileList.count() > 1)
    
    def onProfileItemDoubleClicked(self, item):
        self.view.profileList.editItem(item)
    
    def onCollectionsCurrentItemChanged(self, newItem, previousItem):
        if newItem is not None:
            self.view.collectionDeleteButton.setEnabled(self.view.collectionsList.count() > 1)
            self.view.collectionUpButton.setEnabled(self.view.collectionsList.count() > 1 and self.view.collectionsList.currentRow() > 0)
            self.view.collectionDownButton.setEnabled(self.view.collectionsList.count() > 1 and self.view.collectionsList.currentRow() < self.view.collectionsList.count() - 1)
    
    def onCollectionsItemDoubleClicked(self, item):
        self.view.collectionsList.editItem(item)
    
    def onSelectionButtonToggled(self, on):
        if on:
            profile = self.getCurrentProfile()
            profile[CONFIG_BOOKS_KEY] = CONFIG_BOOKS_VALUE_SELECTION
            
            self.view.collectionAddButton.setDisabled(True)
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
    
    def onCollectionsButtonToggled(self, on):
        if on:
            autoEdit = False
            profile = self.getCurrentProfile()
            booksValue = self.model.getBooksValue(profile)
            
            if (CONFIG_BOOKS_VALUE_SELECTION in booksValue) or (len(booksValue) is 0):
                profile[CONFIG_BOOKS_KEY] = [DEFAULT_COLLECTION_NAME]
                booksValue = self.model.getBooksValue(profile)
                autoEdit = True
            
            self.view.collectionDeleteButton.setDisabled(True)
            self.view.collectionUpButton.setDisabled(True)
            self.view.collectionDownButton.setDisabled(True)
            self.view.collectionsList.clear()
            
            for index, collection in enumerate(booksValue):
                self.view.collectionsList.addItem(collection)
                item = self.view.collectionsList.item(self.view.collectionsList.count() - 1)
                item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
                
            if autoEdit:
                self.view.collectionsList.editItem(self.view.collectionsList.item(0))
                
            self.view.collectionAddButton.setEnabled(True)
    
    def onOneButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE not in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY].insert(0, CONFIG_STRATEGY_VALUE_ONE)
        else:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY].remove(CONFIG_STRATEGY_VALUE_ONE)
    
    def onNewestButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_ONE, CONFIG_STRATEGY_VALUE_NEWEST]
            else:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_NEWEST]
    
    def onOldestButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_ONE, CONFIG_STRATEGY_VALUE_OLDEST]
            else:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_OLDEST]
    
    def onRandomButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            if CONFIG_STRATEGY_VALUE_ONE in profile[CONFIG_STRATEGY_KEY]:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_ONE, CONFIG_STRATEGY_VALUE_RANDOM]
            else:
                profile[CONFIG_STRATEGY_KEY] = [CONFIG_STRATEGY_VALUE_RANDOM]
    
    def onDateButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[CONFIG_ORDER_KEY] = CONFIG_ORDER_VALUE_DATE
    
    def onAuthorButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[CONFIG_ORDER_KEY] = CONFIG_ORDER_VALUE_AUTHOR
    
    def onTitleButtonToggled(self, on):
        profile = self.getCurrentProfile()
            
        if on:
            profile[CONFIG_ORDER_KEY] = CONFIG_ORDER_VALUE_TITLE
    
    def onCollectionDeleteButtonClicked(self):
        profile = self.getCurrentProfile()
        currentRow = self.view.collectionsList.currentRow()
        previousCollection = self.view.collectionsList.takeItem(currentRow)
        newCollection = self.view.collectionsList.item(self.view.collectionsList.currentRow())
        self.onCollectionsCurrentItemChanged(newCollection, previousCollection)
        
        booksValue = self.model.getBooksValue(profile)
        booksValue.remove(str(previousCollection.text()))
        
        del previousCollection
    
    def onProfileDeleteButtonClicked(self):
        profile = self.getCurrentProfile()
        currentRow = self.view.profileList.currentRow()
        previousProfile = self.view.profileList.takeItem(currentRow)
        newProfile = self.view.profileList.item(self.view.profileList.currentRow())
        self.onProfileCurrentItemChanged(newProfile, previousProfile)
        
        del self.model.config[str(previousProfile.text())]
        
        del previousProfile
    
    def onCollectionUpButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        collectionItem = self.view.collectionsList.takeItem(currentRow)
        self.view.collectionsList.insertItem(currentRow - 1, collectionItem)
        self.view.collectionsList.setCurrentRow(currentRow - 1)
        
        collection = self.getCurrentCollection()
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        index = booksValue.index(collection)
        booksValue.pop(index)
        booksValue.insert(index - 1, collection)
    
    def onCollectionDownButtonClicked(self):
        currentRow = self.view.collectionsList.currentRow()
        collectionItem = self.view.collectionsList.takeItem(currentRow)
        self.view.collectionsList.insertItem(currentRow + 1, collectionItem)
        self.view.collectionsList.setCurrentRow(currentRow + 1)
        
        collection = self.getCurrentCollection()
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        index = booksValue.index(collection)
        booksValue.pop(index)
        booksValue.insert(index + 1, collection)
    
    def onCollectionAddButtonClicked(self):
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        name = self.getUniqueName(DEFAULT_COLLECTION_NAME, booksValue)
        booksValue.append(name)
        
        self.view.collectionsList.addItem(name)
        lastRow = self.view.collectionsList.count() - 1
        self.view.collectionsList.setCurrentRow(lastRow)
        item = self.view.collectionsList.item(lastRow)
        item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        self.view.collectionsList.editItem(item)
    
    def onProfileAddButtonClicked(self):
        name = self.getUniqueName(DEFAULT_PROFILE_NAME, self.model.config.keys())
        profile = {}
        self.model.normalize_config_profile(name, profile)
        self.model.config[name] = profile
        
        self.view.profileList.addItem(name)
        lastRow = self.view.profileList.count() - 1
        self.view.profileList.setCurrentRow(lastRow)
        item = self.view.profileList.item(lastRow)
        item.setFlags(item.flags() | QtCore.Qt.ItemIsEditable)
        self.view.profileList.editItem(item)
    
    def onCollectionsItemChanged(self, item):
        profile = self.getCurrentProfile()
        booksValue = self.model.getBooksValue(profile)
        row = self.view.collectionsList.row(item)
        oldName = booksValue[row]
        newName = str(item.text()).strip()
        
        if newName != str(item.text()): # Ignore added leading and trailing spaces
            item.setText(QString(newName))
        
        if newName is None or len(newName) is 0:
            newName = oldName
            item.setText(QString(oldName))
        
        if newName != oldName:
            uniqueName = self.getUniqueName(newName, booksValue)
            booksValue[row] = uniqueName
            item.setText(QString(uniqueName))
    
    def onProfileItemChanged(self, item):
        import copy
        
        profile = self.model.currentProfile
        oldName = self.model.getNameValue(profile)
        newName = str(item.text()).strip()
        
        if newName != str(item.text()): # Ignore added leading and trailing spaces
            item.setText(QString(newName))
        
        if newName is None or len(newName) is 0:
            newName = oldName
            item.setText(QString(oldName))
        
        if newName != oldName:
            uniqueName = self.getUniqueName(newName, self.model.config.keys())
            newProfile = copy.copy(profile)
            self.model.normalize_config_profile(uniqueName, newProfile)
            self.model.config[uniqueName] = newProfile
            self.model.setProfile(uniqueName)
            del self.model.config[oldName]
            item.setText(QString(uniqueName))
    
    def getUniqueName(self, name, existingNames):
        namePrefix = name
        tryName = name
        index = 1
        
        while tryName in existingNames:
            tryName = namePrefix + ':' + str(index)
            index = index + 1
        
        return tryName
