/*
* Copyright (C) 2010 Andy M. aka h1uke	h1ukeguy @ gmail.com
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <sys/stat.h>
#include <sys/time.h>
#include <signal.h>

#include <linux/input.h>
#include <libgen.h>

#include <sys/file.h>
#include <poll.h>

#include "version.h"

#include "launchpad.h"
#include "inifile.h"

#include "keydefs.h"
#include "asciitab.h"

#include "screen.h"
#include "fntdefs.h"

#include "rce.h"

static char proc_dir[MAXPATH] ;

static int introducer ;			/* hotkey sequence introducer code */
static int trailer ;			/* hotkey sequence trailer code */
static int hot_interval ;		/* msecs to stay hot after introducer code has recognized */

static int introducer_just_pressed = 0 ;
static int hotkey_mode = 0 ;
static int npressed = 0 ;		/* number of keys currently pressed */

static short hot_sequence[MAXSEQ] ;	/* input codes collected while in hot mode */
static int hot_sequence_length = 0 ;

/*
 * keyboard and fiveway input event file name pointers, if given in a command line
 */
static char *pkb_file = NULL ;
static char *pfw_file = NULL ;
//static char *pk3vol_file = NULL ;

static int fd_kbd = -1 ;
static int fd_fw = -1 ;
static int fd_k3_vol = -1 ;		/* dedicated event file is used for volume buttons on Kindle3 .. */

static int fd_prockeypad = -1 ;
static int fd_procfiveway = -1 ;

static int do_daemonize = 1 ;
static int running_as_daemon = 0 ;

static FILE * diag_fp = NULL ;

#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1
#define LOCKFILE_NAME  "/var/run/launchpad.pid"

static void daemonize( const char *lockfile ) ;
void send_key(short code, int mode) ;
void close_diag_fp() ;

static volatile int keepgoing = 1 ;
static volatile int reinit = 0 ;

static int input_captured = 0 ;

/* number of next key input events to be skipped.
 * These events are generated by Amazon Framework when
 * simulating FW input by writing to /proc/fiveway.
 * Besides the number of weird keystrokes to skip is calculated
 * correctly, we do not use it anymore -- found a better way.
 */
static volatile int skip_next_events ;

static void open_event_files(void)
{
	if (pkb_file != NULL)
	{
		if (*pkb_file != '\0')
			fd_kbd = open(pkb_file, O_RDONLY | O_NONBLOCK, 0);
		else
			fd_kbd = -1 ;
	}
	else
		fd_kbd = open(keyboard_events(), O_RDONLY | O_NONBLOCK , 0);

	if (pfw_file != NULL)
	{
		if (*pfw_file != '\0')
			fd_fw = open(pfw_file, O_RDONLY | O_NONBLOCK, 0);
		else
			fd_fw = -1 ;
	}
	else
		fd_fw = open(fiveway_events(), O_RDONLY | O_NONBLOCK, 0);

	fd_k3_vol = open(k3_vol_events(), O_RDONLY | O_NONBLOCK, 0);

}

static void close_event_files()
{
	if (fd_kbd != -1)
	{
		close(fd_kbd) ;
		fd_kbd = -1 ;
	}
	if (fd_fw != -1)
	{
		close(fd_fw) ;
		fd_fw = -1 ;
	}
	if (fd_k3_vol != -1)
	{
		close(fd_k3_vol) ;
		fd_k3_vol = -1 ;
	}
}

static void close_procwrite_files()
{
	if (fd_prockeypad != -1)
	{
		close(fd_prockeypad) ;
		fd_prockeypad = -1 ;
	}
	if (fd_procfiveway != -1)
	{
		close(fd_procfiveway) ;
		fd_procfiveway = -1 ;
	}
}


static void capture_input(void)
{
	int on = 1 ;

	if (fd_kbd != -1)
	{
		if (ioctl(fd_kbd, EVIOCGRAB, on)) {
    		perror("Capture kbd input:");
		}
	}
	if (fd_fw != -1)
	{
		if (ioctl(fd_fw, EVIOCGRAB, on)) {
    		perror("Capture fw input:");
		}
	}
	if (fd_k3_vol != -1)
	{
		if (ioctl(fd_k3_vol, EVIOCGRAB, on)) {
    		perror("Capture k3_vol input:");
		}
	}

	input_captured = 1 ;
}

static void release_input(void)
{
	int off = 0 ;

	if (input_captured != 0)
	{
		if (fd_kbd != -1)
		{
			if (ioctl(fd_kbd, EVIOCGRAB, off)) {
				perror("Release kbd input:");
			}
		}
		if (fd_fw != -1)
		{
			if (ioctl(fd_fw, EVIOCGRAB, off)) {
				perror("Release fw input:");
			}
		}
		if (fd_k3_vol != -1)
		{
			if (ioctl(fd_k3_vol, EVIOCGRAB, off)) {
				perror("Release k3_vol input:");
			}
		}

		input_captured = 0 ;
	}
}

static void engage_hotkey_timer(int msecs) ;

static void alarm_handler(int x)
{
	struct action * pa ;
	int i ;

	struct itimerval itv ;

	if (getitimer(ITIMER_REAL, &itv) == 0)
	{
		if ((itv.it_value.tv_sec != 0) || (itv.it_value.tv_usec != 0))
		{
			/* fake alarm, timer has not expired yet */
			return ;
		}
	}

	engage_hotkey_timer(0) ;
	release_input() ;

	if (hotkey_mode)
	{
		hotkey_mode = 0 ;
		if (hot_sequence_length > 0)
		{
			pa = find_sequence_action(&hot_sequence[0], hot_sequence_length) ;
			hot_sequence_length = 0 ;
			if (pa)
			{
				skip_next_events = execute_action(pa, fd_kbd, fd_fw ) ;
			}
			else
			{
				DIAG_PRINT("alarm: Unknown hotkey sequence entered: ") ;
				for (i = 0; i < hot_sequence_length; i++)
				{
					send_key((int) hot_sequence[i], SENDMODE_NORMAL) ;
					DIAG_PRINT("%d ", hot_sequence[i]) ;
				}
				DIAG_PRINT("\n") ;
			}
		}
	}
}

static void engage_hotkey_timer(int msecs)
{
	struct itimerval old ;
	struct itimerval new ;

	new.it_interval.tv_sec = 0 ;
	new.it_interval.tv_usec = 0 ;
	new.it_value.tv_sec = msecs / 1000 ;
	new.it_value.tv_usec = (msecs % 1000) * 1000 ;

	if (msecs != 0)
		signal(SIGALRM, alarm_handler) ;
	else
		signal(SIGALRM, SIG_DFL) ;

	setitimer(ITIMER_REAL, &new, &old) ;
}

static void process_input_event(struct input_event *pev)
{
	int i ;

	if (pev->type == EV_KEY)
	{
		/* below we ignore autorepeat events ... */
		if (pev->value == 1)	/* key pressed */
		{
			npressed += 1 ;

			if (hotkey_mode == 0)
			{
				if ((npressed == 1) && (pev->code == introducer))
					introducer_just_pressed = 1 ;
				else
				{
					introducer_just_pressed = 0 ;
					hotkey_mode = 0 ;
				}
			}
			else
			{
				if (pev->code == trailer)
				{
					struct action * pa ;

					engage_hotkey_timer(0) ;
					release_input() ;

					hotkey_mode = 0 ;
					if (hot_sequence_length > 0)
					{
						pa = find_sequence_action(&hot_sequence[0], hot_sequence_length) ;
						hot_sequence_length = 0 ;
						if (pa)
						{
							skip_next_events = execute_action(pa, fd_kbd, fd_fw ) ;
						}
						else
						{
							DIAG_PRINT("trailer: unknown hotkey sequence entered\n") ;
							for (i = 0; i < hot_sequence_length; i++)
							{
								send_key((int) hot_sequence[i], SENDMODE_NORMAL) ;
								DIAG_PRINT("%d ", hot_sequence[i]) ;
							}
							DIAG_PRINT("\n") ;
						}
					}
				}
				else
				{
					hot_sequence[hot_sequence_length] = get_key_substitute_code(pev->code) ;
					hot_sequence_length += 1 ;
				}
			}
		}
		else if (pev->value == 0)	/* key released */
		{
			npressed -= 1 ;
			if ((npressed == 0) && (pev->code == introducer) && (introducer_just_pressed != 0))
			{
				introducer_just_pressed = 0;
				capture_input() ;
				hotkey_mode = 1 ;
				engage_hotkey_timer(hot_interval) ;
			}
		}
	}
}

static void hup_handler(int x)
{
	reinit = 1 ;
	keepgoing = 0 ;
}

static void int_handler(int x)
{
	keepgoing = 0 ;
	DIAG_PRINT("\nSIGINT -- Exiting...\n") ;
}

static void term_handler(int x)
{
	keepgoing = 0 ;
	DIAG_PRINT("\nSIGTERM -- Exiting...\n") ;
}

char *my_process_directory(void)
{
	return &proc_dir[0] ;
}

int main(int argc, char **av)
{
	char inipath[MAXPATH] ;
	int rc ;
	int i ;

	int rce_sock = -1 ;

    if (readlink ("/proc/self/exe", proc_dir, sizeof(proc_dir)-1) != -1)
    {
    	strcpy(inipath, proc_dir) ;
    	rc = strlen(inipath) ;
    	strncat(inipath, ".ini", MAXPATH - rc - 1) ;

        dirname(proc_dir);
        strcat(proc_dir, "/");
    }
    else
    {
    	strncpy(inipath, av[0], MAXPATH-1) ;
    	strcpy(proc_dir, inipath) ;

    	rc = strlen(inipath) ;
    	strncat(inipath, ".ini", MAXPATH - rc - 1) ;

        dirname(proc_dir);
        strcat(proc_dir, "/");
    }

	for (i = 1; i < argc; i++)
	{
		if (strcmp(av[i], "-d") == 0)
			do_daemonize = 0 ;
		else if (strncmp(av[i], "-kb", 3) == 0) /* see if keyboard event file name is specified */
		{
			pkb_file = &av[i][3] ;
		}
		else if (strncmp(av[i], "-fw", 3) == 0) /* see if fiveway event file name is specified */
		{
			pfw_file = &av[i][3] ;
		}
	}

	if (do_daemonize)
	{
		daemonize(LOCKFILE_NAME) ;
		running_as_daemon = 1 ;
	}

again:
	signal(SIGINT, int_handler) ;
	signal(SIGTERM, term_handler) ;
	signal(SIGHUP, hup_handler) ;

	keepgoing = 1 ;
	reinit = 0 ;
	npressed = 0 ;
	hot_sequence_length = 0 ;
	hotkey_mode = 0 ;
	introducer_just_pressed = 0 ;
	skip_next_events = 0 ;
	rce_sock = -1 ;

	DIAG_PRINT("--- Launchpad version %s built on %s %s\n\n", VERSION, __DATE__, __TIME__) ;

	if ((rc = launchpad_init(proc_dir)) >= 0)
	{

		introducer = inroducer_key_code() ;
		trailer = trailer_key_code() ;
		hot_interval = hot_interval_msec() ;

		open_event_files() ;

		if ((fd_kbd != -1) || (fd_fw != -1))
		{
			int numBytes ;
			struct input_event kbbuf[2];
			struct input_event fwbuf[2];
			struct input_event k3volbuf[2];

			fd_set rxfds ;
			struct timeval tv ;
			int maxfd ;

			rce_sock = rce_init() ;
			if (rce_sock != -1)
				rce_start(rce_sock) ;

			fd_prockeypad = open(PROC_KEYPAD_FILE, O_WRONLY /*| O_NONBLOCK*/) ;
			fd_procfiveway = open(PROC_FIVEWAY_FILE, O_WRONLY /*| O_NONBLOCK*/) ;

			close_diag_fp() ;

			keepgoing = 1 ;

			for (; keepgoing != 0; )
			{
				maxfd = 0 ;
				FD_ZERO(&rxfds) ;

				if (fd_kbd != -1)
				{
					FD_SET(fd_kbd, &rxfds) ;
					maxfd = fd_kbd ;
				}

				if (fd_fw != -1)
				{
					FD_SET(fd_fw, &rxfds) ;
					if (maxfd < fd_fw)
						maxfd = fd_fw ;
				}

				if (fd_k3_vol != -1)
				{
					FD_SET(fd_k3_vol, &rxfds) ;
					if (maxfd < fd_k3_vol)
						maxfd = fd_k3_vol ;
				}

				if (maxfd != 0)
				{
					tv.tv_sec = 1 ;
					tv.tv_usec = 0 ;

					rc = select(maxfd + 1, &rxfds, NULL, NULL, &tv) ;
					if (rc > 0)		/* there is some data waiting */
					{
						if (fd_kbd != -1)
						{
							if (FD_ISSET(fd_kbd, &rxfds))
							{
								numBytes = read(fd_kbd, &kbbuf[0], sizeof(struct input_event) * 2) ;

								for (i = 0; i < 2; i++)
								{
									if (numBytes < sizeof(struct input_event))
										break ;
									switch(kbbuf[i].code)
									{
										case KPKEY_FW_PRESS:
										case KPKEY_FW_UP:
										case KPKEY_FW_RIGHT:
										case KPKEY_FW_DOWN:
										case KPKEY_FW_LEFT:
										case K3KEY_FW_UP:
										case K3KEY_FW_DOWN:
										case K3KEY_FW_PRESS:
											/* skip this -- keypad can't generate FW codes */
											////DIAG_PRINT("KB: skipped weird FW key %d\n", kbbuf[i].code) ;
											break ;
										default:
											process_input_event(&kbbuf[i]) ;
											break ;
									}
									numBytes -= sizeof(struct input_event) ;
								}
							}
						}

						if (fd_fw != -1)
						{
							if (FD_ISSET(fd_fw, &rxfds))
							{
								numBytes = read(fd_fw, &fwbuf[0], sizeof(struct input_event) * 2) ;
								for (i = 0; i < 2; i++)
								{
									if (numBytes < sizeof(struct input_event))
									break ;

									switch(fwbuf[i].code)
									{
										case KPKEY_FW_PRESS:
										case KPKEY_FW_UP:
										case KPKEY_FW_RIGHT:
										case KPKEY_FW_DOWN:
										case KPKEY_FW_LEFT:
										case K3KEY_FW_UP:
										case K3KEY_FW_DOWN:
										case K3KEY_FW_PRESS:
											process_input_event(&fwbuf[i]) ;
											break ;
										default:
											/* skip this -- FW can't generate keypad codes */
											//DIAG_PRINT("FW: skipped weird KB key %d\n", fwbuf[i].code) ;
											break ;
									}
									numBytes -= sizeof(struct input_event) ;
								}
							}
						}

						if (fd_k3_vol != -1)
						{
							if (FD_ISSET(fd_k3_vol, &rxfds))
							{
								numBytes = read(fd_k3_vol, &k3volbuf[0], sizeof(struct input_event) * 2) ;
								for (i = 0; i < 2; i++)
								{
									if (numBytes < sizeof(struct input_event))
										break ;

									switch(k3volbuf[i].code)
									{
										case K3KEY_VPLUS:
										case K3KEY_VMINUS:
											process_input_event(&k3volbuf[i]) ;
											break ;
										default:
											/* skip this -- K3 Vol can't generate keypad codes */
											//DIAG_PRINT("K3Vol: skipped weird KB key %d\n", k3volbuf[i].code) ;
											break ;
									}
									numBytes -= sizeof(struct input_event) ;
								}
							}
						}
					}
					else
					{	/* make sure everything is in sync sync state */
						if (rc == 0)
						{
							npressed = 0 ;
							hot_sequence_length = 0 ;
							skip_next_events = 0 ;
						}
					}
				}
			}

		}
		else
			DIAG_PRINT("** cannot open both input event files, exiting...\n") ;

		engage_hotkey_timer(0) ;
		release_input() ;

		if (rce_sock != -1)
			rce_stop(rce_sock) ;

		close_event_files() ;
		close_procwrite_files() ;

		launchpad_deinit() ;

		if (reinit)
		{
			//DIAG_PRINT("Reinit...\n") ;
			reinit = 0 ;
			goto again ;
		}
	}
	else
		DIAG_PRINT("** cannot initialize hotkey sequence database\n") ;

	return rc ;
}

void msdelay(int msecs)
{
	struct timespec tv , rem;
	unsigned long v = msecs ;

	tv.tv_nsec = (v%1000) * 1000000 ;
	tv.tv_sec = v/1000 ;

	while (nanosleep(&tv, &rem) == EINTR)
		tv = rem ;
}

void send_key(short code, int mode)
{
	char *pname = input_key_name(code) ;
	char tmpbuf[MAXPATH] ;
	int len ;

	static const char *sendmode[3] =
	{
			"send",
			"sendshift",
			"sendalt"
	};
	char *pmode = (char *) sendmode[0] ;

	if ((pname != NULL) && ((strncasecmp(pname, "FW_", 3)==0) || (strncasecmp(pname, "k3_FW_", 6)==0)) )
	{
		len = sprintf(&tmpbuf[0], "send %d\n", code) ;

		if (fd_procfiveway != -1)
		{
			write(fd_procfiveway, tmpbuf, len) ;
		}
	}
	else
	{
		if ((mode>=SENDMODE_NORMAL) && (mode<=SENDMODE_ALT))
		{
			pmode = (char *) sendmode[mode] ;
		}

		len = sprintf(&tmpbuf[0], "%s %d\n", pmode, code) ;

		if (fd_prockeypad != -1)
		{
			write(fd_prockeypad, tmpbuf, len) ;
		}
	}
}

void send_keys(short *p, int len)
{
	int i ;

	for (i=0; i < len; i++)
	{
		send_key(*p, SENDMODE_NORMAL) ;
		++p ;
	}
}

void send_ascii_string(char *p)
{
	char c ;

	short char_keypad_code(char asciichar) ;
	int  char_send_mode(char asciichar) ;
	int kdelay ;

	kdelay = key_delay_msec() ;

	while (*p)
	{
		c = (*p) & 0x7f ;

		if (c == '\\')
		{
			if ((*(p+1)) != '\0')
			{
				++p ;
				c = (*p) & 0x7f ;
			}
		}
		send_key(char_keypad_code(c), char_send_mode(c)) ;
		++p ;

		msdelay(kdelay) ;
	}
};

int send_kindle_input_key(char c)
{
	struct kindle_syms * psym ;
	short chorz, cvert ;
	int nhorz, nvert ;
	int rc = 0 ;
	int kdelay ;

	extern int is_this_kindle3(void) ;

	kdelay = key_delay_msec() ;

	c &= 0x7f ;
	if ((psym = char_kindle_fw_sequence(c)) != NULL)
	{
		if (psym->hsteps < 0)
		{
			nhorz = -(psym->hsteps) ;
			chorz = ((is_this_kindle3() == 0) ? KPKEY_FW_LEFT : K3KEY_FW_LEFT) ;
		}
		else
		{
			nhorz = psym->hsteps ;
			chorz = ((is_this_kindle3() == 0) ? KPKEY_FW_RIGHT : K3KEY_FW_RIGHT) ;
		}
		if (psym->vsteps < 0)
		{
			nvert = -(psym->vsteps) ;
			cvert = ((is_this_kindle3() == 0) ? KPKEY_FW_UP : K3KEY_FW_UP) ;
		}
		else
		{
			nvert = psym->vsteps ;
			cvert = ((is_this_kindle3() == 0) ? KPKEY_FW_DOWN : K3KEY_FW_DOWN) ;
		}

		/* framework injects extra FW keys, so try to compensate.. */
		rc = nhorz + nvert  + 1;

		send_key((is_this_kindle3() == 0) ? KPKEY_SYM : K3KEY_SYM, SENDMODE_NORMAL) ;
		msdelay(kdelay) ;

		for (; nhorz > 0; --nhorz)
		{
			send_key(chorz, SENDMODE_NORMAL) ;
			msdelay(kdelay) ;
		}
		for (; nvert > 0; --nvert)
		{
			send_key(cvert, SENDMODE_NORMAL) ;
			msdelay(kdelay) ;
		}


		send_key(((is_this_kindle3() == 0) ? KPKEY_FW_PRESS : K3KEY_FW_PRESS), SENDMODE_NORMAL) ;
		msdelay(kdelay) ;

		if (is_this_kindle3() != 0)
		{
			send_key(K3KEY_SYM, SENDMODE_NORMAL) ;
			msdelay(kdelay) ;
		}
	}
	else
	{
		send_key(char_keypad_code(c), char_send_mode(c)) ;
	}

	return rc ;
}
int send_kindle_ascii_string(char *ptok)
{
	char c ;
	int toskip = 0 ;

	int kdelay ;

	kdelay = key_delay_msec() ;

	while ((c = *ptok++) != '\0')
	{
		toskip += send_kindle_input_key(c) ;
		msdelay(kdelay) ;
	}

	return toskip ;
}

/*****************  daemonize-related stuff ********************************/

int has_console(void)
{
	return((int) (running_as_daemon == 0)) ;
}

static void child_handler(int signum)
{
    switch(signum) {
    	case SIGALRM: exit(EXIT_FAILURE); break;
    	case SIGUSR1: exit(EXIT_SUCCESS); break;
    	case SIGCHLD: exit(EXIT_FAILURE); break;
    }
}
static void daemon_handler(int signum)
{
    switch(signum)
    {
    	case SIGTERM:
    		unlink(LOCKFILE_NAME) ;
    		exit(EXIT_FAILURE);
    		break;
    }
}


static void daemonize( const char *lockfile )
{
    pid_t pid, sid, parent;
    int lfp = -1;
    struct stat statbuf ;
    char pidbuf[16] ;
    char tmpbuf[32] ;
    int n ;

    /* already a daemon */
    if ( getppid() == 1 ) return;

    if (stat(lockfile, &statbuf) == 0)
    {
        lfp = open(lockfile,O_RDWR,0640);
        if ( lfp < 0 ) {
            DIAG_PRINT("unable to open lock file %s, code=%d (%s)\n",
                    lockfile, errno, strerror(errno) );
            exit(EXIT_FAILURE);
        }
        pidbuf[0] = 0 ;
        read(lfp, &pidbuf[0], sizeof(pidbuf)-1) ;
        close(lfp) ;
        pidbuf[sizeof(pidbuf)-1] = '\0' ;
        sprintf(tmpbuf, "/proc/%s", pidbuf) ;
    }

    /* Create the lock file as the current user */
    if ( lockfile && lockfile[0] ) {
        lfp = open(lockfile,O_RDWR|O_CREAT,0640);
        if ( lfp < 0 ) {
            DIAG_PRINT("unable to create lock file %s, code=%d (%s)\n",
                    lockfile, errno, strerror(errno) );
            exit(EXIT_FAILURE);
        }
    }


    /* Trap signals that we expect to recieve */
    signal(SIGCHLD,child_handler);
    signal(SIGUSR1,child_handler);
    signal(SIGALRM,child_handler);

    /* Fork off the parent process */
    pid = fork();
    if (pid < 0) {
        DIAG_PRINT("unable to fork daemon, code=%d (%s)\n",
                errno, strerror(errno) );

        close(lfp) ;
        unlink(lockfile) ;
        exit(EXIT_FAILURE);
    }
    /* If we got a good PID, then we can exit the parent process. */
    if (pid > 0) {

        /* Wait for confirmation from the child via SIGTERM or SIGCHLD, or
           for two seconds to elapse (SIGALRM).  pause() should not return. */
        alarm(2);
        pause();

        exit(EXIT_FAILURE);
    }

    /* At this point we are executing as the child process */
    parent = getppid();

    /* Cancel certain signals */
    signal(SIGCHLD,SIG_DFL); /* A child process dies */
    signal(SIGTSTP,SIG_IGN); /* Various TTY signals */
    signal(SIGTTOU,SIG_IGN);
    signal(SIGTTIN,SIG_IGN);

    signal(SIGTERM, daemon_handler);

    /* Create a new SID for the child process */
    sid = setsid();
    if (sid < 0) {
        DIAG_PRINT("unable to create a new session, code %d (%s)\n",
                errno, strerror(errno) );
        exit(EXIT_FAILURE);
    }

    n = sprintf(pidbuf, "%d", getpid()) ;
    if ((n > 0) && ((write(lfp, pidbuf, n+1)) > n))
    {
    	close(lfp) ;
    	lfp = -1 ;
    }
    else
    {
    	close(lfp) ;
    	unlink(lockfile) ;
        DIAG_PRINT("unable to write pid %s to %s -- %s\n",
                pidbuf, lockfile, strerror(errno) );
        exit(EXIT_FAILURE);
    }
}

int is_this_kindle3()
{
	int rc = 0 ;

	/* if all three - /dev/input/event0 , /dev/input/event1 */
	if ((fd_kbd != -1) && (fd_fw != -1))
	{
		if (fd_k3_vol != -1)	/* and /dev/input/event2  are open */
			rc = 1 ;			/* then we are running on Kindle3 !! */
	}

	return rc ;
}

FILE * get_diag_fp()
{
	FILE *rc = stderr ;

	if (has_console() == 0)
	{
		if (diag_fp == NULL)
		{
			char tmpbuf[MAXPATH] ;

			strcpy(&tmpbuf[0], proc_dir) ;
			strcat(&tmpbuf[0], "launchpad.log") ;
			if ((diag_fp = fopen(tmpbuf, "w")) != NULL)
				rc = diag_fp;
			else
				fprintf(stderr, "Can't open diag file %s\n", &tmpbuf[0]) ;
		}
		else
			rc = diag_fp ;
	}

	return rc ;
}

void close_diag_fp()
{
	if (diag_fp != NULL)
	{
		fclose(diag_fp) ;
		diag_fp = NULL ;
	}
}
