/*
 * File Name: views_order.c
 */

/*
 * This file is part of mackx_patch.
 *
 * mackx_patch is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * mackx_patch is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2010 Marcel Hendrickx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gconf/gconf-client.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <stdlib.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "main.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct
    {
        int position;
        char *name;
    } ViewOrderItem;
//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------
enum
{
   POSITION_COLUMN,
   NAME_COLUMN,
   N_COLUMNS
};

// NOTE: These are copied from ctb/include/filemodel.h
// special item filenames
#define SPECIAL_ITEM_NAME "special"
#define SPECIAL_BOOKS "books"
#define SPECIAL_IMAGES "images"
#define SPECIAL_NEWS "news"
#define SPECIAL_HELP "help"
#define SPECIAL_PERSONAL "personal"
#define SPECIAL_DIR "dir"
#define SPECIAL_SHORTCUTS "shortcuts"
#define SPECIAL_NOTES "notes"
#define SPECIAL_SETTINGS "settings"
#define SPECIAL_SEARCH "search"
#define SPECIAL_RECENT "recent"

#define CTB_PATCH_GCONF_PATH  "/apps/er/patch/ctb"
static const gchar      *REGKEY_VIEWS_DIR = CTB_PATCH_GCONF_PATH "/views";

#define GCONF_BUFFER_SIZE                   128
//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------
static const int        BUTTONS_H_SPACING       =  20;
static GtkWidget        *g_entry                =  NULL;
static GtkWidget        *g_button_set           = NULL;  // button "Set"
static GtkWidget        *g_tree                 = NULL;  // tree view

static ViewOrderItem g_internal_items[] = {
    { 10,    "lastread" },
    { 20,    SPECIAL_BOOKS },
    { 30,    SPECIAL_NEWS },
    { 40,    SPECIAL_IMAGES },
    { 50,    "store" },
    { 60,    SPECIAL_PERSONAL },
    { 70,    SPECIAL_NOTES },
    { 80,    SPECIAL_RECENT },
    { 90,    SPECIAL_SHORTCUTS },
    {100,    SPECIAL_DIR },
    {110,    SPECIAL_SEARCH },
    {120,    SPECIAL_HELP },
    {130,    SPECIAL_SETTINGS },
    { -1,    NULL}
};

static GArray           *g_views_info = NULL;    // element: ViewOrderItem*

//============================================================================
// Local Function Definitions
//============================================================================

// signal handlers
/* Prototype for selection handler callback */
static void on_set_clicked (GtkButton *button, gpointer user_data);
static void tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data);
static void tree_selection_activated_cb (GtkTreeView        *treeview,
						                 GtkTreePath        *path,
						                 GtkTreeViewColumn  *col,
						                 gpointer            userdata);
static void add_hardcoded_views();
static void add_views_from_gconf();


static void vieworder_populate_tree_model (GtkTreeStore *store);
static gint compare_views(gconstpointer a, gconstpointer b);
static gboolean foreach_func (GtkTreeModel *model,
  			                  GtkTreePath  *path,
			                  GtkTreeIter  *iter,
			                  gpointer      user_data);


//============================================================================
// Functions Implementation
//============================================================================

//
// Load the information from the internal views and the user views (gconf) into g_views_info
void load_vieworder_settings()
{
    LOGPRINTF("entry");
    g_views_info = g_array_new( TRUE, TRUE, sizeof(ViewOrderItem) );
    g_assert(g_views_info);

    // the internal views with default values
    add_hardcoded_views();
    // the user added views and user overwrites of default values
    add_views_from_gconf();

    // sort them on priority
    g_array_sort (g_views_info, compare_views);
}

// store user view info and modified internal views in gconf
void save_vieworder_settings()
{
    GtkTreeModel *model;

    LOGPRINTF("entry");
	if (g_tree)
	{
		model = gtk_tree_view_get_model(GTK_TREE_VIEW (g_tree));
		// traverse each item in the tree_store and check if it is changed (w.r.t. g_views_info)
		gtk_tree_model_foreach(model, foreach_func, NULL);
	}
	else
	{
		LOGPRINTF("No tree_view!!");
	}
}

// Create a tree_view and add the items from g_view_info to it
GtkWidget* create_vieworder_items(void)
{
    LOGPRINTF("entry");
	GtkTreeStore *store;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;
	GtkTreeSelection *select;

	/* Create a model.  We are using the store model for now, though we
	* could use any other GtkTreeModel */
	store = gtk_tree_store_new (N_COLUMNS,
                                G_TYPE_UINT,
                                G_TYPE_STRING);

	/* custom function to fill the model with data */
	vieworder_populate_tree_model (store);

	/* Create a view */
	g_tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
	g_signal_connect (G_OBJECT (g_tree), "row-activated",
					  G_CALLBACK (tree_selection_activated_cb),
					  NULL);

	/* The view now holds a reference.  We can get rid of our own
	* reference */
	g_object_unref (G_OBJECT (store));

	/* Setup the selection handler */
	select = gtk_tree_view_get_selection (GTK_TREE_VIEW (g_tree));
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (G_OBJECT (select), "changed",
					  G_CALLBACK (tree_selection_changed_cb),
					  NULL);

	/* First column.. Position in the Desktop */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Position", renderer,
                                                       "text", POSITION_COLUMN,
                                                       NULL);

	/* Add the column to the view. */
	gtk_tree_view_append_column (GTK_TREE_VIEW (g_tree), column);

	/* Name/Identifier of the item */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
                                                       "text", NAME_COLUMN,
                                                       NULL);

	/* Add the column to the view. */
	gtk_tree_view_append_column (GTK_TREE_VIEW (g_tree), column);


    return g_tree;
}

// create items to edit the position of an item
GtkWidget* create_changeorder_items(void)
{
    GtkWidget *label1;
    GtkWidget *widget;
    GtkBox      *hbox;
    GtkWidget   **p_widget;

    LOGPRINTF("entry");
    //             |
    //             |-- hbox
    //                   |-- label
    //                   |-- entry
    widget = gtk_hbox_new(FALSE, BUTTONS_H_SPACING);
    hbox = GTK_BOX(widget);

    label1 = gtk_label_new("Priority");
    gtk_box_pack_start (hbox, label1, TRUE, TRUE, 0);

    g_entry = gtk_entry_new();
    //gtk_entry_set_text ( GTK_ENTRY (g_entry), old_value );
    gtk_editable_select_region (GTK_EDITABLE (g_entry), 0, -1);
    gtk_entry_set_activates_default (GTK_ENTRY (g_entry), TRUE);

    gtk_box_pack_start (hbox, g_entry, TRUE, TRUE, 0);

    p_widget = &g_button_set;
    widget = gtk_button_new_with_label("Set");
    gtk_box_pack_end(hbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "clicked", G_CALLBACK(on_set_clicked),    NULL);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;

    gtk_widget_show(label1);
    gtk_widget_show(g_entry);
    
    return GTK_WIDGET(hbox);

}

static void vieworder_populate_tree_model (GtkTreeStore *store)
{
	GtkTreeIter   iter;

    LOGPRINTF("entry");
    
    ViewOrderItem *details = (ViewOrderItem*) g_views_info->data;
    while ( details && details->name )
    {
        gtk_tree_store_append (store, &iter, NULL);  /* Acquire an iterator */
        gtk_tree_store_set (store, &iter,
                            POSITION_COLUMN, details->position,
                            NAME_COLUMN, details->name,
                            -1);
        details++;
    }
}

// button "Set" clicked
static void on_set_clicked (GtkButton *button, gpointer user_data)
{
    GtkTreeSelection *selection;
	GtkTreeIter iter;
	GtkTreeModel *model;
	guint position;
	gchar *name;

    LOGPRINTF("entry");
    
    // get the value in the entry and store it for the active item
    position = atoi(gtk_entry_get_text(GTK_ENTRY (g_entry)));
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(g_tree));
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, NAME_COLUMN, &name, 
										  -1);
        gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                            POSITION_COLUMN, position,
                            -1);

		LOGPRINTF("Values %d %s\n", position, name);

		g_free (name);
	}
}

// row selected
static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	guint position;
	gchar *name;

    LOGPRINTF("entry");
	if (gtk_tree_selection_get_selected (selection, &model, &iter))
	{
		gtk_tree_model_get (model, &iter, POSITION_COLUMN, &position,
		                                  NAME_COLUMN, &name, 
										  -1);

		LOGPRINTF("Values %d %s\n", position, name);

		g_free (name);
	}
}

// row double clicked
static void
tree_selection_activated_cb (GtkTreeView        *treeview,
						     GtkTreePath        *path,
						     GtkTreeViewColumn  *col,
						     gpointer            userdata)
{
    GtkTreeModel *model;
    GtkTreeIter   iter;
 
    LOGPRINTF("A row has been double-clicked!\n");
 
    model = gtk_tree_view_get_model(treeview);
 
    if (gtk_tree_model_get_iter(model, &iter, path))
    {
        gchar *name;
	    guint position;
        gchar position_str[20];
 
		gtk_tree_model_get (model, &iter, POSITION_COLUMN, &position,
		                                  NAME_COLUMN, &name, 
										  -1);
 
        LOGPRINTF("Double-clicked row contains name %s\n", name);
        
        sprintf(position_str, "%d", position);
        gtk_entry_set_text ( GTK_ENTRY (g_entry), position_str );
        
        gtk_widget_grab_focus (g_entry);

        g_free(name);
    }
}

// add info from the internal views with default settings to g_views_info
static void add_hardcoded_views()
{
    ViewOrderItem      *known_item = NULL;
    ViewOrderItem      item;

    // fill array with hardcoded view-orders
    for ( known_item = g_internal_items ; known_item->name ; known_item++ )
    {
        LOGPRINTF("HC-Add: [%s]=%d; ", known_item->name, known_item->position);
        // set details
        memset(&item, 0x00, sizeof(item));
        item.position = known_item->position;
        item.name = known_item->name;

        // add to array
        g_array_append_val(g_views_info, item);
    }
}

// read single string from gconf
static gchar *read_gconf_value(const gchar *dir,
                               const gchar *key,
                               GConfClient *client)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    gchar* value;
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);
    value = gconf_client_get_string(client, buffer, NULL);
    return value;
}

// write string to gconf
static gboolean write_gconf_value(const gchar *dir,
                                   const gchar *key,
                                   GConfClient *client,
								   gchar *value)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);
    return gconf_client_set_string(client, buffer, value, NULL);
}

// read int from gconf
static int read_gconf_int(const gchar *dir,
                          const gchar *key,
                          GConfClient *client)
{
    int value = 0;
    gchar buffer[GCONF_BUFFER_SIZE];
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);

    value = gconf_client_get_int(client, buffer, NULL);
    return value;
}

// write int to gconf
static gboolean write_gconf_int(const gchar *dir,
                                const gchar *key,
                                GConfClient *client,
						        int value)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);

    return gconf_client_set_int(client, buffer, value, NULL);
}

// search the item with the given name in g_views_info
ViewOrderItem* find_info_with_name (const gchar *name )
{

    g_assert(g_views_info);

    //LOGPRINTF("find_info_with_name: name [%s]", name);
    if (name == NULL)
    {
        return NULL;
    }

    ViewOrderItem *ret = NULL;
    ViewOrderItem *details = (ViewOrderItem*) g_views_info->data;
    while ( details && details->name )
    {
        //LOGPRINTF("Checking:[%s]\n", details->name);
        if (details->name)
        {
            if (strcmp(details->name, name) == 0)
            {
                ret = details;
                break;
            }
            else
            {
                details++;
            }
        }
        else
        {
            details++;
        }
    }

    //LOGPRINTF("Returning:%d\n", (int)ret);
    return ret;
}

// adds a single entry from gconf to g_views_info (called for every directory in gconf)
static void add_view_from_gconf(const gchar *dir,
                                GConfClient *client)
{
    gchar* buffer = NULL;
    gchar* filename = NULL;
    int mode = 1;
    int position;

    // File Name (extracted from the key)
    char *t = strrchr(dir, '/');
    if (t)
    {
        filename = g_strdup(&t[1]);
    }
    if (!filename) return;
    //LOGPRINTF("FileName:[%s];", filename);
    
    // Mode (only parameter allowed for pre-defined views)
    buffer = read_gconf_value(dir, "mode", client);
    if (!buffer) return;
    if (strcmp(buffer, "enabled") == 0) { mode = 1; }
    else { mode = 0; }
    g_free(buffer);
    //LOGPRINTF("Mode:[%d];", mode);
    
    // Position
    position = read_gconf_int(dir, "position", client);
    //LOGPRINTF("Position:[%d];", position);
    
    // the entries for the build-in types only have a mode field (if any)
    // modify the mode field to be able to hide certain views in the Home screen
    ViewOrderItem *entry = find_info_with_name(filename);
    if (entry)
    {
        // existing item, check if hardcoded item is enabled
        if (mode == 0)
        {
            LOGPRINTF("GC-Change: [%s] disable; ", filename);
            entry->position = 0;
        }
        else if (position != 0) // if position is 0 and mode is 'enable' then position was not in gconf! so use default
        {
            LOGPRINTF("GC-Change: [%s]=%d; ", filename, position);
            entry->position = position;
        }
    }
    else
    {
        ViewOrderItem item;

        LOGPRINTF("GC-Add: [%s]=%d; ", filename, position);
        // new item
        memset(&item, 0x00, sizeof(item));
        item.position = position;
        item.name = g_strdup(filename);
        // add to array
        g_array_append_val(g_views_info, item);
    }
}

// read user views from gconf and add them to g_views_info, also overrides
// default values from internal types when set in gconf
static void add_views_from_gconf()
{
    GSList *types = NULL;
    GConfClient *client = gconf_client_get_default();

    gboolean exists = gconf_client_dir_exists(client, REGKEY_VIEWS_DIR, NULL);
    if (exists == FALSE) {
        ERRORPRINTF("gconf: %s does not exist", REGKEY_VIEWS_DIR);
        goto unref;
    }

    types = gconf_client_all_dirs(client, REGKEY_VIEWS_DIR, NULL);
    if (types == NULL) {
        ERRORPRINTF("gconf: cannot read dir entries of %s", REGKEY_VIEWS_DIR);
        goto unref;
    }

    g_slist_foreach (types, (GFunc)add_view_from_gconf, client);
    g_slist_foreach (types, (GFunc)g_free, NULL);
    g_slist_free(types);
unref:
    g_object_unref(client);
}

// compare function to sort g_views_info
static gint compare_views(gconstpointer a, gconstpointer b)
{
    ViewOrderItem* view_a = ((ViewOrderItem*)(a));
    ViewOrderItem* view_b = ((ViewOrderItem*)(b));
    if (view_a->position == view_b->position)
    {
        return 0;
    }
    else if (((view_a->position > view_b->position) || (view_a->position == 0)) && (view_b->position != 0))
    {
        // if position is zero, put it at end
        return 1;
    }
    return -1;
}

static gboolean foreach_func (GtkTreeModel *model,
  			                  GtkTreePath  *path,
			                  GtkTreeIter  *iter,
			                  gpointer      user_data)
{
	gchar *name;
	guint position;

	// Get the value from the tree_store
	gtk_tree_model_get( model, iter, 
						POSITION_COLUMN, &position,
		                NAME_COLUMN, &name, 
						-1 );

    LOGPRINTF("entry:%s", name);
    
	// get the value from g_views_info
	ViewOrderItem* item = find_info_with_name(name);
	
	// store when value has changed
	if (position != item->position)
	{
		GConfClient *client = gconf_client_get_default();
		gchar buffer[GCONF_BUFFER_SIZE];
		// construct dir
		g_assert(strlen(REGKEY_VIEWS_DIR) + 1 + strlen(name) < GCONF_BUFFER_SIZE);
		snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", REGKEY_VIEWS_DIR, name);
		
		write_gconf_int(buffer, "position", client, position);
		if (position != 0)
		{
            LOGPRINTF("Changed:%s-%d-enabled", name, position);
			write_gconf_value(buffer, "mode", client, "enabled");
		}
		else
		{
            LOGPRINTF("Changed:%s-%d-disabled", name, position);
			write_gconf_value(buffer, "mode", client, "disabled");
		}
	}

	g_free(name); /* gtk_tree_model_get made copies of       */
				  /* the strings for us when retrieving them */

   return FALSE; /* do not stop walking the store, call us with next row */
}
 
