#ifndef __DB_H__
#define __DB_H__

/**
 * File Name  : db.h
 *
 * Description: Metadb interface functions
 */

/*
 * This file is part of hello-world.
 *
 * hello-world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello-world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <glib.h>

// ereader include files, between < >
#include <libermetadb/ermetadb.h>

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Macro Definitions
//---------------------------------------------------------------------------- 

// some empty macros that show the usaga of paramerts in functions
// IN   : input parameter
// OUT  : output parameter
// INOUT: input parameter that is changed by the function

#define IN
#define OUT
#define INOUT

//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// struct to hold thumbnail data
typedef struct {
    guchar* data;   // data is in png format
    gint size;
} thumbType;


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//============================================================================
// Public Functions
//============================================================================

// Note: there are example functions to illustrate how to retreive
// metadata from the database using libermetadb, see db.c for implemetation
// Use your own (and better) defined functions to retrieve and save 
// file metadata from and to the database

// database lifecycle -> open and close

/**---------------------------------------------------------------------------
 *
 * Name :  open_database
 *
 * @brief  Opens metadata database in directory
 *
 * @param  dir  directory where the database resides. This is the same 
 *              place as where the files are stored.
 *
 * @return pointer to a valid erMedadb object on success, NULL on failure
 *
 *--------------------------------------------------------------------------*/ 
erMetadb open_database(const char* IN dir);

/**---------------------------------------------------------------------------
 *
 * Name :  close_database
 *
 * @brief  Closes an open metadata database in directory
 *
 * @param  db     pointer to a valid erMetadb object
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/ 
void close_database(erMetadb IN db); 

/**---------------------------------------------------------------------------
 *
 * Name :  load_file_metadata
 *
 * @brief  Loads the file metadata from database
 *
 * @param  db       pointer to a valid erMetadb object
 * @param  filepath filename for which to load metadata
 * @param  filename filename for which to load metadata
 * @param  title    pointer to GString to store the title
 *                  NULL -> title should not be retreived
 * @param  author   pointer to GString to store the author
 *                  NULL -> author should not be retreived
 *
 * @return FALSE on failure, TRUE on success
 *
 *--------------------------------------------------------------------------*/ 
gboolean load_file_metadata(erMetadb        IN  db,        
                            const gchar*    IN  filepath,
                            const gchar*    IN  filename,
                            GString*        OUT title,
                            GString*        OUT author,
                            thumbType*      OUT small,
                            thumbType*      OUT medium);


/**---------------------------------------------------------------------------
 *
 * Name :  save_file_metadata
 *
 * @brief  Saves the file metadata to database
 *
 * @param  db       pointer to a valid erMetadb object
 * @param  filepath filepath for which to save metadata
 * @param  filename filename for which to save metadata
 * @param  title    pointer to GString to store the title
 *                  NULL -> title should not be stored
 * @param  author   pointer to GString to store the author
 *                  NULL -> author should not be stored
 *
 * @return FALSE on failure, TRUE on success
 *
 *--------------------------------------------------------------------------*/ 
gboolean save_file_metadata(erMetadb        IN db,
                            const gchar*    IN filepath,
                            const gchar*    IN filename,
                            GString*        IN title,
                            GString*        IN author,
                            thumbType*      IN small,
                            thumbType*      IN medium);

/**---------------------------------------------------------------------------
 *
 * Name :  pixbuf_draw_rectangle
 *
 * @brief  Draw a rectangle border in a pixbuf
 *
 * @param  pixbuf    pointer to GdkPixbuf to be converted to BLOB
 * @param  xx        x position offset
 * @param  yy        y position offset
 * @param  ww        width in pixels
 * @param  hh        height pixels
 * @param  color     RGB color 
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/ 
void pixbuf_draw_rectangle (GdkPixbuf* pixbuf, 
                            gint xx, 
                            gint yy, 
                            gint ww, 
                            gint hh, 
                            gint color);

/**---------------------------------------------------------------------------
 *
 * Name :  pixbuf_to_blob
 *
 * @brief  Converts a GdkPixbuf* to a BLOB buffer
 *
 * @param  pixbuf    pointer to GdkPixbuf to be converted to BLOB
 * @param  buffer    buffer in where the data of the BLOB is stored
 * @param  buf_size  size of the buffer after conversion
 *
 * @return FALSE on failure, TRUE on success
 *
 *--------------------------------------------------------------------------*/ 
gboolean pixbuf_to_blob(GdkPixbuf* IN pixbuf, gchar** OUT buffer, gsize* OUT buf_size);


G_END_DECLS

#endif
