/*
 * File Name: views.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2010-2011 Marcel Hendrickx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

#include <gconf/gconf-client.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <string.h>

#include "ctb_log.h"
#include "views.h"
#include "i18n.h"

#include "filetypes.h"
#include "filemodel.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// detailed info per view.
typedef struct
    {
        const char* title;
        int position;
        const char* tag_filter;
        ctb_sort_order_t order;
        gboolean fixed_order;
        const char* filename;
        char* subtitle;
        const char* path;
        const char* filename_icon;
    } ViewModeInfo;


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

#define TAG_BOOK     "book"
#define TAG_IMAGE    "image"
#define TAG_NEWS     "news"
#define TAG_PERSONAL "personal"
#define TAG_HELP     "help"
#define TAG_NOTE     "note"

// Location to store the configuration data. Storing it in /apps/er/ has the advantage that
// the data is handled like all other config data. Non-default data is written into dr.ini
// and can there be changed by the user with a simple editor.
// I preferred to have a separate patch.ini file, but that would require patching sysd to
// write/read the file at shutdown/startup.
#define CTB_PATCH_GCONF_PATH  "/apps/er/patch/ctb"
static const gchar      *REGKEY_VIEWS_DIR = CTB_PATCH_GCONF_PATH "/views";

// known views
// The position parameter is choosen to leave room between each item, to allow user defined views to be added
// between the predefined. Only the mode of pre-defined views can be changed, all other parameters that are 
// specified in the dr.ini/gconf are ignored.
static ViewModeInfo g_viewmodes[] =
    {   // view,             title,               position tag_filter,      default sort,          fixed,    filename,            subtitle,          path,          icon
        //                                                                  order                  order
        [DESKTOP_VIEW]  = { N_("Home"),               0,      NULL,         CTB_SORT_BY_NAME,       TRUE,    "home", "subtitle", NULL, "icon" },
        [SETTINGS_VIEW] = { N_("Settings"),           130,    NULL,         CTB_SORT_BY_NAME,       TRUE,    SPECIAL_SETTINGS,  N_("Find and edit the settings used on this device"), "/usr/share/ctb/settings", "settings" },  
        [BOOKS_VIEW]    = { N_("Books"),              20,     TAG_BOOK,     CTB_SORT_BY_NAME,       FALSE,   SPECIAL_BOOKS,     N_("See all the books that have been saved to your library"), NULL, "books" },
        [NEWS_VIEW]     = { N_("News"),               30,     TAG_NEWS,     CTB_SORT_BY_DATE_ADDED, FALSE,   SPECIAL_NEWS,      N_("Read the latest newspapers and other periodicals"), NULL, "news" },
        [IMAGES_VIEW]   = { N_("Images"),             40,     TAG_IMAGE,    CTB_SORT_BY_NAME,       FALSE,   SPECIAL_IMAGES,    N_("View your saved pictures and images"), NULL, "images" },
        [PERSONAL_VIEW] = { N_("Personal Documents"), 60,     TAG_PERSONAL, CTB_SORT_BY_NAME,       FALSE,   SPECIAL_PERSONAL,  N_("See all the files saved in Personal Documents"), NULL, "personal" },
        [DIR_VIEW]      = { N_("SD Card"),            100,    NULL,         CTB_SORT_BY_NAME,       FALSE,   SPECIAL_DIR,       N_("Browse and navigate the folders saved on the SD card"), NULL, "sdcard" },
        [SHORTCUT_VIEW] = { N_("Shortcuts"),          90,     NULL,         CTB_SORT_BY_NAME,       FALSE,   SPECIAL_SHORTCUTS, N_("Shortcuts that you've made to your documents are found here"), DIR_SHORTCUTS, "shortcuts" },
        [NOTES_VIEW]    = { N_("Notes"),              70,     TAG_NOTE,     CTB_SORT_BY_DATE_ADDED, FALSE,   SPECIAL_NOTES,     N_("Keep track of your notes with this handy notepad"), DIR_NOTES, "notes" },
        [HELP_VIEW]     = { N_("Help"),               120,    TAG_HELP,     CTB_SORT_BY_NAME,       FALSE,   SPECIAL_HELP,      N_("Learn more about the device and features"), NULL, "help" },
        [RECENT_VIEW]   = { N_("Recently Added"),     80,     NULL,         CTB_SORT_BY_DATE_ADDED, TRUE,    SPECIAL_RECENT,    N_("View the last 15 items added to the device"), NULL, "recent" },
        [SEARCH_VIEW]   = { N_("Search Results"),     110,    NULL,         CTB_SORT_BY_NAME,       FALSE,   SPECIAL_SEARCH,    N_("Search for books, news items, or images"), NULL, "search" },
        [LASTREAD_VIEW] = { N_("Last Read"),          10,     NULL,         CTB_SORT_BY_NAME,       FALSE,   "lastread", "subtitle", NULL, "icon" }, // dummy view for sorting
        [STORE_VIEW]    = { N_("Store"),              50,     NULL,         CTB_SORT_BY_NAME,       FALSE,   "store", "subtitle", NULL, "icon" }, // dummy view for sorting
                          { NULL,                     0,      NULL,         CTB_SORT_BY_NAME,       FALSE,   NULL, NULL, NULL, NULL }
    };


#define GCONF_BUFFER_SIZE                   128

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GArray           *g_views_info = NULL;    // element: ViewModeInfoExt*

static int g_view_number = VIEWMODE_USER_OFFSET;

//============================================================================
// Local Function Definitions
//============================================================================

static ViewModeInfoExt* find_view_info (const gchar *title );
static gint compare_views(gconstpointer a, gconstpointer b);


//============================================================================
// Functions Implementation
//============================================================================

static gchar *read_gconf_value(const gchar *dir,
                               const gchar *key,
                               GConfClient *client)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    gchar* value;
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);

    value = gconf_client_get_string(client, buffer, NULL);
    return value;
}


static gboolean read_gconf_boolean(const gchar *dir,
                                   const gchar *key,
                                   GConfClient *client)
{
    gboolean value = FALSE;
    gchar buffer[GCONF_BUFFER_SIZE];
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);

    value = gconf_client_get_bool(client, buffer, NULL);
    return value;
}


static gboolean read_gconf_int(const gchar *dir,
                               const gchar *key,
                               GConfClient *client)
{
    int value = 0;
    gchar buffer[GCONF_BUFFER_SIZE];
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);

    value = gconf_client_get_int(client, buffer, NULL);
    return value;
}


static void add_view_from_gconf(const gchar *dir,
                                GConfClient *client)
{
    gchar* buffer = NULL;
    gchar* title = NULL;
    ViewMode style; // style on which this viewmode is based
    gchar* tag_filter = NULL; // used in style TAG
    ctb_sort_order_t order;
    gboolean fixed_order;
    gchar* filename = NULL;
    gchar* filename_icon = NULL;
    gchar* subtitle = NULL;
    gchar* path = NULL;
    int mode = VIEW_ENABLE;
    int position;

    // File Name (extracted from the key)
    char *t = strrchr(dir, '/');
    if (t)
    {
        filename = g_strdup(&t[1]);
    }
    if (!filename) goto out_error;
    
    // Mode (allowed for pre-defined views)
    buffer = read_gconf_value(dir, "mode", client);
    if (!buffer) goto out_error;
    if (strcmp(buffer, "enabled") == 0) { mode = VIEW_ENABLE; }
    else { mode = VIEW_DISABLE; }
    g_free(buffer);
    
    // the entries for the build-in types only have a mode field and possibly a position (if any)
    // modify the mode field to be able to hide certain views in the Home screen
    ViewModeInfoExt *entry = view_find_info_with_name(filename);
    if (entry)
    {
        entry->mode = mode;
    }

    // Position (allowed for pre-defined views)
    position = read_gconf_int(dir, "position", client);
    if ((entry) && (position > 0))
    {
        entry->position = position;
    }
    
    // Title
    title = read_gconf_value(dir, "title", client);
    if (!title) goto out_error;

    // Style
    buffer = read_gconf_value(dir, "style", client);
    if (!buffer) goto out_error;
    if (strcmp(buffer, "tag") == 0) { style = BOOKS_VIEW; }
    else if (strcmp(buffer, "date") == 0) { style = RECENT_VIEW; }
    else if (strcmp(buffer, "dir") == 0) { style = DIR_VIEW; }
    else { style = BOOKS_VIEW; }
    g_free(buffer);

    // Tag Filter
    tag_filter = read_gconf_value(dir, "tag_filter", client);
    if (!tag_filter) goto out_error;

    // Order
    buffer = read_gconf_value(dir, "sort_order", client);
    if (!buffer) goto out_error;
    if (strcmp(buffer, "name") == 0) { order = CTB_SORT_BY_NAME; }
    else if (strcmp(buffer, "date_read") == 0) { order = CTB_SORT_BY_DATE_READ; }
    else if (strcmp(buffer, "date_added") == 0) { order = CTB_SORT_BY_DATE_ADDED; }
    else { order = CTB_SORT_BY_NAME; }
    g_free(buffer);
    
    // Fixed Order
    fixed_order = read_gconf_boolean(dir, "fixed_order", client);

    // Icon
    filename_icon = read_gconf_value(dir, "icon", client);
    if (!filename_icon) goto out_error;
    
    // Subtitle
    subtitle = read_gconf_value(dir, "subtitle", client);
    if (!subtitle) goto out_error;
    
    // Path
    path = read_gconf_value(dir, "path", client);
    if (!path) goto out_error;
    
    // Flags
    buffer = read_gconf_value(dir, "flags", client);
    // optional field (added later)
    if (buffer) 
    {
        if (strcmp(buffer, "autoremove") == 0) { mode |= VIEW_AUTOREMOVE; }
    }
    
    // Check if there is already an entry with the same name, only add new ones!!
    // (this protects pre-defined views from being overwritten)
    if (find_view_info(title) == NULL)
    {
        ViewModeInfoExt   details;
        memset(&details, 0x00, sizeof(details));
        details.title            = g_strdup(title      );
        details.number           = g_view_number++;
        details.mode             = mode;
        details.position         = position;
        details.style            = style;
        details.tag_filter       = g_strdup(tag_filter );
        details.order            = order;
        details.fixed_order      = fixed_order;
        details.filename         = g_strdup(filename   );
        details.filename_icon    = g_strdup(filename_icon);
        details.subtitle         = g_strdup(subtitle   );
        details.path             = g_strdup(path       );

        // add to array
        g_array_append_val(g_views_info, details);
    }
    // else already exists
    
out_error:
    g_free(path);
    g_free(subtitle);
    g_free(filename_icon);
    g_free(tag_filter);
    g_free(title);
}


static void add_views_from_gconf()
{
    GSList *types = NULL;
    GConfClient *client = gconf_client_get_default();

    gboolean exists = gconf_client_dir_exists(client, REGKEY_VIEWS_DIR, NULL);
    if (exists == FALSE) {
        ERRORPRINTF("gconf: %s does not exist", REGKEY_VIEWS_DIR);
        goto unref;
    }

    types = gconf_client_all_dirs(client, REGKEY_VIEWS_DIR, NULL);
    if (types == NULL) {
        ERRORPRINTF("gconf: cannot read dir entries of %s", REGKEY_VIEWS_DIR);
        goto unref;
    }

    g_slist_foreach (types, (GFunc)add_view_from_gconf, client);
    g_slist_foreach (types, (GFunc)g_free, NULL);
    g_slist_free(types);
unref:
    g_object_unref(client);
}


static void add_hardcoded_views()
{
    ViewModeInfo      *known_view = NULL;
    ViewModeInfoExt   details;
    int number = 0;

    // fill array with hardcoded filetypes
    for ( known_view = g_viewmodes ; known_view->title ; known_view++, number++ )
    {
        // set details
        memset(&details, 0x00, sizeof(details));
        details.title            = g_strdup(known_view->title      );
        details.number           = number; // number in g_viewmode
        details.mode             = VIEW_ENABLE; // default enabled
        details.position         = known_view->position;
        details.style            = (ViewMode)number; // for pre-defined views style == number
        details.tag_filter       = g_strdup(known_view->tag_filter );
        details.order            = known_view->order;
        details.fixed_order      = known_view->fixed_order;
        details.filename         = known_view->filename;
        details.filename_icon    = known_view->filename_icon;
        details.subtitle         = known_view->subtitle;
        details.path             = known_view->path;

        // add to array
        g_array_append_val(g_views_info, details);
    }
}

// This function is called when a settings is changed with the settings dialog
static void on_gconf_key_changed(GConfClient *client,
                                 guint cnxn_id,
                                 GConfEntry *entry,
                                 gpointer user_data)
{
    const gchar* key = gconf_entry_get_key(entry);
    const gchar* path = REGKEY_VIEWS_DIR;
    gchar name[40];
    int i;
    
    // skip common part
    while (*key && (*key == *path)) {key++; path++;}
    if (*key) key++; // skip '/'
    // copy name of view
    i=0;
    while (*key && (*key != '/')) {name[i++] = *key; key++;}
    name[i] = '\0';

    ViewModeInfoExt *view = view_find_info_with_name(name);
    if (!view) {printf("Invalid!"); return;} // no valid entry?
    const GConfValue* value = gconf_entry_get_value(entry);

    // check which attribute changed
    if (*key) key++; // skip '/'
    if (strcmp(key, "mode") == 0)
    {
        const gchar *str = gconf_value_get_string(value);
        if (strcmp(str, "enabled") == 0) { view->mode |= VIEW_ENABLE; }
        else { view->mode &= ~VIEW_ENABLE; }
    }
    else if (strcmp(key, "position") == 0)
    {
        int position = gconf_value_get_int(value);
        view->position = position;
    }
    printf("\n");
}

void views_prepare_views(void)
{
    // sort them on priority
    g_array_sort (g_views_info, compare_views);
}

void views_init ( void )
{
    g_views_info = g_array_new( TRUE, TRUE, sizeof(ViewModeInfoExt) );
    g_assert(g_views_info);

    add_hardcoded_views();
    add_views_from_gconf();

    // sort them on priority
    g_array_sort (g_views_info, compare_views);
    
    // HACK: adjust number, since that will be used (currently) to determine order
    ViewModeInfoExt *details = (ViewModeInfoExt*) g_views_info->data;

    while ( details && details->title )
    {
        details++;
    }
    int watch = 1;
    if (watch) {
        GConfClient *client = gconf_client_get_default();
        g_assert(client);

        gconf_client_add_dir(client,
                             REGKEY_VIEWS_DIR,
                             GCONF_CLIENT_PRELOAD_NONE,
                             NULL);

        gconf_client_notify_add(client,
                                REGKEY_VIEWS_DIR,
                                on_gconf_key_changed,
                                NULL,
                                NULL,
                                NULL);
    }
}

static ViewModeInfoExt* find_view_info (const gchar *title )
{

    g_assert(g_views_info);

    if (title == NULL)
    {
        title = "";
    }

    ViewModeInfoExt *ret = NULL;
    ViewModeInfoExt *details = (ViewModeInfoExt*) g_views_info->data;
    while ( details && details->title )
    {
        if (strcmp(details->title, title) == 0)
        {
            ret = details;
            break;
        }
        else
        {
            details++;
        }
    }

    return ret;
}
int view_get_mode(const gchar *filename)
{
    int ret = 1; // enable (when nothing specified)
    ViewModeInfoExt* view_info = view_find_info_with_name(filename);
    
    if (view_info) ret = view_info->mode;
    
    return ret; 
}

ViewModeInfoExt* view_find_info_with_name (const gchar *filename )
{

    g_assert(g_views_info);

    if (filename == NULL)
    {
        filename = "";
    }

    ViewModeInfoExt *ret = NULL;
    ViewModeInfoExt *details = (ViewModeInfoExt*) g_views_info->data;
    while ( details && details->title )
    {
        if (details->filename)
        {
            if (strcmp(details->filename, filename) == 0)
            {
                ret = details;
                break;
            }
            else
            {
                details++;
            }
        }
        else
        {
            details++;
        }
    }

    return ret;
}

ViewModeInfoExt* view_find_info_with_number (int number )
{

    g_assert(g_views_info);

    ViewModeInfoExt *ret = NULL;
    ViewModeInfoExt *details = (ViewModeInfoExt*) g_views_info->data;
    while ( details && details->title )
    {
        if (details->number == number)
        {
            ret = details;
            break;
        }
        else
        {
            details++;
        }
    }

    return ret;
}

ViewModeInfoExt* view_find_info_on_index (unsigned int index )
{

    g_assert(g_views_info);

    ViewModeInfoExt *ret = NULL;
    ViewModeInfoExt *details = (ViewModeInfoExt*) g_views_info->data;
    if (index < g_views_info->len)
    {
        ret = &details[index];
    }

    return ret;
}

static gint compare_views(gconstpointer a, gconstpointer b)
{
    ViewModeInfoExt* view_a = ((ViewModeInfoExt*)(a));
    ViewModeInfoExt* view_b = ((ViewModeInfoExt*)(b));
    if (view_a->position > view_b->position)
    {
        return 1;
    }
    else if (view_a->position == view_b->position)
    {
        return 0;
    }
    return -1;
}

