/*
 * File Name: actions.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include <stdio.h>
#include <strings.h>
#include <string.h>

#include "tags.h"
#include "filetypes.h"

static const struct {
    char *tag;
    char *special_dir;
    char *exts[32];
} extension_tag_map[] = {
    { "personal",    "Personal Documents", { NULL } },
    { "help",        "Help",               { NULL } },
    { "book",        "Books",              {"pdf", "epub", "pdb",
#if MACHINE_HAS_ACSM
        "acsm",
#endif
        NULL}},
    { "image",       "Images",             { "jpg", "jpeg", "png", "bmp", "tiff", "gif", NULL } },
    { "news",        "News",               { "np", "inp", NULL } },
    { "note",        "Notes",              { "note", NULL } },
    { NULL, NULL, {} }
};


static const char* get_tag_for_extension(const char* ext) 
{
    int i, j;
    for (i = 0; extension_tag_map[i].tag; i++) 
    {
        for (j = 0; extension_tag_map[i].exts[j]; j++)
        {
            if (strcasecmp(extension_tag_map[i].exts[j], ext) == 0)
            {
                return extension_tag_map[i].tag;
            }
        }
    }
    return NULL;
}


static const char* get_tag_for_dir(const char* dir, const char* basedir) 
{
    char fulldir[256];
    int i;
    for (i=0; extension_tag_map[i].tag; i++) 
    {
        snprintf(fulldir, sizeof(fulldir), "%s/%s", basedir, extension_tag_map[i].special_dir);
        int len = strlen(fulldir);
        if (strncasecmp(fulldir, dir, len) == 0) {
            if (dir[len] == 0 || dir[len] == '/') {
                return extension_tag_map[i].tag;
            }
        }
    }
    return NULL;
}


const gchar* get_tag_for_file(const gchar* filename,
                              const gchar* dir,
                              const char* basedir)
{
    static const char* empty = "";

    // 1) Check if file is in a special folder
    const char *ctag = get_tag_for_dir(dir, basedir);
    if (ctag) return ctag;

    // 2) If not found: Check if file has a specific extension
    const char *ext = g_extension_pointer(filename);
    ctag = get_tag_for_extension(ext);
    if (ctag) return ctag;

    return empty;
}

