/*
 * File Name: lastapps.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2010/2011 Marcel Hendrickx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <gconf/gconf-client.h>
#include <stdlib.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "ctb_log.h"
#include "lastapps.h"
#include "i18n.h"
#include "filetypes.h"
#include "filemodel.h"
#include "ipc.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

// CAREFULL: These values need to match with the values in settings!
// for soring retrieving last appc info from registry
#define KEYS_EREADER                "/apps/er"
#define KEYS_LASTAPPS               KEYS_EREADER "/lastapps" 
#define KEY_LASTAPP_MODE            KEYS_LASTAPPS "/mode"
#define KEY_LASTAPP_CARD            KEYS_LASTAPPS "/cardmode"
#define KEY_LASTAPP_ACTIVE          KEYS_LASTAPPS "/active"
#define KEY_LASTAPP_1               KEYS_LASTAPPS "/1"
#define KEY_LASTAPP_2               KEYS_LASTAPPS "/2"
#define KEY_LASTAPP_3               KEYS_LASTAPPS "/3"
#define KEY_LASTAPP_4               KEYS_LASTAPPS "/4"
#define KEY_LASTAPP_5               KEYS_LASTAPPS "/5"

static const gchar      *REGVAL_LASTAPPS_OFF   = "off";
static const gchar      *REGVAL_LASTAPPS_LAST  = "last";
static const gchar      *REGVAL_LASTAPPS_OPEN  = "open";
static const gchar      *REGVAL_LASTAPPS_ALL   = "all";
static const gchar      *REGVAL_LASTAPPS_APP   = "app";

static const gchar      *REGVAL_LASTAPPS_CARD_NORMAL   = "normal";
static const gchar      *REGVAL_LASTAPPS_CARD_RESTORE  = "restore";

#define KEY_LASTAPP_OFF     0
#define KEY_LASTAPP_LAST    1
#define KEY_LASTAPP_OPEN    2
#define KEY_LASTAPP_ALL     3
#define KEY_LASTAPP_APP     4

#define KEY_LASTAPP_CARD_NORMAL     0
#define KEY_LASTAPP_CARD_RESTORE    1

static int g_lastapp_mode = KEY_LASTAPP_OFF;
static int g_card_mode = KEY_LASTAPP_CARD_NORMAL;

static int    g_active = 0; // no active app
// TODO: make this an array so it can easily be extended
static gchar *g_lastapp1 = NULL;
static gchar *g_lastapp2 = NULL;
static gchar *g_lastapp3 = NULL;
static gchar *g_lastapp4 = NULL;
static gchar *g_lastapp5 = NULL;

//============================================================================
// Local Function Definitions
//============================================================================
static void lastapps_start_open (void);
static void lastapps_start_all (void);
static void get_lastappsmode_from_registry ( int *mode, int *card_mode );
static void get_lastapps_from_registry ( void );
static void clear_lastapps_from_registry(void);


//============================================================================
// Functions Implementation
//============================================================================

// init (called when executing start)
static void lastapps_init(void)
{
    // read mode from registry
    get_lastappsmode_from_registry(&g_lastapp_mode, &g_card_mode);
}

void lastapps_quit(void)
{
    // Release memory from global vars
    if (g_lastapp1) { g_free(g_lastapp1); g_lastapp1 = NULL;}
    if (g_lastapp2) { g_free(g_lastapp2); g_lastapp2 = NULL;}
    if (g_lastapp3) { g_free(g_lastapp3); g_lastapp3 = NULL;}
    if (g_lastapp4) { g_free(g_lastapp4); g_lastapp4 = NULL;}
    if (g_lastapp5) { g_free(g_lastapp5); g_lastapp5 = NULL;}
}

gboolean lastapps_trigger_delay(gpointer data)
{
    // send trigger for delay
    lastapps_start(LASTAPPS_TRIGGER_DELAY);
    
    return FALSE; // stop timer
}

// Start the last docs/apps depending on mode
void lastapps_start (int trigger)
{
    static int first_time = 1;
    static int triggers = 0L;
    
    LOGPRINTF("entry(%d): %d mode: %d cardmode: %d", trigger, first_time, g_lastapp_mode, g_card_mode);
    
    // update trigger and check if function should continue
    triggers |= trigger;
    
    if (triggers == LASTAPPS_TRIGGER_START_DELAY)
    {
        // TODO: make timeout configurable
        g_timeout_add(1000, lastapps_trigger_delay, NULL);
    }
    
    if (triggers != LASTAPPS_TRIGGER_ALL) return; // early exit
    // Check if this is the first time after start-up
    if (first_time == 1)
    {
        // get data from registry
        lastapps_init();
        
        // Do the start-up behaviour
        first_time = 0;
        switch (g_lastapp_mode)
        {
            case KEY_LASTAPP_OFF:
            case KEY_LASTAPP_LAST:
            case KEY_LASTAPP_APP:
                // No action for these modes (yet)
                break;
            case KEY_LASTAPP_OPEN:
                lastapps_start_open();
                break;
            case KEY_LASTAPP_ALL:
                lastapps_start_all();
                break;
        }
    }
    else if (g_card_mode == KEY_LASTAPP_CARD_RESTORE)
    {
        // A (new) card has been (re-)mounted and documents need to be restored
        // Or the usb has been unmounted
        lastapps_start_all();
    }
}

//============================================================================
// Local Functions Implementation
//============================================================================

// On entry filepath should look like (within the []):
// [xournal /media/mmcblk0p1/Programs/2009-08-19-Note-11-22.xoj]
// [uds /media/mmcblk0p1/E1 'Factors and measures of business process modelling model building through a multiple case study', Bandara, W., Gable, G.G. & Rosemann, M.,.pdf]
// or [xournal] for app started without document
// or [none] indicating nothing has to be done
static int lastapps_open_one_file(char *filepath)
{
    gchar            *filename       = NULL;
    gchar            *dirname        = NULL;
    const gchar      *viewer_cmd     = NULL;
    gchar            *cmd_line       = NULL;
    gchar            *p              = NULL; // temp local var
    int              ret             = ER_OK;

    LOGPRINTF("lastapps_open_one_file:entry: [%s]", filepath);

    // 'none' indicates that at that index there is no valid app
    if (filepath && (strcmp(filepath, "none") != 0))
    {
        // Check if it is a document
        // Assume that command line is constructed out of app<space>document
        p = strchr(filepath, ' ');
        if (p != NULL)
        {
            // Drop the application part, since we can retrieve it form the extension
            // of the file (if started from the CTB!!)
            filepath = p;
            filepath++; // from here filepath points to the 'cleanup' path of the document to start
            if (filepath)
            {
                // extract directory and filename
                dirname = g_path_get_dirname(filepath);
                // TODO: check if path refers to file and not to directory
                filename = g_path_get_basename(filepath);
                
                // find program to start using the extension of the file
                p = g_strrstr(filename, "."); // first . from end
                if (p)
                {
                    p++; // skip '.'
                    // Get the viewer needed for this extension
                    viewer_cmd = get_viewer_from_file_extension(p);
                    
                    // construct the command to execute with viewer-command and file-path
                    cmd_line = g_strdup_printf("%s \"%s\"", viewer_cmd, filepath);

                    WARNPRINTF("entry: cmd_line [%s] dirname [%s] filename [%s]",
                                          cmd_line,     dirname,     filename);
                    
                    // Now start the command using sysd so it will also be shown in the popupmemu
                    gchar *ret_message = NULL;
                    gint errcode = ipc_sys_start_task(cmd_line, dirname, filename, NULL, &ret_message);
                    if ( errcode > 0 )
                    {
                        ERRORPRINTF("cannot launch viewer, cmd_line [%s], errcode %d [%s]", cmd_line, errcode, ret_message);
                    }
                    else
                    {
                        LOGPRINTF("ipc_sys_start_task=[%d] = OK", errcode);
                    }
                    
                    // cleanup 'our' memory
                    g_free(ret_message);
                    g_free(cmd_line);
                }
                else
                {
                    // No extension
                    ret = ER_FAIL;
                    WARNPRINTF("The file %s has no extension\n", filename);
                }
                
                g_free(dirname);
                g_free(filename);
            }
            else
            {
                // only an app + space, no document!! (not expected, so ignored)
                ret = ER_FAIL;
            }
        }
        else
        {
            // It is an application without arguments, so we only have the name of the app
            // and not the full path
            // TODO: implement!!
            // How: ? make exception for system apps
            //        search in file-extension list on the app for other registered apps
            
            // CONSIDERATION: It might seem strange to start an app without a document, but
            //                the user probably opened a document last time, which will
            //                probably be in the 'most recent used files' of the application
            ret = ER_FAIL;
        }
    }
    else
    {
        ret = ER_FAIL;
    }
    
    return ret;
}

// Open the document that was active at shutdown
static void lastapps_start_open (void)
{
    gchar *filepath = NULL;
    
    LOGPRINTF("lastapps_start_open:entry");

    // Get the settings from the Registry (stored in global vars)
    get_lastapps_from_registry();
    
    // Clear registry to avoid that we get into an endless cycle when the document
    // crashes the DR and it restarts.
    clear_lastapps_from_registry();
    
    // Only if there is an active app/document for this mode
    if (g_active > 0)
    {
        // get correct file
        if (g_active == 1)      filepath = g_lastapp1;
        else if (g_active == 2) filepath = g_lastapp2;
        else if (g_active == 3) filepath = g_lastapp3;
        else if (g_active == 4) filepath = g_lastapp4;
        else if (g_active == 5) filepath = g_lastapp5;
        
        // now open the app
        lastapps_open_one_file(filepath);
    }
    else
    {
        WARNPRINTF("No Active file at shutdown");
    }
}

// Open all documents that were open at shutdown
static void lastapps_start_all (void)
{
    gchar *filepath = NULL;
    
    LOGPRINTF("lastapps_start_all:entry");

    // Get the settings from the Registry
    get_lastapps_from_registry();
    // Clear registry to avoid that we get into an endless cycle when the document
    // crashes the DR and it restarts.
    clear_lastapps_from_registry();
    
    // First do the non-active documents
    if (g_active != 1) lastapps_open_one_file(g_lastapp1);
    if (g_active != 2) lastapps_open_one_file(g_lastapp2);
    if (g_active != 3) lastapps_open_one_file(g_lastapp3);
    if (g_active != 4) lastapps_open_one_file(g_lastapp4);
    if (g_active != 5) lastapps_open_one_file(g_lastapp5);
    
    // Now the active document!
    if (g_active == 1)      filepath = g_lastapp1;
    else if (g_active == 2) filepath = g_lastapp2;
    else if (g_active == 3) filepath = g_lastapp3;
    else if (g_active == 4) filepath = g_lastapp4;
    else if (g_active == 5) filepath = g_lastapp5;
    
    // now open the app
    lastapps_open_one_file(filepath);
}

// registry access: get recent files mode
static void get_lastappsmode_from_registry ( int *mode, int *card_mode )
{
    GConfClient         *client   = NULL;
    gchar               *val      = NULL;
    
    LOGPRINTF("get_lastappsmode_from_registry:entry");

    // init to off in case there are problems retrieving the correct values from
    // the registry
    *mode = KEY_LASTAPP_OFF;
    *card_mode = KEY_LASTAPP_CARD_NORMAL;
    
    // get registry value
    client = gconf_client_get_default();
    val = gconf_client_get_string(client, KEY_LASTAPP_MODE, NULL);
    
    // convert string value to viewtype
    if (val)
    {
        WARNPRINTF("Registry:%s", val);
        
        if ( strcmp(val, REGVAL_LASTAPPS_OFF) == 0 )
        {
            // Do not use the last apps option
            *mode = KEY_LASTAPP_OFF;
        }
        else if ( strcmp(val, REGVAL_LASTAPPS_LAST) == 0 )
        {
            // Start last started app
            *mode = KEY_LASTAPP_LAST;
        }
        else if ( strcmp(val, REGVAL_LASTAPPS_OPEN) == 0 )
        {
            // Start last open/active app
            *mode = KEY_LASTAPP_OPEN;
        }
        else if ( strcmp(val, REGVAL_LASTAPPS_ALL) == 0 )
        {
            // Start all open apps
            *mode = KEY_LASTAPP_ALL;
        }
        else if ( strcmp(val, REGVAL_LASTAPPS_APP) == 0 )
        {
            // Start selected app
            *mode = KEY_LASTAPP_APP;
        }
        else 
        {
            WARNPRINTF("Unknown mode:%s", val);
        }
        g_free(val);
    } 
    else 
    {
        WARNPRINTF("error fetching GConf key %s", KEY_LASTAPP_MODE);
    }

    // get registry value
    client = gconf_client_get_default();
    val = gconf_client_get_string(client, KEY_LASTAPP_CARD, NULL);
    
    g_object_unref(client);

    if (val)
    {
        WARNPRINTF("Registry:%s", val);
        
        if ( strcmp(val, REGVAL_LASTAPPS_CARD_NORMAL) == 0 )
        {
            // Use the normal behaviour for card removal/insertion
            *card_mode = KEY_LASTAPP_CARD_NORMAL;
        }
        else if ( strcmp(val, REGVAL_LASTAPPS_CARD_RESTORE) == 0 )
        {
            // Save and Restore files before/after card removal
            *card_mode = KEY_LASTAPP_CARD_RESTORE;
        }
        else 
        {
            WARNPRINTF("Unknown mode:%s", val);
        }
        g_free(val);
    } 
    else 
    {
        WARNPRINTF("error fetching GConf key %s", KEY_LASTAPP_CARD);
    }
}

static void get_lastapps_from_registry(void)
{
    GConfClient         *client   = NULL;
    
    LOGPRINTF("get_lastapps_from_registry:entry");

    // Release old-memory from global vars
    if (g_lastapp1) { g_free(g_lastapp1); g_lastapp1 = NULL;}
    if (g_lastapp2) { g_free(g_lastapp2); g_lastapp2 = NULL;}
    if (g_lastapp3) { g_free(g_lastapp3); g_lastapp3 = NULL;}
    if (g_lastapp4) { g_free(g_lastapp4); g_lastapp4 = NULL;}
    if (g_lastapp5) { g_free(g_lastapp5); g_lastapp5 = NULL;}

    // get registry value
    client = gconf_client_get_default();
    g_active   = gconf_client_get_int(client, KEY_LASTAPP_ACTIVE, NULL);
    g_lastapp1 = gconf_client_get_string(client, KEY_LASTAPP_1, NULL);
    g_lastapp2 = gconf_client_get_string(client, KEY_LASTAPP_2, NULL);
    g_lastapp3 = gconf_client_get_string(client, KEY_LASTAPP_3, NULL);
    g_lastapp4 = gconf_client_get_string(client, KEY_LASTAPP_4, NULL);
    g_lastapp5 = gconf_client_get_string(client, KEY_LASTAPP_5, NULL);
    // TODO: error handling
    
    LOGPRINTF("%d", g_active);
    LOGPRINTF("%s", g_lastapp1);
    LOGPRINTF("%s", g_lastapp2);
    LOGPRINTF("%s", g_lastapp3);
    LOGPRINTF("%s", g_lastapp4);
    LOGPRINTF("%s", g_lastapp5);

    g_object_unref(client);

}

// clear active and set apps to none
static void clear_lastapps_from_registry(void)
{
    GConfClient         *client   = NULL;

    LOGPRINTF("entry");

    client = gconf_client_get_default();
    
    // TODO: handle errors
    gconf_client_set_int(client, KEY_LASTAPP_ACTIVE, 0, NULL);
    gconf_client_set_string(client, KEY_LASTAPP_1, "none", NULL);
    gconf_client_set_string(client, KEY_LASTAPP_2, "none", NULL);
    gconf_client_set_string(client, KEY_LASTAPP_3, "none", NULL);
    gconf_client_set_string(client, KEY_LASTAPP_4, "none", NULL);
    gconf_client_set_string(client, KEY_LASTAPP_5, "none", NULL);
    
    g_object_unref(client);
}

