#ifndef __FILEMODEL_H__
#define __FILEMODEL_H__

/**
 * File Name  : filemodel.h
 *
 * Description: Content browser data for file model
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <gtk/gtk.h>

#include <libermetadb/ermetadb.h>

G_BEGIN_DECLS

// special item filenames
#define SPECIAL_ITEM_NAME "special"
#define SPECIAL_BOOKS "books"
#define SPECIAL_IMAGES "images"
#define SPECIAL_NEWS "news"
#define SPECIAL_HELP "help"
#define SPECIAL_PERSONAL "personal"
#define SPECIAL_DIR "dir"
#define SPECIAL_SHORTCUTS "shortcuts"
#define SPECIAL_NOTES "notes"
#define SPECIAL_SETTINGS "settings"
#define SPECIAL_SEARCH "search"
#define SPECIAL_RECENT "recent"
#define SPECIAL_UPDIR ".."

// special directory locations
#define DIR_DESKTOP_TEMPLATE        SYSCONFDIR "/desktop"
#define DIR_DESKTOP_INTERNAL        "/var/tmp/"         // NOT really used
#define DIR_STORAGE_MNT             "/media"                        // mountpoint for removable storage devices
#define DIR_LIBRARY                 DIR_STORAGE_MNT "/mmcblk0p1"    // mountpoint for internal SD card

// columns in file model
typedef enum
        {
            MODCOL_FILENAME = 0,
            MODCOL_TITLE,
            MODCOL_FILETYPE,
            MODCOL_FILETYPE_DISPLAY,
            MODCOL_DIRECTORY_PATH,
            MODCOL_IS_DIRECTORY,
            MODCOL_FILESIZE,
            MODCOL_FILEDATE,
            MODCOL_SUBTITLE,
            MODCOL_THUMBNAIL,
            MODCOL_TOGGLED,
            N_FILEMODEL_COLUMNS
        } filemodel_column_t;

#define FILEMODEL_COLUMN_TYPES                                 \
            G_TYPE_STRING,      /* MODCOL_FILENAME         */  \
            G_TYPE_STRING,      /* MODCOL_TITLE */  \
            G_TYPE_STRING,      /* MODCOL_FILETYPE         */  \
            G_TYPE_STRING,      /* MODCOL_FILETYPE_DISPLAY */  \
            G_TYPE_STRING,      /* MODCOL_DIRECTORY_PATH   */  \
            G_TYPE_BOOLEAN,     /* MODCOL_IS_DIRECTORY     */  \
            G_TYPE_STRING,      /* MODCOL_FILESIZE         */  \
            G_TYPE_STRING,      /* MODCOL_FILEDATE         */  \
            G_TYPE_STRING,      /* MODCOL_SUBTITLE           */  \
            GDK_TYPE_PIXBUF,    /* MODCOL_THUMBNAIL        */  \
            G_TYPE_INT,         /* MODCOL_TOGGLED          */  \
            NULL  /* end of list */

// thumbnail size
typedef enum
        {
            MODTHUMB_MINI = 0,
            MODTHUMB_SMALL,
            MODTHUMB_MEDIUM,
            MODTHUMB_LARGE,
            N_FILEMODEL_THUMB_SIZES
        } filemodel_thumbsize_t;

// entry in filelist delivered by filemodel_get_filenames_xx()
typedef struct
        {
            GString     *filename;
            GString     *filename_display;
            GString     *filetype;
            GString     *directory_path;
            gboolean    is_directory;
        } filelist_entry_t;

typedef enum
        {
            BROWSE_MODE = 0,
            DELETE_MODE,
        } ctb_viewmodes_t;

// sorting order in content browser views
typedef enum
        {
            CTB_SORT_BY_NAME = 0,
            CTB_SORT_BY_TYPE,
            CTB_SORT_BY_SIZE,
            CTB_SORT_BY_DATE_ADDED,
            CTB_SORT_BY_DATE_READ,
            CTB_SORT_BY_AUTHOR,
            N_CTB_SORT_ORDER
        } ctb_sort_order_t;

typedef enum
        {
            DESKTOP_VIEW = 0,
            SETTINGS_VIEW,
            BOOKS_VIEW,
            NEWS_VIEW,
            IMAGES_VIEW,
            PERSONAL_VIEW,
            DIR_VIEW,
            SHORTCUT_VIEW,
            NOTES_VIEW,
            HELP_VIEW,
            RECENT_VIEW,
            SEARCH_VIEW,
            LASTREAD_VIEW,
            STORE_VIEW,
        } ViewMode;

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//============================================================================
// Public Functions
//============================================================================


// creates initial desktop etc
void filemodel_init();

void filemodel_quit();


/**---------------------------------------------------------------------------
 *
 * Name :  get_filemodel
 *
 * @brief  Get a pointer to the file model
 *
 * @param  none
 *
 * @return GtkTreeModel interface on the filestore object
 *
 *--------------------------------------------------------------------------*/
GtkTreeModel* get_filemodel();


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_get_database
 *
 * @brief  Get the currently open metadata database
 *
 * @param  none
 *
 * @return Currently open metadata database,
 *         or NULL if no database open
 *
 *--------------------------------------------------------------------------*/
erMetadb filemodel_get_database();


// DISPLAY ATTRIBUTES

gboolean filemodel_set_sortorder(ctb_sort_order_t sort_order,
                                 const gchar* cursor_item,
                                 gboolean updateScreen);

// filters all db results to be of indicated type
void filemodel_set_viewmode2(ViewMode newmode);
ViewMode filemodel_get_viewmode2(void);

void filemodel_set_search_filter(const gchar* search_string);

const gchar* filemodel_get_title();

const gchar* filemodel_get_subtitle();

const gchar* filemodel_get_menu_content();

/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_set_viewsize
 *
 * @brief  Set number of items in view, same number must be loaded in model
 *
 * @param  [in]  n_items - number of items to be loaded in model (> 0)
 *         [in]  updateScreen - whether to load data in gui model
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
gboolean filemodel_set_viewsize(const int n_items, gboolean updateScreen);

// if updateScreen is False, caller has to make other calls that update the screen
void filemodel_set_viewmode(ctb_viewmodes_t newmode, gboolean updateScreen);


/**---------------------------------------------------------------------------
 *
 * Name :  filemodel_set_thumbsize
 *
 * @brief  Set thumbnail to be loaded in model
 *
 * @param  [in]  thumb_size - thumbnail size to be loaded in model
 * @param  [in]  reload - whether to reload data from metadb.
 *          If more calls follow that also cause a reload, use false here.
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void filemodel_set_thumbsize(const filemodel_thumbsize_t thumb_size, gboolean reload);


// NAVIGATION WITHIN DIRECTORY
gboolean filemodel_has_prev_page();

gboolean filemodel_has_next_page();

void filemodel_page_previous();

void filemodel_page_next();

void filemodel_update_pagecounter();
/*
 * Name: filemodel_scroll_to_letter
 *
 * @param  [in]  letter  initial of file/dir to jump to
 * @param  [in]  jump_to_dir  whether to prefer a dir or not
 * @parem [out]  same_page whether index is on same page
 *
 * @return display index of letter (-1 if not found)
 * The column that's being examined depends on the current
 * sort order (Name/Author). */
int filemodel_scroll_to_letter(gchar letter, gboolean jump_to_dir, gboolean* same_page);

void filemodel_scroll_to_filename(const gchar *filename);

// returns index for filename in currently displayed items (page)
// 0 is first item
int filemodel_get_display_index(const gchar *filename);


// NAVIGATION BETWEEN DIRECTORIES
// change model to desktop
int filemodel_chdir_desktop();

// enter a directory
int filemodel_chdir_down(const gchar *dir, const gchar *last_item);

// goes back a dir. Returns cursor_item name of parent dir
gchar *filemodel_chdir_up();

// reloads data from metadata.dd. Does not load items into GUI
// offset stays the same. Returns whether data has been reloaded
gboolean filemodel_resync(gboolean force_reload);

gboolean filemodel_current_is_desktop();

const gchar* filemodel_get_current_dir();

// Returns the current directory depth. Desktop has depth 0.
int filemodel_get_dir_depth();

// returns first letter of first visible entry. -1 if no items
// or if first item has no letter (e.g. no author)
int filemodel_get_first_alpha_index();

// returns an array of FILEMODEL_NUM_ALPHA long. The array indicates
// whether entries are active or not. The returned list does not need
// to be freed.
#define ALPHA_HAS_DIR 0x01
#define ALPHA_HAS_FILE 0x02
#define FILEMODEL_NUM_ALPHA 27
const gchar* filemodel_get_alpha_list();

gboolean filemodel_show_alphabar();

gboolean filemodel_has_searchbutton();

// SELECTION stuff
// toggled entry for deletion. Index is offset in visible window.
void filemodel_toggle_entry(int index, int state, GtkTreeIter *iter);

// delete all the toggled entries. Also resyncs with metadb
void filemodel_delete_toggled();

// return the number of toggled entries
int filemodel_num_toggled();

// set the text that will be shown on first item in Delete mode
void filemodel_set_delete_text(const gchar* text);

//#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
void filemodel_update_last_read(const filelist_entry_t *fileinfo);
//#endif

// FILELIST_ENTRY methods
filelist_entry_t* iter_to_entry(GtkTreeModel *model, GtkTreeIter *iter);

filelist_entry_t* filelist_entry_new();

void filelist_entry_free(filelist_entry_t *thiz);

filelist_entry_t* filelist_entry_copy(const filelist_entry_t *src);

gboolean filemodel_window_is_on_top();

void filemodel_set_window_is_on_top(gboolean is_on_top );

#if (TIMING_ON)
void start_duration_timer();
void stop_duration_timer();
#endif

int filemodel_is_user_view(const char* filename);

// view mode functions
// Convert a user viewmode to one of the base viewmodes
ViewMode       viewmode_get_style(int viewmode);
int            viewmode_get_mode(int viewmode);
const char*    viewmode_get_tag_filter(int viewmode);


G_END_DECLS

#endif

