/*
 * File Name: tasks.c
 */

/*
 * This file is part of sysd.
 *
 * sysd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * sysd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <glib.h>
#include <hal/libhal.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "busy.h"
#include "devicelist.h"
#include "hal.h"
#include "ipc.h"
#include "process.h"
#include "system.h"
#include "tasks.h"
#include "xwindow.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct
{
    proc_t          *proc;
    Window          window;
    gchar           *application;
    gchar           *document;
    gchar           *label;
    gchar           *image;
} task_t;

typedef struct
{
    task_t          *task;
    eripc_context_t *context;
    gchar           *message_id; 
} request_t;


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// time to wait for application started
static const gint TIMEOUT_CHILD_NEW         = 30;
static const gint TIMEOUT_CHILD_CLOSE       = 5;
static const gint TIMEOUT_CHECK_RESTART_UDS = 5;

static const gint TIMER_CHECK_WINDOW_MS     = 500;
static const guint MAX_TASKS                 = 4;


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

#define FOLDER_XID -1

static GSList           *g_tasklist = NULL;
static gboolean         g_forced_close = FALSE;
static gint             g_child_terminate_source = 0;
static gint             g_check_child_window_source = 0;
static gint             g_task_pending_source = 0;


//============================================================================
// Local Function Definitions
//============================================================================

static gboolean child_new               (const char *command,
                                         const char *working_dir,
                                         const char *application,
                                         const char *document,
                                         const char *label, 
                                         const char *image, 
                                         eripc_context_t *context,
                                         const char *message_id);

static gboolean child_activate          (task_t *task);
static gboolean child_close             (task_t *task);

static gboolean child_window_open       (task_t *task, request_t *request);
static gboolean child_window_close      (task_t *task, request_t *request);

static void     on_window_open_callback (eripc_context_t *context,
                                         const eripc_event_info_t *info,
                                         request_t *request);

static void     on_window_close_callback(eripc_context_t *context,
                                         const eripc_event_info_t *info,
                                         request_t *request);

static gboolean on_check_child_window   (gpointer data);
static gboolean on_child_new_timeout    (gpointer data);
static gboolean on_child_term_timeout   (gpointer data);

/*
static gboolean on_check_restart_uds    (gpointer data);
static gboolean is_uds_in_use           (void);
*/

static task_t   *tasklist_add           (proc_t *proc,
                                         const char *application, 
                                         const char *document, 
                                         const char *label, 
                                         const char *image, 
                                         Window window);

static gboolean tasklist_set_first      (task_t *task);
static gboolean tasklist_remove_pid     (GPid pid);
static gboolean tasklist_remove_window  (Window window);
static task_t   *find_valid_task        (task_t *task);
static void     free_task               (task_t *task);

static task_t *get_task                 (const char *application, const char *document);
static task_t *get_task_by_xid          (Window xid);

static gboolean parse_commandline       (const gchar *command_line, 
                                         gchar **application, 
                                         gchar **document, 
                                         gchar **arguments);

static request_t *new_request           (task_t *task,
                                         eripc_context_t *context, 
                                         const char *message_id);

static void     free_request            (request_t *request);


//============================================================================
// Functions Implementation
//============================================================================

gboolean task_start(const char *command_line, 
                    const char *working_dir, 
                    const char *label, 
                    const char *image, 
                    eripc_context_t *context, 
                    const char *message_id)
{
    LOGPRINTF("entry: `%s`", command_line);
    
    gboolean retval       = FALSE;
    gchar    *application = NULL;
    gchar    *document    = NULL;

    parse_commandline(command_line, &application, &document, NULL);

    LOGPRINTF("app [%s] wd [%s] doc [%s] label [%s] image [%s]", application, working_dir, document, label, image);
    
    task_t *task = get_task(application, document);
    if (task)
    {
        LOGPRINTF("task exists, reopen or activate");
        // task already exists, reopen or activate
        //
        if (task->proc && task->proc->ipc_service && task->proc->is_multidoc)
        {
            request_t *request = new_request(task, context, message_id);
            retval = child_window_open(task, request);
        }
        else
        {
            retval = child_activate(task);
            if (context && message_id)
            {
                // return result to caller
                ipc_send_reply_task_start(context, message_id, 0, NULL);
            }
        }
    }
    else
    {
        proc_t *proc = process_get_by_name(application);
        if (proc)
        {
            // process is running but no task yet, create new
            //
            if (proc->ipc_service && proc->is_multidoc)
            {
                LOGPRINTF("task %s, %s not found, create it...", application, document);
                task_t *task = tasklist_add(proc, application, document, label, image, 0);
                request_t *request = new_request(task, context, message_id);
                retval = child_window_open(task, request);
            }
            else
            {
                if (proc->is_multidoc) {
                    LOGPRINTF("%s, %s is currently running but is not a task, ignored.", application, document);
                    // process running, singledoc and no task found for it
                    // this is either a window-less process or a system process
                    // in either case; we ignore this start command.
                } else {
                    // TODO REFACTOR (common code)
                    // process not running, create new
                    LOGPRINTF("process %s running, but not for current doc, start new", application);
                    retval = child_new(command_line, working_dir, application, document, label, image, context, message_id);
                }
            }
        }
        else
        {
            // process not running, create new
            //
            LOGPRINTF("process %s, not currently running start new", application);
            retval = child_new(command_line, working_dir, application, document, label, image, context, message_id);
        }
    }

    // free allocated memory
    g_free(application);
    g_free(document);

    LOGPRINTF("leave: retval %d", retval);
    
    return retval;
}


gboolean task_stop(const char *command_line,
                   eripc_context_t *context, 
                   const char *message_id)
{
    LOGPRINTF("entry: `%s`", command_line);

    gboolean retval       = FALSE;
    gchar    *application = NULL;
    gchar    *document    = NULL;
    
    parse_commandline(command_line, &application, &document, NULL);
                     
    task_t *task = get_task(application, document);
    if (task)
    {
        if (task->proc && task->proc->ipc_service && task->proc->is_multidoc)
        {
            request_t *request = new_request(task, context, message_id);
            LOGPRINTF("found multidoc window, close child window");
            retval = child_window_close(task, request);
        }
        else
        {
            LOGPRINTF("found singledoc window, close child");
            retval = child_close(task);
            ipc_send_reply(context, message_id, retval);
        }
    }

    // free allocated memory
    g_free(application);
    g_free(document);
    
    LOGPRINTF("leave: retval %d", retval);
    
    return retval;
}


gboolean task_rename(int xid, const char *new_document, const char* new_label)
{
    LOGPRINTF("[%d] %s '%s'", xid, new_document, new_label);
    task_t *task = get_task_by_xid(xid);
    if (task == NULL) {
        ERRORPRINTF("unknown task: xid=%d", xid);
        return FALSE;
    }
    g_free(task->document);
    task->document = g_strdup(new_document);
    g_free(task->label);
    task->label = g_strdup(new_label);
    ipc_menu_rename_task(xid, task->label);
    return TRUE;
}


void task_activate_by_xid(int xid)
{
    LOGPRINTF("entry xid=%d", xid);
    if (xid == FOLDER_XID) 
    {
        process_activate_ctb();
        return; 
    }
    task_t *task = get_task_by_xid(xid);
    if (task)
    {
        LOGPRINTF("found task, activate window %d", (gint) task->window);
        child_activate(task);
    } 
    else 
    {
        WARNPRINTF("task not found, xid=%d", xid);
    }
}


gboolean task_activate(const char *command_line)
{
    LOGPRINTF("entry: `%s`", command_line);

    gboolean retval       = FALSE;
    gchar    *application = NULL;
    gchar    *document    = NULL;
    
    parse_commandline(command_line, &application, &document, NULL);
                     
    task_t *task = get_task(application, document);
    if (task)
    {
        LOGPRINTF("found task, activate window %d", (gint) task->window);
        retval = child_activate(task);
    }
    else 
    {
        WARNPRINTF("task not found cmdline='%s'", command_line);
    }

    // free allocated memory
    g_free(application);
    g_free(document);
    
    LOGPRINTF("leave: retval %d", retval);
    
    return retval;
}


#if (TESTING_ON)
void print_task_list()
{
    GSList *task_ptr = g_tasklist;
    
    int i = 0;
    printf("%s() TASKS\n", __func__);
    while (task_ptr)
    {
        task_t *task = (task_t *) task_ptr->data;
        int pid = (task->proc) ? task->proc->pid : 0;
        printf("  [%d] proc_pid=%d  window=%d  app=%s  doc=%s\n", i, pid,
                (guint)task->window, task->application, task->document);

        task_ptr = task_ptr->next;
        i++;
    }
}
#endif


const gchar *task_service_by_window(Window xid)
{
    LOGPRINTF("entry: %d", (gint) xid);

    task_t *task = get_task_by_xid(xid);
    if (task && task->proc && task->proc->ipc_service) 
    {
        return task->proc->ipc_service;
    }
    return NULL;
}


gboolean task_add(const char *application, 
                  const char *document, 
                  const char *label, 
                  const char *image, 
                  const char *ipc_service, 
                  gint        pid, 
                  Window      xid)
{
    LOGPRINTF("entry");
    
    gboolean retval = FALSE;
    
    task_t *task = get_task(application, document);
    proc_t *proc = process_get_by_pid(pid);
    if (proc && !task)
    {
        LOGPRINTF("task %s, %s, not found, add it...", application, document);
        task = tasklist_add(proc, application, document, label, image, xid);
        if (task)
        {
            retval = TRUE;
        }
    }

    return retval;
}


gboolean task_cleanup_pid(GPid pid)
{
    LOGPRINTF("entry: forced %d", g_forced_close);

    gboolean result = tasklist_remove_pid(pid);

    if (result && !g_forced_close)
    {
#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
        // automatically return to application that was on top in the stack
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        // return to folder in content browser
        process_activate_ctb();
#endif
    }
    
    g_forced_close = FALSE;
    
    return result;
}


gboolean task_cleanup_window(Window xid)
{
    LOGPRINTF("entry: xid %d, forced %d", (int) xid, g_forced_close);
 
    gboolean result = tasklist_remove_window(xid);

    if (result && !g_forced_close)
    {
#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
        // automatically return to application that was on top in the stack
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        // return to folder in content browser
        process_activate_ctb();
#endif
    }
    
    g_forced_close = FALSE;
    
    return result;
}


gboolean task_startup_completed(proc_t *proc, Window window)
{
    LOGPRINTF("entry");
    
    GSList *task_ptr = g_tasklist;
    while (task_ptr)
    {
        task_t *task = (task_t *) task_ptr->data;

        if (task->proc == proc)
        {
            if (window != 0)
            {
                task->window = window;
            	LOGPRINTF("assigned window %d to %s, %s", (gint) task->window, task->application, task->document);
            }

            // add to task manager
            LOGPRINTF("task_startup_completed: %s, %s", task->application, task->document);
            ipc_menu_add_task((gint)task->window, task->label, task->application, task->document);

            return TRUE;
        }
        task_ptr = task_ptr->next;
    }
    return FALSE;
}


char* testing_task_get_list()
{
#if (TESTING_ON)
    char buffer[1000];
    char* cp = buffer;
    GSList *iter = g_tasklist;
    while (iter)
    {
        task_t* task = (task_t*) iter->data;
        strcpy(cp, task->application);
        cp += strlen(task->application);
        *cp++ = '|';
        strcpy(cp, task->document);
        cp += strlen(task->document);
        *cp++ = '\n';

        iter = g_slist_next(iter);
    }
    *cp = 0;
    char* result = g_strdup(buffer);
    return result;
#endif
    return g_strdup("no debug");
}


//============================================================================
// Local Function Implementation
//============================================================================

static gboolean child_new(const char *command,
                          const char *working_dir,
                          const char *application,
                          const char *document,
                          const char *label, 
                          const char *image, 
                          eripc_context_t *context,
                          const char *message_id)
{
    LOGPRINTF("entry");

    gboolean retval = FALSE;

    if (   g_check_child_window_source == 0
        && g_task_pending_source       == 0 )
    {
     
        if (working_dir && working_dir[0]=='\0')
        {
            working_dir = NULL;
        }
        
        proc_t *proc = process_start(command, working_dir, NULL, NULL, PS_WAIT_STARTED);
        if (proc != NULL)
        {
            task_t    *task = tasklist_add(proc, application, document, label, image, 0);
            request_t *request = new_request(task, context, message_id);

            // show busy indication
            busy_add_foreground(0, BUSY_DIALOG_DELAYED, NULL);

            LOGPRINTF("pending task adding, wait for window id");
           
            // periodically check window list
            g_check_child_window_source = g_timeout_add(TIMER_CHECK_WINDOW_MS, on_check_child_window, request);            
            
            // set maximum time to wait for new window
            g_task_pending_source = g_timeout_add_seconds(TIMEOUT_CHILD_NEW, on_child_new_timeout, request);
            
            retval = TRUE;
        }
        else 
        {
            LOGPRINTF("error starting, return failure");
            // return FAILED immediately        
        }
    }
    else
    {
        ERRORPRINTF("still waiting for another task to be started, refusing task [%s]", label);
    }
    
    return retval;
}


static gboolean child_window_open(task_t     *task,
                                  request_t  *request)
{
    LOGPRINTF("entry");
    
    gboolean retval = FALSE;
    
    if (task->proc && task->proc->ipc_service)
    {
        LOGPRINTF("pending task added, wait for on_window_open_callback");
        retval = ipc_send_open(task->proc->ipc_service, task->document, on_window_open_callback, request);
    }

    return retval;
}
    

static request_t *new_request(task_t *task,
                              eripc_context_t *context, 
                              const char *message_id)
{
    LOGPRINTF("entry");
    
    request_t *request  = g_new0(request_t, 1);
    if (!request)
    {
        ERRORPRINTF("mem alloc failed");
        return FALSE;
    }
    request->task       = task;
    request->context    = context;
    request->message_id = g_strdup(message_id);
    
    return request;
}


static void free_request(request_t *request)
{
    LOGPRINTF("entry");
    
    g_free(request->message_id);
    g_free(request);
    request = NULL;
}


static gboolean child_window_close(task_t    *task,
                                   request_t *request)
{
    LOGPRINTF("entry");
    
    gboolean retval = FALSE;

    if (task->proc && task->proc->ipc_service)
    {
        LOGPRINTF("wait for on_window_close_callback");
        retval = ipc_send_close(task->proc->ipc_service, task->document, on_window_close_callback, request);
    }
    
    return retval;
}


static gboolean child_activate(task_t *task)
{
    LOGPRINTF("entry");
    
    // set task to head of list
    tasklist_set_first(task);
    
    // activate X11 window 
    window_activate(task->window);

    return TRUE;
}


static gboolean child_close(task_t *task)
{
    LOGPRINTF("entry");
    
    if (task->proc)
    {
        // terminate child
        kill(task->proc->pid, SIGTERM);
    }
    
    // set maximum time to wait for child termination
    g_child_terminate_source = g_timeout_add_seconds(TIMEOUT_CHILD_CLOSE, on_child_term_timeout, task);
    
    return TRUE;
}


static gboolean on_child_new_timeout(gpointer data)
{
    LOGPRINTF("entry");
    
    request_t *request = data;
    
    g_return_val_if_fail(request != NULL, FALSE);
    
    if (g_check_child_window_source)
    {
        g_source_remove(g_check_child_window_source);
        g_check_child_window_source = 0;
    }
    
    if (g_task_pending_source)
    {
        g_source_remove(g_task_pending_source);
        g_task_pending_source = 0;
    }    
    
    task_t *task = find_valid_task(request->task);
    if (task != NULL)
    {
        tasklist_remove_pid(request->task->proc->pid);
    }   
    
    // return result to caller
    if (request->context && request->message_id)
    {
        ipc_send_reply_task_start(request->context, request->message_id, 2, NULL);
    }
    free_request(request);
    
    // remove busy indication
    busy_remove_foreground(0);

    // stop and destroy this timer
    return FALSE; 
}


static void check_num_tasks()
{
    if (g_slist_length(g_tasklist) > MAX_TASKS)
    {
        LOGPRINTF("maximum number of tasks reached, remove oldest");
        
        // get last task in list
        GSList *list_last_task = g_slist_last(g_tasklist);
        task_t *last_task = list_last_task->data;

        g_forced_close = TRUE;
        
        // close last task
        if (last_task->proc && last_task->proc->ipc_service
                && last_task->proc->is_multidoc)
        {
            request_t *request = new_request(last_task, NULL, NULL);
            child_window_close(last_task, request);
        }
        else
        {
            child_close(last_task);
        }            
    }
}


static gboolean on_check_child_window(gpointer data)
{
    LOGPRINTF("entry");
    
    request_t *request = data;

    if (request != NULL)
    {
        task_t *task = find_valid_task(request->task);
        if (task != NULL)
        {
            LOGPRINTF("looking for %s", task->application);

            Window child_window = get_application_window(task->application);
            if (child_window == 0)
            {
                //  do nothing (wait for next call)
                return TRUE;
            }
            else
            {
                check_num_tasks();

                // add window id to task info
                if (task->window == 0) {
                    task->window = child_window;
                }

                if (request->context && request->message_id)
                {
                    // return success to caller
                    ipc_send_reply_task_start(request->context, request->message_id, 0, NULL);
                }
                free_request(request);
            }
        }
        else
        {
            WARNPRINTF("task not found");

            if (request->context && request->message_id)
            {
                // return failed to caller
                ipc_send_reply_task_start(request->context, request->message_id, 3, NULL);
            }
            free_request(request);
        }
    }
    else
    {
        WARNPRINTF("request not found");
    }

    // stop and destroy timer
    g_check_child_window_source = 0;
    
    // stop pending timeout
    if (g_task_pending_source)
    {
        g_source_remove(g_task_pending_source);
        g_task_pending_source = 0;
    }

    // remove busy indication
    busy_remove_foreground(0);

    return FALSE;
}


static void on_window_open_callback(eripc_context_t *context,
                                    const eripc_event_info_t *info,
                                    request_t *request)
{
    LOGPRINTF("entry request [%p]", request);
    
    gboolean retval = FALSE;
    gchar *errmsg = NULL;
    const eripc_arg_t *arg_array = info->args;
    
    g_return_if_fail(request != NULL);
    
    if (info->event_type != ERIPC_EVENT_REPLY)
    {
        WARNPRINTF("invalid event: %d", info->event_type);
    }
    else if ((arg_array == NULL) || (arg_array[0].type != ERIPC_TYPE_INT))
    {
        WARNPRINTF("invalid arguments in reply, task not added");
    }
    else if (arg_array[0].value.i < 0)
    {
        LOGPRINTF("child returned failure, task not added");
        
        if (arg_array[1].type == ERIPC_TYPE_STRING)
        {
            LOGPRINTF("error message: %s", arg_array[1].value.s);
            errmsg = g_strdup(arg_array[1].value.s);
        }
        
        task_t *new_task = find_valid_task(request->task);
        if (new_task)
        {
            tasklist_remove_window(new_task->window);
        }
    }
    else    
    {
        task_t *old_task  = NULL;
        task_t *task_data = NULL;
        GSList *task      = g_tasklist;
        
        Window child_window = arg_array[0].value.i;
    
        // TODO
        // - add check if window exists in windowlist
        
        LOGPRINTF("task started in window: %d", (int) child_window);

        // check if window is already in tasklist
        //
        while ( (task != NULL) && (task->data != NULL) )
        {
            task_data = task->data;

            LOGPRINTF("check %s, %s; window %d", task_data->application, task_data->document, (int) task_data->window);

            if (task_data->window == child_window)
            {
                // task with this window found
                old_task = task_data;
                break;
            }
            
            task = task->next;
        }

        task_t *new_task = find_valid_task(request->task);
        if (new_task && (child_window > 0))
        {
            LOGPRINTF("task %s, %s; window %d", new_task->application, new_task->document, (int) child_window);
            
            if (old_task == new_task)
            {
                // task is already in task manager, activate it
                child_activate(old_task);
            }
            else
            {
                if (old_task)
                {
                    // new task will replace old task
                    LOGPRINTF("remove old task %s, %s: window %d", old_task->application, old_task->document, (int) old_task->window);
                    tasklist_remove_window(old_task->window);
                }
                else
                {
                    // task is new, check current number of tasks
                    check_num_tasks();
                }

                // add to task manager
                LOGPRINTF("on_window_open_callback: %s, %s", new_task->application, new_task->document);
                ipc_menu_add_task(child_window, new_task->label, new_task->application, new_task->document);

                // save new window
                new_task->window = child_window;

                if (old_task)
                {
                    // activate existing window 
                    LOGPRINTF("reactivate existing window");
                    window_activate(child_window);
                }
            }
        }
            
        // set success
        retval = TRUE;
    }

    if (request != NULL)
    {
        if (request->message_id != NULL)
        {
            // return result to caller
            if (retval == TRUE)
            {
                LOGPRINTF("return result to caller [%d], message_id [%s]", 0, request->message_id);
                ipc_send_reply_task_start(request->context, request->message_id, 0, NULL);
            }
            else
            {
                LOGPRINTF("return result to caller [%d], message_id [%s]", 4, request->message_id);
                ipc_send_reply_task_start(request->context, request->message_id, 4, errmsg);            
            }
        }
        free_request(request);
    }
    
    g_free(errmsg);
}


static void on_window_close_callback(eripc_context_t *context,
                                     const eripc_event_info_t *info,
                                     request_t *request)
{
    LOGPRINTF("entry request [%p]", request);
    
    gboolean retval = FALSE;
    const eripc_arg_t *arg_array = info->args;
    
    if (request == NULL)
    {
        WARNPRINTF("request is null");
    }
    if (find_valid_task(request->task) == NULL)
    {
        WARNPRINTF("task not found [%p]", request->task);
    }
    else if (info->event_type != ERIPC_EVENT_REPLY)
    {
        WARNPRINTF("invalid event: %d", info->event_type);
    }
    else if ((arg_array == NULL) || (arg_array[0].type != ERIPC_TYPE_BOOL))
    {
        WARNPRINTF("invalid arguments in reply");
    }
    else if (arg_array[0].value.b == FALSE)
    {
        LOGPRINTF("child returned failure");
    }
    else
    {
        // TODO
        // - check if window is really closed
        
        // remove task with this window from list
        tasklist_remove_window(request->task->window);
        retval = TRUE;
    }
    
    if (request != NULL)
    {
        if (request->message_id != NULL)
        {
            // return result to caller
            LOGPRINTF("return result to caller [%d], message_id [%s]", retval, request->message_id);
            ipc_send_reply(request->context, request->message_id, retval);
        }
        free_request(request);
    }
}



static gboolean on_child_term_timeout(gpointer data)
{
    LOGPRINTF("entry");
    
    task_t *task = data;
    
    if (task->proc && (task->proc->pid > 0))
    {
        // termination timed out, now do it the really hard way
        kill(task->proc->pid, SIGKILL);
    }
    
    // reset terminated handler
    g_child_terminate_source = 0;
    
    // stop and destroy timer
    return FALSE;
}


static task_t *tasklist_add(proc_t *proc, const char *application, const char *document,
                            const char *label, const char *image, Window window)
{
    LOGPRINTF("entry");
   
    task_t *task = g_new0 (task_t, 1);
    if (!task)
    {
        ERRORPRINTF("mem alloc failed");
        return NULL;
    }
    
    task->proc        = proc;
    task->application = g_strdup(application);
    task->document    = g_strdup(document);
    task->label       = g_strdup(label);
    task->image       = g_strdup(image);
    task->window      = window;
    
    g_tasklist = g_slist_prepend(g_tasklist, task);

    LOGPRINTF("done. task %p [#tasks: %d]", task, g_slist_length(g_tasklist));
    
    return task;
}


static void free_task(task_t *task)
{
    g_free(task->application);
    g_free(task->document);
    g_free(task->label);
    g_free(task->image);
    g_free(task);
    task = NULL;
}


static gboolean tasklist_set_first(task_t *task)
{
    LOGPRINTF("entry");
   
    task_t *task_data = NULL;
    GSList *cur_task = g_tasklist;
    
    while ( (cur_task != NULL) && (cur_task->data != NULL) )
    {
        task_data = cur_task->data;
        if (task_data == task)
        {
            ipc_menu_set_first_task(task->window);

            g_tasklist = g_slist_delete_link(g_tasklist, cur_task);
            g_tasklist = g_slist_prepend(g_tasklist, task);
            LOGPRINTF("task set to first [#tasks: %d]", g_slist_length(g_tasklist));
            return TRUE;
        }
        
        cur_task = cur_task->next;
    }
    
    LOGPRINTF("no first task found [#tasks: %d]", g_slist_length(g_tasklist));
    return FALSE;
}


static gboolean tasklist_remove_pid(GPid pid)
{
    LOGPRINTF("entry");
    
    GSList *task = g_tasklist;
    gint    npre = g_slist_length(g_tasklist);
    
    while ( (task != NULL) && (task->data != NULL) )
    {
        task_t *task_data = task->data;

        if (task_data->proc && (task_data->proc->pid == pid))
        {
            ipc_menu_remove_task(task_data->window);
            free_task(task_data);
    
            g_tasklist = g_slist_delete_link(g_tasklist, task);
            
            // start over iteration
            task = g_tasklist;
        }
        else
        {
            task = task->next;
        }
    }
    
    gint npost = g_slist_length(g_tasklist);
    
    if (npre != npost)
    {
        LOGPRINTF("removed %d tasks, %d remaining", npre-npost, npost);
        return TRUE;
    }
    else
    {
        LOGPRINTF("no tasks found for pid %d", pid);
        return FALSE;
    }
}


static gboolean tasklist_remove_window(Window window)
{
    LOGPRINTF("entry: xid %d", (int) window);
    
    GSList *task = g_tasklist;
    gint    npre = g_slist_length(g_tasklist);
    gboolean is_uds_task = FALSE;
    
    while ( (task != NULL) && (task->data != NULL) )
    {
        task_t *task_data = task->data;

        if (task_data->window == window)
        {
            ipc_menu_remove_task(task_data->window);
             
            if (strcmp(task_data->application, "uds") == 0)
            {
                is_uds_task = TRUE;
            }
            free_task(task_data);
            
            g_tasklist = g_slist_delete_link(g_tasklist, task);
            
            // start over iteration
            task = g_tasklist;
        }
        else
        {
            task = task->next;
        }
    }
    
    gint npost = g_slist_length(g_tasklist);
/*    
    if (is_uds_task && !is_uds_in_use())
    {
        // start timer to check if any windows for UDS remain
        LOGPRINTF("no task found for UDS, start timer");
        g_timeout_add_seconds(TIMEOUT_CHECK_RESTART_UDS, on_check_restart_uds, NULL);
    }
*/

    if (npre != npost)
    {
        LOGPRINTF("removed %d tasks, %d remaining", npre-npost, npost);
        return TRUE;
    }
    else
    {
        LOGPRINTF("no tasks found for window %d", (int) window);
        return FALSE;
    }
}


static task_t *find_valid_task(task_t *task)
{
    if (task == NULL) return NULL;

    task_t *task_data = NULL;
    GSList *cur_task = g_tasklist;
    
    while ( (cur_task != NULL) && (cur_task->data != NULL) )
    {
        task_data = cur_task->data;
        if (task_data == task)
        {
            return task_data;
        }
        cur_task = cur_task->next;
    }
    
    return NULL;
}


static task_t *get_task_by_xid(Window xid)
{
    GSList *task_ptr = g_tasklist;
    
    while (task_ptr)
    {
        task_t *cur_task = (task_t *) task_ptr->data;

        if (cur_task->window == xid) return cur_task;
        task_ptr = task_ptr->next;
    }
    return NULL;
}


static task_t *get_task(const char *application, const char *document)
{
    LOGPRINTF("entry");
    
    GSList *task_ptr = g_tasklist;
    gchar  *baseapp  = g_path_get_basename(application);
    
    while (task_ptr)
    {
        task_t *cur_task = (task_t *) task_ptr->data;

        if (baseapp && cur_task->application && (strcmp(cur_task->application, baseapp) == 0) && 
            ((document == NULL) || (cur_task->document && (strcmp(cur_task->document, document) == 0))))
        {
            g_free(baseapp);
            return cur_task;
        }
        task_ptr = task_ptr->next;
    }
    
    g_free(baseapp);
    return NULL;
}


gboolean parse_commandline(const gchar *command_line, gchar **application, gchar **document, gchar **arguments)
{
    char   **argv_ptr   = NULL;
    GError *error       = NULL;
    gchar  *app         = NULL;
    gchar  *doc         = NULL;
    gchar  *arg         = NULL;
    gint   argc         = 0;
    
    LOGPRINTF("enter");
    
    g_shell_parse_argv(command_line, &argc, &argv_ptr, &error);
    if (error)
    {
        WARNPRINTF("Execution of '%s' failed with error: %s", command_line, error->message);
        g_error_free(error);
        return FALSE;
    }
    
    // split command line
    //
    app = g_path_get_basename(argv_ptr[0]);
    
    if (argc > 1)
    {
        doc  = g_strdup(argv_ptr[argc-1]);
        
        // join all arguments except application
        arg = g_strjoinv(" ", &argv_ptr[1]); 
    }
    
    // return arguments
    //
    if (application != NULL)
    {
        LOGPRINTF("app `%s`", app);
        *application = app;
    }
    else 
    {
        g_free(app);
    }
        
    if (document != NULL)
    {
        LOGPRINTF("doc `%s`", doc);
        *document = doc;
    }
    else 
    {
        g_free(doc);
    }
        
    if (arguments != NULL)
    {
        LOGPRINTF("arg `%s`", arg);
        *arguments = arg;
    }
    else 
    {
        g_free(arg);
    }

    g_strfreev(argv_ptr);
    
    return TRUE;
}
