/*
 * File Name: taskbar.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

#include "config.h"
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW

#include <string.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <libergtk/ergtk.h>
#include <gconf/gconf-client.h>

#include "ipc.h"
#include "log.h"
#include "i18n.h"
#include "taskbar.h"

#define WINDOW_WIDTH  40
#define NUM_TASKS 5
#define FOLDER_XID -1
#define TASKBAR_HEIGHT 38

// for retrieving last apps info from registry
#define KEYS_EREADER                "/apps/er"
#define KEYS_LASTAPPS               KEYS_EREADER "/lastapps" 
#define KEY_LASTAPP_MODE			KEYS_LASTAPPS "/mode"
#define KEY_LASTAPP_CARD			KEYS_LASTAPPS "/cardmode"
#define KEY_LASTAPP_ACTIVE			KEYS_LASTAPPS "/active"
#define KEY_LASTAPP_1  			    KEYS_LASTAPPS "/1"
#define KEY_LASTAPP_2  			    KEYS_LASTAPPS "/2"
#define KEY_LASTAPP_3  			    KEYS_LASTAPPS "/3"
#define KEY_LASTAPP_4  			    KEYS_LASTAPPS "/4"
#define KEY_LASTAPP_5  			    KEYS_LASTAPPS "/5"

static GtkWidget *g_notebook = NULL;
static GtkWidget *g_window = NULL;
static int g_labelheight = -1;

struct task {
    int xid;
    int active;
    GtkWidget* labelWidget;
    char *document;
};

static GList *tasks = NULL;
static guint numtasks = 0;
static gboolean g_enabled = TRUE;


void testing_taskbar_print()
{
#if (LOGGING_ON)
    int i = 0;
    GList *iter = tasks;
    printf("------ Tasks ------\n");
    while (iter) {
        const struct task *t = (struct task*)iter->data;
        GList *children = gtk_container_get_children(GTK_CONTAINER(t->labelWidget));
        const char* text = gtk_label_get_text(GTK_LABEL(children->data));
        printf("  [%d]  %8d  '%s'  %s\n", i, t->xid, text, t->active ? "ACTIVE" : "");
        iter = iter->next;
        i++;
    }
#endif
}


static void taskbar_resize(int screenheight) {
    int width;
    int height = screenheight;
    gtk_widget_set_size_request(g_window, WINDOW_WIDTH, screenheight);
    gtk_window_get_size(GTK_WINDOW(g_window), &width, &height);

    g_labelheight = (height - 100) / NUM_TASKS;
    GList *iter = tasks;
    while (iter) {
        struct task *t = (struct task*)iter->data;
        gtk_widget_set_size_request(t->labelWidget, -1, g_labelheight);
        iter = iter->next;
    }
}


static void on_size_allocate(GtkWidget *widget, GtkAllocation *allocation, gpointer data)
{
    static gint height = 0;

    if (allocation->height != height) {
        height = allocation->height;
        taskbar_resize(allocation->height);
    }
}

static void set_active_task(struct task* activeTask, gboolean update_gui)
{
    GList *iter = tasks;
    int page = 0;
    int activePage = 0;
    
    while (iter) {
        struct task *t = (struct task*)iter->data;
        if (t == activeTask) {
            if (t->active == 1) return;
            t->active = 1;
            activePage = page;
        } else t->active = 0;
        page++;
        iter = iter->next;
    }
    if (update_gui) gtk_notebook_set_current_page(GTK_NOTEBOOK(g_notebook), activePage);
#if (LOGGING_ON)
    testing_taskbar_print();
#endif
}


static struct task* page2task(int page_num) {
    GList *iter = tasks;
    int cur_page = 0;
    while (cur_page != page_num) {
        iter = iter->next;
        cur_page++;
    }
    return (struct task*)iter->data;
}


static void task_clicked(GtkNotebook *widget, GtkNotebookPage *page, guint page_num, gpointer user_data)
{
    LOGPRINTF("Task selected page=%d", page_num);
    g_assert (page_num <= numtasks);
  
    if (!ipc_is_enabled()) return;

    struct task* t = page2task(page_num);
    if (t->active) return;
    set_active_task(t, FALSE);
    ipc_send_task_activated(t->xid);
}


void taskbar_create()
{
    GtkWidget *window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    g_window = window;

    GdkColor color;
    gdk_color_parse ("#bbbbbb", &color);
    gtk_widget_modify_bg(window, GTK_STATE_NORMAL, &color);

    gtk_window_set_type_hint(GTK_WINDOW(window), GDK_WINDOW_TYPE_HINT_DOCK);
    gtk_window_set_decorated(GTK_WINDOW(window), FALSE);
    gtk_window_set_accept_focus(GTK_WINDOW(window), FALSE);
    gtk_widget_set_size_request(g_window, WINDOW_WIDTH, gdk_screen_height() - TASKBAR_HEIGHT);
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_window_move(GTK_WINDOW(g_window), gdk_screen_width() - WINDOW_WIDTH, 0);

    GtkWidget *alignment = gtk_alignment_new(0, 0, 1, 1);
    gtk_alignment_set_padding(GTK_ALIGNMENT(alignment), 0, 0, 0, 5);
    gtk_container_add(GTK_CONTAINER(window), alignment);
    gtk_widget_show(alignment);

    GtkWidget *notebook = gtk_notebook_new();
    gtk_widget_set_name(notebook, "irex-taskbar-notebook");
    gtk_notebook_set_show_border(GTK_NOTEBOOK(notebook), FALSE);
    gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_RIGHT);
    gtk_container_add(GTK_CONTAINER(alignment), notebook);
    g_notebook = notebook;

    taskbar_resize(gdk_screen_height() - TASKBAR_HEIGHT);

    // add first special task
    taskbar_add_task(FOLDER_XID, _("Folder"), NULL, NULL);

    gtk_widget_show_all(window);
    gdk_flush();
    g_signal_connect(G_OBJECT(notebook), "switch-page", G_CALLBACK(task_clicked), NULL);
    g_signal_connect(window, "size-allocate", G_CALLBACK(on_size_allocate), NULL);
}


void taskbar_enable(gboolean enabled)
{
    if (g_enabled == enabled) return;

    g_enabled = enabled;
    gtk_widget_set_sensitive(g_notebook, enabled); 
}


static GtkWidget* create_page(const char* labelname)
{
    GtkWidget* labelbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_set_size_request(labelbox, -1, g_labelheight);

    GtkWidget *label = gtk_label_new(labelname);
    gtk_label_set_angle(GTK_LABEL(label), 270);
    gtk_label_set_single_line_mode(GTK_LABEL(label), TRUE);
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    gtk_box_pack_start(GTK_BOX(labelbox), label, TRUE, TRUE, 15);

    GtkWidget* page = gtk_vbox_new(FALSE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(page), 0);
    gtk_widget_show(page);
    gtk_widget_show(label);
    gtk_widget_show(labelbox);
    gtk_notebook_append_page(GTK_NOTEBOOK(g_notebook), page, labelbox);
    gtk_notebook_set_tab_label_packing(GTK_NOTEBOOK(g_notebook), page, FALSE, FALSE, GTK_PACK_START);

    return labelbox;
}


void taskbar_update_locale()
{
    GList *elem = g_list_first(tasks);
    struct task *t = (struct task*)elem->data;
    GList *children = gtk_container_get_children(GTK_CONTAINER(t->labelWidget));
    gtk_label_set_text(GTK_LABEL(children->data), _("Folder"));
}


static GList* find_task(int xid)
{
    GList *iter = tasks;
    while (iter) {
        struct task *t = (struct task*)iter->data;
        if (xid == t->xid) return iter;
        iter = iter->next;
    }
    return NULL;
}


void taskbar_add_task(int xid, const char* label, const char* application, const char* document)
{
    LOGPRINTF("xid=%d  label=%s apllication=%s document=%s",  xid, label, application, document);

    if (numtasks == NUM_TASKS + 1) {
        ERRORPRINTF("%s: error already has 1 extra task!", __func__);
        return;
    }
    numtasks++;
    if (find_task(xid)) {
        ERRORPRINTF("task [%d] already known!", xid);
        return;
    }
    struct task *newtask = malloc(sizeof(struct task));
    newtask->xid = xid;
    newtask->active = 0;
    tasks = g_list_append(tasks, newtask);
    newtask->labelWidget = create_page(label);
    // Putting application and document in one string (again) this will allow
    // easy reuse of the lastapp-patch code form the R1.7 firmware
    newtask->document = g_strdup_printf("%s %s", application, document);
    set_active_task(newtask, TRUE);
}


static void remove_page(GList* elem)
{
    GList *iter = tasks;
    int page = 0;
    while (iter) {
        if (iter == elem) {
            gtk_notebook_remove_page(GTK_NOTEBOOK(g_notebook), page);
            return;
        }
        page++;
        iter = iter->next;
    }
}


void taskbar_remove_task(int xid)
{
    LOGPRINTF("xid=%d",  xid);

    if (xid == FOLDER_XID) {
        ERRORPRINTF("cannot removed fixed task!");
        return;
    }
    GList *elem = find_task(xid);
    if (!elem) {
        ERRORPRINTF("task [%d] unknown!",xid);
        return;
    }
    // if removing extra task, switch to last task,
    // otherwise switch to 'Folder' on close
    GList *folder;
    if (numtasks == NUM_TASKS + 1) folder = g_list_last(tasks); 
    else folder = g_list_first(tasks); 
    set_active_task(folder->data, TRUE);

    remove_page(elem);
    {
        struct task *t = (struct task*)elem->data;
        g_free(t->document); // release memory of document-name
    }
    free(elem->data);
    tasks = g_list_remove(tasks, elem->data);
    numtasks--;
#if (LOGGING_ON)
    testing_taskbar_print();
#endif
}

//------------------------- Functions for Last Opened documents/applications
void taskbar_store_tasks(int mode)
{
	char *last_apps[5];
    gint i = 0;
    GList *iter = tasks;
	gint active = -1; // -1 no active
	
	(void)mode; //0=shutdown, 1=eject, 2=usb removal
	
	for(i=0; i<5; i++)
	{
		// init all apps to none
		last_apps[i] = NULL;
	}
    
    LOGPRINTF("------ Tasks ------ Mode(%d)", mode);
	
	i=0;
    while (iter) 
    {
        const struct task *t = (struct task*)iter->data;
        LOGPRINTF("  [%d]  %8d  %s [%s]\n", i, t->xid, t->active ? "ACTIVE" : "", t->document);

		// skipping previous folder item <-- Check assumption!!!!
		if ((i >= 1) && (i < 5))
		{
            // store full document name
			last_apps[i-1] = g_strdup_printf("%s", t->document);
            if (t->active)
            {
                active = i;
            }
		}

        iter = iter->next;
        i++;
    }
	for(i=0; i<5; i++)
	{
		// fill the rest
		if (last_apps[i] == NULL) { last_apps[i] = g_strdup("none"); }
	}

	// now save them in the registry
	{
		GConfClient         *client   = NULL;

        LOGPRINTF("ACTIVE:%d\n", active);
        LOGPRINTF("APP0:[%s]\n", last_apps[0]);
		LOGPRINTF("APP1:[%s]\n", last_apps[1]);
		LOGPRINTF("APP2:[%s]\n", last_apps[2]);
		LOGPRINTF("APP3:[%s]\n", last_apps[3]);
		LOGPRINTF("APP4:[%s]\n", last_apps[4]);

		client = gconf_client_get_default();
		
		// store in gconf
		if (!gconf_client_set_int(client, KEY_LASTAPP_ACTIVE, active, NULL)) {ERRORPRINTF("gconf error");}
		if (!gconf_client_set_string(client, KEY_LASTAPP_1, last_apps[0], NULL)) {ERRORPRINTF("gconf error");}
		if (!gconf_client_set_string(client, KEY_LASTAPP_2, last_apps[1], NULL)) {ERRORPRINTF("gconf error");}
		if (!gconf_client_set_string(client, KEY_LASTAPP_3, last_apps[2], NULL)) {ERRORPRINTF("gconf error");}
		if (!gconf_client_set_string(client, KEY_LASTAPP_4, last_apps[3], NULL)) {ERRORPRINTF("gconf error");}
		if (!gconf_client_set_string(client, KEY_LASTAPP_5, last_apps[4], NULL)) {ERRORPRINTF("gconf error");}
		
		g_object_unref(client);
	}
	
	for(i=0; i<5; i++)
	{
		// free all memory
		g_free(last_apps[i]);
	}
}

void taskbar_rename_task(int xid, const char* label)
{
    LOGPRINTF("xid=%d  label='%s'", xid, label);
    GList *elem = find_task(xid);
    if (!elem) {
        ERRORPRINTF("task [%d] unknown!", xid);
        return;
    }
    const struct task *task = (struct task*)elem->data;
    GList *children = gtk_container_get_children(GTK_CONTAINER(task->labelWidget));
    gtk_label_set_text(GTK_LABEL(children->data), label);
}


void taskbar_select_task(int xid)
{
    LOGPRINTF("xid=%d", xid);
    GList *task = find_task(xid);
    if (!task) {
        ERRORPRINTF("task [%d] unknown!", xid);
        return;
    }
    set_active_task(task->data, TRUE);
}

#endif

void taskbar_select_folder()
{
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    taskbar_select_task(FOLDER_XID);
#endif
}


