/*
 * File Name: dialog.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <gtk/gtk.h>
#include <string.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>
#include <liberipc/eripc.h>
#include <liberutils/display_utils.h>

// local include files, between " "
#include "log.h"
#include "dialog.h"
#include "i18n.h"
#include "ipc.h"
#include "popup.h"
#include "statusbar.h"
#include "taskbar.h"


//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

// screen layout
#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
static const int        TEXT_TOP_PADDING         =  40 + 5;
static const int        TEXT_BOTTOM_PADDING      =  830 + 5;
static const int        TEXT_LEFT_PADDING        =  150 + 5;
static const int        TEXT_RIGHT_PADDING       =  100 + 5;

static const int        TITLE_TOP_PADDING        =  0;
static const int        TITLE_BOTTOM_PADDING     =  0;
static const int        TITLE_LEFT_PADDING       =  0;
static const int        TITLE_RIGHT_PADDING      =  0;

static const int        MESSAGE_TOP_PADDING      =   0;
static const int        MESSAGE_BOTTOM_PADDING   =   0;
static const int        MESSAGE_LEFT_PADDING     =   0;
static const int        MESSAGE_RIGHT_PADDING    =   0;
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static const int        TITLE_TOP_PADDING        =   0;
static const int        TITLE_BOTTOM_PADDING     =  25;
static const int        TITLE_LEFT_PADDING       =  25;
static const int        TITLE_RIGHT_PADDING      =  25;

static const int        MESSAGE_TOP_PADDING      =   0;
static const int        MESSAGE_BOTTOM_PADDING   =  80;
static const int        MESSAGE_LEFT_PADDING     =   0;
static const int        MESSAGE_RIGHT_PADDING    =   0;
#else
#error "Unhandled machine type"
#endif

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget *dialog_wait    = NULL;
static GtkWidget *splash_window  = NULL;
static GtkWidget *splash_title   = NULL;
static GtkWidget *splash_message = NULL;
static GdkPixbuf *splash_pixbuf  = NULL;
static gboolean  g_got_control   = FALSE;

typedef struct
{
    GtkWidget       *widget;
    eripc_context_t *context;
    gchar           *message_id; 
} dialog_t;

static dialog_t *dialog_response = NULL;
static dialog_t *dialog_info     = NULL;
static gboolean g_is_rotated     = FALSE;
static gboolean g_temp_rotation  = FALSE;
static gboolean g_expect_expose  = FALSE;
static gboolean g_ignore_next_expose = FALSE;


//============================================================================
// Local Function Definitions
//============================================================================

static void create_splash(void);
static void show_splash(const char *background, const char *title, const char *message);
static void redraw_splash(void);

static void on_dialog_response(GtkDialog *dialog, gint rc, gpointer data);
static void on_dialog_close   (GtkDialog *dialog, gint rc, gpointer data);

static void dialog_message_confirm_destroy(void);
static void dialog_message_info_destroy(void);


//============================================================================
// Functions Implementation
//============================================================================

void dialog_create(void)
{
    LOGPRINTF("entry");
    
    gchar *orient = ipc_get_orientation();
    if (orient && (strcmp(orient, "portrait") !=0))
    {
        g_is_rotated = TRUE;
    }
    g_free(orient);
    
    create_splash();
}


gboolean dialog_splash_remove()
{
    LOGPRINTF("entry");
    
    if (g_temp_rotation)
    {
        // set display back to landscape
        ipc_set_orientation("landscape");
        g_ignore_next_expose = TRUE;
        return FALSE;
    }
    
    gtk_widget_hide(splash_window);
    ipc_send_request_popup("unblock");

    if (g_got_control) {
        display_splash_unlock();
        display_return_control();
        g_got_control = FALSE;
    }

    return TRUE;
}


gboolean dialog_splash_show(const char *type)
{
    LOGPRINTF("entry");
    gboolean retval = TRUE;
    
    g_return_val_if_fail(type != NULL, FALSE);
    
    if (strcmp(type,"usbconnect") == 0)
    {
        show_splash("usbconnect", 
                _("Connected"), 
                _("Tip: Eject the device from your computer to resume use while charging."));
    }
    else if (strcmp(type,"restart") == 0)
    { 
        show_splash("softwareupdate", 
                _("Installing Firmware"), 
                _("In a few seconds, the device will restart "
                  "to install the new firmware."));
    }
    else if (strcmp(type,"shutdown") == 0)
    { 
		// Shutting down, save list of open tasks/documents
		// This function is called both for manual and automatic shutdown
		taskbar_store_tasks(0); //indicate shutdown

        show_splash("shutdown", 
                _("Goodbye"),
                _("In a few seconds, the device will turn off."));
    }
    else if (strcmp(type, "indexing") == 0)
    {
        show_splash("indexing", 
                _("Indexing Files"),
                _("Please wait while indexing files..."));
    }
    else if (strcmp(type, "batterylow") == 0)
    {
        show_splash("batterylow",
                _("Low Battery"),
                _("Connect to a power source to keep using the device."));
    }
    else if (strcmp(type, "safelyremove") == 0)
    {
        show_splash("nosd", 
                _("Eject Memory Card"), 
                _("You can now safely remove the card from the device."));
    }
    else if (strcmp(type, "nosd") == 0)
    {
        show_splash("nosd",
                _("No Memory Card Found"),
                _("Insert a card into the device to continue use."));
    }
    else if (strcmp(type, "sdfull") == 0)
    {
        show_splash("nosd",
                _("Memory Card Full"),
                _("Free space by deleting files to continue use."));
    }
    else if (strcmp(type, "sdreadonly") == 0)
    {
        show_splash("nosd",
                _("Memory Card Locked"),
                _("Remove the write lock to continue use."));
    }
    else 
    {
        WARNPRINTF("splash screen not found: %s", type);
        retval = FALSE;
    }
    
    return retval;
}


void dialog_wait_show(const char *message)
{
    LOGPRINTF("entry");
    dialog_wait_close();
    dialog_wait = ergtk_busy_dialog_new(message);
    gtk_widget_show(dialog_wait);
}


GtkWidget *dialog_message_info(GtkMessageType type,
                               const char *title, 
                               const char *message,
                               eripc_context_t *context,
                               const char *message_id)
{
    LOGPRINTF("entry");
    
    GtkWidget *dialog = NULL;

    dialog_message_info_destroy();

    ipc_send_request_popup("block");

    dialog = gtk_message_dialog_new(g_statusbar_window, 
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    type, 
                                    GTK_BUTTONS_OK,
                                    message);
    
    // make sure the dialog appears on top
    gtk_window_set_transient_for(GTK_WINDOW(dialog), NULL);
    gtk_window_set_keep_above(GTK_WINDOW(dialog), TRUE);
    
    if (title)
    {
        gtk_window_set_title(GTK_WINDOW(dialog), title);
    }
    
    dialog_info = g_new0(dialog_t, 1);
    if (!dialog_info)
    {
        ERRORPRINTF("mem alloc failed");
        return FALSE;
    }
    dialog_info->widget  = dialog;
    dialog_info->context = context;
    dialog_info->message_id = g_strdup(message_id);

    // return immediately and capture response through callback
    g_signal_connect(dialog, "response", G_CALLBACK (on_dialog_close), NULL);
    gtk_widget_show(dialog);
    
    return dialog;
}


gboolean dialog_message_confirm(const char *title, 
                                const char *message, 
                                eripc_context_t *context,
                                const char *message_id,
                                const char *button_no,
                                const char *button_yes)
{
    LOGPRINTF("entry");
    
    GtkWidget *widget = NULL;
    
    dialog_message_confirm_destroy();
    
    ipc_send_request_popup("block");
    
    widget = gtk_message_dialog_new(g_statusbar_window, 
                                    GTK_DIALOG_DESTROY_WITH_PARENT,
                                    GTK_MESSAGE_QUESTION, 
                                    GTK_BUTTONS_NONE,
                                    message);
    
    gtk_dialog_add_buttons(GTK_DIALOG(widget), 
                           button_no,  GTK_RESPONSE_NO, 
                           button_yes, GTK_RESPONSE_YES, 
                           NULL);
            
    // make sure the dialog appears on top
    gtk_window_set_transient_for(GTK_WINDOW(widget), NULL);
    gtk_window_set_keep_above(GTK_WINDOW(widget), TRUE);
   
    if (title)
    {
        gtk_window_set_title(GTK_WINDOW(widget), title);
    }
    
    dialog_response = g_new0(dialog_t, 1);
    if (!dialog_response)
    {
        ERRORPRINTF("mem alloc failed");
        return FALSE;
    }
    dialog_response->widget  = widget;
    dialog_response->context = context;
    dialog_response->message_id = g_strdup(message_id);
        
    // return immediately and capture response through callback
    g_signal_connect(widget, "response", G_CALLBACK (on_dialog_response), NULL);
    gtk_widget_show(widget);
          
    return TRUE;
}


void dialog_wait_close(void)
{
    LOGPRINTF("entry");
    if (dialog_wait)
    {
        gtk_widget_destroy(dialog_wait);
        dialog_wait = NULL;
    }
}


void dialog_message_confirm_close(void)
{
    if (dialog_response && dialog_response->widget)
    {
        gtk_dialog_response(GTK_DIALOG(dialog_response->widget), GTK_RESPONSE_CLOSE);
    }
}


void dialog_message_info_close(void)
{
    if (dialog_info && dialog_info->widget)
    {
        gtk_dialog_response(GTK_DIALOG(dialog_info->widget), GTK_RESPONSE_CLOSE);
    }
}


void dialog_rotated(const char *orientation)
{
    LOGPRINTF("entry");
    
    if (strcmp(orientation, "portrait") == 0)
    {
        g_is_rotated = FALSE;
        if (g_temp_rotation)
        {
            // show completed window
            redraw_splash();
            gtk_widget_show(splash_window);
            display_update_keep_splash_lock();
        }
    }
    else if (g_str_has_prefix(orientation, "landscape"))
    {
        g_is_rotated = TRUE;
        if (g_temp_rotation)
        {
            g_temp_rotation = FALSE;
            dialog_splash_remove();
        }
    }
}


//============================================================================
// Local Function Implementation
//============================================================================

static void redraw_splash(void)
{
    LOGPRINTF("entry");
    
    GdkRectangle rect;
    rect.x = 0;
    rect.y = 0;
    rect.width  = 1024;
    rect.height = 1268;
    gtk_widget_show(splash_window);
    gdk_window_invalidate_rect(splash_window->window, &rect, TRUE);
    
    g_expect_expose = TRUE;
}


static void show_splash(const char *image, const char *title, const char *message)
{
    LOGPRINTF("entry");
    
    g_return_if_fail(splash_window != NULL);
    
    display_splash_lock();
    display_gain_control();
    g_got_control = TRUE;
    
    // close dialogs
    dialog_wait_close();
    dialog_message_info_close();
    dialog_message_confirm_close();
        
    // replace image
    gchar *background_file = NULL;
    if (strcmp(image, "usbconnect") == 0
#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
        || strcmp(image, "batterylow") == 0
        || strcmp(image, "nosd") == 0
        || strcmp(image, "softwareupdate") == 0
#endif    
        )
    {
        background_file = g_strdup_printf("/usr/share/popupmenu/background_%s.png", image);
    }
    else
    {
        background_file = g_strdup("/usr/share/popupmenu/background_general.png");
    }

    if (splash_pixbuf) g_object_unref(splash_pixbuf);
    splash_pixbuf = gdk_pixbuf_new_from_file(background_file, NULL);
    if (!splash_pixbuf)
    {
        WARNPRINTF("splash background not found [%s]", background_file);
    }
    g_free(background_file);
    
    // replace title and message
    gtk_label_set_text(GTK_LABEL(splash_title), title);
    gtk_label_set_text(GTK_LABEL(splash_message), message);
    
    if (g_is_rotated)
    {
        // set display to portrait mode temporarily
        ipc_set_orientation("portrait");
        g_ignore_next_expose = TRUE;
        g_temp_rotation = TRUE;
    }
    else
    {
        redraw_splash();
    }

    ipc_send_request_popup("block");
}


static gboolean on_splash_expose(GtkWidget      *widget,
                                 GdkEventExpose *event,
                                 gpointer       data)
{ 
    LOGPRINTF("entry: expect expose [%d] is rotated [%d] temp_rotation [%d] ignore next [%d]", g_expect_expose, g_is_rotated, g_temp_rotation, g_ignore_next_expose);
    
    if (g_ignore_next_expose)
    {
        g_ignore_next_expose = FALSE;
        return FALSE;
    }

    if (g_expect_expose)
    {
        gdk_draw_pixbuf(widget->window, NULL, splash_pixbuf, 0, 0, 0, 0, -1, -1, GDK_RGB_DITHER_MAX, 0, 0);
        
        GtkWidget *child = gtk_bin_get_child(GTK_BIN(widget));
        if (child)
        {
            gtk_container_propagate_expose(GTK_CONTAINER(widget), child, event);
        }

        if (!g_is_rotated)
        {
            display_update_keep_splash_lock();
        }

        g_expect_expose = FALSE;
    }
    return TRUE;
}


static void create_splash()
{
    LOGPRINTF("entry");
    
    GtkWidget *window     = NULL;
    GtkWidget *widget     = NULL;
    GtkBox    *vbox       = NULL;
    GtkWidget *alignment  = NULL;

    // object hierarchy:
    //    
    // window
    //  |
    widget = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    window = widget;
    splash_window = GTK_WIDGET(widget);
    gtk_window_set_resizable(GTK_WINDOW(splash_window), FALSE);
    gtk_window_fullscreen(GTK_WINDOW(splash_window)); 
    gtk_window_set_type_hint(GTK_WINDOW(splash_window), GDK_WINDOW_TYPE_HINT_SPLASHSCREEN);

#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
    //    |--alignment
    //       |
    widget = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
    gtk_alignment_set_padding(GTK_ALIGNMENT(widget),
            TEXT_TOP_PADDING,
            TEXT_BOTTOM_PADDING,
            TEXT_LEFT_PADDING,
            TEXT_RIGHT_PADDING);
    gtk_container_add(GTK_CONTAINER(window), widget);
    gtk_widget_show(widget);
    alignment = widget;
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);   
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(window), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);   
    //             |
    //             |-- alignment (filler)
    //             |    
    widget = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_box_pack_start(vbox, widget, TRUE, TRUE, 0);
    gtk_widget_show(widget);
#else
#error "Unhandled machine type"
#endif     
    //             |
    //             |-- alignment
    //             |     |
    widget = gtk_alignment_new(0.5, 0.5, 0.5, 0.5);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               TITLE_TOP_PADDING,
                               TITLE_BOTTOM_PADDING,
                               TITLE_LEFT_PADDING,
                               TITLE_RIGHT_PADDING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    alignment = widget;
    
    //             |     |    
    //             |     |-- label
    //             |        
    widget = gtk_label_new(NULL);
#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
    int LABEL_WIDTH = 768 - TEXT_LEFT_PADDING - TEXT_RIGHT_PADDING 
                          - TITLE_LEFT_PADDING - TITLE_RIGHT_PADDING; 
    gtk_widget_set_size_request(widget, LABEL_WIDTH, -1);
    gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_CENTER);
    gtk_label_set_line_wrap(GTK_LABEL(widget), TRUE);
#endif    
    gtk_widget_set_name(widget, "irex-splash-title");
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);
    splash_title = widget;

    //             |
    //             |-- alignment
    //                   |
    widget = gtk_alignment_new(0.5, 0.5, 0.5, 0.5);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget),
                               MESSAGE_TOP_PADDING,
                               MESSAGE_BOTTOM_PADDING,
                               MESSAGE_LEFT_PADDING,
                               MESSAGE_RIGHT_PADDING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    gtk_widget_show(widget);
    alignment = widget;
    
    //                   |    
    //                   |-- label
    //                     
    widget = gtk_label_new(NULL);
#if MACHINE_IS_DR800SG || MACHINE_IS_DR800S || MACHINE_IS_DR800SW
    LABEL_WIDTH = 768 - TEXT_LEFT_PADDING - TEXT_RIGHT_PADDING 
                      - MESSAGE_LEFT_PADDING - MESSAGE_RIGHT_PADDING; 
    gtk_widget_set_size_request(widget, LABEL_WIDTH, -1);
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    gtk_widget_set_size_request(widget, 1024, -1);
#else
#error "Unhandled machine type"
#endif    
    gtk_widget_set_name(widget, "irex-splash-message");
    gtk_label_set_justify(GTK_LABEL(widget), GTK_JUSTIFY_CENTER);
    gtk_label_set_line_wrap(GTK_LABEL(widget), TRUE);
    gtk_container_add(GTK_CONTAINER(alignment), widget);
    gtk_widget_show(widget);   
    splash_message = widget;

    // force non-decorated full screen splash
    gint w = -1;
    gint h = -1;
    gdk_window_get_geometry(gdk_get_default_root_window(), NULL, NULL, &w, &h, NULL);
    gtk_widget_set_size_request(splash_window, w,  h);
    g_signal_connect(GTK_OBJECT(splash_window), "expose_event", G_CALLBACK (on_splash_expose), NULL);
}


static void on_dialog_response(GtkDialog *widget, gint rc, gpointer data)
{
    gboolean  retval  = FALSE;
    
    LOGPRINTF("entry");

    if (GTK_WIDGET(widget) != dialog_response->widget)
    {
        WARNPRINTF("huh? expected these to be the same");
    }
    
    if ((rc == GTK_RESPONSE_OK) || (rc == GTK_RESPONSE_YES))
    {
        retval = TRUE;
		// This function is currently used (via dialog_message_confirm) for 
		// usb-connection and firmware update. We only want to catch the first
		// one, but can not determine which case it is used for.
		// After a firmware update the lastapps-mechanism willnot be there anymore,
		// so it will not harm very much.
		taskbar_store_tasks(2); // indicate usb removal
    }
    
    if (dialog_response->context)
    {
        ipc_send_reply(dialog_response->context, dialog_response->message_id, retval);
    }

    // remove dialog and its data
    dialog_message_confirm_destroy();
}


static void on_dialog_close(GtkDialog *widget, gint rc, gpointer data)
{
    LOGPRINTF("entry");

    if (GTK_WIDGET(widget) != dialog_info->widget)
    {
        WARNPRINTF("huh? expected these to be the same");
    }

    if (dialog_info->context)
    {
        ipc_send_reply(dialog_info->context, dialog_info->message_id, TRUE);
    }
    
    // remove dialog and its data
    dialog_message_info_destroy();
}


static void dialog_message_confirm_destroy(void)
{
    LOGPRINTF("entry");
    if (dialog_response != NULL)
    {
        gtk_widget_destroy(dialog_response->widget);
        g_free(dialog_response->message_id);
        g_free(dialog_response);
        dialog_response = NULL;
        ipc_send_request_popup("unblock");
    }
}


static void dialog_message_info_destroy(void)
{
    LOGPRINTF("entry");
    if (dialog_info != NULL)
    {
        gtk_widget_destroy(dialog_info->widget);
        g_free(dialog_info->message_id);
        g_free(dialog_info);
        dialog_info = NULL;
        ipc_send_request_popup("unblock");
    }
}
