#ifndef __MENUSTORE_H__
#define __MENUSTORE_H__

/**
 * File Name  : menustore.h
 *
 * Description: 
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include <glib.h>

G_BEGIN_DECLS

//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

enum menu_state
{
  MENU_STATE_NORMAL = 0,
  MENU_STATE_SELECTED,
  MENU_STATE_DISABLED,
  MENU_STATE_ALTERNATE
};

enum
{
  TREE_COL_TYPE = 0,
  TREE_COL_NAME,
  TREE_COL_TEXT,
  TREE_COL_ICON,
  TREE_COL_STATE,
  TREE_COL_IMAGE_NORMAL,
  TREE_COL_IMAGE_ALTERNATE,
  TREE_COL_ITER,
  TREE_NUM_COLS
};

//----------------------------------------------------------------------------
// Functions
//----------------------------------------------------------------------------

typedef void (*taskbar_cb_t)();
void menustore_create(taskbar_cb_t cb);

void menustore_set_text();

const char* menustore_get_current_menu();

void menustore_set_current_menu(const char* name);

const char* menustore_get_current_service();


/**---------------------------------------------------------------------------
 *
 * Name :  menustore_add_group
 *
 * @brief  Add a menu group to the store
 *
 * @param  iname New group name
 * @param  pname Parent group name it belongs to or NULL for root
 * @param  text Label text
 * @param  icon Icon name
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_add_group(const char *iname, const char *pname, const char *text, const char *icon);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_add_item
 *
 * @brief  Add a menu item to a group
 *
 * @param  iname New item name
 * @param  pname Parent group name it belongs to
 * @param  text Label text
 * @param  icon Icon name
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_add_item(const char *iname, const char *pname, const char *text, const char *icon);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_add_menu
 *
 * @brief  Create a new erMyThing object
 *
 * @param  iname New menu set
 * @param  label Label text
 * @param  service IPC service name
 * @param  group1 Name of group in 1st row
 * @param  group2 Name of group in 2nd row
 * @param  group3 Name of group in 3rd row
 * @param  group4 Name of group in 4th row
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_add_menu(const char *iname, const char *label, const char *service, 
                            const char *group1, const char *group2,
                            const char *group3, const char *group4);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_remove_group
 *
 * @brief  Remove menu group and everything below it
 *
 * @param  name Name of menu group
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_remove_group(const char *name);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_remove_item
 *
 * @brief  Remove a single item
 *
 * @param  iname New item name
 * @param  pname Parent group name it belongs to
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_remove_item(const char *iname, const char* pname);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_remove_menu
 *
 * @brief  Remove menu set
 *
 * @param  name Name of menu set
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_remove_menu(const char *name);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_set_group_state
 *
 * @brief  Set the group state
 *
 * @param  name Name of group
 * @param  state New state
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_set_group_state(const char *name, const char *state);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_set_group_label
 *
 * @brief  Set the group label
 *
 * @param  name Name of group
 * @param  state New label text
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_set_group_label(const char *name, const char *label);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_set_item_state
 *
 * @brief  Set the item state
 *
 * @param  iname New item name
 * @param  pname Parent group name it belongs to
 * @param  state New state
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_set_item_state(const char *iname, const char *pname, const char *state);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_set_item_label
 *
 * @brief  Set the item label text
 *
 * @param  iname New item name
 * @param  pname Parent group name it belongs to
 * @param  label New label text
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_set_item_label(const char *iname, const char *pname, const char *label);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_add_toolbar_item
 *
 * @brief  Add an existing menu item to the toolbar, items are added in order of call
 *
 * @param  menu  Menu name of the item
 * @param  group Group name of the item
 * @param  item  Item name to be added
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_add_toolbar_item(const char *menu, const char *group, const char *item);

/**---------------------------------------------------------------------------
 *
 * Name :  menustore_remove_toolbar_items
 *
 * @brief  Clear the toolbar items from a menu
 *
 * @param  menu  Menu name of which the toolbar items will be cleared
 *
 * @return Returns TRUE on succes, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean menustore_remove_toolbar_items(const char *name);


gboolean menustore_popup_has_changed();

void menustore_clear_popup_changed();

typedef void (*tool_func_t)(const char *iname, const char *pname, const char *mname, const char *state, const char *service);
gboolean menustore_activate_item_iter(gpointer user_data, tool_func_t func);

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
#define MENUSTORE_NUM_TOOLS 16      // nr of tools with fullbar
#define MENUSTORE_SMALL_TOOLS 5     // nr of tools with limited space

gboolean menustore_toolbar_has_changed();

void menustore_clear_toolbar_changed();

int menustore_get_tool_limit();

GdkPixbuf* menustore_get_tool_icon(int index);

void menustore_activate_toolitem(int index, tool_func_t func);

void menustore_activate_item(const char* pname, const char* iname, tool_func_t func);
#endif

typedef gpointer (*add_item_func)(const char* name,
                                  const char* text,
                                  enum menu_state state,
                                  GdkPixbuf* img,
                                  gpointer menu_data,
                                  gpointer user_data);
typedef void (*separator_func)(gpointer user_data);
void menustore_fill_menu(add_item_func item_cb,
                         add_item_func submenu_cb,
                         separator_func separator_cb,
                         gpointer user_data);

#if (TESTING_ON)
const char* testing_menustore_print();

const char* testing_menustore_get_route(const char* pname, const char* iname);
#endif

G_END_DECLS
    
#endif

