/*
 * File Name: ipc.c
 */

/*
 * This file is part of hello-world-settings.
 *
 * hello-world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello-world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <stdio.h>
#include <stdlib.h>
#include <gdk/gdkx.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >
#include <liberipc/eripc.h>

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

typedef struct 
{
    eripc_handler_t *handler;
    const gchar     *message_name;
    const gchar     *interface;
    gint            handler_id;
} functionEntry;


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// IPC application
// TODO: set DBUS_APPL_NAME to your application name, use lower-case and digits only
//       or set to PACKAGE_NAME when this is lower-case and digits only
#define DBUS_APPL_NAME                  "helloworld" "settings"            // lower-case and digits only
#define DBUS_SERVICE                     "com.irexnet." DBUS_APPL_NAME
#define DBUS_PATH                       "/com/irexnet/" DBUS_APPL_NAME
#define DBUS_INTERFACE                   "com.irexnet." DBUS_APPL_NAME

// IPC system control
#define DBUS_SERVICE_SYSTEM_CONTROL     "com.irexnet.sysd"

// IPC popup menu
#define DBUS_SERVICE_POPUP_MENU         "com.irexnet.popupmenu"


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_context_t  *eripcContext = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static void on_menu_item         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_window_activated  ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_window_deactivated( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_prepare_standby   ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_changed_locale    ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_changed_orientation(eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

// Exported DBUS API list
static functionEntry    service_functions[] =
        {
            // message handlers (method calls to this service)
            { on_menu_item,           "menuItemActivated",      NULL                        },
            { on_window_activated,    "activatedWindow",        NULL                        },
            { on_window_deactivated,  "deactivatedWindow",      NULL                        },
            // signal handlers (broadcasted from given service)
            { on_prepare_standby,     "sysPrepareStandby",      DBUS_SERVICE_SYSTEM_CONTROL },
            { on_changed_locale,      "sysChangedLocale",       DBUS_SERVICE_SYSTEM_CONTROL },
            { on_changed_orientation, "sysChangedOrientation",  DBUS_SERVICE_SYSTEM_CONTROL },
            { NULL }
        };


//============================================================================
// Functions Implementation
//============================================================================


//----------------------------------------------------------------------------
// Generic
//----------------------------------------------------------------------------

// initialise
void ipc_set_services (void)
{
    gint          i;
    eripc_error_t retval;

    LOGPRINTF("entry");
    g_assert(eripcContext == NULL);

    // initialise IPC context
    eripcContext = eripc_init(DBUS_APPL_NAME, "1.0", NULL);
    if (eripcContext == NULL) 
    {
        ERRORPRINTF("Failed to initialize eripc context\n");
        exit(1);
    }

    // register message/signal handlers
    for (i = 0 ; service_functions[i].handler != NULL ; i++)
    {
        if (service_functions[i].interface)
        {
            // install signal handler
            retval = eripc_set_signal_handler( eripcContext,
                                               service_functions[i].handler,
                                               NULL,
                                               ERIPC_BUS_SESSION,
                                               service_functions[i].interface,
                                               service_functions[i].message_name,
                                               &(service_functions[i].handler_id) );
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                ERRORPRINTF( "eripc_set_signal_handler [%s] returns [%d]",
                             service_functions[i].message_name,
                             retval );
            }
        }
        else
        {
            // install message handler
            retval = eripc_set_message_handler( eripcContext,
                                                service_functions[i].handler,
                                                NULL,
                                                ERIPC_BUS_SESSION,
                                                DBUS_INTERFACE,
                                                service_functions[i].message_name,
                                                &(service_functions[i].handler_id) );
              
            if (retval != ERIPC_ERROR_SUCCESS) 
            {
                ERRORPRINTF( "eripc_set_message_handler [%s] returns [%d]",
                             service_functions[i].message_name,
                             retval );
            }
        }
    }
}


// un-initialise
void ipc_unset_services (void)
{
    gint i;

    for (i = 0 ; service_functions[i].handler_id != 0 ; i++)
    {
        eripc_unset_handler(eripcContext, service_functions[i].handler_id);
    }    
}


//----------------------------------------------------------------------------
// System control
//----------------------------------------------------------------------------

// report "application started"
void ipc_sys_startup_complete ( void )
{
    eripc_error_t   result;

    // parameters for ipc message
    const int       my_pid = getpid();
    const int       my_xid = GDK_WINDOW_XID(g_main_window->window);
    const gboolean  is_multidoc = FALSE;

    LOGPRINTF("entry");
    g_assert( eripcContext );

    // broadcast signal over session bus
    result = eripc_send_signal_varargs( eripcContext,
                                        ERIPC_BUS_SESSION,
                                        DBUS_PATH,
                                        DBUS_SERVICE_SYSTEM_CONTROL,
                                        "startupComplete",
                                        ERIPC_TYPE_STRING, DBUS_APPL_NAME,
                                        ERIPC_TYPE_INT,    my_pid,
                                        ERIPC_TYPE_BOOL,   is_multidoc,
                                        ERIPC_TYPE_STRING, DBUS_SERVICE,
                                        ERIPC_TYPE_INT,    my_xid,
                                        ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
    } 
}


//----------------------------------------------------------------------------
// Popup menu
//----------------------------------------------------------------------------

// add a menu set
gboolean ipc_menu_add_menu( const char *name,
                            const char *group1,
                            const char *group2,
                            const char *group3 )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] groups [%s] [%s] [%s]",
                       name,     group1, group2, group3 );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, DBUS_SERVICE,
                                 ERIPC_TYPE_STRING, group1,
                                 ERIPC_TYPE_STRING, group2,
                                 ERIPC_TYPE_STRING, group3,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// add a menu group
gboolean ipc_menu_add_group( const char *name,
                             const char *parent, 
                             const char *image )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] image [%s]",
                       name,     parent,     image       );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addGroup",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, image,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// add a menu item
gboolean ipc_menu_add_item( const char *name,
                            const char *parent, 
                            const char *image  )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] image [%s]",
                       name,     parent,     image       );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "addItem",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, "",
                                 ERIPC_TYPE_STRING, image,
                                 ERIPC_TYPE_INVALID);

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set text for a menu set
gboolean ipc_menu_set_menu_label ( const char *name,
                                   const char *label )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] label [%s]", name, label );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setMenuLabel",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, label,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set text for a menu group
gboolean ipc_menu_set_group_label ( const char *name,
                                    const char *label )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] label [%s]", name, label );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setGroupLabel",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, label,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set text for an menu item
gboolean ipc_menu_set_item_label ( const char *name,
                                   const char *parent, 
                                   const char *label )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] label [%s]", name, parent, label );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setItemLabel",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, label,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// show the given menu set
gboolean ipc_menu_show_menu( const char *name )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s]", name );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "showMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// remove the given menu set
gboolean ipc_remove_menu( const char *name )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s]", name );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "removeMenu",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    }

    return ok;
}


// set the state of a menu group
gboolean ipc_menu_set_group_state( const char *name,
                                   const char *state )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] state [%s]", name, state );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setGroupState",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}


// set the state of a menu item
gboolean ipc_menu_set_item_state( const char *name,
                                  const char *parent,
                                  const char *state  )
{
    gboolean      ok = TRUE;    // return value
    eripc_error_t result;

    LOGPRINTF( "entry: name [%s] parent [%s] state [%s]", name, parent, state );
    g_assert( eripcContext );

    result = eripc_send_varargs( eripcContext,
                                 NULL,                  // reply handler
                                 NULL,                  // user data
                                 ERIPC_BUS_SESSION,
                                 DBUS_SERVICE_POPUP_MENU,
                                 "setItemState",
                                 ERIPC_TYPE_STRING, name,
                                 ERIPC_TYPE_STRING, parent,
                                 ERIPC_TYPE_STRING, state,
                                 ERIPC_TYPE_INVALID );

    if (result != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("eripc_send_varargs returns [%d]", result);
        ok = FALSE;
    } 

    return ok;
}

//============================================================================
// Local Function Implementation
//============================================================================

//----------------------------------------------------------------------------
// Signal/message handlers
//----------------------------------------------------------------------------

/* @brief Called when a menu items is activated in Popup menu
 *
 * Application (callee) should handle the item depending on the current state.
 */
static void on_menu_item ( eripc_context_t          *context,
                           const eripc_event_info_t *info,
                           void                     *user_data )
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if ((arg_array[0].type == ERIPC_TYPE_STRING) && 
        (arg_array[1].type == ERIPC_TYPE_STRING) && 
        (arg_array[2].type == ERIPC_TYPE_STRING) && 
        (arg_array[3].type == ERIPC_TYPE_STRING))
    {
        const char        *item      = arg_array[0].value.s;
        const char        *group     = arg_array[1].value.s;
        const char        *menu      = arg_array[2].value.s;
        const char        *state     = arg_array[3].value.s;
        
        if (item && group && menu && state)
        {
            menu_on_item_activated( item, group, menu, state );
        }
    }
}


/* @brief Called after a window was activated (set to the foreground)
 *
 * Application (callee) should set its context for the given window and set the 
 * Popupmenu menu context.
 */
static void on_window_activated( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    LOGPRINTF("entry");
    gboolean          result      = FALSE; 
    const eripc_arg_t *arg_array  = info->args;

    if (arg_array[0].type == ERIPC_TYPE_INT)
    {
        // TODO: Replace implementation

        menu_show();

        result = TRUE;
    }

    // return result to caller
    eripc_reply_bool(context, info->message_id, result);
}


/* @brief Called after a window was deactivated (set to the background)
 *
 * Application (callee) may adapt its context and free resources.
 */  
static void on_window_deactivated( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data )
{
    LOGPRINTF("entry");
    gboolean          result      = FALSE; 
    const eripc_arg_t *arg_array  = info->args;

    if (arg_array[0].type == ERIPC_TYPE_INT)
    {
        // TODO: Replace implementation

        result = TRUE;
    }
    
    // return result to caller
    eripc_reply_bool(context, info->message_id, result);
}


/* @brief Called just before the system enters standby mode
 * 
 * Application must commit changes (flush) of all its open files. Failing to 
 * handle this signal may result in unsaved data or currupt files.
 */
static void on_prepare_standby ( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    LOGPRINTF("entry");

    // TODO: Add implementation here
}


/* @brief Called when the system's locale has changed
 *
 * Application should load language dependent screen texts and probably set new 
 * labels for its menu items; to activate a new locale application should call:
 *             g_setenv("LANG", new_locale, TRUE);
 *             setlocale(LC_ALL, "");
 */
static void on_changed_locale ( eripc_context_t          *context,
                                const eripc_event_info_t *info,
                                void                     *user_data )
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if (arg_array[0].type == ERIPC_TYPE_STRING)
    {
        const char *locale = arg_array[0].value.s;
        if (locale)
        {
            const char *old_locale = g_getenv("LANG");
            if (!old_locale || (strcmp(old_locale, locale) != 0))
            {
                main_set_locale(locale);
            }
        }
    }
}


/* @brief Called when the display's orientation has changed
 *
 * Application may need to adapt its screen size, coordinates and/or origin.
 * Possible values: "portrait", "landscape_clockwise", "landscape_anticlockwise"
 */
static void on_changed_orientation ( eripc_context_t          *context,
                                     const eripc_event_info_t *info,
                                     void                     *user_data )
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;
    
    if (arg_array[0].type == ERIPC_TYPE_STRING)
    {
        const char *orientation = arg_array[0].value.s;
        if (orientation)
        {
            // TODO: Replace implementation
        }
    }
}
