/*
 * File Name: fileview.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include <libergtk/ergtk.h>
#include <liberutils/display_utils.h>

#include "ctb_log.h"
#include "ctb_actions.h"
#include "fileview.h"
#include "filefind.h"
#include "filetypes.h"
#include "measures.h"
#include "i18n.h"
#include "ipc.h"
#include "menu.h"
#include "views.h"
#include "db.h"

#define UNUSED(x) (void)(x)

//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

// actions when item clicked
typedef int (*ctb_item_callback)    ( const GString             *directory,
                                      const filelist_entry_t    *file_info );
//
typedef struct
        {
            const char        *name;          // name of this action
            ctb_item_callback item_callback;  // call for each item, parm is an item in the directory
        } ctb_action_t;

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// registry key "view type"
static const gchar      *REGKEY_VIEWTYPE         = "/apps/er/sys/ctb/viewtype";
static const gchar      *REGVAL_VIEWTYPE_ICON    = "iconview";
static const gchar      *REGVAL_VIEWTYPE_LIST    = "listview";
static const gchar      *REGVAL_VIEWTYPE_CONTENT = "listcontent";
static const gchar      *REGVAL_VIEWTYPE_AUTO    = "auto-select";

// registry key "startup view"
static const gchar      *REGKEY_STARTUPMODE      = "/apps/er/patch/ctb/startupmode";
static const gchar      *REGKEY_STARTUPDIR       = "/apps/er/patch/ctb/startupdir";

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

// screen elements
static GtkWidget        *g_main_window          = NULL;
static GtkWidget        *g_alphabar             = NULL;   // alpha search bar
static GtkWidget        *g_background           = NULL;
static ctb_viewtypes_t   g_current_view         = -1;
static GtkWidget        *g_iconview             = NULL;   // iconview widget
static GtkWidget        *g_listview             = NULL;   // listview widget
static GtkWidget        *g_eventbox             = NULL;
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static GtkWidget        *g_home_label           = NULL;
static GtkWidget        *g_home_separator       = NULL;
static GtkWidget        *g_home_eventbox         = NULL;
#endif
static GtkWidget        *g_title_label          = NULL;
static GtkWidget        *g_subtitle_label       = NULL;
static GtkWidget        *g_search_button        = NULL;
static GtkWidget        *g_search_dialog        = NULL;
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static GtkWidget        *g_shortcut_button      = NULL;
#endif

static gboolean         g_is_user_selected_view = FALSE;  // user has selected the current view
static ctb_viewmodes_t  g_viewmode              = BROWSE_MODE;
static GtkTreeViewColumn *g_toggleColumn        = NULL;   // extra column needed for selection stuff
static int g_handler_id                         = 0;      // id for the activated handler
static gboolean         g_got_display_control   = FALSE;  // have we taken display control
static gboolean         g_cursor_in_progress    = FALSE;
static gboolean         g_operation_in_progress = FALSE;
static int g_cursor_update_type = DM_HINT_FULL;

static int g_width = 0;
static int g_height = 0;

typedef enum
        {
            INITIAL = 0,
            CONFIRM,
        } delete_state;
static delete_state g_delete_state = -1;

// alpha_bar variables
static const char* letters[FILEMODEL_NUM_ALPHA] = {"#", "A", "B", "C", "D", "E", "F", "G", "H", "I", "J", "K", "L", "M", "N", "O", "P", "Q", "R", "S", "T", "U", "V", "W", "X", "Y", "Z"};
static GtkWidget *g_labels[FILEMODEL_NUM_ALPHA];
static int g_selected_letter = -1;
static int g_last_clicked_index = -1;

// startup view
static ViewMode gStartupViewMode = -1;
static gchar    *gStartupViewDir = NULL;

//============================================================================
// Local Function Definitions
//============================================================================

static gchar* get_filename_at_cursor();
static void set_cursor_at_filename(const gchar *cursor_item);
static void set_cursor(gint row, gint column);
static void setViewMode(ctb_viewmodes_t newmode, gboolean updateScreen);
static void scroll_to_letter(gchar letter, gboolean jump_to_dir);
static void alphabar_update();
static void alphabar_select(int new_index);
static void handle_activate_item();

//============================================================================
// Functions Implementation
//============================================================================

static void listview_set_column_headers()
{

    if (g_current_view != CTB_LISTVIEW) return;

    static const struct
    {
        const char *header;
        guint index;
    }           column_headers[] =
                {
                    { N_("Name"),           2 },
                    { N_("Type"),           4 },
                    { N_("Size"),           6 },
                    { N_("Date Modified"),  8 },
                    { NULL, 0 }  // end of list
                };

    int i;
    for (i = 0 ; column_headers[i].header ; i++)
    {
        GtkTreeViewColumn *column = gtk_tree_view_get_column( GTK_TREE_VIEW(g_listview), column_headers[i].index );
        if (column)
        {
            gtk_tree_view_column_set_title(column, _(column_headers[i].header));
        }
    }
}


static void update_titles()
{
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    gtk_label_set_text(GTK_LABEL(g_home_label), _("Home"));
    if (filemodel_current_is_desktop()) {
        gtk_widget_hide(g_home_eventbox);
        gtk_widget_hide(g_home_separator);
    } else {
        gtk_widget_show(g_home_eventbox);
        gtk_widget_show(g_home_separator);
    }
#endif
    gtk_label_set_text(GTK_LABEL(g_title_label), filemodel_get_title());
    gtk_label_set_text(GTK_LABEL(g_subtitle_label), filemodel_get_subtitle());
}


static void update_gui()
{
    g_last_clicked_index = -1;
    if (filemodel_has_searchbutton()) {
        gtk_widget_show(g_search_button);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        gtk_widget_show(g_shortcut_button);
#endif
    } else {
        gtk_widget_hide(g_search_button);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        gtk_widget_hide(g_shortcut_button);
#endif
    }

    alphabar_update();
    update_titles();
}


void fileview_update_screen_texts()
{
    listview_set_column_headers();
    fileview_refresh(TRUE);
    filemodel_set_delete_text(_("Finish Delete"));
    update_titles();
}


void fileview_grab_focus( void )
{
    switch (g_current_view)
    {
        case CTB_ICONVIEW:
            gtk_widget_grab_focus(GTK_WIDGET(g_iconview));
            break;
        case CTB_LISTVIEW:
        case CTB_CONTENTVIEW:
            gtk_widget_grab_focus(GTK_WIDGET(g_listview));
            break;
        default:
            break;
    }
}


static void update_padding()
{
    int left = WINDOW_H_PADDING;
    int right = WINDOW_H_PADDING;

    // alphabar visible
    if (GTK_WIDGET_VISIBLE(g_alphabar)) right -= 20;

    // icon/contenview
    switch (g_current_view) {
        case CTB_ICONVIEW:
            left += 20;
            if (!GTK_WIDGET_VISIBLE(g_alphabar)) left += 20;
            break;
        default:
            break;
    }

    guint delta = (ipc_sys_is_in_portrait_mode()) ? 0 : 5;
    gtk_alignment_set_padding( GTK_ALIGNMENT(g_background),
                               ( WINDOW_TOP_PADDING - delta),
                               WINDOW_BOTTOM_PADDING,
                               left, right);
}


void fileview_set_sort_order (const ctb_sort_order_t sort_order)
{
    fileview_stop_update_display();
    gchar *cursor_item = get_filename_at_cursor();

    gboolean order_has_changed = filemodel_set_sortorder( sort_order, cursor_item, TRUE);
    if (order_has_changed) {
        gtk_label_set_text(GTK_LABEL(g_subtitle_label), filemodel_get_subtitle());
        alphabar_update();
    }
    set_cursor_at_filename(cursor_item);
    g_free(cursor_item);
}


static void update_view_for_desktop()
{
    if (g_current_view != CTB_CONTENTVIEW && g_current_view != CTB_ICONVIEW)
    {
        fileview_set_view_type(CTB_CONTENTVIEW, FALSE);
    }
}


void fileview_dir_up()
{
    g_assert(g_viewmode == BROWSE_MODE);
    gchar* cursor_item = NULL;

    if (filemodel_get_dir_depth() == 1)
    {
        // next dir up is the desktop
        update_view_for_desktop();
    }

    cursor_item = filemodel_chdir_up();
    update_gui();

    if (cursor_item) {
        alphabar_select(filemodel_get_first_alpha_index());
        set_cursor_at_filename(cursor_item);
        g_free(cursor_item);
    } else
        ERRORPRINTF("no cursor set!");
}


void fileview_dir_down(const gchar *dir, const char* cursorname)
{
    LOGPRINTF("dir=%s  cursor=%s", dir, cursorname ? cursorname : "NULL");
    g_assert(g_viewmode == BROWSE_MODE);

    filemodel_chdir_down(dir, cursorname);

    update_gui();
    set_cursor(0, 0);
}


void fileview_show_desktop()
{
    fileview_stop_update_display();
    // cancel possible DELETE_MODE
    setViewMode(BROWSE_MODE, FALSE);
    update_view_for_desktop();
    filemodel_chdir_desktop();

    update_gui();
    set_cursor(0, 0);
}


ctb_viewtypes_t get_viewtype_from_registry ( void )
{
    LOGPRINTF("");

    // get registry value
    GConfClient *client = gconf_client_get_default();
    gchar* val = gconf_client_get_string(client, REGKEY_VIEWTYPE, NULL);
    g_object_unref(client);

    // convert string value to viewtype
    ctb_viewtypes_t view = CTB_AUTOSELECT;
    if (val)
    {
        if ( strcmp(val, REGVAL_VIEWTYPE_ICON) == 0 )
        {
            view = CTB_ICONVIEW;
        }
        else if ( strcmp(val, REGVAL_VIEWTYPE_LIST) == 0 )
        {
            view = CTB_LISTVIEW;
        }
        else if ( strcmp(val, REGVAL_VIEWTYPE_CONTENT) == 0 )
        {
            view = CTB_CONTENTVIEW;
        }
        g_free(val);
    } else {
        WARNPRINTF("error fetching GConf key %s", REGKEY_VIEWTYPE);
    }

    // for auto-select choose view depending on stylus availability
    if (view == CTB_AUTOSELECT)
    {
        const device_caps_t *dev_caps = ipc_sys_get_device_capabilities();
        g_assert(dev_caps);
        if (dev_caps->has_stylus)
        {
            view = CTB_ICONVIEW;
        }
        else
        {
            view = CTB_LISTVIEW;
        }
    }

    return view;
}

static void get_startup_view_from_registry ( void )
{
    LOGPRINTF("");

    // get registry value
    GConfClient *client = gconf_client_get_default();
    int mode   = gconf_client_get_int(client, REGKEY_STARTUPMODE, NULL);
    gchar* dir = gconf_client_get_string(client, REGKEY_STARTUPDIR, NULL);
    g_object_unref(client);
    
    gStartupViewMode = (ViewMode)mode;
    if (gStartupViewDir != NULL) 
    {
        g_free(gStartupViewDir); // release old value
        gStartupViewDir = NULL;
    }
    gStartupViewDir = dir;
    
}

void fileview_media_mounted (const gchar *mountpoint)
{
    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    ctb_viewtypes_t view = get_viewtype_from_registry();
    if (view != CTB_AUTOSELECT)
    {
        fileview_set_view_type(view, FALSE);
        g_is_user_selected_view = FALSE;
    }

    // Not starting the default view, but starting the one defined in the registry
    get_startup_view_from_registry();

    // Use mode and dir from gconf.
    // Not all views are supported yet!
    // TODO: Since a lot of knowledge from the model is needed, this code should better 
    //       be in filemodel.c
    //       This does not work for user views, can style be used?
    // TODO: The user get view-numbers based on their order in gconf, so in theory this order
    //       could change? (and it will when new views are added!!)
    //printf("Mode:%d, Dir:%s\n", (int)gStartupViewMode, gStartupViewDir); 
    switch (viewmode_get_style(gStartupViewMode))
    {
        // handle the different views, see example code in ctb_actions.c->handle_special_item()
        case DIR_VIEW:
            // this code needs to handle the dir view and all views derived from it!
            filemodel_set_viewmode2(DIR_VIEW);
            if (gStartupViewDir)
            {
                fileview_dir_down(gStartupViewDir, SPECIAL_DIR);
            }
            else
            {
                fileview_dir_down("/media/mmcblk0p1/Programs", SPECIAL_DIR);
            }
            break;
        case BOOKS_VIEW:
            // This code needs to handle the books-view and all 'tag'-views derived from it
            filemodel_set_viewmode2(gStartupViewMode);
            fileview_dir_down("/media/mmcblk0p1/Programs", SPECIAL_BOOKS);
            break;
        case RECENT_VIEW:
            // This code needs to handle the recent-view and all 'date'-views derived from it
            filemodel_set_viewmode2(gStartupViewMode);
            fileview_dir_down("/media/mmcblk0p1/Programs", SPECIAL_RECENT);
            break;
        // These views are not supported yet and default to the desktop-view
        case SETTINGS_VIEW:
        case NEWS_VIEW:
        case IMAGES_VIEW:
        case PERSONAL_VIEW:
        case HELP_VIEW:
        case NOTES_VIEW:
        case SHORTCUT_VIEW:
        case SEARCH_VIEW:
        case DESKTOP_VIEW:
            fileview_show_desktop();
            break;
        default:
            ERRORPRINTF("Unkown style for viewmode: %d", gStartupViewMode);
            break;
        
    }
}


static void save_viewtype_to_registry ( ctb_viewtypes_t view )
{
    GConfClient  *client  = NULL;
    gchar        *val_old = NULL;
    const gchar  *val_new = NULL;
    GError       *err     = NULL;

    LOGPRINTF("entry");

    // convert viewtype to string value
    switch (view)
    {
        case CTB_ICONVIEW:
            val_new = REGVAL_VIEWTYPE_ICON;
            break;
        case CTB_LISTVIEW:
            val_new = REGVAL_VIEWTYPE_LIST;
            break;
        case CTB_CONTENTVIEW:
            val_new = REGVAL_VIEWTYPE_CONTENT;
            break;
        default:
            val_new = REGVAL_VIEWTYPE_AUTO;
    }

    // save to registry
    client = gconf_client_get_default();
    val_old = gconf_client_get_string(client, REGKEY_VIEWTYPE, NULL);
    if (   val_old == NULL
        || strcmp(val_old, val_new) != 0 )
    {
        gboolean ok = gconf_client_set_string(client, REGKEY_VIEWTYPE, val_new, &err);
        if ( !ok )
        {
            ERRORPRINTF("cannot write registry key [%s] - error [%s]", REGKEY_VIEWTYPE, err->message);
        }
        g_clear_error(&err);
    }
    g_object_unref(client);

    g_free(val_old);
}

static void save_startup_view_to_registry ( void )
{
    GConfClient  *client  = NULL;
    GError       *err     = NULL;

    LOGPRINTF("entry");

    // save to registry
    client = gconf_client_get_default();
    ViewMode mode_old   = (ViewMode)gconf_client_get_int(client, REGKEY_STARTUPMODE, NULL);
    // TODO: Should use strings for every mode to allow better error checking. When key is not present
    //       0 is returned, which is a valid viewmode for DeskTop, which is the default view, so no 
    //       real problem so far
    gchar* dir_old = gconf_client_get_string(client, REGKEY_STARTUPDIR, NULL);
    if (   dir_old == NULL
        || strcmp(dir_old, gStartupViewDir) != 0 )
    {
        gboolean ok = gconf_client_set_string(client, REGKEY_STARTUPDIR, gStartupViewDir, &err);
        if ( !ok )
        {
            ERRORPRINTF("cannot write registry key [%s] - error [%s]", REGKEY_STARTUPDIR, err->message);
        }
        g_clear_error(&err);
    }
    if (mode_old != gStartupViewMode)
    {
        gboolean ok = gconf_client_set_int(client, REGKEY_STARTUPMODE, gStartupViewMode, &err);
        if ( !ok )
        {
            ERRORPRINTF("cannot write registry key [%s] - error [%s]", REGKEY_STARTUPMODE, err->message);
        }
        g_clear_error(&err);
    }
    g_object_unref(client);

    g_free(dir_old);
}


void fileview_save_view_type()
{
    if (g_is_user_selected_view)
    {
        save_viewtype_to_registry(g_current_view);
        g_is_user_selected_view = FALSE;
    }
}


void fileview_media_unmounted ()
{
    LOGPRINTF("entry");
    fileview_save_view_type();
    save_startup_view_to_registry();
    if (g_search_dialog) {
        gtk_widget_destroy(g_search_dialog);
        g_search_dialog = NULL;
    }
    setViewMode(BROWSE_MODE, FALSE);
    filemodel_chdir_desktop();
}


static void get_cursor ( gint *row, gint *column )
{
    // select first item and set cursor
    switch (g_current_view)
    {
        case CTB_ICONVIEW:
            g_assert(g_iconview);
            ergtk_icon_view_get_cursor( ERGTK_ICON_VIEW(g_iconview), row, column );
            break;
        case CTB_LISTVIEW:
        case CTB_CONTENTVIEW:
            g_assert(g_listview);
            ergtk_list_view_get_cursor( ERGTK_LIST_VIEW(g_listview), row);
            *column = 0;
            break;
        default:
            g_assert_not_reached();
    }
}


void fileview_stop_update_display()
{
    LOGPRINTF("got_control=%d", g_got_display_control);
    if (!g_got_display_control) {
        g_got_display_control = TRUE;
        display_gain_control();
    }
}


static gboolean update_display(gpointer data)
{
    LOGPRINTF("got_control=%d", g_got_display_control);
    if (g_got_display_control) {
        display_update_return_control((gint) data);
        g_got_display_control = FALSE;
        STOP_TIMER();
    }
    return FALSE;
}


static void on_idle_update_display(gint type)
{
    if (g_got_display_control) {
        g_idle_add(update_display, (gpointer) type);
    }
}


static gboolean on_idle_set_cursor( gpointer data )
{
    gulong  row_col = (gulong)data;
    gint    row    = (row_col >> 8) & 0xFF;
    gint    column =  row_col       & 0x00FF;

    // select first item and set cursor
    switch (g_current_view)
    {
        case CTB_ICONVIEW:
            g_assert(g_iconview);
            ergtk_icon_view_set_cursor( ERGTK_ICON_VIEW(g_iconview), row, column );
            break;
        case CTB_LISTVIEW:
        case CTB_CONTENTVIEW:
            g_assert(g_listview);
            ergtk_list_view_set_cursor( ERGTK_LIST_VIEW(g_listview), row);
            break;
        default:
            g_assert_not_reached();
    }
    g_cursor_in_progress = FALSE;
    on_idle_update_display(g_cursor_update_type);
    g_cursor_update_type = DM_HINT_FULL;
    return FALSE;  // don't call me again
}


static void set_cursor(gint row, gint column)
{
    if (!g_cursor_in_progress) {
        gulong row_col = ((row & 0xFF) << 8) | (column & 0xFF);

        g_idle_add(on_idle_set_cursor, (gpointer)row_col);
        g_cursor_in_progress = TRUE;
        g_operation_in_progress = FALSE;
    }
}


static int find_nearest_active(int index) {
    const gchar* list = filemodel_get_alpha_list();
    if (index >= 1 && list[index-1]) return index-1;
    if (index <= FILEMODEL_NUM_ALPHA-1 && list[index+1]) return index+1;
    return -1;
}


static gboolean on_letter_click(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
    UNUSED(widget);
    UNUSED(event);
    START_TIMER();
    const gchar* list = filemodel_get_alpha_list();
    int index = (int)user_data;
    if (list[index] == 0) {
        index = find_nearest_active(index);
    }
    if (index != -1) {
        fileview_stop_update_display();
        if (index != g_selected_letter) alphabar_select(index);

        // clicking on same letter twice jumps to file with initial, otherwise jump to directory
        gboolean jump_to_dir = TRUE;
        if (index == g_last_clicked_index) {
            jump_to_dir = FALSE;
            g_last_clicked_index = -1;
        } else {
            g_last_clicked_index = index;
        }
        scroll_to_letter(letters[index][0], jump_to_dir);
    }
    return FALSE;
}


// updates all letters with active/inactive state
static void alphabar_update()
{
    if (!filemodel_show_alphabar()) {
        gtk_widget_hide(g_alphabar);
        update_padding();
        return;
    } else {
        gtk_widget_show(g_alphabar);
        update_padding();
    }

    const char* list = filemodel_get_alpha_list();
    int selected = -1;
    int i;
    for (i=0; i<FILEMODEL_NUM_ALPHA; i++) {
        if (list[i] == 0) {
            gtk_widget_set_name(g_labels[i], "irex-alphabar-label-inactive");
        } else {
            // select first active letter by default
            if (selected == -1) selected = i;
            gtk_widget_set_name(g_labels[i], "irex-alphabar-label-active");
        }
    }
    if (selected != -1) {
        alphabar_select(selected);
    } else {
        alphabar_select(-1);
    }
}


static void alphabar_select(int new_index) {
    if (!GTK_WIDGET_VISIBLE(g_alphabar)) return;
    g_assert(new_index <= FILEMODEL_NUM_ALPHA);

    // deselect previously selected
    if (g_selected_letter != -1) {
        gtk_label_set_text(GTK_LABEL(g_labels[g_selected_letter]), letters[g_selected_letter]);
        g_selected_letter = -1;
    }
    if (new_index == -1) return;

    // select new entry
    const char* list = filemodel_get_alpha_list();
    g_assert(list[new_index] != 0);
    g_selected_letter = new_index;
    char buffer[4];
    snprintf(buffer, sizeof(buffer), "-%s-", letters[g_selected_letter]);
    gtk_label_set_text(GTK_LABEL(g_labels[g_selected_letter]), buffer);
}


static GtkWidget *create_alpha_bar()
{
    GtkWidget* vbox = gtk_vbox_new(TRUE, 0);
    g_alphabar = vbox;
    gtk_widget_set_size_request(vbox, 36, -1);  // force width to avoid later updates
    
    int i; 
    for (i=0; i<FILEMODEL_NUM_ALPHA; i++) {
        GtkWidget* letterbox = gtk_event_box_new();
        gtk_box_pack_start(GTK_BOX(vbox), letterbox, TRUE, TRUE, 0);
        GtkWidget* label = gtk_label_new(letters[i]);
        gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
        gtk_widget_set_name(label, "irex-alphabar-label-inactive");
        g_labels[i] = label;
        gtk_container_add(GTK_CONTAINER(letterbox), label); 
        g_signal_connect(GTK_EVENT_BOX(letterbox), "button-press-event",
            G_CALLBACK(on_letter_click), (void*)i);
    }
    return vbox;
}


#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static gboolean on_shortcut_clicked (GtkWidget* button, GdkEventButton* event, gpointer data)
{
    fileview_stop_update_display();
    if (!filemodel_current_is_desktop()) {
        filemodel_chdir_desktop();
    }
    filemodel_set_viewmode2(SHORTCUT_VIEW);
    char buffer[512];
    sprintf(buffer, "%s/%s", ipc_get_media(), DIR_SHORTCUTS);
    fileview_dir_down(buffer, SPECIAL_SHORTCUTS);
    return FALSE;
}
#endif


static gboolean on_file_search_clicked (GtkWidget* button, GdkEventButton* event, gpointer data)
{
    fileview_show_search_dialog();
    return FALSE;
}


#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static gboolean on_home_clicked (GtkWidget* button, GdkEventButton* event, gpointer data)
{
    fileview_show_desktop();
    return FALSE;
}
#endif


static GtkWidget* create_screen_layout()
{
    // object hierarchy:
    //     background (alignment)
    //       |
    GtkWidget *background = gtk_alignment_new(0.0, 0.0, 1.0, 1.0);

    guint delta = (ipc_sys_is_in_portrait_mode()) ? 0 : 5;
    gtk_alignment_set_padding( GTK_ALIGNMENT(background),
                               (WINDOW_TOP_PADDING - delta),
                               WINDOW_BOTTOM_PADDING,
                               WINDOW_H_PADDING,
                               WINDOW_H_PADDING  );
    g_background = background;
    gtk_widget_show(background);
    //       |
    //       |-- vbox
    //             |
    GtkWidget *widget = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(background), widget);
    gtk_widget_show(widget);
    GtkBox *vbox = GTK_BOX(widget);
    //             |
    //             |-- hbox
    //             |     |
    widget = gtk_hbox_new(FALSE, TITLE_SPACING);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, TITLE_V_PADDING);
    gtk_widget_set_size_request(widget, -1, TITLE_HEIGHT);
    GtkBox *hbox = GTK_BOX(widget);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    //             |     |
    //             |     | -- home event box
    GtkWidget* home_event_box = gtk_event_box_new();
    gtk_box_pack_start(hbox, home_event_box, FALSE, FALSE, 0);
    gtk_widget_set_events(home_event_box, GDK_BUTTON_PRESS_MASK);
    g_signal_connect(home_event_box, "button-press-event" , G_CALLBACK(on_home_clicked), NULL);
    gtk_widget_show(home_event_box);
    g_home_eventbox = home_event_box;
    //             |     |
    //             |     | -- home_label
    GtkWidget* home_label = gtk_label_new(NULL);
    gtk_label_set_ellipsize(GTK_LABEL(home_label), PANGO_ELLIPSIZE_NONE);
    gtk_widget_set_name(home_label, "irex-ctb-title");
    gtk_container_add(GTK_CONTAINER(home_event_box), home_label);
    g_home_label = home_label;
    //             |     |
    //             |     | -- home_separator
    GtkWidget* home_separator = gtk_image_new_from_file(DATADIR"/pathbar-separator.png");
    gtk_widget_show(home_separator);
    gtk_box_pack_start(hbox, home_separator, FALSE, FALSE, 0);
    g_home_separator = home_separator;
#endif
    //             |     |
    //             |     | -- title_label
    GtkWidget* title_label = gtk_label_new(NULL);
    gtk_label_set_ellipsize(GTK_LABEL(title_label), PANGO_ELLIPSIZE_NONE);
    gtk_widget_set_name(title_label, "irex-ctb-title");
    gtk_box_pack_start(hbox, title_label, FALSE, FALSE, 0);
    g_title_label = title_label;
    //             |     |
    //             |     | -- subtitle_label
    GtkWidget* subtitle_label = gtk_label_new(NULL);
    gtk_label_set_ellipsize(GTK_LABEL(subtitle_label), PANGO_ELLIPSIZE_END);
    gtk_misc_set_alignment(GTK_MISC(subtitle_label), 0.0, 0.5);
    gtk_widget_set_name(subtitle_label, "irex-ctb-subtitle");
    gtk_box_pack_start(hbox, subtitle_label, TRUE, TRUE, 0);
    g_subtitle_label = subtitle_label;
    //             |    |
    char path[512];
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    //             |    | -- Shortcut event box "button"
    GtkWidget* sc_event_box = gtk_event_box_new();
    gtk_box_pack_end(hbox, sc_event_box, FALSE, FALSE, 0);
    gtk_widget_show(sc_event_box);
    snprintf (path, 512, "%s/%s", DATADIR, "icon-shortcut-button.png");
    GtkWidget* sc_image = gtk_image_new_from_file( (gchar*) path );
    gtk_container_add(GTK_CONTAINER(sc_event_box), sc_image);
    gtk_widget_show(sc_image);
    gtk_widget_set_events(sc_event_box, GDK_BUTTON_PRESS_MASK);
    g_signal_connect( sc_event_box, "button-press-event" , G_CALLBACK(on_shortcut_clicked), NULL);
    g_shortcut_button = sc_event_box;
#endif
    //             |    |
    //             |    | -- File Find event box "button"
    GtkWidget* ff_event_box = gtk_event_box_new();
    gtk_box_pack_end(hbox, ff_event_box, FALSE, FALSE, 0);
    gtk_widget_show(ff_event_box);
    snprintf (path, 512, "%s/%s", DATADIR, "icon-search-button.png");
    GtkWidget* search_image = gtk_image_new_from_file( (gchar*) path );
    gtk_container_add(GTK_CONTAINER(ff_event_box), search_image);
    gtk_widget_show(search_image);
    gtk_widget_set_events(ff_event_box, GDK_BUTTON_PRESS_MASK);
    g_signal_connect( ff_event_box, "button-press-event" , G_CALLBACK(on_file_search_clicked), NULL);
    g_search_button = ff_event_box;
    //             |
    gtk_widget_show_all(widget);
    //             |-- hbox2
    //             |     |
    GtkWidget* hbox2 = gtk_hbox_new(FALSE, 20); 
    gtk_box_pack_start(vbox, hbox2, TRUE, TRUE, 0);
    gtk_widget_show(hbox2);
    //             |     |
    //             |     | -- g_eventbox
    widget = gtk_event_box_new();
    gtk_widget_show(widget);
    gtk_box_pack_start(GTK_BOX(hbox2), widget, TRUE, TRUE, 0);
    g_eventbox = widget;
    //             |     |
    //             |     | -- alpha_bar
    GtkWidget *alphabar = create_alpha_bar();
    gtk_box_pack_start(GTK_BOX(hbox2), alphabar, FALSE, FALSE, 0);
    gtk_widget_show_all(alphabar);

    return background;
}


static void on_iconview_item_activated ( GtkIconView *iconview,
                                         GtkTreePath *path,
                                         gpointer    user_data )
{
    handle_activate_item();
}


static void fileview_page_previous(int new_row, int new_col) {
    fileview_stop_update_display();
    filemodel_page_previous();
    alphabar_select(filemodel_get_first_alpha_index());
    set_cursor(new_row, new_col);
}


static void fileview_page_next(int new_row, int new_col) {
    fileview_stop_update_display();
    filemodel_page_next();
    alphabar_select(filemodel_get_first_alpha_index());
    set_cursor(new_row, new_col);
}


static void on_iconview_navigate_cursor ( erGtkIconView         *er_iconview,
                                          erGtkIconViewKeyPress keycode,
                                          gpointer              user_data )
{
    // old iconview details
    gint        old_row;
    gint        old_col;
    gint        num_rows;
    gint        num_cols;
    gint        num_items;

    // new iconview details
    gboolean    prev_page = FALSE;      // goto previous page
    gboolean    next_page = FALSE;      // goto next page
    gint        new_row = 0;
    gint        new_col = 0;

    LOGPRINTF("entry: keycode [%d]", keycode);

    // get iconview details
    ergtk_icon_view_get_cursor(er_iconview, &old_row, &old_col);
    ergtk_icon_view_get_view_size(er_iconview, &num_rows, &num_cols, &num_items);

    // determine new cursor position
    switch (keycode)
    {
        case ERGTK_ICON_VIEW_PRESS_SHORT_UP:
            // previous page, last row, same column
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_DOWN:
            // next page, first row, same column
            next_page = TRUE;
            new_row   = 0;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_LEFT:
            // previous page, last item
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = num_cols - 1;
            break;

        case ERGTK_ICON_VIEW_PRESS_SHORT_RIGHT:
            // next page, first item
            next_page = TRUE;
            new_row   = 0;
            new_col   = 0;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_UP:
            // previous page, same position
            prev_page = TRUE;
            new_row   = old_row;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_DOWN:
            // next page, same position
            next_page = TRUE;
            new_row   = old_row;
            new_col   = old_col;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_LEFT:
            // previous page, start of last row
            prev_page = TRUE;
            new_row   = num_rows - 1;
            new_col   = 0;
            break;

        case ERGTK_ICON_VIEW_PRESS_LONG_RIGHT:
            // next page, end of first row
            next_page = TRUE;
            new_row   = 0;
            new_col   = num_cols - 1;
            break;

        default:
            ERRORPRINTF("illegal erGtkIconViewKeyPress [%d]", keycode);
            ;  // ignore
    }

    // move page as needed, set new cursor position
    if ( prev_page  &&  filemodel_has_prev_page() )
    {
        fileview_page_previous(new_row, new_col);
    }
    else if ( next_page  &&  filemodel_has_next_page())
    {
        fileview_page_next(new_row, new_col);
    }
}


static void update_view_size()
{
    int num_items = 1;

    switch (g_current_view)
    {
        case CTB_ICONVIEW:
            if ( g_iconview  &&  GTK_WIDGET_VISIBLE(g_iconview) )
            {
                int num_columns = 1;
                int num_rows  = 1;
                erGtkIconView *iconview = ERGTK_ICON_VIEW(g_iconview);
                ergtk_icon_view_get_view_size(iconview, &num_columns, &num_rows, NULL);
                num_items = num_rows * num_columns;
            }
            break;
        case CTB_LISTVIEW:
        case CTB_CONTENTVIEW:
            if ( g_listview  &&  GTK_WIDGET_VISIBLE(g_listview) )
            {
                int num_rows  = 1;
                erGtkListView *listview = ERGTK_LIST_VIEW(g_listview);
                ergtk_list_view_get_view_size(listview, &num_rows, &num_items);
                num_items = num_rows;
            }
            break;
        default:
            g_assert_not_reached();
    }

    if (num_items > 0) {
        gboolean changed = filemodel_set_viewsize(num_items, !g_operation_in_progress);
        if (changed && !g_operation_in_progress) {
            alphabar_select(filemodel_get_first_alpha_index());
            set_cursor(0,0);
        }
    }
}


static void on_listview_size_allocate ( GtkWidget     *widget,
                                        GtkAllocation *allocation,
                                        gpointer      user_data   )
{
    g_width = allocation->width;
    g_height = allocation->height;
    if (filemodel_window_is_on_top()) update_view_size();
}


static void on_listview_row_activated ( GtkTreeView       *view,
                                        GtkTreePath       *path,
                                        GtkTreeViewColumn *column,
                                        gpointer          user_data )
{
    handle_activate_item();
}


static void on_listview_navigate_cursor ( erGtkListView         *er_listview,
                                          erGtkListViewKeyPress keycode,
                                          gpointer              user_data )
{
    // old listview details
    gint        old_row;
    gint        num_rows;
    gint        num_items;

    // new listview details
    gboolean    prev_page = FALSE;      // goto previous page
    gboolean    next_page = FALSE;      // goto next page
    gint        new_row = 0;

    LOGPRINTF("entry: keycode [%d]", keycode);

    // get listview details
    ergtk_list_view_get_cursor(er_listview, &old_row);
    ergtk_list_view_get_view_size(er_listview, &num_rows, &num_items);

    // determine new cursor position
    switch (keycode)
    {
        case ERGTK_LIST_VIEW_PRESS_SHORT_UP:
            // previous page, last row
            prev_page = TRUE;
            new_row   = num_rows - 1;
            break;

        case ERGTK_LIST_VIEW_PRESS_SHORT_DOWN:
            // next page, first row
            next_page = TRUE;
            new_row   = 0;
            break;

        case ERGTK_LIST_VIEW_PRESS_LONG_UP:
            if (filemodel_has_prev_page()) {
                // previous page, same row
                prev_page = TRUE;
                new_row   = old_row;
            } else {
                // first row on same page
                set_cursor(new_row, 0);
                return;
            }
            break;

        case ERGTK_LIST_VIEW_PRESS_LONG_DOWN:
            if (filemodel_has_next_page()) {
                // next page, same row
                next_page = TRUE;
                new_row   = old_row;
            } else {
                // last row on same page
                set_cursor(40, 0);
                return;
            }
            break;

        default:
            ERRORPRINTF("illegal erGtkListViewKeyPress [%d]", keycode);
            ;  // ignore
    }

    // move page as needed, set new cursor position
    if ( prev_page  &&  filemodel_has_prev_page() )
    {
        fileview_page_previous(new_row, 0);
    }
    else if ( next_page  &&  filemodel_has_next_page() )
    {
        fileview_page_next(new_row, 0);
    }
}


static gboolean isFirstItem(GtkTreeModel *model, GtkTreeIter* iter)
{
    GtkTreeIter first;
    gtk_tree_model_get_iter_first(model, &first);
    if (iter->user_data == first.user_data) return TRUE;
    return FALSE;
}


static int iter_to_index(GtkTreeModel* model, const GtkTreeIter *iter)
{
    int index = 0;
    GtkTreeIter iter2;
    gtk_tree_model_get_iter_first(model, &iter2);
    while (gtk_tree_model_iter_next(model, &iter2)) {
        index++;
        if (iter->user_data == iter2.user_data) break;
    }
    return index;
}


static void reset_delete_state()
{
    if (g_delete_state != INITIAL) {
        g_delete_state = INITIAL;
        filemodel_set_delete_text(_("Finish Delete"));
    }
}


static void handle_delete_activated(GtkTreeModel* model, GtkTreeIter* iter)
{
    START_TIMER();
    fileview_stop_update_display();
    if (isFirstItem(model, iter)) {    // special item (Delete action)
        // check whether first click or confirm
        if (g_delete_state == INITIAL) {
            int count = filemodel_num_toggled();
            if (count > 0) {
                g_delete_state = CONFIRM;
                char buf[128];
                // TRANSLATORS: this is an ngettext(3) call which needs a singular and plural translation
                snprintf(buf, 127, ngettext("Finish Delete - delete %d item?", "Finish Delete - delete %d items?", count), count);
                filemodel_set_delete_text(buf);
            } else {
                setViewMode(BROWSE_MODE, TRUE);
                return;
            }
        } else if (g_delete_state == CONFIRM) {
            ipc_sys_busy(TRUE);
            filemodel_delete_toggled();
            alphabar_update();
            setViewMode(BROWSE_MODE, TRUE);
            ipc_sys_busy(FALSE);
            return;
        }
    } else {    // normal item
        int toggled;
        gtk_tree_model_get (model, iter, MODCOL_TOGGLED, &toggled, -1);
        if (toggled != -1) {        // -1 means not allowed
            toggled = 1 - toggled;
            filemodel_toggle_entry(iter_to_index(model, iter), toggled, iter);
            reset_delete_state();
        }
    }
    on_idle_update_display(DM_HINT_CURSOR);
}


static void on_iconview_item_activated_delete(GtkIconView *iconview,
                                              GtkTreePath *path,
                                              gpointer    user_data )
{
    GtkTreeModel* model = get_filemodel();
    GtkTreeIter iter;
    gtk_tree_model_get_iter(model, &iter, path);

    handle_delete_activated(model, &iter);
}


static void on_contentview_row_activated_delete( GtkTreeView       *view,
                                                 GtkTreePath       *path,
                                                 GtkTreeViewColumn *column,
                                                 gpointer          user_data )
{
    GtkTreeSelection *selection = gtk_tree_view_get_selection(view);
    GtkTreeModel *model;
    GtkTreeIter iter;
    if (!gtk_tree_selection_get_selected(selection, &model, &iter)) {
        WARNPRINTF("no row selected");
        return;
    }
    handle_delete_activated(model, &iter);
}


static void toggle_cell_function (GtkTreeViewColumn *col,
                                    GtkCellRenderer   *renderer,
                                    GtkTreeModel      *model,
                                    GtkTreeIter       *iter,
                                    gpointer           user_data)
{
    int toggled;
    gtk_tree_model_get (model, iter, MODCOL_TOGGLED, &toggled, -1);

    GdkPixbuf *icon = NULL;
    gboolean show = TRUE;
    
    if (toggled == -1) {
        show = FALSE;
    } else {
        filemodel_thumbsize_t size = MODTHUMB_MINI;
        switch (g_current_view)
        {
            case CTB_LISTVIEW:
                size = MODTHUMB_MINI;
                break;
            case CTB_CONTENTVIEW:
                size = MODTHUMB_SMALL;
                break;
            default:
                size = MODTHUMB_MEDIUM;
                break;
        }
        icon = get_icon_delete_toggle(toggled, size);
    }

    g_object_set(G_OBJECT(renderer),
        "visible", show,
        "pixbuf", icon,
        NULL);
}


static void setViewModeIconView(ctb_viewmodes_t newmode)
{
    switch (newmode) {
        case BROWSE_MODE:
        {
            // change activated handler
            if (g_handler_id != 0) g_signal_handler_disconnect(g_iconview, g_handler_id);
            g_handler_id = g_signal_connect(G_OBJECT(g_iconview), "item-activated",  G_CALLBACK(on_iconview_item_activated),  NULL);
            break;
        }
        case DELETE_MODE:
        {
            // change activated handler
            if (g_handler_id != 0) g_signal_handler_disconnect(g_iconview, g_handler_id);
            g_handler_id = g_signal_connect(G_OBJECT(g_iconview), "item-activated",  G_CALLBACK(on_iconview_item_activated_delete),  NULL);
            break;
        }
    }
}


static void setViewModeListView(ctb_viewmodes_t newmode, int position, int width)
{
    GtkTreeView *view = GTK_TREE_VIEW(g_listview);
    switch (newmode) {
        case BROWSE_MODE:
        {
            // remove toggle column
            if (g_toggleColumn) gtk_tree_view_remove_column(view, g_toggleColumn);
            g_toggleColumn = NULL;

            // change activated handler
            if (g_handler_id != 0) g_signal_handler_disconnect(g_listview, g_handler_id);
            g_handler_id = g_signal_connect(g_listview, "row-activated", G_CALLBACK(on_listview_row_activated), NULL);
            break;
        }
        case DELETE_MODE:
        {
            // add toggle column
            GtkCellRenderer *renderer = gtk_cell_renderer_pixbuf_new();
            GtkTreeViewColumn *column = gtk_tree_view_column_new();
            gtk_tree_view_column_pack_start (column, renderer, TRUE);
            gtk_tree_view_column_set_cell_data_func(column, renderer, toggle_cell_function, NULL, NULL);
            gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
            gtk_tree_view_column_set_fixed_width(column, width);
            gtk_tree_view_insert_column(view, column, position);
            g_toggleColumn = column;

            // change activated handler
            if (g_handler_id != 0) g_signal_handler_disconnect(g_listview, g_handler_id);
            g_handler_id = g_signal_connect(g_listview, "row-activated", G_CALLBACK(on_contentview_row_activated_delete), NULL);
            break;
        }
    }
}


static void setViewMode(ctb_viewmodes_t newmode, gboolean updateScreen)
{
    if (g_viewmode == newmode) return;

    g_viewmode = newmode;
    filemodel_set_viewmode(newmode, updateScreen);
    reset_delete_state();

    if (newmode == DELETE_MODE) {
        gtk_widget_hide(g_search_button);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        gtk_widget_hide(g_shortcut_button);
#endif
    } else {
        gtk_widget_show(g_search_button);
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        gtk_widget_show(g_shortcut_button);
#endif
    }

    switch (g_current_view)
    {
        case CTB_ICONVIEW:
            setViewModeIconView(newmode);
            break;
        case CTB_LISTVIEW:
            setViewModeListView(newmode, -1, 50);
            break;
        case CTB_CONTENTVIEW:
            setViewModeListView(newmode, 3, 80);
            break;
        default:
            g_assert_not_reached();
    }

    if (updateScreen) set_cursor(0, 0);
}


void show_error_dialog (const gchar *msg)
{
    GtkWidget *dialog = gtk_message_dialog_new(
                            GTK_WINDOW(g_main_window),
                            GTK_DIALOG_DESTROY_WITH_PARENT,
                            GTK_MESSAGE_ERROR,
                            GTK_BUTTONS_OK,
                            msg );
    gtk_window_set_deletable( GTK_WINDOW(dialog), FALSE );
    ipc_menu_block();
    gtk_dialog_run( GTK_DIALOG(dialog) );
    fileview_stop_update_display();
    ipc_menu_unblock();
    gtk_widget_destroy( dialog );
    // Note: sometimes we only get partial update, force fullscreen one
    on_idle_update_display(DM_HINT_FULL);
}


void fileview_show_search_dialog()
{ 
    ipc_menu_block();
    GtkWidget* dialog = filefind_dialog_create(g_main_window);
    g_search_dialog = dialog;
    gint result = gtk_dialog_run(GTK_DIALOG(dialog));
    if (result != GTK_RESPONSE_NONE) fileview_stop_update_display();
    switch (result)
    {
        case GTK_RESPONSE_ACCEPT:
        {
            if (!filemodel_current_is_desktop()) {
                filemodel_chdir_desktop();
            }
            const gchar* search_string = filefind_get_text(); 
            filemodel_set_search_filter(search_string);
            filemodel_set_viewmode2(SEARCH_VIEW);
            fileview_dir_down(DIR_LIBRARY, SPECIAL_SEARCH);
        }
            break;
        default:
            // do nothing
            break;
    }
    ipc_menu_unblock();
    if (g_search_dialog) {
        gtk_widget_destroy(dialog);
        g_search_dialog = NULL;
    }
    fileview_grab_focus();
    if (result != GTK_RESPONSE_ACCEPT && result != GTK_RESPONSE_NONE) {
        // Note: sometimes we only get partial update, force fullscreen one
        on_idle_update_display(DM_HINT_FULL);
    }
}

gchar *fileview_show_properties_dialog(const gchar          *title,
                                       const gchar          *old_title,
                                       const gchar          *old_author,
                                       const gchar          *old_tag,
                                       gchar                **new_title,
                                       gchar                **new_author,
                                       gchar                **new_tag)
{
    GtkWidget *dialog;
    GtkWidget *label;
    GtkWidget *entry_title;
    GtkWidget *entry_author;
    GtkWidget *entry_tag;
    gchar     *new_value = NULL;

    LOGPRINTF("entry: %s %s %s\n", old_title, old_author, old_tag);
    // object hierarchy:
    //       dialog
    //         |
    /* Create the widgets */
    dialog = gtk_dialog_new_with_buttons (title,
                                          GTK_WINDOW(g_main_window),
                                          GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                          GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
                                          NULL);
    gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
    gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2); /* 2 * 5 + 2 = 12 */
    gtk_container_set_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);

    //--------------------------------------------------------------------------
    label = gtk_label_new("Title");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0);

    entry_title = gtk_entry_new();
    gtk_entry_set_text ( GTK_ENTRY (entry_title), old_title );
    gtk_editable_select_region (GTK_EDITABLE (entry_title), 0, -1);
    gtk_entry_set_activates_default (GTK_ENTRY (entry_title), TRUE);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), entry_title, TRUE, TRUE, 0);

    gtk_widget_show(label);
    gtk_widget_show(entry_title);

    //--------------------------------------------------------------------------
    label = gtk_label_new("Author");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0);

    entry_author = gtk_entry_new();
    gtk_entry_set_text ( GTK_ENTRY (entry_author), old_author );
    gtk_editable_select_region (GTK_EDITABLE (entry_author), 0, -1);
    gtk_entry_set_activates_default (GTK_ENTRY (entry_author), TRUE);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), entry_author, TRUE, TRUE, 0);

    gtk_widget_show(label);
    gtk_widget_show(entry_author);

    //--------------------------------------------------------------------------
    label = gtk_label_new("Give Value for New Tag(s)");
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), label, TRUE, TRUE, 0);

    entry_tag = gtk_entry_new();
    gtk_entry_set_text ( GTK_ENTRY (entry_tag), old_tag );
    gtk_editable_select_region (GTK_EDITABLE (entry_tag), 0, -1);
    gtk_entry_set_activates_default (GTK_ENTRY (entry_tag), TRUE);

    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), entry_tag, TRUE, TRUE, 0);

    gtk_widget_show(label);
    gtk_widget_show(entry_tag);

    //--------------------------------------------------------------------------
    gtk_widget_grab_focus (entry_title);

    if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
    {
        // TODO: handle empty strings??
        *new_title  = g_strdup (gtk_entry_get_text ( GTK_ENTRY (entry_title) )); 
        *new_author = g_strdup (gtk_entry_get_text ( GTK_ENTRY (entry_author) )); 
        *new_tag    = g_strdup (gtk_entry_get_text ( GTK_ENTRY (entry_tag) ));
        new_value = *new_title;
    }

    gtk_widget_destroy(dialog);
    
    return new_value;
}

void fileview_refresh(gboolean force_reload)
{
    LOGPRINTF("");

    if (filemodel_window_is_on_top()) fileview_stop_update_display();

    // cancel possible DELETE_MODE
    setViewMode(BROWSE_MODE, FALSE);

    gchar *cursor_item = get_filename_at_cursor();
    if (filemodel_resync(force_reload)) {
        alphabar_update();
        filemodel_scroll_to_filename(cursor_item);
        alphabar_select(filemodel_get_first_alpha_index());
    }

    set_cursor_at_filename(cursor_item);
    g_free(cursor_item);
}


void fileview_set_on_top(gboolean ontop)
{
    if (ontop) {
        START_TIMER();
    } else {
        STOP_TIMER();
    }
    filemodel_set_window_is_on_top(ontop);
}


void fileview_toggle_delete_mode()
{
    fileview_stop_update_display();
    ctb_viewmodes_t newmode = BROWSE_MODE;
    switch (g_viewmode) {
    case BROWSE_MODE:
        newmode = DELETE_MODE;
        break;
    case DELETE_MODE:   // fallthrough
    default:
        newmode = BROWSE_MODE;
        break;
    }
    setViewMode(newmode, TRUE);
}


static filelist_entry_t* get_entry_at_cursor()
{
    filelist_entry_t *entry = NULL;

    switch (g_current_view) {
        case CTB_ICONVIEW:
        {
            GtkIconView *iconview = GTK_ICON_VIEW(g_iconview);
            GtkTreePath *path = NULL;
            gboolean cursor = gtk_icon_view_get_cursor(iconview, &path, NULL);
            if (cursor) {
                GtkTreeModel* model = get_filemodel();
                GtkTreeIter iter;
                gtk_tree_model_get_iter(model, &iter, path);
                entry = iter_to_entry(model, &iter);
            }
            gtk_tree_path_free(path);
            break;
        }
        case CTB_LISTVIEW:
        case CTB_CONTENTVIEW:
        {
            GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(g_listview));
            GtkTreeModel *model;
            GtkTreeIter iter;
            if (gtk_tree_selection_get_selected(selection, &model, &iter)) {
                entry = iter_to_entry(model, &iter);
            }
            break;
        }
        default:
            break;
    }

    if (entry) {
        if (strcmp(entry->directory_path->str, ".") == 0) {
            g_string_assign(entry->directory_path, filemodel_get_current_dir());
        }
    }
    return entry;
}


void fileview_create_shortcut()
{
    filelist_entry_t *entry = get_entry_at_cursor();
    if (entry) {
        create_shortcut_item(entry);
        filelist_entry_free(entry);
    }
}

void fileview_edit_properties( void )
{
    filelist_entry_t *entry = get_entry_at_cursor();
    if (entry) {
        edit_properties_item(entry);
        filelist_entry_free(entry);
        fileview_refresh(FALSE); // title/author changed
    }
}

void fileview_set_as_startup_view( void )
{
    // Accepting all views here, see fileview_media_mounted() for the supported views
    gStartupViewMode = filemodel_get_viewmode2();
    if (gStartupViewDir) g_free(gStartupViewDir);
    gStartupViewDir = g_strdup(filemodel_get_current_dir());
    
    // store in registry
    save_startup_view_to_registry();
}

static void handle_activate_item()
{
    START_TIMER();
    filelist_entry_t *entry = get_entry_at_cursor();
    if (entry) {
        activate_item(entry);
        // handle 'autoremove' views
        if (viewmode_get_mode(filemodel_get_viewmode2()) & VIEW_AUTOREMOVE)
        {
            int rc;
            metadata_table *values = NULL;
            
            // current view is a autoremove view, so remove the tag of this view
            // from the started file
            // get the current values
            rc = db_query_get_tag(entry->filename->str, entry->directory_path->str, &values);
            if (values)
            {
                const metadata_cell *cell = (const metadata_cell*) (values->cell_data->data);
                if (cell->type != METADATA_TEXT) {
                    printf("illegal cell type [%d] for tag", cell->type);
                    metadata_table_free(values);
                }
                else
                {
                    char *tags = cell->value.v_text->str; // the tags from the document
                    if (tags)
                    {
                        // create empty string with length of original to strip out the tag
                        gchar *new_tag = g_strnfill(strlen(tags)+1, '\0');
                        gchar *needle;
                        
                        // remove tag of this view from it
                        while(tags)
                        {
                            // 1) "tag bbbbb"
                            // 2) "aaaaaa tag bbbbbb"
                            // 3) "aaaaaa tag"
                            needle = strstr(tags, viewmode_get_tag_filter(filemodel_get_viewmode2()));
                            if (needle)
                            {
                                // copy from tags to needle to new_tag
                                if (tags != needle) // tags starts with the tag
                                {
                                    strncat(new_tag, tags, (needle-tags));
                                }
                                // move to next position (after previous tag)
                                tags = needle + strlen(viewmode_get_tag_filter(filemodel_get_viewmode2()));
                            }
                            else
                            {
                                // copy from tags to end of string to new_tag
                                if (strlen(tags) > 0)
                                {
                                    strcat(new_tag, tags);
                                }
                                tags = NULL; // reached end of tags
                            }
                        }
                        
                        // strip space
                        g_strstrip(new_tag);
                        
                        // modify with returned value
                        if (strlen(new_tag) > 0)
                        {
                            db_query_update_tag(entry->filename->str, entry->directory_path->str, new_tag);
                        }
                        g_free(new_tag);
                    }
                    if (values) metadata_table_free(values);
                }
            }
        }
        filelist_entry_free(entry);
    }
}


void fileview_show_prev_page()
{
    START_TIMER();
    gint row    = 0;
    gint column = 0;
    get_cursor(&row, &column);
    fileview_page_previous(row, column);
}


void fileview_show_next_page()
{
    START_TIMER();
    gint row    = 0;
    gint column = 0;
    get_cursor(&row, &column);
    fileview_page_next(row, column);
}


static gchar* get_filename_at_cursor()
{
    gchar *filename = NULL;
    filelist_entry_t *entry = get_entry_at_cursor();
    if (entry) {
        filename = g_strdup(entry->filename->str);
        filelist_entry_free(entry);
    }
    return filename;
}


static void set_cursor_at_index(int index)
{
    gint  row   = 0;
    gint  col   = 0;

    if (index > 0)
    {
        switch (g_current_view)
        {
            case -1:
                WARNPRINTF("viewtype not set");
                break;
            case CTB_ICONVIEW:
            {
                g_assert(g_iconview);
                erGtkIconView *iconview = ERGTK_ICON_VIEW(g_iconview);
                gint n_cols;
                ergtk_icon_view_get_view_size(iconview, NULL, &n_cols, NULL);
                if (n_cols != 0) {  // when switching view, n_cols is sometimes zero!!
                    row = index / n_cols;
                    col = index % n_cols;
                }
                break;
            }
            case CTB_LISTVIEW:
            case CTB_CONTENTVIEW:
                row = index;
                col = 0;
                break;
            default:
                g_assert_not_reached();
        }
    }
    set_cursor(row, col);
}


static void scroll_to_letter(gchar letter, gboolean jump_to_dir)
{
    gboolean same_page = FALSE;
    int index = filemodel_scroll_to_letter(letter, jump_to_dir, &same_page);
    if (same_page) g_cursor_update_type = DM_HINT_CURSOR;
    else g_cursor_update_type = DM_HINT_FULL;
    set_cursor_at_index(index);
}


static void set_cursor_at_filename(const gchar *cursor_item)
{
    int index = filemodel_get_display_index(cursor_item);
    set_cursor_at_index(index);
}


static gboolean on_idle_set_cursor_at_filename ( gpointer data )
{
    gchar *filename = (gchar*)data;

    filemodel_scroll_to_filename(filename);
    alphabar_select(filemodel_get_first_alpha_index());
    set_cursor_at_filename(filename);

    g_free(filename);

    return FALSE;  // don't call me again
}


static void fileview_set_cursor_at_filename ( const gchar *filename )
{
    if (filename && *filename) {
        g_idle_add(on_idle_set_cursor_at_filename, g_strdup(filename));
    }
}


int fileview_get_xid()
{
    return GDK_WINDOW_XID(g_main_window->window);
}


void fileview_move_up(guint delta)
{
    if (delta > WINDOW_TOP_PADDING )
    {
        delta = WINDOW_TOP_PADDING;
    }
    gtk_alignment_set_padding( GTK_ALIGNMENT(g_background),
                               (WINDOW_TOP_PADDING - delta),
                               WINDOW_BOTTOM_PADDING,
                               WINDOW_H_PADDING,
                               WINDOW_H_PADDING  );
}

void fileview_move_undo()
{
    gtk_alignment_set_padding( GTK_ALIGNMENT(g_background),
                               WINDOW_TOP_PADDING,
                               WINDOW_BOTTOM_PADDING,
                               WINDOW_H_PADDING,
                               WINDOW_H_PADDING  );
}


static void on_iconview_size_allocate ( GtkWidget     *widget,
                                        GtkAllocation *allocation,
                                        gpointer      user_data   )
{
    g_width = allocation->width;
    g_height = allocation->height;
    if (filemodel_window_is_on_top()) update_view_size();
}


static GtkWidget* create_iconview()
{
    GtkWidget *widget = ergtk_icon_view_new_with_model( get_filemodel() );
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
    g_object_set(G_OBJECT(widget), "navigate-mode", "simple-navigate-mode", NULL);
#endif 
    g_signal_connect(G_OBJECT(widget), "size-allocate",   G_CALLBACK(on_iconview_size_allocate),   NULL);
    g_handler_id = g_signal_connect(G_OBJECT(widget), "item-activated",  G_CALLBACK(on_iconview_item_activated),  NULL);
    g_signal_connect(G_OBJECT(widget), "navigate-cursor", G_CALLBACK(on_iconview_navigate_cursor), NULL);
    gtk_widget_set_name(widget, "iconview-irex-ctb");
    GtkIconView *view = GTK_ICON_VIEW(widget);
    ergtk_icon_view_set_focus_mode(ERGTK_ICON_VIEW(widget), TRUE, FALSE);
    gtk_icon_view_set_selection_mode(GTK_ICON_VIEW(widget), GTK_SELECTION_SINGLE);
    gtk_icon_view_set_spacing(view, 8);    // pixels between icon and text

    // add filename column
    gtk_icon_view_set_text_column(view, MODCOL_TITLE);
    // add thumbnail column
    gtk_icon_view_set_pixbuf_column(view, MODCOL_THUMBNAIL);

    return widget;
}


static GtkWidget* create_listview()
{
    GtkWidget *view = ergtk_list_view_new_with_model( get_filemodel() );
    g_signal_connect(G_OBJECT(view), "size-allocate",   G_CALLBACK(on_listview_size_allocate),   NULL);
    g_handler_id = g_signal_connect(G_OBJECT(view), "row-activated",   G_CALLBACK(on_listview_row_activated),   NULL);
    g_signal_connect(G_OBJECT(view), "navigate-cursor", G_CALLBACK(on_listview_navigate_cursor), NULL);
    gtk_widget_set_name(view, "listview-irex-ctb");

    erGtkListView *er_listview = ERGTK_LIST_VIEW(view);
    GtkTreeView *treeview = GTK_TREE_VIEW(view);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
    ergtk_list_view_set_focus_mode(er_listview, TRUE, FALSE);
    gtk_tree_view_set_fixed_height_mode(treeview, TRUE);
    // NOTE: customized for DR1000's
    ergtk_list_view_set_row_height(er_listview, 34);

    // icon column
    GtkCellRenderer *renderer = gtk_cell_renderer_pixbuf_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   0,
                  "ypad",   0,
                  "xalign", 0.5,
                  "yalign", 0.5,
                  NULL );
    GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes( NULL,
                                                       renderer,
                                                       "pixbuf", MODCOL_THUMBNAIL,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "expand", FALSE,
                  NULL );
    gtk_tree_view_column_set_fixed_width(column, 50);
    ergtk_list_view_append_column(er_listview, column);

    // filename column
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0,
                  "yalign",        0.5,
                  "ellipsize",     PANGO_ELLIPSIZE_END,
                  "ellipsize-set", TRUE,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_TITLE,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "expand", TRUE,
                  NULL );
    ergtk_list_view_append_column(er_listview, column);

    // filetype column
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0,
                  "yalign",        0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILETYPE_DISPLAY,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "expand", FALSE,
                  NULL );
    gtk_tree_view_column_set_fixed_width(column, 120);
    ergtk_list_view_append_column(er_listview, column);

    // filesize column
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   0,
                  "ypad",   0,
                  "xalign", 1.0,
                  "yalign", 0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILESIZE,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "alignment", 0.0,
                  NULL );
    gtk_tree_view_column_set_fixed_width(column, 80);
    ergtk_list_view_append_column(er_listview, column);

    // filedate column
    renderer = gtk_cell_renderer_text_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   0,
                  "ypad",   0,
                  "xalign", 0.5,
                  "yalign", 0.5,
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text", MODCOL_FILEDATE,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  NULL );
    // note: required width depends on locale, this should fit all
    gtk_tree_view_column_set_fixed_width(column, 250);
    ergtk_list_view_append_column(er_listview, column);

    return view;
}


static GtkWidget* create_contentview()
{
    GtkWidget *view = ergtk_list_view_new_with_model( get_filemodel() );
    g_signal_connect(G_OBJECT(view), "size-allocate",   G_CALLBACK(on_listview_size_allocate),   NULL);
    g_handler_id = g_signal_connect(view, "row-activated",   G_CALLBACK(on_listview_row_activated),   NULL);
    g_signal_connect(G_OBJECT(view), "navigate-cursor", G_CALLBACK(on_listview_navigate_cursor), NULL);
    gtk_widget_set_name(view, "contentview-irex-ctb");

    erGtkListView *er_listview = ERGTK_LIST_VIEW(view);
    GtkTreeView *treeview = GTK_TREE_VIEW(view);
    gtk_tree_view_set_headers_visible(treeview, FALSE);
    GtkTreeSelection *selection = gtk_tree_view_get_selection(treeview);
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);
    ergtk_list_view_set_focus_mode(er_listview, TRUE, FALSE);
    gtk_tree_view_set_fixed_height_mode(treeview, TRUE);

    // icon column
    GtkCellRenderer *renderer = gtk_cell_renderer_pixbuf_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",   10,
                  "ypad",   0,
                  "xalign", 0.5,
                  "yalign", 0.5,
                  NULL );
    GtkTreeViewColumn *column = gtk_tree_view_column_new_with_attributes( NULL,
                                                       renderer,
                                                       "pixbuf", MODCOL_THUMBNAIL,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "expand", FALSE,
                  NULL );
    gtk_tree_view_column_set_fixed_width(column, 90);
    ergtk_list_view_append_column(er_listview, column);

    // title + author column
    renderer = ergtk_cell_renderer_text_new(2);
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0, // left
                  "yalign",        1.0, // bottom
                  "ellipsize",     PANGO_ELLIPSIZE_END,
                  "ellipsize-set", TRUE,
                  "font-0",        "Normal 10",
                  "font-1",        "Normal italic 9",
                  "height-0",      32,
                  "height-1",      22,
                  "foreground-0",  "black",
                  "foreground-1",  "#555555",
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       "text-0", MODCOL_TITLE,
                                                       "text-1", MODCOL_SUBTITLE,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "expand", TRUE,
                  NULL );
    gtk_tree_view_column_set_fixed_width(column, 300);
    ergtk_list_view_append_column(er_listview, column);

    // border column (invisible)
    // Note: add this to the GtkTreeView to avoid separator column from erGtkListView
    renderer = ergtk_cell_renderer_border_new();
    g_object_set( G_OBJECT(renderer),
                  "xpad",          0,
                  "ypad",          0,
                  "xalign",        0.0,
                  "yalign",        0.5,
                  "border-width",  2,
                  "border-offset", 3,
                  "border-color",  "dark grey",
                  NULL );
    column = gtk_tree_view_column_new_with_attributes( "",
                                                       renderer,
                                                       NULL );
    g_object_set( G_OBJECT(column),
                  "sizing", GTK_TREE_VIEW_COLUMN_FIXED,
                  "expand", FALSE,
                  NULL );
    gtk_tree_view_column_set_fixed_width(column, 10);
    gtk_tree_view_append_column(treeview, column);

    return view;
}


void fileview_set_view_type (const ctb_viewtypes_t view, gboolean do_cursor)
{
    if (g_viewmode == DELETE_MODE) {
        WARNPRINTF("cannot switch viewtype in delete mode");
        return;
    }

    LOGPRINTF("old=%d -> new=%d", g_current_view, view);
    g_is_user_selected_view = TRUE;
    if ( view == g_current_view ) return;
    fileview_stop_update_display();

    gchar *cursor_item = NULL;
    if (do_cursor) cursor_item = get_filename_at_cursor();

    // delete old view
    switch (g_current_view) {
        case CTB_ICONVIEW:
            if (g_iconview) {
                gtk_widget_destroy(g_iconview);
                g_iconview = NULL;
            }
            break;
        case CTB_LISTVIEW:
        case CTB_CONTENTVIEW:
            if (g_listview) {
                gtk_widget_destroy(g_listview);
                g_listview = NULL;
            }
            break;
        default:
            break;
    }

    g_current_view = view;
    
    // create new view
    GtkWidget *widget = NULL;
    switch (view)
    {
        case CTB_ICONVIEW:
            filemodel_set_thumbsize(MODTHUMB_MEDIUM, do_cursor);
            widget = create_iconview();
            g_iconview = widget;
            g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), &widget);
            gtk_widget_show(widget);
            gtk_container_add(GTK_CONTAINER(g_eventbox), widget);
            if (g_width != 0) gtk_widget_set_size_request(widget, g_width, g_height);
            break;
        case CTB_LISTVIEW:
            filemodel_set_thumbsize(MODTHUMB_MINI, do_cursor);
            widget = create_listview();
            g_listview = widget;
            g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), &widget);
            gtk_widget_show(widget);
            gtk_container_add(GTK_CONTAINER(g_eventbox), widget);
            listview_set_column_headers();
            break;
        case CTB_CONTENTVIEW:
            filemodel_set_thumbsize(MODTHUMB_SMALL, do_cursor);
            widget = create_contentview();
            g_listview = widget;
            g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), &widget);
            gtk_widget_show(widget);
            gtk_container_add(GTK_CONTAINER(g_eventbox), widget);
            if (g_width != 0) gtk_widget_set_size_request(widget, g_width, g_height);
            break;
        default:
            break;
    }

    update_padding();
    menu_select_view_type(view);
    if (do_cursor) {
        g_operation_in_progress = TRUE;
        fileview_set_cursor_at_filename(cursor_item);
        g_free(cursor_item);
    }
}


static void on_destroy (GtkWidget * widget, gpointer data)
{
    gtk_main_quit();
}


void fileview_create()
{
    // create the top level window 
    GtkWidget *window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), PACKAGE " " VERSION);
    gtk_window_maximize(GTK_WINDOW(window));
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(window), 0);
    gtk_window_set_modal(GTK_WINDOW(window), TRUE);
    g_signal_connect(G_OBJECT(window), "destroy", G_CALLBACK(on_destroy), NULL);
    g_main_window = window;

    GtkWidget *widget = create_screen_layout();
    gtk_container_add(GTK_CONTAINER(window), widget);

    ctb_viewtypes_t view = get_viewtype_from_registry();
    fileview_set_view_type(view, FALSE);
    g_is_user_selected_view = FALSE;

    fileview_show_desktop();
    gtk_widget_show(window);
}


void fileview_init()
{
    filemodel_init();
    reset_delete_state();
}

