#ifndef __CTB_LOG_H__
#define __CTB_LOG_H__

/**
 * File Name  : ctb_log.h
 *
 * Description: Macros for debug printing
 *
 * The value of XXX_ON (with XXX one of LOGGING, WARNING or ERROR)
 * determines whether a message will be printed. The value as defined
 * in this file can be overruled by defining them before the inclusion of this file.
 *
 * This file is best included in c files only.
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

#include <stdio.h>
#include "config.h"

#define LOG_PREFIX PACKAGE_NAME

/* To capture logging when running this component on the DR hardware, use this flag.
   See main() for the location of the log file.
   TODO: get the selection of the log-file out of main(). (Or learn how to use syslog on the target :-)
 */
#ifndef USE_FILELOG
#define USE_FILELOG 0
#endif

#ifndef USE_SYSLOG
#define USE_SYSLOG 0
#endif
 
#ifndef LOGGING_ON
#define LOGGING_ON 0
#endif

#ifndef TIMING_ON
#define TIMING_ON 0
#endif

#ifndef WARNING_ON
#define WARNING_ON 1
#endif

#ifndef ERROR_ON
#define ERROR_ON 1
#endif

#if (TIMING_ON)
#define START_TIMER() start_duration_timer()
#define STOP_TIMER() stop_duration_timer()
#else
#define START_TIMER()  do {} while(0)
#define STOP_TIMER() do {} while(0)
#endif

#if (USE_SYSLOG)
#include <syslog.h>
#define LOG_OPEN(X) openlog(X, LOG_PID | LOG_NDELAY, LOG_USER)
#define LOG_CLOSE() closelog()
#elif (USE_FILELOG)
extern FILE *fplog;
#define LOG_OPEN(X) fplog = fopen(X, "w");
#define LOG_CLOSE() {fclose(fplog); fplog = NULL;}
#else
#define LOG_OPEN(X) do {} while (0)
#define LOG_CLOSE() do {} while (0)
#endif

#if (LOGGING_ON)
#if (USE_SYSLOG)
#define LOGPRINTF(format, args...) syslog(LOG_INFO | LOG_USER, __FILE__ ":%d,%s() " format "\n", __LINE__, __func__ , ##args)
#else
#define LOGPRINTF(format, args...) fprintf(stderr, "(" LOG_PREFIX "_L)" __FILE__ ":%d,%s() " format "\n", __LINE__, __func__ , ##args)
#endif
#else
#define LOGPRINTF(format, args...) do {} while (0)
#endif

#if (WARNING_ON)
#if (USE_SYSLOG)
#define WARNPRINTF(format, args...) syslog(LOG_WARNING | LOG_USER, __FILE__ ":%d,%s() " format "\n", __LINE__, __func__ , ##args)
#elif (USE_FILELOG)
#define WARNPRINTF(format, args...) if (fplog) {fprintf(fplog, "(" LOG_PREFIX "_W)" __FILE__ ":%d,%s() " format "\n", __LINE__, __func__ , ##args); fflush(fplog);}
#else
#define WARNPRINTF(format, args...) fprintf(stderr, "(" LOG_PREFIX "_W)" __FILE__ ":%d,%s() " format "\n", __LINE__, __func__ , ##args)
#endif
#else
#define WARNPRINTF(format, args...) do {} while (0)
#endif

#if (ERROR_ON)
#include <errno.h>
#include <string.h>
#if (USE_SYSLOG)
#define ERRORPRINTF(format, args...) syslog(LOG_ERR | LOG_USER, __FILE__ ":%d,%s() --- " format "\n", __LINE__, __func__ , ##args)
#define ERRNOPRINTF(format, args...) syslog(LOG_ERR | LOG_USER, __FILE__ ":%d,%s() --- " format ", errno [%d] [%s]\n", __LINE__, __func__ , ##args, errno, strerror(errno))
#else
#define ERRORPRINTF(format, args...) fprintf(stderr, "(" LOG_PREFIX "_E)" __FILE__ ":%d,%s() --- " format "\n", __LINE__, __func__ , ##args)
#define ERRNOPRINTF(format, args...) fprintf(stderr, "(" LOG_PREFIX "_E)" __FILE__ ":%d,%s() --- " format ", errno [%d] [%s]\n", __LINE__, __func__ , ##args, errno, strerror(errno))
#endif
#else
#define ERRORPRINTF(format, args...) do {} while (0)
#define ERRNOPRINTF(format, args...) do {} while (0)
#endif

#endif // __CTB_LOG_H__
