#!/media/mmcblk0p1/Programs/_lua/bin/lua

----------------------------------------------------------------------
-- INBOX Manager: manages files in INBOX folder
--
-- Copyright (C) 2011  Iñigo Serna <inigoserna@gmail.com>
-- Released under GPL v3+
--
-- Time-stamp: <>
----------------------------------------------------------------------


require("lfs")

require("lgob.gtk")
require("lgob.gdk")
require("lgob.pango")


----------------------------------------------------------------------
----- Constants
local SDCARD_DIR = "/media/mmcblk0p1"
local LUADIR =  SDCARD_DIR .. "/Programs/_lua"
local INBOX_DIR = SDCARD_DIR .. "/INBOX"
local INITIAL_DESTDIR = SDCARD_DIR
local SHORTCUTS_DIR = SDCARD_DIR .. "/System/Shortcuts"
local MDBINDEX_CMD = "/usr/bin/mdbindex ".. SDCARD_DIR
local MTIME_FMT = "%d/%m/%Y"
local SHORTCUT_TEMPLATE = [[
[Desktop Entry]
Version=1.0
Type=%s
%s
Name=%s
]]
local pix_set = gdk.Pixbuf.new_from_file(LUADIR .. "/share/pixmaps/selected.png")
local pix_noset = gdk.Pixbuf.new_from_file(LUADIR .. "/share/pixmaps/noselected.png")
local TVCOL_SET, TVCOL_NAME, TVCOL_DATE, TVCOL_SHORTCUT, TVCOL_DEST = 0, 1, 2, 3, 4
local TVCOL_W = { SET=25, NAME=400, DATE=120, SHORTCUT=80 }


----------------------------------------------------------------------
----- Global variables
local model, dlg_folder, btn_proceed


----------------------------------------------------------------------
----- Utils
local function get_items(path)
   local fs = {}
   for f in lfs.dir(path) do
      if f ~= "." and f ~= ".." and f ~= "metadata.db" then
         local attr = lfs.attributes(path..'/'..f)
         if attr.mode == 'directory' then f = f .. '/' end
         fs[f] = attr.modification or 0
      end
   end
   return fs
end


-- format_datetime: return a formated string of date
local function timestamp2str(timestamp)
   return os.date(MTIME_FMT, timestamp)
end


-- create a DR shortcut
local function create_shortcut(file, dest)
   local shortcut, type, url, buf
   if file:sub(#file) == '/' then
      file = file:sub(1, #file-1)
      shortcut = SHORTCUTS_DIR .. '/' .. file .. '_001.directory'
      type = 'Directory'
      dest = dest:gsub(SDCARD_DIR, "../..")
      url = 'Path='..dest..'/'..file --:gsub(' ', '%%20')
   else
      shortcut = SHORTCUTS_DIR .. '/' .. file .. '_001.desktop'
      type = 'Link'
      url = 'URL=file:'..dest..'/'..file --:gsub(' ', '%%20')
   end
   buf = string.format(SHORTCUT_TEMPLATE, type, url, file)
   local f, err, errno = io.open(shortcut, "w")
   if not f then return err, errno end
   f:write(buf)
   f:close()
end


----------------------------------------------------------------------
----- gtk+ helper functions
local mygtk = {
   -- create a label with markup
   Markup_new = function(txt)
                   local label = gtk.Label.new()
                   label:set_markup(txt)
                   return label
                end
}


-- splash
local function ui_splash(title, subtitle)
   local splash = gtk.Window.new(gtk.WINDOW_TOPLEVEL)
   splash:set("window-position", gtk.WIN_POS_CENTER, "decorated", false,
              "width-request", 550, "height-request", 150,
              "resizable", false, "border-width", 10)
   local frame = gtk.Frame.new()
   local vbox = gtk.VBox.new(true, 0)
   vbox:set("border-width", 10)
   local line1 = mygtk.Markup_new(title)
   vbox:pack_start(line1, true, true, 10)
   local line2 = mygtk.Markup_new(subtitle)
   line2:set("ellipsize", pango.ELLIPSIZE_START)
   vbox:pack_start(line2, true, true, 10)
   frame:add(vbox)
   splash:add(frame)
   splash:show_all()
   while gtk.events_pending() do
      gtk.main_iteration()
   end
   return splash
end


function populate_model()
   local iter = gtk.TreeIter.new()
   local fs = get_items(INBOX_DIR)
   model:clear()
   for f, t in pairs(fs) do
      model:append(iter)
      model:set(iter, TVCOL_SET, pix_noset, TVCOL_NAME, f,
                TVCOL_DATE, timestamp2str(t), TVCOL_SHORTCUT, false,
                TVCOL_DEST, nil)
   end
end


----------------------------------------------------------------------
----- Callbacks
function cb_tv_click(view, ev)
    local _, x, y = gdk.Event.get_coords(ev)
    local _, path = view:get_path_at_pos(x, y)
    if not path then return true end
    local iter = gtk.TreeIter.new()
    model:get_iter_from_string(iter, path)
    -- return if clicked over shortcut? column
    if x > (TVCOL_W.SET+TVCOL_W.NAME+TVCOL_W.DATE) then
       btn_proceed:grab_focus()
       return false
    end
    -- clear if set
    if x < TVCOL_W.SET and model:get(iter, TVCOL_SET) == pix_set then
        model:set(iter, TVCOL_SET, pix_noset)
        model:set(iter, TVCOL_DEST, nil)
        btn_proceed:grab_focus()
        return true
    end
    -- get destination folder and update model row
    ret = dlg_folder:run()
    dlg_folder:hide()
    if ret == gtk.RESPONSE_CANCEL then -- cancel
       btn_proceed:grab_focus()
       return true
    end
    dest = dlg_folder:get_filename()
    if dest == model:get(iter, TVCOL_DEST) or dest == INBOX_DIR then return true end -- not modified
    if dest == nil or dest == '' then
        model:set(iter, TVCOL_SET, pix_noset)
        model:set(iter, TVCOL_DEST, nil)
    else
        model:set(iter, TVCOL_SET, pix_set)
        model:set(iter, TVCOL_DEST, dest)
    end
    btn_proceed:grab_focus()
    return true -- don't propagate signal, stop here
end


function do_move()
   local splash = ui_splash("<big><b>Copying files</b></big>", "<i>Please wait</i>")
   local iter = gtk.TreeIter.new()
   model:get_iter_first(iter)
   while iter do
      if model:get(iter, TVCOL_SET) == pix_set then
         src = INBOX_DIR .. '/' .. model:get(iter, TVCOL_NAME)
         dest = model:get(iter, TVCOL_DEST)
         cmd = 'mv "' .. src .. '" "' .. dest .. '"'
         os.execute(cmd)
         if model:get(iter, TVCOL_SHORTCUT) then
            create_shortcut(model:get(iter, TVCOL_NAME), dest)
         end
      end
      if not model:iter_next(iter) then break end
   end
   os.execute(MDBINDEX_CMD)
   splash:hide()
   splash:destroy()
   gtk.main_quit()
end


----------------------------------------------------------------------
----- UI
function ui_init()
    local win = gtk.Window.new()
    win:set("title", "INBOX Manager", "border-width", 20)
    win:connect("delete-event", function() gtk.main_quit() end)
    local vbox = gtk.VBox.new(false, 0)
    win:add(vbox)
    dlg_folder = gtk.FileChooserDialog.new("Select the destination folder", win,
                                           gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                           "gtk-cancel", gtk.RESPONSE_CANCEL,
                                           "gtk-ok", gtk.RESPONSE_OK)
    dlg_folder:set_current_folder(INITIAL_DESTDIR)

    -- header: title and quit button
    local hbox = gtk.HBox.new(false, 10)
    local evbox = gtk.EventBox.new()
    evbox:modify_bg(gtk.STATE_NORMAL, gdk.color_parse("black"))
    local title = mygtk.Markup_new("<span size='xx-large' color='white' weight='bold'>INBOX Manager</span>")
    evbox:add(title)
    hbox:pack_start(evbox, true, true, 0)
    local evbox_quit = gtk.EventBox.new()
    evbox_quit:set_events(gdk.BUTTON_PRESS_MASK)
    evbox_quit:connect("button-press-event", function() gtk.main_quit() end)
    local img = gtk.Image.new_from_file(LUADIR .. "/share/pixmaps/quit.png")
    evbox_quit:add(img)
    hbox:pack_end(evbox_quit, false, false, 0)
    vbox:pack_start(hbox, false, false, 10)

    -- treeview
    local view = gtk.TreeView.new()
    view:connect("button-press-event", cb_tv_click, view)
    view:set("rules-hint", true, "model", model)
    local scrollwin = gtk.ScrolledWindow.new()
    scrollwin:set("hscrollbar-policy", gtk.POLICY_NEVER,
                  "vscrollbar-policy", gtk.POLICY_AUTOMATIC)
    scrollwin:add(view)
    vbox:pack_start(scrollwin, true, true, 0)
    local cr_set = gtk.CellRendererPixbuf.new()
    local col_set = gtk.TreeViewColumn.new_with_attributes(" ", cr_set, "pixbuf", TVCOL_SET)
    col_set:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", TVCOL_W.SET)
    view:append_column(col_set)
    local cr_name = gtk.CellRendererText.new()
    cr_name:set("ellipsize-set", true, "ellipsize", pango.ELLIPSIZE_MIDDLE)
    local col_name = gtk.TreeViewColumn.new_with_attributes("Name", cr_name, "text", TVCOL_NAME)
    col_name:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", TVCOL_W.NAME, "alignment", 0)
    view:append_column(col_name)
    local cr_added = gtk.CellRendererText.new()
    local col_added = gtk.TreeViewColumn.new_with_attributes("Date", cr_added, "text", TVCOL_DATE)
    col_added:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", TVCOL_W.DATE, "alignment", 0.5)
    view:append_column(col_added)
    local cr_shortcut = gtk.CellRendererToggle.new()
    cr_shortcut:connect("toggled",
                        function(data, path)
                           local iter = gtk.TreeIter.new()
                           model:get_iter_from_string(iter, path)
                           model:set(iter, 3, not model:get(iter, 3))
                        end)
    local col_shortcut = gtk.TreeViewColumn.new_with_attributes("Shortcut?", cr_shortcut, "active", TVCOL_SHORTCUT)
    col_shortcut:set("sizing", gtk.TREE_VIEW_COLUMN_FIXED, "fixed-width", TVCOL_W.SHORTCUT, "alignment", 0.5)
    view:append_column(col_shortcut)

    -- buttons
    local hbox = gtk.HBox.new(true, 25)
    btn_proceed = gtk.Button.new_with_mnemonic("Move selected items")
    btn_proceed:connect("clicked", do_move)
    btn_proceed:grab_focus()
    hbox:add(btn_proceed)
    vbox:pack_start(hbox, false, false, 10)

    -- end
    model = gtk.ListStore.new("GdkPixbuf", "gchararray", "gchararray", "gboolean", "gchararray")
    populate_model()
    view:set_model(model)
    model:set_sort_column_id(1, gtk.SORT_ASCENDING)
    win:fullscreen()
    win:show_all()
 end



----------------------------------------------------------------------
----- Main
ui_init()
btn_proceed:grab_focus()
gtk.main()


----------------------------------------------------------------------
