/*
 * File Name: ipc.c
 */

/*
 * This file is part of sysd.
 *
 * sysd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * sysd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// ereader include files, between < >
#include <liberipc/eripc.h>
#include <liberipc/eripc_support.h>
#include <liberkeyb/erkeyb-client.h>

// local include files, between " "
#include "log.h"
#include "busy.h"
#include "ipc.h"
#include "connections.h"
#include "hal.h"
#include "process.h"
#include "system.h"
#include "tasks.h"
#include "wacom.h"
#include "xwindow.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

#define CHECK_ARG_STRING(x) \
    if (arg_array[x].type != ERIPC_TYPE_STRING || arg_array[x].value.s == NULL) { \
        ERRORPRINTF("arg %d is not a string", x); \
        return; \
    }

#define CHECK_ARG_INT(x) \
    if (arg_array[x].type != ERIPC_TYPE_INT) { \
        ERRORPRINTF("arg %d is not an integer", x); \
        return; \
    }

#define CHECK_ARG_BOOL(x) \
    if (arg_array[x].type != ERIPC_TYPE_BOOL) { \
        ERRORPRINTF("arg %d is not a boolean", x); \
        return; \
    }

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

static const char *charge_text[]       = { "", "low", "charging", "discharging", "full" };
static const char *orientation_text[]  = { "portrait", "landscape_clockwise", "landscape_anticlockwise" };

static const gchar *browser_path       = "/usr/bin/erbrowser";
static const gchar *browser_options    = "-n"; // start without navigation bar

static const gchar *ade_fulfill_path    = "/usr/bin/adobe-fulfill";
static const gchar *downloadmgr_path    = "/usr/bin/downloadmgr";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

eripc_client_context_t *eripcClient = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static void cb_sys_get_battery_state    (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_get_orientation      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_get_pageturn_inverted(eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_get_card_mountpoint  (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_get_device_capabilities (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_set_busy_led         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_set_bg_busy          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_reset_bg_busy        (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_set_fg_busy          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_reset_fg_busy        (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_beep                 (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_rotate               (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_lock_sensors         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_set_stylus           (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_get_stylus           (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_card_mount           (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_card_unmount         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_shutdown             (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_standby              (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_sys_set_keyboard         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_task_start               (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_task_activate            (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_task_stop                (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_task_rename              (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_menu_request_popup       (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_menu_item_activated      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_status_item_activated    (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_startup_complete         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_open_url                 (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_opened_window            (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_closed_window            (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void on_confirm_usbconnect       (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void on_confirm_install_drz      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void on_confirm_install_update   (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_connect             (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_disconnect          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_status              (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_status_request      (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_signal              (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_add_profile         (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void cb_conn_edit_profile        (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);
static void testing_list_tasks          (eripc_context_t *context, const eripc_event_info_t *info, void *user_data);

static const gchar *medium_service      (const char *medium);
static gint sender_from_service         (const char *service);

// Exported DBUS API list
static eripc_callback_function_t service_functions[] = {
        // message handlers (method calls to this service)
        { cb_sys_get_battery_state,         "sysGetBatteryState",           NULL,              0 },
        { cb_sys_get_orientation,           "sysGetOrientation",            NULL,              0 },
        { cb_sys_get_pageturn_inverted,     "sysGetPageturnInverted",       NULL,              0 },
        { cb_sys_get_card_mountpoint,       "sysGetCardMountPoint",         NULL,              0 },
        { cb_sys_get_device_capabilities,   "sysGetDeviceCapabilities",     NULL,              0 },
        { cb_sys_beep,                      "sysBeep",                      NULL,              0 },
        { cb_sys_set_busy_led,              "sysSetBusyLed",                NULL,              0 },
        { cb_sys_set_bg_busy,               "sysSetBgBusy",                 NULL,              0 },
        { cb_sys_reset_bg_busy,             "sysResetBgBusy",               NULL,              0 },
        { cb_sys_set_fg_busy,               "sysSetBusy",                   NULL,              0 },
        { cb_sys_reset_fg_busy,             "sysResetBusy",                 NULL,              0 },
        { cb_sys_rotate,                    "sysRotate",                    NULL,              0 },
        { cb_sys_lock_sensors,              "sysLockSensors",               NULL,              0 },
        { cb_sys_set_stylus,                "sysSetStylus",                 NULL,              0 },
        { cb_sys_get_stylus,                "sysGetStylus",                 NULL,              0 },
        { cb_sys_card_mount,                "sysCardMount",                 NULL,              0 },
        { cb_sys_card_unmount,              "sysCardUnmount",               NULL,              0 },
        { cb_sys_shutdown,                  "sysShutdown",                  NULL,              0 },
        { cb_sys_standby,                   "sysStandby",                   NULL,              0 },
        { cb_sys_set_keyboard,              "sysSetKeyboard",               NULL,              0 },
        { cb_task_start,                    "startTask",                    NULL,              0 },
        { cb_task_activate,                 "activateTask",                 NULL,              0 },
        { cb_task_stop,                     "stopTask",                     NULL,              0 },
        { cb_task_rename,                   "renameTask",                   NULL,              0 },
        { cb_open_url,                      "openUrl",                      NULL,              0 },
        { cb_opened_window,                 "openedWindow",                 NULL,              0 },
        { cb_closed_window,                 "closedWindow",                 NULL,              0 },
        { cb_menu_request_popup,            "menuRequestPopup",             NULL,              0 },
        { cb_menu_item_activated,           "menuItemActivated",            NULL,              0 },
        { cb_status_item_activated,         "statusItemActivated",          NULL,              0 },
        { cb_conn_connect,                  "connConnect",                  NULL,              0 },
        { cb_conn_disconnect,               "connDisconnect",               NULL,              0 },
        { cb_conn_status,                   "connConnectionStatus",         NULL,              0 },
        { cb_conn_signal,                   "connConnectionSignal",         NULL,              0 },
        { cb_conn_status_request,           "connConnectionStatusRequest",  NULL,              0 },
        { cb_conn_add_profile,              "connAddProfile",               NULL,              0 },
        { cb_conn_edit_profile,             "connEditProfile",              NULL,              0 },
        // signal handlers (broadcasted from given service)
        { cb_startup_complete,              "startupComplete",              DBUS_INTERFACE,    0 },
        { cb_closed_window,                 "closedWindow",                 DBUS_INTERFACE,    0 },
        // for testing
        { testing_list_tasks,               "testingListTasks",             NULL,              0 },
        { NULL, NULL, NULL, 0 }
    };


//============================================================================
// Functions Implementation
//============================================================================

void ipc_set_services(void)
{
    eripcClient = eripc_client_context_new(
                    DBUS_APPL_NAME,
                    "1.0",
                    DBUS_SERVICE,
                    DBUS_PATH,
                    DBUS_INTERFACE,
                    service_functions);
    
    erkeyb_client_init();
}


void ipc_unset_services(void)
{
    erkeyb_client_term();
    eripc_client_context_free(eripcClient, service_functions);
}


void ipc_send_battery_state(gint battery_level, enum state_charge charge_state, gint time_left)
{
    LOGPRINTF("entry");
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysBatteryState",
                              ERIPC_TYPE_INT, battery_level,
                              ERIPC_TYPE_STRING, charge_text[charge_state],
                              ERIPC_TYPE_INT, time_left,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysBatteryState: %d, %s", battery_level, charge_text[charge_state]);
}


void ipc_send_prepare_standby()
{
    LOGPRINTF("entry");
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysPrepareStandby",
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysPrepareStandby");
}


void ipc_send_prepare_unmount(const char *mount_point)
{
    LOGPRINTF("entry");

    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysPrepareUnmount",
                              ERIPC_TYPE_STRING, mount_point,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysPrepareUnmount %s", mount_point);
}


void ipc_send_volume_mounted(const char *mount_point)
{
    LOGPRINTF("entry");
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysVolumeMounted",
                              ERIPC_TYPE_STRING, mount_point,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysVolumeMounted %s", mount_point);
}


void ipc_send_volume_mounted_to(const char *service, const char *mount_point)
{
    eripc_error_t retval;

    LOGPRINTF("entry");
    
    retval = eripc_send_string(eripcClient->context, 
                               NULL,
                               NULL,
                               ERIPC_BUS_SESSION,
                               service,
                               "sysVolumeMounted",
                               mount_point);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return;
    }

    LOGPRINTF("Sent sysVolumeMounted %s to %s", mount_point, service);
}


void ipc_send_volume_unmounted(const char *mount_point)
{
    LOGPRINTF("entry");
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysVolumeUnmounted",
                              ERIPC_TYPE_STRING, mount_point,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysVolumeUnmounted %s", mount_point);
}

void ipc_refresh_ctb()
{
    eripc_send_varargs(eripcClient->context,
                      NULL,
                      NULL,
                      ERIPC_BUS_SESSION,
                      DBUS_SERVICE_CTB,
                      "filesystemChanged", ERIPC_TYPE_INVALID);
}


void ipc_send_usb_state(const char *state)
{
    LOGPRINTF("state = '%s'", state);
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysUsbState",
                              ERIPC_TYPE_STRING, state,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysUsbState %s", state);
}


void ipc_send_changed_locale(const char *locale)
{
    LOGPRINTF("entry");
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysChangedLocale",
                              ERIPC_TYPE_STRING, locale,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysChangedLocale %s", locale);
}


void ipc_send_demo_mode(gboolean demo_mode)
{
    LOGPRINTF("entry");

    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context,
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysChangeDemoMode",
                              ERIPC_TYPE_BOOL, demo_mode,
                              ERIPC_TYPE_INVALID);

    LOGPRINTF("Sent signal sysChangeDemoMode %d", demo_mode);
}


void ipc_send_changed_pageturn_inverted(gboolean is_inverted)
{
    LOGPRINTF("entry");

    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysChangedPageturnInverted",
                              ERIPC_TYPE_BOOL, is_inverted,
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysChangedPageturnInverted %d", (gint)is_inverted);
}


void ipc_send_changed_orientation(guint orientation)
{
    LOGPRINTF("entry");

    if (orientation >= sizeof(orientation_text))
    {
        ERRORPRINTF("Orientation %d unknown, signal not sent.", orientation);
        return;
    }
    
    // broadcast signal over session bus
    //
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysChangedOrientation",
                              ERIPC_TYPE_STRING, orientation_text[orientation],
                              ERIPC_TYPE_INVALID);
    
    LOGPRINTF("Sent signal sysChangedOrientation %s", orientation_text[orientation]);
}


static void ipc_send_window_show(const char *ipc_interface, gint window)
{
    LOGPRINTF("entry");
    
    eripc_error_t retval;
    retval = eripc_send_int(eripcClient->context,
                            NULL,
                            NULL,
                            ERIPC_BUS_SESSION,
                            ipc_interface,
                            "showWindow",
                            window);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_menu_set_item_state(const char *iname, const char *pname, const char *state) 
{
    eripc_menu_set_item_state(eripcClient, iname, pname, state);
}


void ipc_menu_set_statusitem_state(const char *name, const char *state) 
{
    eripc_menu_set_statusitem_state(eripcClient, name, state);
}


void ipc_menu_add_task(gint xid, const char *label, const char *application, const char *document)
{
    LOGPRINTF("entry, xid=%d  label='%s' application=%s document='%s'", xid, label, application, document);
    g_return_if_fail(label != NULL);

    eripc_error_t retval = eripc_send_varargs(eripcClient->context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_MENU,
                                "addTask",
                                ERIPC_TYPE_INT, xid,
                                ERIPC_TYPE_STRING, label,
                                ERIPC_TYPE_STRING, application,
                                ERIPC_TYPE_STRING, document,
                                ERIPC_TYPE_STRING, DBUS_INTERFACE,
                                ERIPC_TYPE_INVALID);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    } 
}


void ipc_menu_set_first_task(int xid) 
{
    LOGPRINTF("xid=%d", xid);

    eripc_error_t retval = eripc_send_int(eripcClient->context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_MENU,
                                "setTaskToTop",
                                xid);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    } 
}


void ipc_menu_remove_task(int xid) 
{
    LOGPRINTF("entry, xid=%d", xid);

    eripc_error_t retval = eripc_send_int(eripcClient->context,
                               NULL,
                               NULL,
                               ERIPC_BUS_SESSION,
                               DBUS_SERVICE_MENU,
                               "removeTask", xid);
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    } 
}


void ipc_menu_rename_task(int xid, const char* label)
{
    LOGPRINTF("entry, xid=%d  label=%s", xid, label);

    eripc_error_t retval = eripc_send_varargs(eripcClient->context,
                               NULL,
                               NULL,
                               ERIPC_BUS_SESSION,
                               DBUS_SERVICE_MENU,
                               "renameTask",
                               ERIPC_TYPE_INT, xid,
                               ERIPC_TYPE_STRING, label,
                               ERIPC_TYPE_INVALID);
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    } 
}


void ipc_menu_show(const char *menu)
{
    eripc_menu_show_menu(eripcClient, menu);
}


void ipc_show_popup(const char *state)
{
    LOGPRINTF("entry");
    
    eripc_error_t retval;
    retval = eripc_send_string(eripcClient->context,
                               NULL,
                               NULL,
                               ERIPC_BUS_SESSION,
                               DBUS_SERVICE_MENU,
                               "setPopupShow",
                               state);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_show_busy(gboolean show_mode, const gchar *message)
{
    LOGPRINTF("entry");
    eripc_error_t retval;

    retval = eripc_send_varargs(eripcClient->context,
                                NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_MENU,
                               "setBusyShow",
                                ERIPC_TYPE_STRING, show_mode ? "show" : "hide",
                                ERIPC_TYPE_STRING, message,
                                ERIPC_TYPE_INVALID);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_show_splash(const char *type)
{
    LOGPRINTF("entry [%s]", type);
    eripc_error_t retval;
    
    g_return_if_fail(type != NULL);

    // it may take a little while for the splash to show up
    // so make sure we don't suspend early
    sys_reset_idle_time();
    
    retval = eripc_send_string(eripcClient->context,
                               NULL,
                               NULL,
                               ERIPC_BUS_SESSION,
                               DBUS_SERVICE_MENU,
                               "setSplashShow", type);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_show_message(const char *type, const void *reply_handler, const void *user_data)
{
    LOGPRINTF("entry");
    eripc_error_t retval;
    
    g_return_if_fail(type != NULL);

    retval = eripc_send_string(eripcClient->context,
                               (eripc_handler_t *) reply_handler,
                               user_data,
                               ERIPC_BUS_SESSION,
                               DBUS_SERVICE_MENU,
                               "setMessageShow",
                               type);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_confirm_usbconnect(gboolean show_mode)
{
    LOGPRINTF("entry");
    eripc_error_t retval;
    
    retval = eripc_send_string(eripcClient->context,
                                show_mode ? &on_confirm_usbconnect : NULL,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_MENU,
                                "confirmUsbConnect",
                                show_mode ? "show" : "hide");

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_confirm_install_drz(gboolean show_mode)
{
    LOGPRINTF("entry");
    eripc_error_t retval;
    
    retval = eripc_send_string(eripcClient->context,
                                &on_confirm_install_drz,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_MENU,
                                "confirmInstallDrz",
                                show_mode ? "show" : "hide");

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
    
    return;
}


void ipc_confirm_install_update(gboolean show_mode)
{
    LOGPRINTF("entry");
    eripc_error_t retval;
    
    retval = eripc_send_string(eripcClient->context,
                                &on_confirm_install_update,
                                NULL,
                                ERIPC_BUS_SESSION,
                                DBUS_SERVICE_MENU,
                                "confirmInstallUpdate",
                                show_mode ? "show" : "hide");

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_ctb_goto(const char *location)
{
    LOGPRINTF("entry");
    
    eripc_error_t retval;
        
    g_return_if_fail(location != NULL);
    
    retval = eripc_send_string(eripcClient->context,
                               NULL, // reply handler
                               NULL, // reply data
                               ERIPC_BUS_SESSION,
                               DBUS_SERVICE_CTB,
                               "gotoLocation",
                               location);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


gboolean ipc_connect(const char *ipc_service, const char *medium, const char *profile)
{
    LOGPRINTF("entry");

    eripc_error_t retval;
        
    g_return_val_if_fail(medium != NULL, FALSE);
    // profile may be NULL
    
    // NOTE: this call slightly differs from eripc_sysd_conn_connect()
    //       in that it uses the IPC SERVICE of the connection manager as
    //       destination for the message 
    retval = eripc_send_varargs(eripcClient->context,
                               NULL,            // no reply handler
                               NULL,            // no reply data
                               ERIPC_BUS_SESSION,
                               ipc_service,
                               "connConnect",
                               ERIPC_TYPE_STRING, ipc_service,
                               ERIPC_TYPE_STRING, medium,
                               ERIPC_TYPE_STRING, profile,
                               ERIPC_TYPE_INVALID);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


gboolean ipc_disconnect(const char *ipc_service)
{
    LOGPRINTF("entry");

    eripc_error_t retval;
        
    // NOTE: this call slightly differs from eripc_sysd_conn_connect()
    //       in that it uses the IPC SERVICE of the connection manager as
    //       destination for the message 
    retval = eripc_send_varargs(eripcClient->context,
                               NULL,            // no reply handler
                               NULL,            // no reply data
                               ERIPC_BUS_SESSION,
                               ipc_service,
                               "connDisconnect",
                               ERIPC_TYPE_STRING, ipc_service,
                               ERIPC_TYPE_INVALID);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


gboolean ipc_send_conn_status( const char *ipc_service,
                               gboolean   is_connected,
                               const char *medium,
                               const char *profile,
                               const char *reason  )
{
    LOGPRINTF("entry");

    eripc_error_t retval;
        
    // NOTE: status is send to a specific service when there was no change is in state (already connected/disconnected)
    retval = eripc_send_varargs(eripcClient->context,
                               NULL,            // no reply handler
                               NULL,            // no reply data
                               ERIPC_BUS_SESSION,
                               ipc_service,
                               "connConnectionStatus",
                               ERIPC_TYPE_BOOL,   is_connected,
                               ERIPC_TYPE_STRING, medium,
                               ERIPC_TYPE_STRING, profile,
                               ERIPC_TYPE_STRING, reason,
                               ERIPC_TYPE_INVALID);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


gboolean ipc_broadcast_conn_status( gboolean   is_connected,
                                    const char *medium,
                                    const char *profile,
                                    const char *reason  )
{
    LOGPRINTF("connected=%d", is_connected);

    // NOTE: status is broadcasted when it was changed
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "connConnectionStatus",
                              ERIPC_TYPE_BOOL,   is_connected,
                              ERIPC_TYPE_STRING, medium,
                              ERIPC_TYPE_STRING, profile,
                              ERIPC_TYPE_STRING, reason,
                              ERIPC_TYPE_INVALID);
    
    return TRUE;
}


gboolean ipc_broadcast_conn_signal( const char *medium,
                                    const int  signal )
{
    LOGPRINTF("%s signal: %d", medium, signal);

    // NOTE: signal is broadcasted when it was changed
    eripc_send_signal_varargs(eripcClient->context, 
                              ERIPC_BUS_SESSION,
                              DBUS_PATH,
                              DBUS_INTERFACE,
                              "sysSignalStrength",
                              ERIPC_TYPE_INT,    signal,
                              ERIPC_TYPE_STRING, medium,
                              ERIPC_TYPE_INVALID);
    
    return TRUE;
}


gboolean ipc_add_profile(const char *medium)
{
    LOGPRINTF("entry");

    eripc_error_t retval;
        
    g_return_val_if_fail(medium != NULL, FALSE);
    
    retval = eripc_send_varargs(eripcClient->context,
                               NULL,            // no reply handler
                               NULL,            // no reply data
                               ERIPC_BUS_SESSION,
                               medium_service(medium),
                               "connAddProfile",
                               ERIPC_TYPE_STRING, medium,
                               ERIPC_TYPE_INVALID);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


gboolean ipc_edit_profile(const char *medium, const char *profile)
{
    LOGPRINTF("entry");

    eripc_error_t retval;
        
    g_return_val_if_fail(medium  != NULL, FALSE);
    g_return_val_if_fail(profile != NULL, FALSE);
    
    retval = eripc_send_varargs(eripcClient->context,
                               NULL,            // no reply handler
                               NULL,            // no reply data
                               ERIPC_BUS_SESSION,
                               medium_service(medium),
                               "connEditProfile",
                               ERIPC_TYPE_STRING, medium,
                               ERIPC_TYPE_STRING, profile,
                               ERIPC_TYPE_INVALID);
    
    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


gboolean ipc_send_open(const char *ipc_interface, 
                       const char *document, 
                       void *callback_handler, 
                       void *callback_data)
{
    LOGPRINTF("entry");
    
    eripc_error_t retval;
        
    g_return_val_if_fail(ipc_interface != NULL, FALSE);
    g_return_val_if_fail(document != NULL, FALSE);
    
    retval = eripc_send_string(eripcClient->context,
                               callback_handler,
                               callback_data,
                               ERIPC_BUS_SESSION,
                               ipc_interface,
                               "openFile",
                               document);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


gboolean ipc_send_close(const char *ipc_interface, 
                        const char *document, 
                        void *callback_handler, 
                        void *callback_data)
{
    LOGPRINTF("entry");
    
    eripc_error_t retval;
        
    g_return_val_if_fail(ipc_interface != NULL, FALSE);
    g_return_val_if_fail(document != NULL, FALSE);
    
    retval = eripc_send_string(eripcClient->context,
                               callback_handler,
                               callback_data,
                               ERIPC_BUS_SESSION,
                               ipc_interface,
                               "closeFile",
                               document);

    if (retval != ERIPC_ERROR_SUCCESS) 
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
        return FALSE;
    }
    
    return TRUE;
}


void ipc_send_reply(eripc_context_t *context, const char *message_id, gboolean result)
{
    LOGPRINTF("entry");
    
    g_return_if_fail(context != NULL);

    if (message_id)
    {
        LOGPRINTF("Sending reply %s to: %s, context %p", (result == TRUE ? "TRUE":"FALSE"), message_id, context);

        eripc_error_t retval = eripc_reply_bool(context, message_id, result);
        if (retval != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(retval));
        }
    }
    else
    {
        LOGPRINTF("Result is %s but no reply was requested", (result == TRUE ? "TRUE":"FALSE"));
    }
}


void ipc_send_reply_task_start(eripc_context_t *context, const char *message_id, gint err_code, gchar *err_msg)
{
    LOGPRINTF("entry");
    

    g_return_if_fail(context != NULL);

    if (message_id)
    {
        LOGPRINTF("Sending reply %d to: %s, context %p", err_code, message_id, context);

        eripc_error_t retval = eripc_reply_varargs(context, message_id, 
                                     ERIPC_TYPE_INT, err_code,
                                     ERIPC_TYPE_STRING, err_msg,
                                     ERIPC_TYPE_INVALID);
        if (retval != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(retval));
        }
    }
    else
    {
        LOGPRINTF("Sending reply %d but no reply was requested", err_code);
    }
}


//============================================================================
// Local Function Implementation
//============================================================================

static void cb_sys_get_battery_state(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;

    if (info->message_id)
    {
        enum state_charge charge_state = 0;
        gint battery_level = 0;
        gint time_left = 0;
        
        sys_get_battery(&battery_level, &charge_state, &time_left);

        LOGPRINTF("Sending battery %d, state %s to, time left %d: %s, context %p", 
                  battery_level, 
                  charge_text[charge_state], 
                  time_left, 
                  info->message_id, 
                  context);

        result = eripc_reply_varargs(context, info->message_id, 
                                      ERIPC_TYPE_INT, battery_level,
                                      ERIPC_TYPE_STRING, charge_text[charge_state], 
                                      ERIPC_TYPE_INT, -1,
                                      ERIPC_TYPE_INVALID);

        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(result));
        }
    }
}


static void cb_sys_get_orientation(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;

    if (info->message_id)
    {
        guint orientation = sys_get_orientation();
        
        if (orientation >= sizeof(orientation_text))
        {
            ERRORPRINTF("Orientation %d unknown, signal not sent.", orientation);
            return;
        }

        LOGPRINTF("Sending orientation %s", orientation_text[orientation]);

        result = eripc_reply_varargs(context, info->message_id, 
                                      ERIPC_TYPE_STRING, orientation_text[orientation], 
                                      ERIPC_TYPE_INVALID);

        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(result));
        }
    }
}


static void cb_sys_get_pageturn_inverted(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;

    if (info->message_id)
    {
        gboolean  is_inverted = sys_get_pageturn_inverted();
        
        LOGPRINTF("Sending pageturn_inverted %d", (gint) is_inverted);

        result = eripc_reply_varargs(context, info->message_id, 
                                      ERIPC_TYPE_BOOL, is_inverted, 
                                      ERIPC_TYPE_INVALID);

        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(result));
        }
    }
}


static void cb_sys_get_card_mountpoint(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    if (info->message_id)
    {
        const char* mountpoint = NULL;
        if (sys_get_card() == STATE_CARD_MOUNTED) mountpoint = MOUNTPOINT_CARD;

        LOGPRINTF("Sending card mountpoint '%s'", mountpoint ? mountpoint : "NULL");

        gboolean result = eripc_reply_string(context, info->message_id, mountpoint);
        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(result));
        }
    }
}


static void cb_sys_get_device_capabilities(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;

    if (info->message_id)
    {
        gboolean has_stylus     = FALSE;
        gboolean has_wifi       = FALSE;
        gboolean has_bluetooth  = FALSE;
        gboolean has_3g         = FALSE;
        
        sys_get_device_capabilities(&has_stylus, &has_wifi, &has_bluetooth, &has_3g);

        LOGPRINTF("Sending device capabilities: stylus %d, wifi %d, bluetooth %d, 3g %d", 
                  has_stylus, has_wifi, has_bluetooth, has_3g);

        result = eripc_reply_varargs(context, info->message_id, 
                                      ERIPC_TYPE_BOOL, has_stylus,
                                      ERIPC_TYPE_BOOL, has_wifi,
                                      ERIPC_TYPE_BOOL, has_bluetooth,
                                      ERIPC_TYPE_BOOL, has_3g,
                                      ERIPC_TYPE_INVALID);

        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(result));
        }
    }
}


/* NOTE:
 * sysSetBusyLed is deprecated but currently supported for backward compatibility
 * Use sysSetBgBusy/sysResetBgBusy and sysSetBusy/sysResetBusy instead
 */
static void cb_sys_set_busy_led(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gchar *busy_mode = arg_array[0].value.s;
    gint sender = sender_from_service(info->service);

    gboolean result = FALSE;
    if (g_ascii_strcasecmp(busy_mode, "on") == 0)
    {
        result = busy_add_foreground(sender, BUSY_DIALOG_NONE, NULL);
    }
    else if (g_ascii_strcasecmp(busy_mode, "off") == 0)
    {
        result = busy_remove_foreground(sender);
    }
    else
    {
        WARNPRINTF("Unknown busy mode: %s", busy_mode);
    }

    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_set_fg_busy(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gint sender = sender_from_service(info->service);
    gchar *busy_mode = arg_array[0].value.s;
    gchar *busy_msg  = NULL;
    if ((arg_array[1].type == ERIPC_TYPE_STRING) && (arg_array[1].value.s != NULL))
    {
        busy_msg = arg_array[1].value.s;
    }
    
    gboolean result = FALSE;
    if (g_ascii_strcasecmp(busy_mode, "nodialog") == 0)
    {
        result = busy_add_foreground(sender, BUSY_DIALOG_NONE, NULL);
    }
    else if (g_ascii_strcasecmp(busy_mode, "delaydialog") == 0)
    {
        result = busy_add_foreground(sender, BUSY_DIALOG_DELAYED, busy_msg);
    }
    else if (g_ascii_strcasecmp(busy_mode, "directdialog") == 0)
    {
        result = busy_add_foreground(sender, BUSY_DIALOG_DIRECT, busy_msg);
    }
    else
    {
        WARNPRINTF("Unknown busy mode: %s", busy_mode);
    }

    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_reset_fg_busy(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;
    gint sender = sender_from_service(info->service);

    result = busy_remove_foreground(sender);

    // send message result back to caller
    ipc_send_reply(context, info->message_id, result);
}



static void cb_sys_set_bg_busy(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;
    gint sender = sender_from_service(info->service);

    result = busy_add_background(sender);

    // send message result back to caller
    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_reset_bg_busy(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;
    gint sender = sender_from_service(info->service);

    result = busy_remove_background(sender);

    // send message result back to caller
    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_beep(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_INT(0);
    CHECK_ARG_STRING(1);
    gboolean result = sys_set_beeper(arg_array[0].value.i, arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_rotate(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gboolean result = sys_set_orientation(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_lock_sensors(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gboolean result = sys_set_sensor_lock(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_set_stylus(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    char *stylus_mode = arg_array[0].value.s;
    gboolean result = TRUE;
    if (g_ascii_strcasecmp(stylus_mode, "enable") == 0)
    {
        wacom_enable();
    }
    else if (g_ascii_strcasecmp(stylus_mode, "disable") == 0)
    {
        wacom_disable();
    }
    else if (g_ascii_strcasecmp(stylus_mode, "suspend") == 0)
    {
        wacom_suspend();
    }
    else if (g_ascii_strcasecmp(stylus_mode, "resume") == 0)
    {
        wacom_resume();
    }
    else if (g_ascii_strcasecmp(stylus_mode, "high") == 0)
    {
        wacom_scan_high();
    }
    else if (g_ascii_strcasecmp(stylus_mode, "normal") == 0)
    {
        wacom_scan_normal();
    }
    else
    {
        WARNPRINTF("Unknown stylus mode: %s", stylus_mode);
        result = FALSE;
    }

    ipc_send_reply(context, info->message_id, result);
}


static void cb_sys_get_stylus(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    gboolean result = FALSE;

    if (info->message_id)
    {
        gboolean stylus_enabled = wacom_is_enabled();

        result = eripc_reply_varargs(context, info->message_id, 
                                     ERIPC_TYPE_BOOL, stylus_enabled,
                                     ERIPC_TYPE_INVALID);

        if (result != ERIPC_ERROR_SUCCESS) 
        {
            ERRORPRINTF("Error sending reply to message: %s", eripc_error_string(result));
        }
    }
}


static void cb_sys_card_mount(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
   
    const eripc_arg_t *arg_array = NULL;
    
    gboolean enable_splash = FALSE;
    arg_array = info->args;

    g_return_if_fail(arg_array[0].type == ERIPC_TYPE_BOOL);   // is_connected
    enable_splash = arg_array[0].value.b ; 

    if ( enable_splash )
    {
        sys_set_enable_index_splash(TRUE);
    }
    else 
    {
        sys_set_enable_index_splash(FALSE);
    }

    hal_remount_all_volumes(NULL, NULL);

    // send message result back to caller
    ipc_send_reply(context, info->message_id, TRUE);
}


static void cb_sys_card_unmount(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    // eject silently
    sys_eject_card(TRUE);

    ipc_send_reply(context, info->message_id, TRUE);
}


static void cb_sys_standby(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");

    // send message result back to caller immediately
    ipc_send_reply(context, info->message_id, TRUE);
    
    sys_standby();
}


static void cb_sys_shutdown(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    
    // send message result back to caller immediately
    ipc_send_reply(context, info->message_id, TRUE);

    sys_standby();
}


static void cb_sys_set_keyboard(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    char *keyboard_mode = arg_array[0].value.s;
    
    gboolean result = TRUE;
    if (g_ascii_strcasecmp(keyboard_mode, "show") == 0)
    {
        erkeyb_client_show();
    }
    else if (g_ascii_strcasecmp(keyboard_mode, "hide") == 0)
    {
        erkeyb_client_hide();
    }
    else
    {
        WARNPRINTF("Unknown keyboard mode: %s", keyboard_mode);
        result = FALSE;
    }

    // send message result back to caller
    ipc_send_reply(context, info->message_id, result);
}


static void cb_task_start(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);    // command_line  string, full path to application with arguments
    CHECK_ARG_STRING(1);    // working_dir   string, current working directory, or NULL to inherit from sysd
    CHECK_ARG_STRING(2);    // label         string, text label shown under icon in popup menu
    CHECK_ARG_STRING(3);    // image         string, full path to icon shown in popup menu
    gboolean result = task_start(arg_array[0].value.s, 
                            arg_array[1].value.s, 
                            arg_array[2].value.s,
                            arg_array[3].value.s,
                            context,
                            info->message_id);
    // send failure to start task back to caller immediately
    // otherwise result is returned asynchronously
    if (result == FALSE)
    {
        ipc_send_reply_task_start(context, info->message_id, 1, NULL);
    }
}


static void cb_task_stop(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);    // cmdline, full path to application with arguments
    gboolean result = task_stop(arg_array[0].value.s, 
                           context,
                           info->message_id);
    // send failure to start task back to caller immediately
    // otherwise result is returned asynchronously
    if (result == FALSE)
    {
        ipc_send_reply(context, info->message_id, FALSE);
    }
}


static void cb_task_rename(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_INT(0);       // xid of task
    CHECK_ARG_STRING(1);    // new document name
    CHECK_ARG_STRING(2);    // new label
    gboolean result = task_rename(arg_array[0].value.i, arg_array[1].value.s, arg_array[2].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_task_activate(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_INT(0);
    int xid = arg_array[0].value.i;
    task_activate_by_xid(xid);
    ipc_send_reply(context, info->message_id, TRUE);
}


static void cb_open_url(eripc_context_t *context,  const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if (!sys_has_network())
    {
        LOGPRINTF("Device does not have networking capabilities, browser not started");
        ipc_send_reply_task_start(context, info->message_id, 5, NULL);
        return;
    }

    // [0] url          string, URL to open
    // [1] name         string, text label shown under icon in popup menu
    // [2] application  string, application name shown in "Back" bar of browser, or NULL to omit this bar
    
    gboolean result = FALSE;
    if ( (arg_array[0].type == ERIPC_TYPE_STRING) && (arg_array[0].value.s != NULL) )
    {
        const gchar *url = arg_array[0].value.s;
        gchar *name = NULL;
        gchar *command_line = NULL;

        if (arg_array[1].type == ERIPC_TYPE_STRING)
        {
            name = arg_array[1].value.s;
        }
        
        if ( (arg_array[2].type == ERIPC_TYPE_STRING) && (arg_array[2].value.s != NULL) )
        {
            // use embedded flag to pass application name in "Back" bar
            command_line = g_strdup_printf("%s %s -e \"%s\" \"%s\"", browser_path, browser_options, arg_array[2].value.s, url);
        }
        else
        {
            command_line = g_strdup_printf("%s %s \"%s\"", browser_path, browser_options, url);
        }
        
        if ((name == NULL) || (name[0] == '\0'))
        {
            // no name given, use URL without protocol
            const gchar *seperator = "://";
            name = strstr(url, "://");
            if (name)
            {
                name += strlen(seperator);
            }
            else
            {
                name = (gchar *)url;
            }
        }

        // TODO: get thumbnail icon to popupmenu folder
        result = task_start(command_line, 
                            NULL, 
                            name,
                            "/usr/share/ctb/icon-html-small.png",
                            context,
                            info->message_id);
        
        g_free(command_line);
    }

    // send failure to start task back to caller immediately
    // otherwise result is returned asynchronously
    if (result == FALSE)
    {
        ipc_send_reply_task_start(context, info->message_id, 1, NULL);
    }
}


static void cb_opened_window(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);    // application   string, name of the application
    CHECK_ARG_STRING(1);    // document      string, document (should be unqiue)
    CHECK_ARG_STRING(2);    // label         string, text label shown under icon in popup menu
    CHECK_ARG_STRING(3);    // image         string, full path to icon shown in popup menu (png, 60x60 pixels)
    CHECK_ARG_STRING(4);    // service       string, IPC service name of the application
    CHECK_ARG_INT(5);       // pid           integer, process id
    CHECK_ARG_INT(6);       // window        integer, window ID (XID)
    gboolean result = task_add(arg_array[0].value.s, 
                          arg_array[1].value.s, 
                          arg_array[2].value.s,
                          arg_array[3].value.s,
                          arg_array[4].value.s,
                          arg_array[5].value.i,
                          arg_array[6].value.i);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_menu_request_popup(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gboolean result = sys_request_popup(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}



static void cb_menu_item_activated(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    CHECK_ARG_STRING(1);
    CHECK_ARG_STRING(2);
    CHECK_ARG_STRING(3);
    const char *item  = arg_array[0].value.s;
    const char *group = arg_array[1].value.s;
    //const char *menu  = arg_array[2].value.s;
    const char *state = arg_array[3].value.s;

    gboolean result = FALSE;
    if (g_ascii_strcasecmp(group, "general") == 0)
    {
        if (g_ascii_strcasecmp(item, "desktop") == 0)
        {
            LOGPRINTF("gotoLocation desktop");
            ipc_ctb_goto("desktop");
            process_activate("ctb");
            result = TRUE;
        }
        else if (g_ascii_strcasecmp(item, "rotate_screen") == 0)
        {
            LOGPRINTF("rotate_screen");
            result = sys_set_orientation("toggle");
        }
        else if (g_ascii_strcasecmp(item, "lock") == 0)
        {
            LOGPRINTF("lock");
            if (g_ascii_strcasecmp(state, "normal") == 0)
            {
                result = sys_set_sensor_lock("lock");
            }
            else if (g_ascii_strcasecmp(state, "selected") == 0)
            {
                result = sys_set_sensor_lock("unlock");
            }
        }
        else if (g_ascii_strcasecmp(item, "eject_card") == 0)
        {
            LOGPRINTF("eject_card");
            sys_eject_card(FALSE);
            result = TRUE;
        }
        else if (g_ascii_strcasecmp(item, "shutdown") == 0)
        {
            LOGPRINTF("shutdown");
            sys_standby();
            result = TRUE;
        }
        else
        {
            WARNPRINTF("Unknown general item: %s", item);
        }
    }
    else if (g_ascii_strcasecmp(group, "tasks") == 0)
    {
        LOGPRINTF("tasks");
        
        if (g_ascii_strcasecmp(item, "back_to_library") == 0)
        {
            process_activate("ctb");
            result = TRUE;
        }
        else 
        {
            // activate task and return result immediately
            result = task_activate(item);
        }
    }

    ipc_send_reply(context, info->message_id, result);
}


static void cb_status_item_activated(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);    // name of activated status item
    CHECK_ARG_STRING(1);    // state of activated status item

    const gchar *name = arg_array[0].value.s;
    // const gchar *status = arg_array[1].value.s;
    LOGPRINTF("name [%s] status [%s]", name, arg_array[1].value.s);
    
    if (strcmp(name, "statusbar_downloadmgr") == 0)
    {
        LOGPRINTF("downloadmgr clicked, try to activate Adobe fulfillment");
        
        // When adobe-fulfill is running, activate it
        // Otherwise, start downloadmgr or activate it when it is running
        if (task_activate(ade_fulfill_path))
        {
            LOGPRINTF("ADE fulfillment activated");
        }
        else
        {
            LOGPRINTF("ADE fulfillment is not activated, try to activate downloadmgr");
            
            if (task_activate(downloadmgr_path))
            {
                // special call to unhide dowloadmgr window
                ipc_send_window_show("com.irexnet.downloadmgr", 0);
                
                LOGPRINTF("downloadmgr activated");
            }
            else 
            {
                if (task_start(downloadmgr_path, NULL, "DownloadMgr", NULL, NULL, NULL))
                {
                    LOGPRINTF("downloadmgr started");
                }
                else
                {
                    WARNPRINTF("failed to start downloadmgr");
                }
            }
        }
    }
    else if (   strcmp(name, "statusbar_wifi"     ) == 0
             || strcmp(name, "statusbar_bluetooth") == 0
             || strcmp(name, "statusbar_3g"       ) == 0 )
    {
        conn_on_statusitem_activated();
    }
    
    // don't return reply for signals
}


static void cb_startup_complete(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);    // name           string, name of application
    CHECK_ARG_INT(1);       // pid            integer, process id
    CHECK_ARG_BOOL(2);      // is_multidoc    boolean, application can open/close documents over IPC
    CHECK_ARG_STRING(3);    // ipc_service    string, IPC service name (may be NULL when is_multidoc is false)
                            // xid            integer, window id
    gint xid = 0;
    if (arg_array[4].type == ERIPC_TYPE_INT)
    {
        xid =  arg_array[4].value.i;
    }
    LOGPRINTF("name [%s] pid [%d] is_multidoc [%d] ipc_service [%s], window [%d]",
                arg_array[0].value.s,
                arg_array[1].value.i,
                arg_array[2].value.b,
                arg_array[3].value.s,
                xid);

    // save sender's pid
    busy_set_pid(sender_from_service(info->service), arg_array[1].value.i);
    process_startup_complete(arg_array[0].value.s, arg_array[1].value.i, arg_array[2].value.b, arg_array[3].value.s, xid);
    // don't return reply for signals
}


static void cb_closed_window(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    gboolean result = FALSE;

    CHECK_ARG_INT(0);
    if (arg_array[0].value.i > 0)
    {
        result = task_cleanup_window(arg_array[0].value.i);
    }

    ipc_send_reply(context, info->message_id, result);
}


static void cb_conn_connect(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;
    
    CHECK_ARG_STRING(0);    // ipc service
    CHECK_ARG_STRING(1);    // medium
    CHECK_ARG_STRING(2);    // profile

    gchar *service = arg_array[0].value.s;
    gchar *medium  = arg_array[1].value.s;
    gchar *profile = arg_array[2].value.s;
    
    LOGPRINTF("MACHINE_NAME [" MACHINE_NAME "] medium [%s]", medium);
    if (!medium || (medium[0]==0))
    {
#if MACHINE_IS_DR1000SW
        medium = "wifi";
#elif MACHINE_IS_DR800SG
        medium = "3g";
#else
        WARNPRINTF("no connectivity for MACHINE_NAME [" MACHINE_NAME "]");
        ipc_send_reply(context, info->message_id, FALSE);
        return;
#endif
    }    

    gboolean result = conn_connect(service, medium, profile);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_conn_disconnect(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gboolean result = conn_disconnect(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_conn_status(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_BOOL(0);      // is_connected
    CHECK_ARG_STRING(1);    // medium
    CHECK_ARG_STRING(2);    // profile
    CHECK_ARG_STRING(2);    // reason (disconnect only)

    gboolean result = conn_set_status( arg_array[0].value.b,
                              arg_array[1].value.s,
                              arg_array[2].value.s,
                              arg_array[3].value.s );
    ipc_send_reply(context, info->message_id, result);
}


static void cb_conn_status_request(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);
    gboolean result = conn_status_request(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_conn_signal(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_INT(0);       // strength
    CHECK_ARG_STRING(1);    // medium

    ipc_broadcast_conn_signal(arg_array[1].value.s, arg_array[0].value.i);
    ipc_send_reply(context, info->message_id, TRUE);
}


static void cb_conn_add_profile(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0); 
    gboolean result = conn_add_profile(arg_array[0].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void cb_conn_edit_profile(eripc_context_t *context, const eripc_event_info_t *info, void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    CHECK_ARG_STRING(0);    // connection medium
    CHECK_ARG_STRING(1);    // network profile
    gboolean result = conn_edit_profile(arg_array[0].value.s, arg_array[1].value.s);
    ipc_send_reply(context, info->message_id, result);
}


static void on_confirm_usbconnect(eripc_context_t *context,
                                  const eripc_event_info_t *info,
                                  void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if (info->event_type != ERIPC_EVENT_REPLY)
    {
        WARNPRINTF("invalid event: %d", info->event_type);
    }
    else if ((arg_array == NULL) || (arg_array[0].type != ERIPC_TYPE_BOOL))
    {
        WARNPRINTF("invalid arguments in reply");
    }
    else if (arg_array[0].value.b == TRUE)
    {
        LOGPRINTF("User confirmed connect");
        // handle connect
        sys_usb_connect();
    }
    else
    {
        sys_usb_no_connect();
    }
}


static void on_confirm_install_update(eripc_context_t *context,
                                      const eripc_event_info_t *info,
                                      void *user_data)
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if (info->event_type != ERIPC_EVENT_REPLY)
    {
        WARNPRINTF("invalid event: %d", info->event_type);
    }
    else if ((arg_array == NULL) || (arg_array[0].type != ERIPC_TYPE_BOOL))
    {
        WARNPRINTF("invalid arguments in reply");
    }
    else if (arg_array[0].value.b == TRUE)
    {
        LOGPRINTF("User confirmed restart to install firmware update");

        // handle shutdown and cold restart to trigger install procedure
        sys_restart();
    }
}


static void on_confirm_install_drz(eripc_context_t *context,
                                   const eripc_event_info_t *info,
                                   void *user_data)
{
    LOGPRINTF("entry");
    
    const eripc_arg_t *arg_array = info->args;
    
    if (info->event_type != ERIPC_EVENT_REPLY)
    {
        WARNPRINTF("invalid event: %d", info->event_type);
    }
    else if ((arg_array == NULL) || (arg_array[0].type != ERIPC_TYPE_BOOL))
    {
        WARNPRINTF("invalid arguments in reply");
    }
    else if (arg_array[0].value.b == TRUE)
    {
        LOGPRINTF("User confirmed to install drz");
        sys_install_drz();
    }
}


static void testing_list_tasks(eripc_context_t *context,
                               const eripc_event_info_t *info,
                               void *user_data)
{
#if (TESTING_ON)
    print_window_list();
    print_process_list();
    print_task_list();
#endif
    char* list = testing_task_get_list();
    eripc_reply_string(context, info->message_id, list);
    g_free(list);
}


static const gchar *medium_service(const char *medium)
{
    if (medium)
    {
        if (strcmp(medium, "wifi") == 0)
        {
            return DBUS_SERVICE_CONN_WIFI;
        }
        else if (strcmp(medium, "bluetooth") == 0)
        {
            return DBUS_SERVICE_CONN_BLUE;
        }
        else if (strcmp(medium, "3g") == 0)
        {
            return DBUS_SERVICE_CONN_3G;
        }
    }
    return NULL;
}


static gint sender_from_service(const char *service)
{
    gint sender = 0;
    gchar *dot = strchr(service, '.');
    if (dot) 
    {
       sender = atoi(dot + 1);
    }
    return sender;
}


void ipc_send_window_deactivated(const char *ipc_interface,  gint window)
{
    LOGPRINTF("entry");

    eripc_error_t retval;
    retval = eripc_send_int(eripcClient->context,
                            NULL,
                            NULL,
                            ERIPC_BUS_SESSION,
                            ipc_interface,
                            "deactivatedWindow",
                            window);

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}


void ipc_send_window_activated(const char *ipc_interface, gint window)
{
    LOGPRINTF("entry");

    eripc_error_t retval;
    retval = eripc_send_int(eripcClient->context,
                            NULL,
                            NULL,
                            ERIPC_BUS_SESSION,
                            ipc_interface,
                            "activatedWindow",
                            window);

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("Error launching eripc handler: %s", eripc_error_string(retval));
    }
}

