#ifndef __IPC_H__
#define __IPC_H__

/**
 * File Name  : ipc.h
 *
 * Description: The dbus-based eripc functions 
 */

/*
 * This file is part of sysd.
 *
 * sysd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * sysd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */


//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <liberipc/eripc.h>
#include "system.h"

G_BEGIN_DECLS


//----------------------------------------------------------------------------
// Definitions
//---------------------------------------------------------------------------- 

/**
 * System Control IPC DBUS interface description
 *    
 * Methods incoming
 * ================
 * Name:     sysGetBatteryState Get the battery level and charge state
 * Args:     (none)
 * Returns:  1) level           integer, current battery level in percent
 *           2) state           string, charge state: "low", "charging", "discharging", "full"
 *
 * Name:     sysGetOrientation  Get the display orientation
 * Args:     (none)
 * Returns:  string, state: "portrait","landscape_clockwise","landscape_anticlockwise"
 *
 * Name:     sysGetPageturnInverted  Get the pageturn mode (inverted or normal)
 * Args:     (none)
 * Returns:  boolean, TRUE if page turning is inverted, FALSE of page turning is normal
 *
 * Name:     sysGetCardMountPoint get mountpoint of SDCard
 * Args:     (none)
 * Returns:  string , mountpoint (eg /media/mmcblk0p1) if mounted, NULL if not.
 *
 * Name:     sysGetDeviceCapabilities  Get the device's capabilities
 * Args:     (none)
 * Returns:  1) has_stylus      boolean, TRUE if device has stylus, FALSE otherwise
 *           2) has_wifi        boolean, TRUE if device has wifi, FALSE otherwise
 *           3) has_bluetooth   boolean, TRUE if device has bluetooth, FALSE otherwise
 *
 * Name:     sysSetBusy         Set busy indication
 * Args:     1) state           string, mode "nodialog", "delaydialog", "directdialog"
 *           2) message         string, message text or NULL for default text
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *     
 * Name:     sysResetBusy       Reset busy indication
 * Args:     (none)
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *     
 * Name:     sysSetBgBusy       Set background busy 
 * Args:     (none)
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *     
 * Name:     sysResetBgBusy     Reset background busy indication
 * Args:     (none)
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *     
 * Name:     sysBeep            Set system beep tone
 * Args:     1) duration_ms     integer, length of tone in ms (use intervals of 10ms) 
 *           2) tone            string, tone pitch: "high", "low"
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *
 * Name:     sysRotate          Set orientation of display
 * Args:     1) mode            string, new mode: "portrait", "landscape", "toggle"
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *
 * Name:     sysLockSensors     Set locking of sensors
 * Args:     1) mode            string, new mode: "lock", "unlock", "toggle"
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *
 * Name:     sysGetStylus       Get stylus mode
 * Args:     none
 * Returns:  boolean, TRUE of enabled, FALSE otherwise
 *
 * Name:     sysSetStylus       Set stylus mode 
 * Args:     1) mode            string, new mode: "enable", "disable", "suspend", "resume", "high", "normal"
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *
 * Name:     sysCardMount       Mount volume(s) on SD card
 * Args:     none
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * Action:   Success means the command was executed and unmount has been started. When a volume
 *           is mounted this is reported through the broadcasted sysVolumeMounted signal
 *
 * Name:     sysCardUnmount     Unmount volume(s) on SD card
 * Args:     none
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * Action:   Success means the command was executed and unmount has been started. When a volume
 *           is unmounted this is reported through the broadcasted sysVolumeUnmounted signal
 *
 * Name:     sysShutdown        Graciously shut down the device
 * Args:     none
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * Action:   Success means the command was executed and shutdown has been initiated.
 *
 * Name:     startTask          Start a task
 * Args:     1) command_line    string, full path to application with arguments
 *           2) working_dir     string, current working directory
 *           3) label           string, text label shown under icon in popup menu
 *           4) image           string, full path to icon shown in popup menu (png, 60x60 pixels)
 * Returns:  1) error_code      integer, (1) error starting application
 *                                       (2) timeout waiting for application window
 *                                       (3) application exited before creating a window 
 *                                       (4) application returned an error (errr_msg follows)
 *           2) error_msg       string, error message text from application or NULL
 * Example:  dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.startTask string:"uds /usr/share/icons/file.png" string:"/usr/bin" string:"Document1" string:open
 *           dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.startTask string:"uds /media/mmcblk0p1/Library/test.txt" string:"/usr/bin" string:"Document1" string:open
 *           dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.startTask string:"firstboot" string:"/usr/bin" string:"firstboot" string:open
 *
 * Name:     stopTask           Stop a running task
 * Args:     1) command_line    string, full path to application with arguments
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * 
 * Name:     activateTask       Activate a running task
 * Args:     1) command_line    string, full path to application with arguments
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * 
 * Name:     menuRequestPopup   Set the popupmenu state when possible
 * Args:     1) state           string, new state: "show", "hide", "toggle", "block", "unblock"
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * 
 * Name:     openUrl            Open URL in the web browser
 * Args:     1) url             string, URL
 *           2) label           string, text label shown under icon in popup menu, or NULL to derive from URL
 *           3) application     string, application name shown in "Back" bar of browser, or NULL to omit this bar
 * Returns:  1) error_code      integer, (1) error starting browser application
 *                                       (2) timeout waiting for browser window
 *                                       (3) browser exited before creating a window 
 *                                       (4) browser returned an error (errr_msg follows)
 *                                       (5) device does not have networking capabilities so browser is not started
 *           2) error_msg       string, error message text from application or NULL
 * Example:  dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.openUrl string:"http://www.irextechnologies.com"
 *           dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.openUrl string:"http://www.irextechnologies.com" string:"iRex homepage" string:"Home"
 *           dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.openUrl string:"file:///media/mmcblk0p1/test.html" string:"Test page"
 *
 * Name:     openedWindow       Notify that a window was opened
 * Args:     1) application     string, base name of the application
 *           2) document        string, full path of the document
 *           3) label           string, text label shown under icon in popup menu
 *           4) image           string, full path to icon shown in popup menu (png, 60x60 pixels)
 *           5) ipc_service     string, IPC service name of the application
 *           6) pid             integer, process id
 *           7) window          integer, window ID (XID)
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * When:     Send when a window was opened which is to be added to the Task Manager of Popupmenu. 
 *           Don't use this call when a window is opened in response to "openFile".
 * Example:  dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.openedWindow string:"uds" string:"/media/mmcblk0p1/eBooks/bla.pdf" string:"bla" string:open string:com.irexnet.uds int32:2455 int32:12345
 * 
 * Name:     closedWindow       Notify that a window was closed
 * Args:     1) window          integer, window ID (XID)
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * When:     Send when a window was closed which is listed in the Task Manager of Popupmenu. 
 *           Don't use this call when a window is closed in response to "closeFile".
 *    
 * Name:     connConnect        Request a network connection
 * Args:     1) ipc_service     string, IPC service name of the application
 *           2) medium          string, connection medium or NULL for automatic selection
 *           3) profile         string, gconf path of network profile to use or NULL for automatic selection
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * Example:  dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.connConnect string:"com.irexnet.erbrowser" string:"wifi" string:""
 *
 * Name:     connDisconnect     Release the network connection
 * Args:     1) ipc_service     string, IPC service name of the application
 * Returns:  boolean, TRUE on success, FALSE otherwise
 * Example:  dbus-send --print-reply --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.connDisconnect string:"com.irexnet.erbrowser"
 *
 * Name:     connConnectionStatus  Set the connection status
 * Args:     1) is_connected    boolean, TRUE if connection is online, FALSE otherwise
 *           2) medium          string, connection medium in use
 *           3) profile         string, gconf path of network profile in use
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *
 * Name:     connConnectionStatusRequest Ask the connection status
 * Args:     none
 * Returns:  boolean, TRUE on success, FALSE otherwise
 *
 *
 * Signal incoming
 * ===============
 *
 * Name:     startupComplete    Indicate that an application has finished startup
 * Args:     1) name            string, name of application (eg "settings" or "ctb")
 *           2) pid             integer, process id
 *           3) is_multidoc     boolean, TRUE if application can open/close documents over IPC, FALSE otherwise
 *           4) ipc_service     string, IPC service name (eg "/com/irexnet/ctb"), or NULL when no IPC services are implemented
 *           5) xid             integer, window id; set to 0 when the application should not be shown in the Task Manager
 *
 * 
 * Method calls outgoing
 * =====================
 *     
 * Name:     openFile           Open a file in a window
 * Args:     1) file            string, path to file
 * Returns:  1) integer, X window on success, -1 on failure
 *           2) string, custom error message or NULL for none
 * When:     Sent when a file, document or url is to be closed but the application (callee)
 * Action:   Application (callee) should create and realise, or reuse an existing window 
 *           for the given file and return the X window id in the method reply. The X window 
 *           can be obtained using GDK_WINDOW_XID(widget->window). When the file is already 
 *           opened by the callee, it may just return its X window id. This call implies that
 *           the window is activated (set to the foreground) so callee should also set its 
 *           context for the given window and set the Popupmenu menu context. 
 *           System Daemon adds a task to Task Manager of Popupmenu, or replaces the task 
 *           when an existing window is returned.
 *      
 * Name:     activatedWindow    Notify that a window was activated
 * Args:     1) window          integer, X window of the file
 * Returns:  boolean, TRUE on success, FALSE on failure
 * When:     Sent after a window was activated (set to the foreground)
 * Action:   Application (callee) should set its context for the given window and set the 
 *           Popupmenu menu context.
 *     
 * Name:     deactivatedWindow  Notify that a window was deactivated
 * Args:     1) window          integer, X window of the file
 * Returns:  boolean, TRUE on success, FALSE on failure
 * When:     Sent after a window was deactivated (set to the background)
 * Action:   Application (callee) may adapt its context and free resources.
 *
 * Name:     closeFile          Close a file
 * Args:     1) file            string, path to file
 * Returns:  boolean, TRUE on success, FALSE on failure
 * When:     Sent when a file, document or url is to be closed by the application (callee)
 * Action:   Application (callee) should close the file and may destroy its window and free 
 *           other resources. System Daemon removes the task from the Task Manager of Popupmenu.
 *
 *
 * Signals outgoing
 * ================
 * 
 * Name:     sysBatteryState
 * Args:     1) level           integer, charge level in percent
 *           2) state           string, state: "low", "charging", "discharging", "full"
 * When:     Sent on state change and when battery level changed n% (default: 5%)
 * Action:   Applications may use this information to inform the user.
 *     
 * Name:     sysPrepareStandby
 * Args:     (none)
 * When:     Sent just before the system enters standby mode
 * Action:   Applications must commit changes (flush) of all its open files. Failing to 
 *           handle this signal may result in unsaved data or currupt files.
 *
 * Name:     sysPrepareUnmount
 * Args:     1) mount_point     string, mount point of volume
 * When:     Sent just before unmounting the volume
 * Action:   Applications must close all its open files on the given volume. Failing to 
 *           handle this signal may result in unsaved data or currupt files.
 * Example:  dbus-send --type=signal --dest=com.irexnet.sysd /com/irexnet/sysd com.irexnet.sysd.sysPrepareUnmount string:/media/mmcblk0p1
 *
 * Name:     sysVolumeMounted
 * Args:     1) mount_point     string, mount point of volume
 * When:     Sent just after a volume is mounted
 * Action:   Applications may use this to add/open the new volume.
 *     
 * Name:     sysVolumeUnmounted
 * Args:     1) mount_point     string, mount point of volume
 * When:     Sent just after unmounting the volume
 * Action:   Typically an application should have responded to a prior sysPrepareUnmount 
 *           signal, but when a device with volumes was removed unexpectedly it may need 
 *           to act on this signal.
 *     
 * Name:     sysChangedLocale
 * Args:     1) locale          string, locale code (ll_CC)
 * When:     Sent when the system's locale has changed
 * Action:   Applications should load language dependent screen texts and probably set new 
 *           labels for its menu items; to activate a new locale application should call:
 *             g_setenv("LANG", new_locale, TRUE);
 *             setlocale(LC_ALL, "");
 *
 * Name:     sysChangedPageturnInverted
 * Args:     1) is_inverted          boolean
 * When:     Sent when the system's pageturnmode  has changed
 * Action:   Applications (UDS) should invert the page turning. If is_inverted == TRUE, turn pages as in book
 *    
 * Name:     sysChangedOrientation
 * Args:     1) state           string, state: "portrait", "landscape_clockwise", "landscape_anticlockwise"
 * When:     Sent when the display's orientation has changed
 * Action:   Applications may need to adapt its screen size, coordinates and/or origin.
 *    
 * Name:     sysUsbState
 * Args:     1) state           string, state: "disconnected", "unmounted", "mounted"
 * When:     Sent on state change of USB connection to computer
 * Action:   Applications may use this information to control end user info.
 *
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include <liberipc/eripc_support.h>


//----------------------------------------------------------------------------
// Forward Declarations
//----------------------------------------------------------------------------

#define DBUS_APPL_NAME          PACKAGE_NAME
                                
#define DBUS_SERVICE            "com.irexnet."  DBUS_APPL_NAME
#define DBUS_PATH               "/com/irexnet/" DBUS_APPL_NAME
#define DBUS_INTERFACE          "com.irexnet."  DBUS_APPL_NAME
                                
#define DBUS_SERVICE_CTB        "com.irexnet.ctb"
#define DBUS_SERVICE_MENU       "com.irexnet.popupmenu"
#define DBUS_SERVICE_CONN_WIFI  "com.irexnet.connwifi"
#define DBUS_SERVICE_CONN_BLUE  "com.irexnet.connblue"
#define DBUS_SERVICE_CONN_3G    "com.irexnet.conn3g"

    
//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------

extern eripc_client_context_t *eripcClient;


//============================================================================
// Public Functions
//============================================================================

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_set_services
 *
 * @brief  Setup IPC connection and register API functions
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_set_services(void);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_unset_services
 *
 * @brief  Unregister API functions
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_unset_services(void);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_battery_state
 *
 * @brief  Send the sysBatteryState signal
 *
 * @param  battery_level Battery capacity remaining percentage
 * @param  state_charge  Battery charge mode
 * @param  time_left     Minutes to empty (discharging) or -1 (charging)
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_battery_state(gint battery_level, enum state_charge, gint time_left);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_prepare_standby
 *
 * @brief  Send the sysPrepareStandby signal
 *
 * @param  --
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_prepare_standby(void);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_prepare_unmount
 *
 * @brief  Send the sysPrepareUnmount signal
 *
 * @param  mount_point Mount point
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_prepare_unmount(const char *mount_point);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_menu_set_item_state
 *
 * @brief  Set new menu item state
 *
 * @param  iname Item name
 * @param  pname Parent group name
 * @param  state State name
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_menu_set_item_state(const char *iname, const char *pname, const char *state);  

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_menu_set_statusitem_state
 *
 * @brief  Set new status item state
 *
 * @param  name Item name
 * @param  state State name
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_menu_set_statusitem_state(const char *name, const char *state);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_volume_mounted
 *
 * @brief  Emit sysVolumeMounted signal
 *
 * @param  mount_point Mount point
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_volume_mounted(const char *mount_point);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_volume_unmounted
 *
 * @brief  Emit sysVolumeUnmounted signal
 *
 * @param  mount_point Mount point
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_volume_unmounted(const char *mount_point);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_changed_locale
 *
 * @brief  Emit sysChangedLocale signal
 *
 * @param  locale Locale string in form ll_CC
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_changed_locale(const char *locale);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_demo_mode
 *
 * @brief  Emit sysChangeDemoMode signal
 *
 * @param  demo_mode to enable/disable demo_mode
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_demo_mode(gboolean demo_mode);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_changed_pageturn_inverted
 *
 * @brief  Emit sysChangedPageturnInverted  signal
 *
 * @param  boolean is_inverted
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_changed_pageturn_inverted( gboolean is_inverted);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_ctb_goto
 *
 * @brief  Set new location for Content Browser
 *
 * @param  location New location to show ("desktop"|"library")
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_ctb_goto(const char *location);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_open
 *
 * @brief  Open new file by application
 *
 * @param  ipc_interface    IPC service name to send message to
 * @param  document         Full path to file
 * @param  callback_handler Function to receive reply
 * @param  callback_data    Data to pass to callback_handler
 *
 * @return TRUE if successfully opened by application, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean ipc_send_open(const char *ipc_interface, 
                       const char *document, 
                       void *callback_handler, 
                       void *callback_data);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_close
 *
 * @brief  Close file by application
 *
 * @param  ipc_interface    IPC service name to send message to
 * @param  document         Full path to file
 * @param  callback_handler Function to receive reply
 * @param  callback_data    Data to pass to callback_handler
 *
 * @return TRUE if successfully closed by application, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
gboolean ipc_send_close(const char *ipc_interface, 
                        const char *document, 
                        void *callback_handler, 
                        void *callback_data);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_reply
 *
 * @brief  Return a IPC reply message
 *
 * @param  context     ERIPC context to use for reply
 * @param  message_id  ERIPC message id to use for reply
 * @param  result      Boolean return value
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_reply(eripc_context_t *context, const char *message_id, gboolean result);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_reply_task_start
 *
 * @brief  Return a IPC reply message to startTask call
 *
 * @param  context     ERIPC context to use for reply
 * @param  message_id  ERIPC message id to use for reply
 * @param  err_code    Integer, return code (0 for success, >0 for error)
 * @param  err_msg     String, custom error message or NULL for none
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_reply_task_start(eripc_context_t *context, const char *message_id, gint err_code, gchar *err_msg);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_menu_add_task
 *
 * @brief  Add a task item
 *
 * @param  xid         window id
 * @param  label       Label text for menu
 * @param  application name of the application
 * @param  document    full-path to file (needed for 'auto-start')
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_menu_add_task(gint xid, const char *label, const char *application, const char *document);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_menu_set_first_task
 *
 * @brief  Set task item to front or list
 * 
 * @param  xid
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_menu_set_first_task(int xid);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_menu_remove_task
 *
 * @brief  Remove a task item
 *
 * @param  application Application of task
 * @param  document Document of task
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_menu_remove_task(int xid);

void ipc_menu_rename_task(int xid, const char* label);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_show_busy
 *
 * @brief  Show a busy/wait prompt
 *
 * @param  show_mode TRUE to show the prompt, FALSE to remove it
 * @param  message   Busy message to show in dialog, or NULL for default text
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_show_busy(gboolean show_mode, const gchar *message);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_show_splash
 *
 * @brief  Show a predefined splash screen
 *
 * @param  type Splash to show ("usbconnect" or "shutdown") or "hide" to remove
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_show_splash(const char *type);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_show_message
 *
 * @brief  Show a predefined message dialog
 *
 * @param  type Message to show ("safelyremove")
 * @param  reply_handler Handler called when a reply is returned
 * @param  user_data Data to pass to the reply handler
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_show_message(const char *type, const void *reply_handler, const void *user_data);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_confirm_usbconnect
 *
 * @brief  Ask user to confirm to connect over USB
 *
 * @param  show_mode TRUE to show the prompt, FALSE to remove it
 *
 * @return TRUE when user confirmed connection, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
void ipc_confirm_usbconnect(gboolean show_mode);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_confirm_install_drz
 *
 * @brief  Ask user to confirm to restart and install drz file(s)
 *
 * @param  show_mode TRUE to show the prompt, FALSE to remove it
 *
 * @return TRUE when user confirmed restart, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
void ipc_confirm_install_drz(gboolean show_mode);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_confirm_install_update
 *
 * @brief  Ask user to confirm to restart and install firmware update
 *
 * @param  show_mode TRUE to show the prompt, FALSE to remove it
 *
 * @return TRUE when user confirmed restart, FALSE otherwise
 *
 *--------------------------------------------------------------------------*/
void ipc_confirm_install_update(gboolean show_mode);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_usb_state
 *
 * @brief  Send the sysUsbState signal
 *
 * @param  state State ("disconnected", "unmounted", "mounted")
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_usb_state(const char *state);

/**---------------------------------------------------------------------------
 *
 * Name :  ipc_send_volume_mounted_to
 *
 * @brief  Send the volumeMounted method call to an application
 *
 * @param  service IPC service
 * @param  mount_point Mount point
 *
 * @return --
 *
 *--------------------------------------------------------------------------*/
void ipc_send_volume_mounted_to(const char *service, const char *mount_point);

void ipc_refresh_ctb();

void ipc_menu_show(const char *menu);
void ipc_show_popup(const char *state);
void ipc_send_changed_orientation(guint orientation);

gboolean ipc_connect(const char *ipc_service, const char *medium, const char *profile);
gboolean ipc_disconnect(const char *ipc_service);
gboolean ipc_send_conn_status(const char *ipc_service, gboolean is_connected, const char *medium, const char *profile, const char *reason);
gboolean ipc_broadcast_conn_status(gboolean is_connected, const char *medium, const char *profile, const char *reason);
gboolean ipc_add_profile(const char *medium);
gboolean ipc_edit_profile(const char *medium, const char *profile);

void ipc_send_window_activated(const char *ipc_interface,  gint window);
void ipc_send_window_deactivated(const char *ipc_interface,  gint window);

G_END_DECLS

#endif /* __IPC_H__ */
