/*
 * File Name: menustore.c
 */

/*
 * This file is part of popupmenu.
 *
 * popupmenu is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * popupmenu is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <gtk/gtk.h>
#include <string.h>

// ereader include files, between < >
#include <libergtk/ergtk.h>

// local include files, between " "
#include "log.h"
#include "menustore.h"
#include "pixlist.h"
#include "i18n.h"
#include "ipc.h"
#include "taskbar.h"

//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

enum menu_type
{
  MENU_TYPE_GROUP = 0,
  MENU_TYPE_ITEM,
};

typedef struct
{
    gchar* name;
    gchar* service;
    GList* groups;      // entries are gchar*
} MenuEntry;

typedef struct
{
    const gchar* name;
    gboolean found;
    GtkTreeIter* iter;
} SearchArgs;

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
typedef struct
{
    const char* pname;
    const char* iname;
} ToolItem;

typedef struct
{
    const char* menu;
    ToolItem items[MENUSTORE_NUM_TOOLS];
} MenuToolItems;
#endif

//----------------------------------------------------------------------------
// Global Variables
//----------------------------------------------------------------------------

GtkTreeStore *menu_store = NULL;
static GList *menu_list = NULL; // entries are MenuEntry*
static gchar *current_menu = NULL;
static taskbar_cb_t taskbar_home_cb = NULL;
static gboolean g_popup_has_changed = FALSE;
static gboolean g_toolbar_has_changed = FALSE;


//============================================================================
// Functions Implementation
//============================================================================

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static MenuToolItems hardcoded_tools[] =
{
    { "ctb_menu_content", {     // desktop + settings
        { "system_top", "desktop" },
        { "ctb_view", "view_small" },
        { "ctb_view", "view_detail" },
        { "ctb_view", "view_content" },
    } },
    { "ctb_menu_content_media", {   // in Folder, except recent view
        { "system_top", "desktop" },
        { "ctb_view", "view_small" },
        { "ctb_view", "view_detail" },
        { "ctb_view", "view_content" },
    } },
    { "ctb_menu_recent_mode", {   // in Recent view
        { "system_top", "desktop" },
        { "ctb_view", "view_small" },
        { "ctb_view", "view_detail" },
        { "ctb_view", "view_content" },
    } },
    { "ctb_menu_delete_mode", {   // in Delete Mode
        { "system_top", "desktop" },
        { "ctb_view", "view_small" },
        { "ctb_view", "view_detail" },
        { "ctb_view", "view_content" },
    } },
    { "uds_menu_reading", {     // normal reading mode
        { "system_top", "desktop" },
        { "system_top", "back_to_library" },
        { "uds_navigation", "search_text" },
        { "uds_navigation", "goto_page" },
        { "pen_functions", "pen" },
        { "pen_functions", "eraser" },
        { "zoom_page", "mode_continuous" },
        { "zoom_shift", "zoom_fit" },
        { "zoom_shift", "zoom_selection" },
        { "zoom_shift", "mode_pan" },
        { "uds_views", "view_annotation" },
        { "uds_views", "view_toc" },
        { "uds_views", "view_thumbnail" },
        { "uds_navigation", "close" },
    } },
    { "uds_menu_annotation", {     // bookmarks view, has pagecounter!!
        { "system_top", "desktop" },
        { "system_top", "back_to_library" },
        { "uds_views", "view_reading" },
        { "uds_views", "view_toc" },
        { "uds_views", "view_thumbnail" },
    } },
    { "uds_menu_toc", {           // TableOfContent mode, has pagecounter!!
        { "system_top", "desktop" },
        { "system_top", "back_to_library" },
        { "uds_views", "view_reading" },
        { "uds_views", "view_annotation" },
        { "uds_views", "view_thumbnail" },
    } },
    { "uds_menu_thumbnail", {     // Thumbnail view, has pagecounter!!
        { "system_top", "desktop" },
        { "system_top", "back_to_library" },
        { "uds_views", "view_reading" },
        { "uds_views", "view_annotation" },
        { "uds_views", "view_thumbnail" },
    } },
    { "uds_menu_bookinfo", {   
        { "system_top", "desktop" },
        { "system_top", "back_to_library" },
        { "uds_views", "view_reading" },
        { "uds_views", "view_annotation" },
        { "uds_views", "view_thumbnail" },
    } },
    { "notepad_menu", {
        { "system_top", "desktop" },
        { "notepad_actions", "clear_page" },
        { "notepad_actions", "insert_page" },
        { "notepad_actions", "delete_page" },
        { "notepad_actions", "rename" },
    } },
};
#endif

const char* menustore_get_current_menu()
{
    return current_menu;
}


void menustore_set_current_menu(const char* name)
{
    //WARNPRINTF("%s current:%s", name, current_menu);
    if (current_menu && name && strcmp(name, current_menu) == 0) return;

    g_free(current_menu);
    if (name == NULL)  current_menu = NULL;
    else current_menu = g_strdup(name);
    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
}


static void add_system_groups()
{
    LOGPRINTF("entry");
    
    menustore_add_group("system_top", NULL, NULL, NULL);
    menustore_add_item( "desktop",         "system_top", _("Go to Home"),    "desktop");
    menustore_add_item( "back_to_library", "system_top", "", "back_to_library");
    menustore_set_item_state("back_to_library", "system_top", "disabled");
    
    menustore_add_group("system_bottom", NULL, NULL, NULL);
    menustore_add_item( "rotate_screen",   "system_bottom", _("Rotate"),        "rotate_screen");
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    menustore_add_item( "lock",            "system_bottom", _("Lock Sensors"),  "lock");
    menustore_add_item( "eject_card",      "system_bottom", _("Safely Remove"), "eject_card");
    menustore_add_item( "shutdown",        "system_bottom", _("Turn Off Device"),"shutdown");
    // start with card unavailable
    menustore_set_item_state("eject_card", "system_bottom", "disabled");
#endif    
}


void menustore_set_text()
{
    LOGPRINTF("entry");

    // update language texts
    menustore_set_item_label("desktop",         "system_top",    _("Go to Home"));
    menustore_set_item_label("back_to_library", "system_top",    "");
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    menustore_set_item_label("lock",            "system_bottom", _("Lock Sensors"));
    menustore_set_item_label("eject_card",      "system_bottom", _("Safely Remove"));
    menustore_set_item_label("shutdown",        "system_bottom", _("Turn Off Device"));
#endif
    menustore_set_item_label("rotate_screen",   "system_bottom", _("Rotate"));
}



void menustore_create(taskbar_cb_t cb)
{
    menu_store = gtk_tree_store_new (TREE_NUM_COLS,
                                     G_TYPE_INT,        // TREE_COL_TYPE
                                     G_TYPE_STRING,     // TREE_COL_NAME
                                     G_TYPE_STRING,     // TREE_COL_TEXT
                                     G_TYPE_STRING,     // TREE_COL_ICON
                                     G_TYPE_INT,        // TREE_COL_STATE
                                     GDK_TYPE_PIXBUF,   // TREE_COL_IMAGE_NORMAL
                                     GDK_TYPE_PIXBUF,   // TREE_COL_IMAGE_ALTERNATE
                                     G_TYPE_POINTER);   // TREE_COL_ITER
    taskbar_home_cb = cb;
    add_system_groups();
    menustore_set_text();
}


static char* type2str[] = {
  [MENU_TYPE_GROUP]   = "GROUP",
  [MENU_TYPE_ITEM]    = " ITEM",
};


static const char *state2str[] =
{
    [MENU_STATE_NORMAL]    = "normal",
    [MENU_STATE_SELECTED]  = "selected",
    [MENU_STATE_DISABLED]  = "disabled",
    [MENU_STATE_ALTERNATE] = "alternate",
};


static enum menu_state string2state(const char* statestr)
{
    if (g_ascii_strcasecmp(statestr, "normal") == 0) return MENU_STATE_NORMAL;
    if (g_ascii_strcasecmp(statestr, "selected") == 0) return MENU_STATE_SELECTED;
    if (g_ascii_strcasecmp(statestr, "disabled") == 0) return MENU_STATE_DISABLED;
    if (g_ascii_strcasecmp(statestr, "alternate") == 0) return MENU_STATE_ALTERNATE;
    ERRORPRINTF("unknown state: %s", statestr);
    return -1;
}


static void pixbuf_unref(GdkPixbuf *img)
{
    if (img != NULL) g_object_unref(img);
}


#if (TESTING_ON)
static int debug_count = 0;

static gboolean printfunc(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer data)
{
    enum menu_type type = MENU_TYPE_ITEM;
    enum menu_state state = MENU_STATE_NORMAL;
    gchar *name = NULL;
    gchar *text = NULL;
    gchar *icon = NULL;
    GdkPixbuf *img_normal = NULL;
    GdkPixbuf *img_alternate = NULL;

    gtk_tree_model_get(model, iter,
                       TREE_COL_TYPE, &type,
                       TREE_COL_NAME, &name,
                       TREE_COL_TEXT, &text,
                       TREE_COL_ICON, &icon,
                       TREE_COL_STATE, &state,
                       TREE_COL_IMAGE_NORMAL, &img_normal,
                       TREE_COL_IMAGE_ALTERNATE, &img_alternate,
                       -1);
    int depth = gtk_tree_path_get_depth(path);
    char indent [20];
    memset(indent, ' ', sizeof(indent));
    indent[(depth-1)*3] = 0;
    gchar* pathstr = gtk_tree_path_to_string(path);
    printf("[%2d %s] %s%s '%s'   icon=%s  images=%p|%p  state=%s\n", ++debug_count, type2str[type],
            indent, name, text, icon, img_normal, img_alternate, state2str[state]);

    pixbuf_unref(img_normal);
    pixbuf_unref(img_alternate);
    g_free(name);
    g_free(text);
    g_free(icon);
    g_free(pathstr);
    return FALSE;
}
#endif


static gboolean compare_func(GtkTreeModel* model, GtkTreePath* path, GtkTreeIter* iter, gpointer data)
{
    SearchArgs* args = (SearchArgs*)data;
    gboolean stop = FALSE;

    gchar *name = NULL;
    GtkTreeIter* iter2;
    gtk_tree_model_get(model, iter,
                       TREE_COL_NAME, &name,
                       TREE_COL_ITER, &iter2,
                       -1);
    if (strcmp(name, args->name) == 0) {
        stop = TRUE;
        args->found = TRUE;
        args->iter = iter2;
    }
    
    g_free(name);
    return stop;
}


static gboolean name_exists(const char* name)
{
    SearchArgs args;
    args.name = name;
    args.found = FALSE;
    args.iter = NULL;
    gtk_tree_model_foreach(GTK_TREE_MODEL(menu_store), compare_func, &args);
    return args.found;
}


static GtkTreeIter *get_iter(const char* name)
{
    SearchArgs args;
    args.name = name;
    args.found = FALSE;
    args.iter = NULL;
    gtk_tree_model_foreach(GTK_TREE_MODEL(menu_store), compare_func, &args);
    if (!args.found) return NULL;
    return args.iter;
}


static GtkTreeIter* get_child_by_iter(const char* iname, GtkTreeIter *parent_iter)
{
    GtkTreeIter child_iter;
    gboolean result = gtk_tree_model_iter_children(GTK_TREE_MODEL(menu_store), &child_iter, parent_iter);
    while (result == TRUE)
    {
        gboolean found = FALSE;
        gchar* child_name = NULL;
        GtkTreeIter* iter2;
        gtk_tree_model_get (GTK_TREE_MODEL(menu_store), &child_iter,
                            TREE_COL_NAME, &child_name,
                            TREE_COL_ITER, &iter2,
                            -1);
        if (strcmp(iname, child_name) == 0) found = TRUE; 
        g_free(child_name);
        if (found) return iter2;
        result = gtk_tree_model_iter_next(GTK_TREE_MODEL(menu_store), &child_iter);
    }
    return NULL;
}


static GtkTreeIter* get_child_by_name(const char* iname, const char* pname)
{
    GtkTreeIter *parent_iter = NULL;
    if (pname) {
        // find parent
        parent_iter = get_iter(pname);
        if (!parent_iter) {
            WARNPRINTF("parent does not exist! (iname=%s, pname=%s)", iname, pname);
            return 0;
        }
    }

    return get_child_by_iter(iname, parent_iter);
}


gboolean menustore_add_group(const char *iname, const char *pname, const char *text, const char *icon)
{
    TRACE("%s() iname=%s  pname=%s  text=%s\n", __func__, iname, pname, text);

    // group name should be unique!
    if (name_exists(iname)) {
        LOGPRINTF("name '%s' already used, re-adding", iname);
        menustore_remove_group(iname);
    }

    // find optional parent
    GtkTreeIter *parent_iter = NULL;
    if (pname != NULL && strlen(pname) > 0) {
        parent_iter = get_iter(pname);
        if (!parent_iter) {
            WARNPRINTF("parent does not exist! (iname=%s, pname=%s)", iname, pname);
            return FALSE;
        }
    }

    // add to store and list
    GtkTreeIter iter;
    gtk_tree_store_append (GTK_TREE_STORE(menu_store), &iter, parent_iter);
    gtk_tree_store_set (GTK_TREE_STORE(menu_store), &iter,
                      TREE_COL_TYPE, MENU_TYPE_GROUP,
                      TREE_COL_NAME, iname,
                      TREE_COL_TEXT, text,
                      TREE_COL_ICON, icon,
                      TREE_COL_STATE, MENU_STATE_NORMAL,
                      TREE_COL_IMAGE_NORMAL, pixlist_icon_state(icon, "normal"),
                      TREE_COL_ITER, gtk_tree_iter_copy(&iter),
                      -1);
    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


gboolean menustore_add_item(const char *iname, const char *pname, const char *text, const char *icon)
{
    TRACE("%s() iname=%s  pname=%s  text=%s\n", __func__, iname, pname, text);

    // check pname
    if (pname == 0 || strlen(pname) == 0) {
        WARNPRINTF("no pname (iname=%s)", iname);
        return FALSE;
    }

    // find parent
    GtkTreeIter *parent_iter = get_iter(pname);
    if (!parent_iter) {
        WARNPRINTF("parent does not exist! (iname=%s, pname=%s)", iname, pname);
        return FALSE;
    }

    // check if child exists
    if (get_child_by_iter(iname, parent_iter)) {
        ERRORPRINTF("name '%s.%s' already used", pname, iname);
        return FALSE;
    }

    // add to store and list
    GtkTreeIter iter;
    gtk_tree_store_append (GTK_TREE_STORE(menu_store), &iter, parent_iter);
    gtk_tree_store_set (GTK_TREE_STORE(menu_store), &iter,
                        TREE_COL_TYPE, MENU_TYPE_ITEM,
                        TREE_COL_NAME, iname,
                        TREE_COL_TEXT, text,
                        TREE_COL_ICON, icon,
                        TREE_COL_STATE, MENU_STATE_NORMAL,
                        TREE_COL_IMAGE_NORMAL, pixlist_icon_state(icon, "normal"),
                        TREE_COL_IMAGE_ALTERNATE, pixlist_icon_state(icon, "alternate"),
                        TREE_COL_ITER, gtk_tree_iter_copy(&iter),
                        -1);
    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


static gboolean check_group(const char* group, const char* iname)
{
    if (strlen(group) > 0 && !name_exists(group))
    {
        WARNPRINTF("cannot find group '%s' for item %s", group, iname);
        return FALSE;
    }
    return TRUE;
}


static void add_menu_group2(GList** list, const char* group)
{
    if (strlen(group) != 0) {
       *list = g_list_prepend(*list, g_strdup(group));
    }
}


static MenuEntry* find_menu(const char* name)
{
    if (name == NULL) return NULL;

    GList *iter = g_list_first(menu_list);
    while (iter != NULL)
    {
        MenuEntry *entry = (MenuEntry *) iter->data;
        if (strcmp(entry->name, name) == 0) return entry;
        iter = g_list_next(iter);
    }
    return NULL;
}


const char* menustore_get_current_service()
{
    MenuEntry *entry = find_menu(current_menu);
    if (entry) return entry->service;
    return "";
}


gboolean menustore_add_menu(const char *iname, const char *ilabel, const char *service,
                            const char *group1,  const char *group2,
                            const char *group3, const char *group4)
{
    TRACE("%s() iname=%s  ilabel=%s  service=%s  groups={%s, %s, %s, %s)\n", __func__,
        iname, ilabel, service, group1, group2, group3, group4);

    // check name
    if (find_menu(iname)) {
        LOGPRINTF("name '%s' already used, re-adding", iname);
        menustore_remove_menu(iname);
    }

    // check groups
    if (!check_group(group1, iname)) return FALSE;
    if (!check_group(group2, iname)) return FALSE;
    if (!check_group(group3, iname)) return FALSE;
    if (!check_group(group4, iname)) return FALSE;

    // add menu to list
    MenuEntry *new_menu = g_new0 (MenuEntry, 1);
    new_menu->name = g_strdup(iname);
    new_menu->service = g_strdup(service);
    new_menu->groups = NULL;
    menu_list = g_list_append(menu_list, new_menu);

    // add groups to menu
    add_menu_group2(&new_menu->groups, group1);
    add_menu_group2(&new_menu->groups, group2);
    add_menu_group2(&new_menu->groups, group3);
    add_menu_group2(&new_menu->groups, group4);

    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


static gboolean check_type(GtkTreeIter* iter, const char* name, enum menu_type type)
{
    enum menu_type realtype = MENU_TYPE_ITEM;
    gtk_tree_model_get (GTK_TREE_MODEL(menu_store), iter,
                        TREE_COL_TYPE, &realtype,
                        -1);
    if (realtype != type)
    {
        WARNPRINTF("'%s' is not of type %s", name, type2str[type]);
        return FALSE;
    }
    return TRUE;
}


gboolean menustore_remove_group(const char *name)
{
    TRACE("%s() name=%s\n", __func__, name);

    // check name
    GtkTreeIter *menu_iter = get_iter(name);
    if (menu_iter == NULL)
    {
        LOGPRINTF("'%s' not found", name);
        return FALSE;
    }

    if (!check_type(menu_iter, name, MENU_TYPE_GROUP)) return FALSE;

    // remove children in this group and below
    GtkTreeIter child_iter;
    if (gtk_tree_model_iter_children(GTK_TREE_MODEL(menu_store), &child_iter, menu_iter))
    {
        enum menu_type type = MENU_TYPE_ITEM;
        gchar* child_name = NULL;
        gtk_tree_model_get (GTK_TREE_MODEL(menu_store), &child_iter,
                            TREE_COL_NAME, &child_name,
                            TREE_COL_TYPE, &type,
                            -1);
        switch (type) {
        case MENU_TYPE_GROUP:
            menustore_remove_group(child_name);
            break;
        case MENU_TYPE_ITEM:
            menustore_remove_item(child_name, name);
            break;
        }
        g_free(child_name);
    }

    // remove group itself
    gtk_tree_store_remove(GTK_TREE_STORE(menu_store), menu_iter);
    gtk_tree_iter_free(menu_iter);

    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


gboolean menustore_remove_item(const char *iname, const char* pname)
{
    TRACE("%s() iname=%s  pname=%s\n", __func__, iname, pname);

    // check parent name
    if (pname == 0 || strlen(pname) == 0) {
        WARNPRINTF("no pname (iname=%s)", iname);
        return FALSE;
    }

    // find parent
    GtkTreeIter *parent_iter = get_iter(pname);
    if (!parent_iter) {
        WARNPRINTF("parent does not exist! (iname=%s, pname=%s)", iname, pname);
        return FALSE;
    }

    // check if child exists
    GtkTreeIter *child_iter = get_child_by_iter(iname, parent_iter);
    if (!child_iter) {
        LOGPRINTF("'%s.%s' not found", pname, iname);
        return FALSE;
    }

    if (!check_type(child_iter, iname, MENU_TYPE_ITEM)) return FALSE;

    // remove item
    gtk_tree_store_remove(GTK_TREE_STORE(menu_store), child_iter);
    gtk_tree_iter_free(child_iter);

    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


static void free_menu_entry(MenuEntry* entry)
{
    g_free(entry->name);
    g_free(entry->service);
    g_free(entry);
}


gboolean menustore_remove_menu(const char *name)
{
    TRACE("%s() name=%s\n", __func__, name);

    GList *iter = g_list_first(menu_list);
    while (iter != NULL)
    {
        MenuEntry *entry = (MenuEntry *) iter->data;
        if (strcmp(entry->name, name) == 0) break;
        iter = g_list_next(iter);
    }

    if (iter == NULL) {
        WARNPRINTF("menu '%s' not found", name);
        return FALSE;
    }

    free_menu_entry(iter->data);
    menu_list = g_list_delete_link(menu_list, iter);

    if (current_menu && strcmp(current_menu, name) == 0)
    {
        menustore_set_current_menu(NULL);
    }

    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


static GdkPixbuf *get_pixbuf(enum menu_state state, GtkTreeIter *menu_iter)
{
    GdkPixbuf *img = NULL;
    switch (state)
    {
        case MENU_STATE_NORMAL:
        case MENU_STATE_SELECTED:
        case MENU_STATE_DISABLED:
            gtk_tree_model_get(GTK_TREE_MODEL(menu_store), menu_iter,
                               TREE_COL_IMAGE_NORMAL, &img,
                               -1);
            break;

        case MENU_STATE_ALTERNATE:
            gtk_tree_model_get(GTK_TREE_MODEL(menu_store), menu_iter,
                               TREE_COL_IMAGE_ALTERNATE, &img,
                               -1);
            break;
    }
    if (img == NULL)
    {
        if (state == MENU_STATE_SELECTED)
        {
            img = pixlist_icon_state("selected", "normal");
        } else {
            img = pixlist_icon_state("blank", "normal");
        }
        if (img) g_object_ref(img);
    }
    return img;
}


static gboolean update_state(GtkTreeIter* iter, const char* iname, const char* pname, const char* stateStr)
{
    enum menu_state state = string2state(stateStr);
    if ((gint) state == -1) return FALSE;

    // update state in menu store
    gtk_tree_store_set(GTK_TREE_STORE(menu_store), iter,
                     TREE_COL_STATE, state,
                     -1);

    g_popup_has_changed = TRUE;
    g_toolbar_has_changed = TRUE;
    return TRUE;
}


gboolean menustore_set_group_state(const char *name, const char *stateStr)
{
    TRACE("%s() name=%s  state=%s\n", __func__, name, stateStr);

    GtkTreeIter *iter = get_iter(name);
    if (iter == NULL)
    {
        LOGPRINTF("'%s' not found", name);
        return FALSE;
    }

    if (!check_type(iter, name, MENU_TYPE_GROUP)) return FALSE;

    return update_state(iter, name, NULL, stateStr);
}


gboolean menustore_set_item_state(const char *iname, const char *pname, const char *stateStr)
{
    TRACE("%s() iname=%s  pname=%s  state=%s\n", __func__, iname, pname, stateStr);

    if ( (pname != NULL) && strcmp(pname, "general")==0 )
    {
        pname = "system_bottom";
    }
    
    // check item
    GtkTreeIter *iter = get_child_by_name(iname, pname);
    if (iter == NULL)
    {
        LOGPRINTF("'%s.%s' not found", pname, iname);
        return FALSE;
    }

    return update_state(iter, iname, pname, stateStr);
}


static void update_label(GtkTreeIter* iter, const char* iname, const char* pname, const char* label)
{
    // update state in menu store
    gtk_tree_store_set(GTK_TREE_STORE(menu_store), iter,
                       TREE_COL_TEXT, label,
                       -1);
    g_popup_has_changed = TRUE;
}


gboolean menustore_set_group_label(const char *name, const char *label)
{
    TRACE("%s() name=%s  label=%s\n", __func__, name, label);

    // check name
    GtkTreeIter *iter = get_iter(name);
    if (iter == NULL)
    {
        LOGPRINTF("'%s' not found", name);
        return FALSE;
    }

    // check type
    if (!check_type(iter, name, MENU_TYPE_GROUP)) return FALSE;

    update_label(iter, name, NULL, label);
    return TRUE;
}


gboolean menustore_set_item_label(const char *iname, const char *pname, const char *label)
{
    TRACE("%s() iname=%s  pname=%s  label=%s\n", __func__, iname, pname, label);

    // check name
    GtkTreeIter *iter = get_child_by_name(iname, pname);
    if (iter == NULL)
    {
        LOGPRINTF("'%s.%s' not found", pname, iname);
        return FALSE;
    }

    update_label(iter, iname, pname, label);
    return TRUE;
}


gboolean menustore_popup_has_changed()
{
    return g_popup_has_changed;
}


void menustore_clear_popup_changed()
{
    g_popup_has_changed = FALSE;
}


#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
gboolean menustore_toolbar_has_changed()
{
    return g_toolbar_has_changed;
}


void menustore_clear_toolbar_changed()
{
    g_toolbar_has_changed = FALSE;
    //WARNPRINTF("clear changed");
}


static MenuToolItems* find_tools_for_menu(const char* menu)
{
    if (menu == NULL) return NULL;

    int num = sizeof(hardcoded_tools) / sizeof(hardcoded_tools[0]);
    int i;
    for (i=0; i<num; i++) {
        const char* hardcoded = hardcoded_tools[i].menu;
        // NOTE: menu name must start with hardcoded name. (for menu_notepad_2831)
        if (strncmp(hardcoded, menu, strlen(hardcoded)) == 0) {
            return &hardcoded_tools[i];
        }
    }
    return NULL;
}


int menustore_get_tool_limit()
{
    // HACK to work around 4/16 slots. If slot[4] exists, we have a long one
    MenuToolItems* tools = find_tools_for_menu(current_menu);
    if (tools == NULL) return 0;

    ToolItem* item = &tools->items[MENUSTORE_SMALL_TOOLS];
    if (item->pname == NULL) return MENUSTORE_SMALL_TOOLS;
    return MENUSTORE_NUM_TOOLS;
}


GdkPixbuf* menustore_get_tool_icon(int index)
{
    g_assert(index < MENUSTORE_NUM_TOOLS);

    MenuToolItems* tools = find_tools_for_menu(current_menu);
    if (tools == NULL) goto blank;

    ToolItem* item = &tools->items[index];
    if (item->pname == NULL || item->iname == NULL) goto blank;
    GtkTreeIter *iter = get_child_by_name(item->iname, item->pname);
    if (iter == NULL) {
        ERRORPRINTF("'%s.%s' not found", item->pname, item->iname);
        goto blank;
    }

    enum menu_state state;
    gchar *icon_name = NULL;
    gtk_tree_model_get(GTK_TREE_MODEL(menu_store), iter,
                       TREE_COL_STATE, &state,
                       TREE_COL_ICON, &icon_name,
                       -1);
    GdkPixbuf* img = pixlist_toolbar_icon(icon_name, state2str[state]);
    g_free(icon_name);
    return img;
blank:
    return pixlist_icon_state("toolbar_blank", "normal");
}


void menustore_activate_toolitem(int index, tool_func_t func)
{
    g_assert(index < MENUSTORE_NUM_TOOLS);

    // check current menu
    MenuToolItems* tools = find_tools_for_menu(current_menu);
    if (tools == NULL) {
        WARNPRINTF("no tool for menu");
        return;
    }

    // find item
    ToolItem* item = &tools->items[index];
    if (item->pname == NULL || item->iname == NULL) return;

    menustore_activate_item(item->pname, item->iname, func);
}
#endif


gboolean menustore_activate_item_iter(gpointer user_data, tool_func_t func)
{
    GtkTreeIter* iter = (GtkTreeIter*) user_data;

    gchar *iname;
    enum menu_type type = MENU_TYPE_ITEM;
    enum menu_state state = MENU_STATE_NORMAL;
    gtk_tree_model_get(GTK_TREE_MODEL (menu_store), iter,
                       TREE_COL_TYPE, &type,
                       TREE_COL_NAME, &iname,
                       TREE_COL_STATE, &state,
                       -1);
    
    if (type != MENU_TYPE_ITEM) {
        goto error;
    }
    if (state == MENU_STATE_DISABLED)
    {
        WARNPRINTF("cannot activate disabled item");
        goto error;
    }

    GtkTreeIter parent_iter;
    gtk_tree_model_iter_parent(GTK_TREE_MODEL (menu_store), 
                               &parent_iter,
                               iter);
    gchar *pname;
    gtk_tree_model_get(GTK_TREE_MODEL (menu_store), &parent_iter,
                       TREE_COL_NAME, &pname,
                       -1);

    if ((strcmp(pname, "system_top") == 0) || (strcmp(pname, "system_bottom") == 0))
    {
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        // special case for Taskbar for "Home" and "Back to category"
        if ( (strcmp(iname, "desktop") == 0) || (strcmp(iname, "back_to_library") == 0) )
        {
            taskbar_home_cb();
        }
#endif
        if (strcmp(iname, "eject_card") == 0)
        {
            // when card is going to be ejected, last_app hack needs to be notified
            taskbar_store_tasks(1); // indicate card eject
        }    
        if (strcmp(iname, "back_to_library") == 0)
        {
            func(iname, "tasks", current_menu, state2str[state], DBUS_SERVICE_SYSTEM_CONTROL);
        }
        else
        {
            func(iname, "general", current_menu, state2str[state], DBUS_SERVICE_SYSTEM_CONTROL);
        }
    }
    else
    {
        func(iname, pname, current_menu, state2str[state], menustore_get_current_service());
    }
    g_free(iname);
    g_free(pname);
    return TRUE;
error:
    g_free(iname);
    return FALSE;
}


void menustore_activate_item(const char* pname, const char* iname, tool_func_t func)
{
    LOGPRINTF("entry [%s.%s]", pname, iname);

    GtkTreeIter *iter = get_child_by_name(iname, pname);
    if (iter == NULL) {
        ERRORPRINTF("'%s.%s' not found", pname, iname);
        return;
    }
    menustore_activate_item_iter(iter, func);
}


static gboolean fill_add_group(const char* group,
                               add_item_func item_cb,
                               add_item_func submenu_cb,
                               separator_func separator_cb,
                               gpointer user_data)
{
    GtkTreeIter *menu_iter = get_iter(group);
    if (menu_iter == NULL)
    {
        WARNPRINTF("entry: name [%s] not found", group);
        return FALSE;
    }

    // check if group has children
    gint num_items = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(menu_store), menu_iter);
    if (num_items == 0)
    {
        WARNPRINTF("given group `%s` has no children", group);
        return FALSE;
    }
    
    gboolean rc = FALSE;

    GtkTreeIter child_iter;
    gboolean result = gtk_tree_model_iter_children(GTK_TREE_MODEL(menu_store), &child_iter, menu_iter);
    while (result == TRUE)
    {
        // get info from child
        gchar *name = NULL;
        gchar *text = NULL;
        enum menu_type  type = MENU_TYPE_ITEM;
        enum menu_state state = MENU_STATE_NORMAL;
        gtk_tree_model_get(GTK_TREE_MODEL(menu_store), &child_iter,
                           TREE_COL_NAME, &name,
                           TREE_COL_TEXT, &text,
                           TREE_COL_STATE, &state,
                           TREE_COL_TYPE, &type,
                           -1);

        if (state != MENU_STATE_DISABLED && strcmp("", text) == 0) {
            LOGPRINTF("empty label for %s.%s", group, name);
        }

        GdkPixbuf *img = get_pixbuf(state, &child_iter);
        GtkTreeIter *iter_copy = gtk_tree_iter_copy(&child_iter);

        int count = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(menu_store), &child_iter);
        if (type == MENU_TYPE_GROUP && count != 0) {
            gpointer user_data2 = submenu_cb(name, text, state, img, iter_copy, user_data);
            separator_cb(user_data2);
            GtkTreeIter child_child;
            gboolean is_next = gtk_tree_model_iter_children(GTK_TREE_MODEL(menu_store), &child_child, &child_iter);
            while (is_next)
            {
                gchar *group_name = NULL;
                gtk_tree_model_get(GTK_TREE_MODEL(menu_store), &child_child,
                                   TREE_COL_NAME, &group_name,
                                   -1);

                // add group to submenu
                gboolean is_added = fill_add_group(group_name, item_cb, submenu_cb, separator_cb, user_data2);
                g_free(group_name);

                is_next = gtk_tree_model_iter_next(GTK_TREE_MODEL(menu_store), &child_child);
                if (is_added && is_next)
                {
                    // group follows, add separator line
                    separator_cb(user_data2);
                }
            }
        } else {
            item_cb(name, text, state, img, iter_copy, user_data);
        }

        pixbuf_unref(img);
        g_free(name);
        g_free(text);

        if (state != MENU_STATE_DISABLED) rc = TRUE;

        result = gtk_tree_model_iter_next(GTK_TREE_MODEL(menu_store), &child_iter);
    }
    return rc;
}


static const GList* get_menu_groups()
{
    MenuEntry *entry = find_menu(current_menu);
    if (entry) return entry->groups;
    return NULL;
}


void menustore_fill_menu(add_item_func item_cb,
                         add_item_func submenu_cb,
                         separator_func separator_cb,
                         gpointer user_data)
{
    fill_add_group("system_top", item_cb, submenu_cb, separator_cb, user_data);
    separator_cb(user_data);

    gboolean is_added = FALSE;
    const GList *groups = get_menu_groups();
    while (groups) {
        is_added = fill_add_group((const char*)groups->data, item_cb, submenu_cb, separator_cb, user_data);

        if (is_added) separator_cb(user_data);

        groups = g_list_next(groups);
    }

    fill_add_group("system_bottom", item_cb, submenu_cb, separator_cb, user_data);
}


#if (TESTING_ON)
static void testing_extra_prints()
{
    printf("Current menu = '%s'  service='%s'\n",
            current_menu, menustore_get_current_service());
    printf("---- %s() MENULIST:\n", __func__);
    GList *iter = g_list_first(menu_list);
    while (iter != NULL)
    {
        MenuEntry *entry = (MenuEntry *) iter->data;
        printf("%s  service=%s\n", entry->name, entry->service);
        GList *iter2 = g_list_first(entry->groups);
        while (iter2) {
            printf("  %s\n", (const char*)iter2->data);
            iter2 = g_list_next(iter2);
        }
        iter = g_list_next(iter);
    }

    printf("\n---- %s() MENU_STORE:\n", __func__);
    debug_count = 0;
    gtk_tree_model_foreach(GTK_TREE_MODEL(menu_store), printfunc, NULL);

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    printf("\n---- %s() HARDCODED TOOLITEMS:\n", __func__);
    int num = sizeof(hardcoded_tools) / sizeof(hardcoded_tools[0]);
    int i;
    for (i=0; i<num; i++) {
        MenuToolItems* tools = &hardcoded_tools[i];

        printf("  %s\n", tools->menu);
        int i;
        for (i=0; i<MENUSTORE_NUM_TOOLS; i++) {
            if (tools->items[i].pname) {
                printf("    [%2d] %s.%s\n", i, tools->items[i].pname, tools->items[i].iname);
            }
        }
    }
#endif
}

static char buffer[1000];
static char* cp = NULL;

gpointer print_item(const char* name,
                    const char* text,
                    enum menu_state state,
                    GdkPixbuf* img,
                    gpointer menu_data,
                    gpointer user_data)
{
    strcpy(cp, name);
    cp += strlen(name);
    *cp++ = '\n';
    return NULL;
}


void print_separator(gpointer user_data) {} // ignore


static gboolean find_route(const char* pname, const char* iname, const char* group, char** route, gboolean top_lvl)
{
    char* orig = *route;
    char *rp = *route;
    GtkTreeIter *group_iter = get_iter(group);
    g_assert(group_iter);

    GtkTreeIter child_iter;
    gboolean result = gtk_tree_model_iter_children(GTK_TREE_MODEL(menu_store), &child_iter, group_iter);
    gboolean found = FALSE;
    while (result == TRUE)
    {
        // get child info
        gchar *name = NULL;
        gchar *text = NULL;
        enum menu_type type = MENU_TYPE_ITEM;
        enum menu_state state = MENU_STATE_NORMAL;
        gtk_tree_model_get(GTK_TREE_MODEL(menu_store), &child_iter,
                           TREE_COL_NAME, &name,
                           TREE_COL_TEXT, &text,
                           TREE_COL_STATE, &state,
                           TREE_COL_TYPE, &type,
                           -1);
        gboolean visible = TRUE;
        if (strcmp(name, iname) == 0 && strcmp(group, pname) == 0) found = TRUE;
        if (state == MENU_STATE_DISABLED || strcmp(text, "") == 0) visible = FALSE;
        g_free(text);
        if (found) {
            if (!visible) {
                // not visible
                *orig++ = 'X';
                *orig = 0;
            } else {
                // match
                *rp++ = 'E';
                *rp = 0;

            }
            g_free(name);
            return TRUE; 
        } else {
            if (visible) {
                // TODO groups can have '' name, does this occur?
                if (type == MENU_TYPE_GROUP) {
                    char* before = rp;
                    if (top_lvl) {
                        *rp++ = 'E';
                        *rp++ = 'D';
                        *route = rp;
                    }
                    // search recursively
                    gboolean found2 = find_route(pname, iname, name, route, FALSE);
                    if (found2) {
                        g_free(name);
                        return TRUE;
                    } else {
                        if (top_lvl) {  // backtrack submenu
                            rp = before;
                            *rp++ = 'D';
                            *route = rp;
                        }
                    }
                } else {
                    *rp++ = 'D';
                    *route = rp;
                }
            } else {
                // skipping
            }
        }

        g_free(name);
        result = gtk_tree_model_iter_next(GTK_TREE_MODEL(menu_store), &child_iter);
    }
    return FALSE;
}


static const char* testing_route_to_item(const char* pname, const char* iname)
{
    char route[128];
    memset(route, 0, sizeof(route));
    char *cp = &route[0];

    gboolean found = find_route(pname, iname, "system_top", &cp, TRUE);
    if (found) goto out_found;

    const GList *groups = get_menu_groups();
    while (groups) {
        found = find_route(pname, iname, (const char*)groups->data, &cp, TRUE);
        if (found) goto out_found;
        groups = g_list_next(groups);
    }

    found = find_route(pname, iname, "system_bottom", &cp, TRUE);
out_found:
    if (found && route[strlen(route)-1] == 'X') found = FALSE;
    if (!found) {
        route[0] = 'X'; 
        route[1] = 0;
    }
    // add spaces to make parsing easier
    static char result[256];
    char *ip = &route[0];
    char *op = &result[0];
    unsigned int i = 0;
    for (i=0; i<strlen(route); i++) {
        *op++ = *ip++; 
        *op++ = ' ';
    }
    *op = 0;
    return result;
}


const char* testing_menustore_print()
{
    testing_extra_prints();

    cp = buffer;
    menustore_fill_menu(print_item, print_item, print_separator, NULL);
    *cp = 0;
    return buffer;
}


const char* testing_menustore_get_route(const char* pname, const char* iname)
{
    return testing_route_to_item(pname, iname);
}

#endif // TESTING_ON

