/*
 * File Name: main.c
 */

/*
 * This file is part of hello-world.
 *
 * hello-world is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * hello-world is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

// system include files, between < >
#include <glib.h>
#include <gtk/gtk.h>
#include <signal.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>

// ereader include files, between < >

// local include files, between " "
#include "log.h"
#include "i18n.h"
#include "ipc.h"
#include "main.h"
#include "menu.h"
#include "db.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

static const char  *rc_filename = DATADIR "/" PACKAGE_NAME ".rc";


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GtkWidget        *g_title = NULL;        // screen title
static GtkWidget        *g_menu  = NULL;        // latest menu selection
static GtkWidget        *g_pageturn = NULL ; 
static GtkWidget        *g_portrait = NULL ; 

static viewtypes_t      g_viewtype = -1;        // currently selected view type


//============================================================================
// Local Function Definitions
//============================================================================

static void     display_view_type       (void);
static void     display_flipbar_portrait(void);
static void     main_set_text           (void);
static gboolean on_startup_complete     (gpointer data);


//============================================================================
// Functions Implementation
//============================================================================

// print usage text and quit
static void usage (const char *argv_0)
{
    static const char *usage_text = 
                        "\n"
                        "usage: %s [options]\n"
                        "\n"
                        "options:\n"
                        "    --help\n"
                        "        Print help text and quit\n";

    printf(usage_text, argv_0);

    exit(1);
}

// get command-line options
static void parse_arguments (int argc, char **argv)
{
    int  i;

    // parse options
    for (i = 1 ; i < argc ; i++)
    {
        if (strcmp(argv[i], "--help") == 0)
        {
            usage(argv[0]);
        }
        else
        {
            ERRORPRINTF("ignore unknown option [%s]", argv[i]);
            usage(argv[0]);
        }
    }
}


// set locale (language)
void main_set_locale (const char *locale)
{
    LOGPRINTF("entry: locale [%s]", locale);

    g_setenv("LANG", locale, TRUE);
    setlocale(LC_ALL, "");

    main_set_text();
    menu_set_text();
}


// create main screen layout
static GtkWidget *create_screen_layout (void)
{
    GtkWidget   *background = NULL;  // return value
    GtkWidget   *widget;
    GtkWidget   **p_widget;
    GtkBox      *vbox;

    LOGPRINTF("entry");

    // object hierarchy:
    //     background (alignment)
    //       |
    widget = gtk_alignment_new(0.0, 0.0, 1.0, 1.0);
    gtk_alignment_set_padding( GTK_ALIGNMENT(widget), 100, 100, 100, 100);
    gtk_widget_show(widget);
    background = widget;
    //       |
    //       |-- vbox
    //             |
    widget = gtk_vbox_new(FALSE, 100);
    gtk_container_add(GTK_CONTAINER(background), widget);
    gtk_widget_show(widget);
    vbox = GTK_BOX(widget);
    //             |
    //             |-- g_title (label)
    //             |
    p_widget = &g_title;
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-hello-title");
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    //             |
    //             |-- g_menu (label)
    p_widget = &g_menu;
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-hello-menu");
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    //             |
    //             |-- g_pageturn (label)
    p_widget = &g_pageturn;
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-hello-menu");
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    //             |
    //             |-- g_portrait (label)
    p_widget = &g_portrait;
    widget = gtk_label_new(NULL);
    gtk_widget_set_name(widget, "irex-hello-menu");
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    //             |
    //             |-- g_volume (label)
    p_widget = &g_volume;
    widget = gtk_label_new(NULL);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    //             |
    //             |-- g_action (label)
    p_widget = &g_action;
    widget = gtk_label_new(NULL);
    gtk_box_pack_start(vbox, widget, FALSE, FALSE, 0);
    g_signal_connect(widget, "destroy", G_CALLBACK(gtk_widget_destroyed), p_widget);
    gtk_widget_show(widget);
    g_assert(*p_widget == NULL);
    *p_widget = widget;
    
    main_set_text();
    
    return background;
}


// set screen texts
static void main_set_text (void)
{
    LOGPRINTF("entry");

    if (g_title)
    {
        gtk_label_set_text( GTK_LABEL(g_title), _("Hello world") );
    }
    display_view_type();
}


// show selected view type
void main_set_view_type ( viewtypes_t view_type )
{
    LOGPRINTF("entry: view_type [%d]", view_type);
    g_return_if_fail(view_type < N_VIEWTYPES);

    g_viewtype = view_type;
    display_view_type();
    display_flipbar_portrait();
    menu_select_view_type(view_type);
}


// show selected view type
static void display_view_type ( void )
{
    gchar       *msg  = NULL;
    gchar       *type = NULL;

    LOGPRINTF("entry: view_type [%d]", g_viewtype);

    if (g_menu)
    {
        switch (g_viewtype)
        {
            case ICONVIEW:
                type = _("thumbnails");
                break;
            case LISTVIEW:
                type = _("details");
                break;
            default:
                type = _("unknown");
        }

        msg = g_strdup_printf( _("Current view type: %d (%s)"), g_viewtype, type);
        gtk_label_set_text( GTK_LABEL(g_menu), msg );
        g_free(msg);
    }
}

static void display_flipbar_portrait( void)
{
    LOGPRINTF("entry: view_type [%d]", g_viewtype);

    gchar *msg = NULL ; 
    gchar *portrait = NULL ; 
    gchar *pageturning = NULL ; 

    gboolean is_inverted = ipc_sys_is_pageturn_inverted();
    gboolean is_portrait = ipc_sys_is_in_portrait_mode();

    if (is_inverted ) 
        pageturning = _("inverted");
    else 
        pageturning = _("normal");

    if (is_portrait ) 
        portrait = _("portrait");
    else 
        portrait = _("landscape");

    msg = g_strdup_printf( _("Pageturn mode: %d (%s)"), is_inverted, pageturning);
    gtk_label_set_text( GTK_LABEL(g_pageturn), msg );
    g_free(msg);

    msg = g_strdup_printf( _("Initial orientation: %d (%s)"), is_portrait, portrait);
    gtk_label_set_text( GTK_LABEL(g_portrait), msg );
    g_free(msg);

}


// terminate application
void main_quit (void)
{
    WARNPRINTF("entry");
    
    menu_destroy();
    
    if (g_main_window)
    {
        gtk_widget_destroy(g_main_window);
        g_main_window = NULL;
    }    
    
    if (gtk_main_level() > 0)
    {
        WARNPRINTF("quit main loop");
        gtk_main_quit();
    }
    else
    {
        WARNPRINTF("no main loop to quit, exit directly");
        _exit(0);
    }
}


// terminate signal
static void on_sigterm (int signo)
{
    WARNPRINTF("    -- entry " PACKAGE_NAME ", my pid [%d]", getpid());

    // stop main process, prepare to quit application
    main_quit();

    WARNPRINTF("    -- leave " PACKAGE_NAME);
}


// report startup completed
static gboolean on_startup_complete (gpointer data)
{
    LOGPRINTF("entry");

    ipc_sys_startup_complete();

    return FALSE; // remove timer source
}


//  do some actions on the metadb
static gboolean do_example_db_actions(gpointer data)
{
    // This function demonstrates the necessary steps to update the metadata
    // of a document to store/update the:
    // - title
    // - author 
    // - thumbnail (two sizes)
    // - number of pages
    // - current page
    
    printf("To run this application, make sure that you have the following files in\n"
            "the \"/media/mmcblk0p1/Examples\" dir:\n"
            " - \"example.epub\" \n"
            " - \"example.png\" \n");

    LOGPRINTF("entry");
    gboolean result = TRUE;
    erMetadb my_db = open_database("/media/mmcblk0p1");
    if ( my_db == NULL ) 
    {
        WARNPRINTF("Could not open metadb at dir %s\n", "/media/mmcblk0p1");
        goto err_open_database;
    }

    const gchar* filepath = "Examples";
    const gchar* filename = "example.epub";
    
#if 0
    GString* title = g_string_new(NULL);
    GString* author = g_string_new(NULL);

    result = load_file_metadata(my_db,   
                                filepath,
                                filename,
                                title,   
                                author,  
                                NULL,    
                                NULL     
                                );

    if (result == FALSE)
    {
        WARNPRINTF(" -- FAIL: load_file_metadata: result = %d, could not load metadata for file: %s\n", result, filename );
        goto err_load_metadata;
    }
#else
    GString* title = g_string_new("This is the title");
    GString* author = g_string_new("I am the author");
#endif 

    // load PNG file and scale it to thumb sizes -- these functions are blocking
    GdkPixbuf* small = NULL;
    GdkPixbuf* medium = NULL;
    small  = gdk_pixbuf_new_from_file_at_scale("/media/mmcblk0p1/Examples/example.png",  60,  60, TRUE, NULL);
    if ( small  == NULL ) { goto err_no_small; }

    medium = gdk_pixbuf_new_from_file_at_scale("/media/mmcblk0p1/EXamples/example.png", 120, 120, TRUE, NULL);
    if ( medium == NULL ) { goto err_no_medium; }

    // draw a rectangle around the thumbs
    gint width = gdk_pixbuf_get_width(small);
    gint height = gdk_pixbuf_get_height(small);
    pixbuf_draw_rectangle(small, 0, 0, width, height, 0x00000000);
    width = gdk_pixbuf_get_width(medium);
    height = gdk_pixbuf_get_height(medium);
    pixbuf_draw_rectangle(medium, 0, 0, width, height, 0x00000000);

    thumbType small_thumb = { NULL, 0 };
    thumbType medium_thumb= { NULL, 0 } ;

    // convert to BLOB
    pixbuf_to_blob(small,  (gchar**) &(small_thumb.data),  (gsize*) &small_thumb.size);
    pixbuf_to_blob(medium,  (gchar**) &(medium_thumb.data),  (gsize*) &medium_thumb.size);

    result = save_file_metadata(my_db,              // erMetadb* database
                                filepath,
                                filename,
                                title,              // GString* title
                                author,             // GString* author
                                &small_thumb,       // thumbType* small
                                &medium_thumb       // thumbType* medium
                                );
    
    if (result == FALSE)
    {
        WARNPRINTF(" -- FAIL: save_file_metadata: result = %d, could not save metadata for file: %s\n", result, filename);
        goto err_save_metadata;
    }

    // clean up
err_save_metadata:
    g_object_unref(medium);
err_no_medium:
    g_object_unref(small);
err_no_small:
#if 0 
err_load_metadata:
#endif 
    g_string_free(author, TRUE);
    g_string_free(title, TRUE);
    close_database(my_db);
err_open_database:
    return FALSE;
}


// main function
int main (int argc, char *argv[])
{
    GtkWidget        *window;
    GtkWidget        *widget;
    struct sigaction on_term;

    // parse command-line arguments
    parse_arguments(argc, argv);

    // catch the SIGTERM signal
    memset(&on_term, 0x00, sizeof(on_term));
    on_term.sa_handler = on_sigterm;
    sigaction(SIGTERM, &on_term, NULL);
#if LOGGING_ON
    sigaction(SIGINT,  &on_term, NULL);
#endif

    // init domain for translations
    textdomain(GETTEXT_PACKAGE);
    
    // init gtk, list the default rc files
    gtk_init(&argc, &argv);
    gchar** files = gtk_rc_get_default_files();
    while( *files )
    {
        WARNPRINTF("gtk_rc_get_default_files [%s]", *files);
        files++;
    }

    // init modules
    g_mountpoint = NULL;
    ipc_set_services();
    menu_init();

    // open the RC file associated with this program
    WARNPRINTF("gtk_rc_parse [%s]", rc_filename);
    gtk_rc_parse(rc_filename);

    // create the top level window 
    window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(window), PACKAGE " " VERSION);
    gtk_window_set_resizable(GTK_WINDOW(window), FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(window), 0);
    g_main_window = window;

    // add screen details
    widget = create_screen_layout();
    gtk_container_add(GTK_CONTAINER(window), widget);
    main_set_view_type(ICONVIEW);

    // make sure everything is visible 
    gtk_widget_show(window);

    if (argc > 0)
    {
        gchar *msg = NULL;
        msg = g_strdup_printf("Started with: %s", argv[1]);
        gtk_label_set_text( GTK_LABEL(g_action), msg);
        LOGPRINTF("%s", msg);
        g_free(msg);
    }
    
    // tell system daemon we are ready to go
    g_idle_add(on_startup_complete, NULL);

    // do some actions on the metadb
    g_idle_add(do_example_db_actions, NULL);
    
    // run the main loop
    LOGPRINTF("before gtk_main");
    gtk_main();
    LOGPRINTF("after gtk_main");

    return 0;
}


// run error dialog
void run_error_dialog (const gchar *msg)
{
    GtkWidget   *dialog = NULL;

    ERRORPRINTF("entry: msg [%s]", msg);

    dialog = gtk_message_dialog_new( GTK_WINDOW(g_main_window),
                                     GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_ERROR,
                                     GTK_BUTTONS_OK,
                                     msg );

    gtk_dialog_run( GTK_DIALOG(dialog) );
    gtk_widget_destroy( dialog );
}
