/*
 * File Name: lastapps_settings.c
 */

/*
 * This file is part of settings.
 *
 * settings is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * settings is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Copyright (C) 2009 Marcel Hendrickx
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

// system include files, between < >
#include <string.h>
#include <stdlib.h>

// ereader include files, between < >

// local include files, between " "
#include "config.h"
#include <gconf/gconf-client.h>
#include "log.h"
#include "ipc.h"
#include "i18n.h"
#include "screen_utils.h"
#include "lastapps_settings.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------

#define GCONF_BUFFER_SIZE                   128

// CAREFULL: These values need to match with the values in popupmenu!
// for soring retrieving last appc info from registry
#define KEYS_EREADER                "/apps/er"
#define KEYS_LASTAPPS               KEYS_EREADER "/lastapps" 
#define KEY_LASTAPP_MODE			"mode"
#define KEY_LASTAPP_CARD			"cardmode"

static const gchar      *REGVAL_LASTAPPS_OFF   = "off";
static const gchar      *REGVAL_LASTAPPS_LAST  = "last";
static const gchar      *REGVAL_LASTAPPS_OPEN  = "open";
static const gchar      *REGVAL_LASTAPPS_ALL   = "all";
static const gchar      *REGVAL_LASTAPPS_APP   = "app";

static const gchar      *REGVAL_LASTAPPS_CARD_NORMAL   = "normal";
static const gchar      *REGVAL_LASTAPPS_CARD_RESTORE  = "restore";

#define KEY_LASTAPP_OFF 	0
#define KEY_LASTAPP_LAST 	1
#define KEY_LASTAPP_OPEN 	2
#define KEY_LASTAPP_ALL 	3
#define KEY_LASTAPP_APP 	4

#define KEY_LASTAPP_CARD_NORMAL 	0
#define KEY_LASTAPP_CARD_RESTORE	1

typedef struct
{
    int lastapp_mode;
	int card_mode;
} LastAppsSettings;

static const int        SCREENITEMS_V_SPACING   =  20;

//----------------------------------------------------------------------------
// Global Constants
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static LastAppsSettings g_orig_lastapps_settings;
static LastAppsSettings g_cur_lastapps_settings;

static GtkWidget* g_lastapps_radios[5]            = {NULL, NULL, NULL, NULL, NULL};
static GtkWidget* g_cardremoval_radios[2]         = {NULL, NULL};


//============================================================================
// Local Function Definitions
//============================================================================

static GtkWidget* create_lastapps_widgets       (GtkWidget* top_level_vbox);
static GtkWidget* create_cardremoval_widgets    (GtkWidget* top_level_vbox);

static void on_lastapps_mode_changed(GtkWidget *widget, gpointer data);
static void on_cardremoval_mode_changed(GtkWidget *widget, gpointer data);
static void init_widgets_with_settings          (void);

static gchar *read_gconf_value(const gchar *dir,
                               const gchar *key,
                               GConfClient *client);
static gboolean write_gconf_value(const gchar *dir,
                                   const gchar *key,
                                   GConfClient *client,
								   const gchar *value);

//============================================================================
// Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--alignment
//       |--vbox
//          |--label
//          |--filler
//          |--Last Apps section vbox
//          |--separator
//          |--Card Removal section vbox
//          |--separator
GtkWidget* create_lastapps_items(void)
{
    GtkWidget* vbox       = NULL;
    GtkWidget* separator  = NULL;

    LOGPRINTF("entry");
    
    // Top level vbox.
    vbox = gtk_vbox_new(FALSE, SCREENITEMS_V_SPACING);

    // The lastapp section
    create_lastapps_widgets(vbox);

    // Add separator here.
    separator = create_separator();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 0);

    // The card removal section.
    create_cardremoval_widgets(vbox);

    // Update widget with current settings.
    init_widgets_with_settings();
    
    return vbox;
}


void load_lastapps_settings()
{
	const char* lastapp_mode_str;
	const char* card_mode_str;
	
	GConfClient *client = gconf_client_get_default();
    
    lastapp_mode_str = read_gconf_value(KEYS_LASTAPPS, KEY_LASTAPP_MODE, client);
    card_mode_str    =  read_gconf_value(KEYS_LASTAPPS, KEY_LASTAPP_CARD, client);
    
    // check if available in registry
    if (!lastapp_mode_str) lastapp_mode_str = REGVAL_LASTAPPS_OFF;
    if (!card_mode_str) card_mode_str = REGVAL_LASTAPPS_CARD_NORMAL;

	LOGPRINTF("entry:%s %s", lastapp_mode_str, card_mode_str);
	
	// Initialise
    g_orig_lastapps_settings.lastapp_mode = KEY_LASTAPP_OFF;
    g_orig_lastapps_settings.card_mode = KEY_LASTAPP_CARD_NORMAL;

	// load settings from gconf
	if (strcmp(lastapp_mode_str, REGVAL_LASTAPPS_OFF) == 0)        g_orig_lastapps_settings.lastapp_mode = KEY_LASTAPP_OFF;
	else if (strcmp(lastapp_mode_str, REGVAL_LASTAPPS_LAST) == 0)  g_orig_lastapps_settings.lastapp_mode = KEY_LASTAPP_LAST;
	else if (strcmp(lastapp_mode_str, REGVAL_LASTAPPS_OPEN) == 0)  g_orig_lastapps_settings.lastapp_mode = KEY_LASTAPP_OPEN;
	else if (strcmp(lastapp_mode_str, REGVAL_LASTAPPS_ALL) == 0)   g_orig_lastapps_settings.lastapp_mode = KEY_LASTAPP_ALL;
	else if (strcmp(lastapp_mode_str, REGVAL_LASTAPPS_APP) == 0)   g_orig_lastapps_settings.lastapp_mode = KEY_LASTAPP_APP;
	
	if (strcmp(card_mode_str, REGVAL_LASTAPPS_CARD_NORMAL) == 0)        g_orig_lastapps_settings.card_mode = KEY_LASTAPP_CARD_NORMAL;
	else if (strcmp(card_mode_str, REGVAL_LASTAPPS_CARD_RESTORE) == 0)  g_orig_lastapps_settings.card_mode = KEY_LASTAPP_CARD_RESTORE;

	LOGPRINTF("mode:%d, %d", g_orig_lastapps_settings.lastapp_mode, g_orig_lastapps_settings.card_mode);
	
    g_cur_lastapps_settings = g_orig_lastapps_settings;
    
    g_object_unref(client);
}


void save_lastapps_settings()
{
	GConfClient *client = gconf_client_get_default();

    // Store in gconf (only if changed)
    if (g_cur_lastapps_settings.lastapp_mode != g_orig_lastapps_settings.lastapp_mode)
    {
		const char *mode_str = REGVAL_LASTAPPS_OFF;
		
		if (g_cur_lastapps_settings.lastapp_mode == KEY_LASTAPP_OFF) mode_str = REGVAL_LASTAPPS_OFF;
		else if (g_cur_lastapps_settings.lastapp_mode == KEY_LASTAPP_LAST) mode_str = REGVAL_LASTAPPS_LAST;
		else if (g_cur_lastapps_settings.lastapp_mode == KEY_LASTAPP_OPEN) mode_str = REGVAL_LASTAPPS_OPEN;
		else if (g_cur_lastapps_settings.lastapp_mode == KEY_LASTAPP_ALL) mode_str = REGVAL_LASTAPPS_ALL;
		else if (g_cur_lastapps_settings.lastapp_mode == KEY_LASTAPP_APP) mode_str = REGVAL_LASTAPPS_APP;

        write_gconf_value(KEYS_LASTAPPS, KEY_LASTAPP_MODE, client, mode_str);
    }

    if (g_cur_lastapps_settings.card_mode != g_orig_lastapps_settings.card_mode)
    {
		const char *mode_str = REGVAL_LASTAPPS_CARD_NORMAL;
		
		if (g_cur_lastapps_settings.card_mode == KEY_LASTAPP_CARD_NORMAL) mode_str = REGVAL_LASTAPPS_CARD_NORMAL;
		else if (g_cur_lastapps_settings.card_mode == KEY_LASTAPP_CARD_RESTORE) mode_str = REGVAL_LASTAPPS_CARD_RESTORE;

        write_gconf_value(KEYS_LASTAPPS, KEY_LASTAPP_CARD, client, mode_str);
    }

    g_object_unref(client);

    LOGPRINTF("Saving recent settings, done.");
}


//============================================================================
// Local Functions Implementation
//============================================================================

// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Last Document Options")
//    |--vbox
//       |--radio
//       |--radio
static GtkWidget* create_lastapps_widgets(GtkWidget* top_level_vbox)
{
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;
    int        i              = 0;

    // The label "Last Document Mode".
    label = gtk_label_new(_("Last Document Mode"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);
    
    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    for (i=0; i<5; i++)
    {
		switch (i)
		{
			case 0:
				// The radio button "Always".
				g_lastapps_radios[i] = gtk_radio_button_new_with_label(NULL, 
											_("Disabled"));
				break;
			case 1:
				// The radio button "Last Opened Document".
				g_lastapps_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_lastapps_radios[0]), 
											_("Last Opened Document"));
				break;
			case 2:
				// The radio button "Document Active on Shutdown".
				g_lastapps_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_lastapps_radios[0]), 
											_("Document Active on Shutdown"));
				break;
			case 3:
				// The radio button "All documents Active on Shutdown".
				g_lastapps_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_lastapps_radios[0]), 
											_("All documents Active on Shutdown"));
				break;
			case 4:
				// The radio button "Selected Application".
				g_lastapps_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_lastapps_radios[0]), 
											_("Selected Application"));
				break;
		}
		
        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_lastapps_radios[i]),
            "toggled",
            G_CALLBACK(on_lastapps_mode_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_lastapps_radios[i], FALSE, FALSE, 0);
        gtk_widget_show(g_lastapps_radios[i]);
    }
    gtk_widget_show(label);
    gtk_widget_show(vbox);
		
    // TODO: edit box for app
    
    return top_level_vbox;
}


// Widget Hierarchy
// |--Top Level vbox
//    |--label ("Card Removal Options")
//    |--vbox
//       |--radio
//       |--radio
static GtkWidget* create_cardremoval_widgets(GtkWidget* top_level_vbox)
{
    GtkWidget* label          = NULL;
    GtkWidget* vbox           = NULL;
    int        i              = 0;

    // The label "Automatic Shutdown".
    label = gtk_label_new(_("Card Removal Mode"));
    gtk_widget_set_name(label, "irex-section-title");
    gtk_widget_set_size_request(label, -1, -1);
    gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), label, FALSE, FALSE, 0);
    
    // The vbox containing radio buttons.
    vbox = gtk_vbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(top_level_vbox), vbox, FALSE, FALSE, 0);

    for (i=0; i<2; i++)
    {
		switch (i)
		{
			case 0:
				// The radio button "Normal".
				g_cardremoval_radios[i] = gtk_radio_button_new_with_label(NULL, 
											_("Normal"));
				break;
			case 1:
				// The radio button "Save and Restore".
				g_cardremoval_radios[i] = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(g_cardremoval_radios[0]), 
											_("Save and Restore"));
				break;
		}
        // Add signal handler.
        g_signal_connect_after(G_OBJECT(g_cardremoval_radios[i]),
            "toggled",
            G_CALLBACK(on_cardremoval_mode_changed),
            (gpointer)i);

        gtk_box_pack_start(GTK_BOX(vbox), g_cardremoval_radios[i], FALSE, FALSE, 0);
        gtk_widget_show(g_cardremoval_radios[i]);
    }
    gtk_widget_show(label);
    gtk_widget_show(vbox);
		
    return top_level_vbox;
}

static void on_lastapps_mode_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_lastapps_settings.lastapp_mode = index;
    }
}

static void on_cardremoval_mode_changed(GtkWidget *widget, gpointer data)
{
    int index = (int)data;
    gboolean is_active = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));

    if (is_active == TRUE)
    {
        g_cur_lastapps_settings.card_mode = index;
    }
}


static void init_widgets_with_settings()
{
	int i;
	
    for (i = 0; i < 5; i++)
    {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_lastapps_radios[i]), 
                (i == g_cur_lastapps_settings.lastapp_mode));
    }
    for (i = 0; i < 2; i++)
    {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(g_cardremoval_radios[i]), 
                (i == g_cur_lastapps_settings.card_mode));
    }
	
	// Disabled not implemented functionality
	gtk_widget_set_sensitive(g_lastapps_radios[1], FALSE); //last read
	//gtk_widget_set_sensitive(g_lastapps_radios[3], FALSE); // all
	gtk_widget_set_sensitive(g_lastapps_radios[4], FALSE); // application
	//gtk_widget_set_sensitive(g_cardremoval_radios[1], FALSE); // restore
}

// read single string from gconf
static gchar *read_gconf_value(const gchar *dir,
                               const gchar *key,
                               GConfClient *client)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    gchar* value;
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);
    value = gconf_client_get_string(client, buffer, NULL);
    return value;
}

// write string to gconf
static gboolean write_gconf_value(const gchar *dir,
                                   const gchar *key,
                                   GConfClient *client,
								   const gchar *value)
{
    gchar buffer[GCONF_BUFFER_SIZE];
    g_assert(strlen(dir) + 1 + strlen(key) < GCONF_BUFFER_SIZE);
    snprintf(buffer, GCONF_BUFFER_SIZE-1, "%s/%s", dir, key);
    return gconf_client_set_string(client, buffer, value, NULL);
}
