/*
 * File Name: menu.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

#include "ctb_log.h"
#include "fileview.h"
#include "i18n.h"
#include "ipc.h"
#include "menu.h"


//----------------------------------------------------------------------------
// Type Declarations
//----------------------------------------------------------------------------


//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// menu groups                              
static const char *GROUP_VIEWS              = "ctb_view";
static const char *GROUP_SORT               = "ctb_sort";

static const char *GROUP_ACTIONS            = "ctb_actions";
static const char *GROUP_EXTRA              = "ctb_extra";
                                            
// menu items                               
//   group "Views"                          
static const char *ITEM_VIEW_ICON           = "view_small";
static const char *ITEM_VIEW_DETAIL         = "view_detail";
static const char *ITEM_VIEW_CONTENT        = "view_content";

//   group "Sort"                           
static const char *ITEM_SORT_NAME           = "sort_name";
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static const char *ITEM_SORT_TYPE           = "sort_type";
static const char *ITEM_SORT_SIZE           = "sort_size";
#endif
static const char *ITEM_SORT_AUTHOR         = "sort_author";
static const char *ITEM_SORT_DATE_ADDED     = "sort_date_added";
static const char *ITEM_SORT_DATE_READ      = "sort_date_read";
//   group "Actions"                        
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static const char *ITEM_ACTION_SHORTCUT     = "create_shortcut";
#endif
static const char *ITEM_ACTION_DELETE       = "delete_mode";
static const char *ITEM_ACTION_PROPERTIES   = "properties";
static const char *ITEM_ACTION_STARTPAGE    = "startpage";
                                            
// item states                              
static const char *STATE_NORMAL             = "normal";
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
static const char *STATE_DISABLED           = "disabled";
#endif
static const char *STATE_SELECTED           = "selected";

// stylus states
static const char *STYLUS_NONE              = "none";

static const char* BACK_TO_LIBRARY          = "back_to_library";
static const char* SYSTEM_TOP               = "system_top";

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static GString  *g_sort_item            = NULL;
static ctb_viewmodes_t g_viewmode = BROWSE_MODE;
static gchar* g_category = NULL;

//============================================================================
// Functions Implementation
//============================================================================

void menu_init()
{
    g_sort_item = g_string_new("");

    // add groups
    const char *group = GROUP_VIEWS;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_VIEW_ICON,         group, "view_small"      );
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    ipc_menu_add_item ( ITEM_VIEW_DETAIL,         group, "view_detail"     );
#endif
    ipc_menu_add_item ( ITEM_VIEW_CONTENT,      group, "view_content"     );

    group = GROUP_SORT;                         
    ipc_menu_add_group( group,                  "",    "folder"          );                                                  
    ipc_menu_add_item ( ITEM_SORT_NAME,         group, "sort_name"       );
    ipc_menu_add_item ( ITEM_SORT_AUTHOR,       group, "sort_author"     );
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    ipc_menu_add_item ( ITEM_SORT_TYPE,         group, "sort_type"       );
    ipc_menu_add_item ( ITEM_SORT_SIZE,         group, "sort_size"       );
#endif
    ipc_menu_add_item ( ITEM_SORT_DATE_ADDED,   group, "sort_date_added" );
    ipc_menu_add_item ( ITEM_SORT_DATE_READ,    group, "sort_date_read" );

    group = GROUP_ACTIONS;
    ipc_menu_add_group( group,                  "",    "folder"          );
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    ipc_menu_add_item ( ITEM_ACTION_SHORTCUT,   group, "create_shortcut" );
#endif
    ipc_menu_add_item ( ITEM_ACTION_DELETE,     group, "delete"          );
    ipc_menu_add_item ( ITEM_ACTION_PROPERTIES, group, "properties"      );
    ipc_menu_add_item ( ITEM_ACTION_STARTPAGE,  group, "startpage"       );

    group = GROUP_EXTRA;
    ipc_menu_add_group( group,                  "",    "folder"          );
    ipc_menu_add_item ( ITEM_ACTION_STARTPAGE,  group, "startpage"       );

    // add menus
    ipc_menu_add_menu(MENU_CONTENT_MEDIA,       GROUP_VIEWS,   GROUP_SORT,    GROUP_ACTIONS);
    ipc_menu_add_menu(MENU_CONTENT,             GROUP_VIEWS,   GROUP_EXTRA,   NULL);
    ipc_menu_add_menu(MENU_CONTENT_DELETE_MODE, GROUP_ACTIONS, NULL,          NULL);
    ipc_menu_add_menu(MENU_CONTENT_RECENT_MODE, GROUP_VIEWS,   GROUP_ACTIONS, NULL);

    ipc_menu_set_item_state(BACK_TO_LIBRARY, SYSTEM_TOP, STATE_NORMAL);

    menu_set_text();
}


static void update_viewmode_text()
{
    const char *group = GROUP_ACTIONS;

    switch (g_viewmode) {
        case BROWSE_MODE:
            ipc_menu_set_item_label(ITEM_ACTION_DELETE, group, _("Delete..."));
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
            ipc_menu_set_item_state(ITEM_ACTION_SHORTCUT, group, STATE_NORMAL);
#endif
            break;
        case DELETE_MODE:
            ipc_menu_set_item_label(ITEM_ACTION_DELETE, group, _("Cancel Delete"));
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
            ipc_menu_set_item_state(ITEM_ACTION_SHORTCUT, group, STATE_DISABLED);
#endif
            break;
    }
}


static void update_category()
{
    if (filemodel_window_is_on_top()) {
        ipc_menu_set_item_label(BACK_TO_LIBRARY, SYSTEM_TOP, "");
    } else {
        char buffer[128];
        buffer[0] = 0;
        if (g_category) {
            /* TRANSLATORS: %s is replaced by the current category
            Example: 'Back to Books' */
            snprintf(buffer, 127, _("Back to %s"), _(g_category));
        }
        ipc_menu_set_item_label(BACK_TO_LIBRARY, SYSTEM_TOP, buffer);
    }
}


void menu_set_category(const char* category)
{
    g_free(g_category);
    g_category = g_strdup(category);
    update_category();
}


// initialise popup menu
void menu_set_text()
{
    const char *group;

    // set groups + items
    group = GROUP_VIEWS;
    ipc_menu_set_group_label( group,                         _("View")              );
#if MACHINE_IS_DR800S || MACHINE_IS_DR800SG || MACHINE_IS_DR800SW
    ipc_menu_set_item_label ( ITEM_VIEW_ICON,         group, _("View as Covers")    );
#elif MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    ipc_menu_set_item_label ( ITEM_VIEW_ICON,         group, _("View as Thumbnails"));
    ipc_menu_set_item_label ( ITEM_VIEW_DETAIL,       group, _("View as Details")   );
#else
#error Unhandled machine type
#endif
    ipc_menu_set_item_label ( ITEM_VIEW_CONTENT,      group, _("View as List")      );
    //
    group = GROUP_SORT;
    ipc_menu_set_group_label( group,                         _("Sort by")           );
    ipc_menu_set_item_label ( ITEM_SORT_NAME,         group, _("Sort by Title")     );
    ipc_menu_set_item_label ( ITEM_SORT_AUTHOR,       group, _("Sort by Author")    );
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    ipc_menu_set_item_label ( ITEM_SORT_TYPE,         group, _("Sort by Type")      );
    ipc_menu_set_item_label ( ITEM_SORT_SIZE,         group, _("Sort by Size")      );
#endif
    ipc_menu_set_item_label ( ITEM_SORT_DATE_ADDED,   group, _("Sort by Date Added")  );
    ipc_menu_set_item_label ( ITEM_SORT_DATE_READ,    group, _("Sort by Recently Opened"));
    //
    group = GROUP_ACTIONS;
    ipc_menu_set_group_label( group,                         _("Actions")           );
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
    ipc_menu_set_item_label ( ITEM_ACTION_SHORTCUT,   group, _("Add Shortcut")      );
#endif
    ipc_menu_set_item_label ( ITEM_ACTION_PROPERTIES, group, _("Properties")        );
    ipc_menu_set_item_label ( ITEM_ACTION_STARTPAGE,  group, _("Set as StartPage")  );
    //
    group = GROUP_EXTRA;
    ipc_menu_set_item_label ( ITEM_ACTION_STARTPAGE,  group, _("Set as StartPage")  );


    update_viewmode_text();
    update_category();
}


void menu_show()
{
    if (!filemodel_window_is_on_top()) return;

    ipc_status_set_stylus( STYLUS_NONE );
    update_category();
    ipc_menu_show_menu(filemodel_get_menu_content());
    filemodel_update_pagecounter();
}


void menu_hide()
{
    if (!filemodel_current_is_desktop()) {
        update_category();
        ipc_menu_updates_finished();
    }
}


void menu_select_view_type (ctb_viewtypes_t view_type)
{
    static const char *item_old = "";
    const char *item_new = "";

    LOGPRINTF("entry: view_type [%d]", view_type);

    switch (view_type)
    {
        case CTB_ICONVIEW:
            item_new = ITEM_VIEW_ICON;
            break;
        case CTB_LISTVIEW:
            item_new = ITEM_VIEW_DETAIL;
            break;
        case CTB_CONTENTVIEW:
            item_new = ITEM_VIEW_CONTENT;
            break;
        default:
            ;  // ignore: handled by assert at function entry
    }

    if ( strcmp(item_old, item_new) != 0 )
    {
        if ( item_old[0] != '\0' )
        {
            ipc_menu_set_item_state ( item_old, GROUP_VIEWS, STATE_NORMAL   );
        }
        ipc_menu_set_item_state ( item_new, GROUP_VIEWS, STATE_SELECTED );

        item_old = item_new;
    }
}


void menu_update_view_mode(ctb_viewmodes_t view_mode)
{
    g_viewmode = view_mode;
    update_viewmode_text();
    ipc_menu_show_menu(filemodel_get_menu_content());
}


// select one of the sorting orders
void menu_select_sort_order ( const ctb_sort_order_t sort_order )
{
    LOGPRINTF("entry: sort_order [%d]", sort_order);
    g_assert(g_sort_item && g_sort_item->str);

    const char *item_new = NULL;
    switch (sort_order)
    {
        case CTB_SORT_BY_NAME:
            item_new = ITEM_SORT_NAME;
            break;
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        case CTB_SORT_BY_TYPE:
            item_new = ITEM_SORT_TYPE;
            break;
        case CTB_SORT_BY_SIZE:
            item_new = ITEM_SORT_SIZE;
            break;
#endif
        case CTB_SORT_BY_DATE_ADDED:
            item_new = ITEM_SORT_DATE_ADDED;
            break;
        case CTB_SORT_BY_DATE_READ:
            item_new =  ITEM_SORT_DATE_READ;
            break;
        case CTB_SORT_BY_AUTHOR:
            item_new = ITEM_SORT_AUTHOR;
            break;
        default:
            g_assert(0);
    }

    if ( strcmp(g_sort_item->str, item_new) != 0 )
    {
        if ( g_sort_item->str[0] )
        {
            ipc_menu_set_item_state ( g_sort_item->str, GROUP_SORT, STATE_NORMAL   );
        }
        ipc_menu_set_item_state ( item_new, GROUP_SORT, STATE_SELECTED );

        g_string_assign( g_sort_item, item_new );
    }
}


void menu_set_current_is_media(const gboolean is_media)
{
    static gboolean prev_is_media = -1;

    if (prev_is_media != is_media)
    {
        prev_is_media = is_media;

#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
        ipc_menu_set_item_state(ITEM_ACTION_SHORTCUT,
                                GROUP_ACTIONS,
                                is_media ? STATE_NORMAL : STATE_DISABLED);
        ipc_menu_set_item_state(ITEM_VIEW_DETAIL,
                                GROUP_VIEWS,
                                is_media ? STATE_NORMAL : STATE_DISABLED);
#endif
        //WARNPRINTF("MH");
        ipc_menu_show_menu(filemodel_get_menu_content());
    }
}


//----------------------------------------------------------------------------
// Callbacks from popupmenu
//----------------------------------------------------------------------------

void menu_on_goto_location(const gchar *location)
{
    START_TIMER();
    g_assert( location  &&  location[0] != '\0' );

    if ( strcmp(location, "desktop") == 0 )
    {
        fileview_show_desktop();
        //WARNPRINTF("MH");
        menu_show();
    }
    else
    {
        ERRORPRINTF("unknown location [%s]", location);
    }
}


void menu_on_item_activated ( const gchar *item,
                              const gchar *group,
                              const gchar *menu,
                              const gchar *state )
{
    START_TIMER();
    gboolean ok = TRUE;

    LOGPRINTF("entry: item [%s] group [%s]", item, group);

    if ( strcmp(group, GROUP_VIEWS) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            ctb_viewtypes_t view_type = 0;
            if (strcmp(item, ITEM_VIEW_ICON) == 0 )
            {
                view_type = CTB_ICONVIEW;
            }
            else if (strcmp(item, ITEM_VIEW_DETAIL) == 0 )
            {
                view_type = CTB_LISTVIEW;
            }
            else if (strcmp(item, ITEM_VIEW_CONTENT) == 0 )
            {
                view_type = CTB_CONTENTVIEW;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }

            if (ok) fileview_set_view_type( view_type, TRUE );
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_SORT) == 0 )
    {
        ctb_sort_order_t sort_order = 0;
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if (      strcmp(item, ITEM_SORT_NAME  ) == 0 )
            {
                sort_order = CTB_SORT_BY_NAME;
            }
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
            else if ( strcmp(item, ITEM_SORT_TYPE  ) == 0 )
            {
                sort_order = CTB_SORT_BY_TYPE;
            }
            else if ( strcmp(item, ITEM_SORT_SIZE  ) == 0 )
            {
                sort_order = CTB_SORT_BY_SIZE;
            }
#endif
            else if ( strcmp(item, ITEM_SORT_DATE_ADDED  ) == 0 )
            {
                sort_order = CTB_SORT_BY_DATE_ADDED;
            }
            else if ( strcmp(item, ITEM_SORT_DATE_READ  ) == 0 )
            {
                sort_order = CTB_SORT_BY_DATE_READ;
            }
            else if ( strcmp(item, ITEM_SORT_AUTHOR) == 0 )
            {
                sort_order = CTB_SORT_BY_AUTHOR;
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }

            if (ok)
            {
                fileview_set_sort_order( sort_order );
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_ACTIONS) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if ( strcmp(item, ITEM_ACTION_DELETE    ) == 0 )
            {
                fileview_toggle_delete_mode();
            }
#if MACHINE_IS_DR1000S || MACHINE_IS_DR1000SW
            else if ( strcmp(item, ITEM_ACTION_SHORTCUT  ) == 0 )
            {
                fileview_create_shortcut();
            }
#endif
            else if ( strcmp(item, ITEM_ACTION_PROPERTIES  ) == 0 )
            {
                fileview_edit_properties();
            }
            else if ( strcmp(item, ITEM_ACTION_STARTPAGE  ) == 0 )
            {
                fileview_set_as_startup_view();
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else if ( strcmp(group, GROUP_EXTRA) == 0 )
    {
        if ( strcmp(state, STATE_NORMAL) == 0 )
        {
            if ( strcmp(item, ITEM_ACTION_STARTPAGE  ) == 0 )
            {
                fileview_set_as_startup_view();
            }
            else
            {
                WARNPRINTF("unexpected menu item [%s] in group [%s]", item, group);
                ok = FALSE;
            }
        }
        else
        {
            WARNPRINTF("ignore state [%s] item [%s][%s]", state, item, group);
        }
    }
    else
    {
        WARNPRINTF("unexpected menu group [%s]", group);
    }
    ipc_menu_updates_finished();
}

