/*
 * File Name: ipc.c
 */

/*
 * This file is part of ctb.
 *
 * ctb is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * ctb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
 
/**
 * Copyright (C) 2008 iRex Technologies B.V.
 * All rights reserved.
 */

//----------------------------------------------------------------------------
// Include Files
//----------------------------------------------------------------------------

#include "config.h"

#include <stdlib.h>
#include <unistd.h>

#include <liberipc/eripc.h>
#include <liberipc/eripc_support.h>
#include <liberutils/er_error.h>
#include <libermetadb/ermetadb.h>

#include <time.h>

#include "ctb_log.h"
#include "ipc.h"
#include "menu.h"
#include "lastapps.h"

//----------------------------------------------------------------------------
// Constants
//----------------------------------------------------------------------------

// IPC content browser
#define DBUS_APPL_NAME                  PACKAGE_NAME
#define DBUS_SERVICE                     "com.irexnet." DBUS_APPL_NAME
#define DBUS_PATH                       "/com/irexnet/" DBUS_APPL_NAME
#define DBUS_INTERFACE                   "com.irexnet." DBUS_APPL_NAME

#define DBUS_SERVICE_SYSTEM_CONTROL     "com.irexnet.sysd"
#define DBUS_SERVICE_POPUP_MENU         "com.irexnet.popupmenu"
#define DBUS_SERVICE_SETTINGS           "com.irexnet.settings"

//----------------------------------------------------------------------------
// Static Variables
//----------------------------------------------------------------------------

static eripc_client_context_t *eripcClient = NULL;
static quit_function g_quit_callback = NULL;
static locale_function g_set_locale_callback = NULL;
static gboolean g_in_portrait = TRUE;

static gchar *g_storage_media = NULL;
static mount_cb_t g_mount_cb = NULL;
static unmount_cb_t g_unmount_cb = NULL;


//============================================================================
// Local Function Definitions
//============================================================================

static void on_goto_location     ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_page_change       ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_filesystem_change ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_window_activated  ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_window_deactivated( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_menu_item         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_mounted           ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );
                                 
static void on_unmounted         ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_prepare_standby ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_changed_locale    ( eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static void on_changed_orientation(eripc_context_t          *context,
                                   const eripc_event_info_t *info,
                                   void                     *user_data );

static gboolean is_in_portrait_orientation();

// Exported DBUS API list
static eripc_callback_function_t service_functions[] = {
//    handler               message_name           interface,                  id
    { on_goto_location,     "gotoLocation",        NULL                        ,0 },
    { on_menu_item,         "menuItemActivated",   NULL                        ,0 },
    { on_mounted,           "sysVolumeMounted",    NULL                        ,0 },
    { on_window_activated,  "activatedWindow",     NULL                        ,0 },
    { on_window_deactivated,"deactivatedWindow",   NULL                        ,0 },
    { on_mounted,           "sysVolumeMounted",    DBUS_SERVICE_SYSTEM_CONTROL ,0 },
    { on_unmounted,         "sysPrepareUnmount",   DBUS_SERVICE_SYSTEM_CONTROL ,0 },
    { on_unmounted,         "sysVolumeUnmounted",  DBUS_SERVICE_SYSTEM_CONTROL ,0 },
    { on_prepare_standby,   "sysPrepareStandby",   DBUS_SERVICE_SYSTEM_CONTROL ,0 },
    { on_changed_locale,    "sysChangedLocale",    DBUS_SERVICE_SYSTEM_CONTROL ,0 },
    { on_page_change,       "pageChange"    ,      NULL                        ,0 },
    { on_filesystem_change, "filesystemChanged",   NULL                        ,0 },
    { on_changed_orientation, "sysChangedOrientation",  DBUS_SERVICE_SYSTEM_CONTROL ,0 },
    { NULL, NULL, NULL, 0 }  // end of list
};


//============================================================================
// Functions Implementation
//============================================================================

typedef unsigned long long u_int64_t;
u_int64_t getCurrentTime()
{
    struct timespec now;

    clock_gettime(CLOCK_MONOTONIC, &now);
    u_int64_t now64 = now.tv_sec;
    now64 *= 1000000;
    now64 += (now.tv_nsec/1000);
    return now64;
}


void ipc_init (quit_function quit_cb,
               locale_function set_locale_cb,
               mount_cb_t mount_cb,
               unmount_cb_t unmount_cb)
{
    g_quit_callback = quit_cb;
    g_set_locale_callback = set_locale_cb;
    g_mount_cb = mount_cb;
    g_unmount_cb = unmount_cb;

    eripcClient = eripc_client_context_new(
                    DBUS_APPL_NAME, 
                    "1.0",
                    DBUS_SERVICE, 
                    DBUS_PATH,
                    DBUS_INTERFACE,
                    service_functions);
    g_in_portrait = is_in_portrait_orientation();
}


void ipc_sys_startup_complete()
{
    eripc_sysd_startup_complete( eripcClient, getpid(), FALSE, fileview_get_xid());
}


gint ipc_sys_start_task ( const gchar  *cmd_line,
                          const gchar  *work_dir,
                          const gchar  *label,
                          const gchar  *thumbnail_path,
                          gchar        **err_message )
{
    LOGPRINTF("entry: cmd_line [%s] work_dir [%s] label [%s] thumbnail_path [%s]",
                       cmd_line,     work_dir,     label,     thumbnail_path       );
    g_assert( cmd_line && *cmd_line );
    return eripc_sysd_start_task( eripcClient, 
                                  cmd_line, 
                                  work_dir, 
                                  label, 
                                  thumbnail_path, 
                                  err_message);
}


gboolean ipc_sys_stop_task ( const gchar  *cmd_line )
{
    LOGPRINTF("entry: cmd_line [%s]", cmd_line );
    g_assert( cmd_line && *cmd_line );
    return eripc_sysd_stop_task( eripcClient, cmd_line );
}


gint ipc_sys_open_url ( const gchar  *url,
                        const gchar  *label,
                        const gchar  *back_text,
                        gchar        **err_message )
{
    LOGPRINTF("entry: url [%s] label [%s] back_text [%s]",
                      url,     label,     back_text     );
    g_assert( url && *url );
    return eripc_sysd_open_url( eripcClient, url, label, back_text, err_message );
}


const device_caps_t* ipc_sys_get_device_capabilities()
{
    static device_caps_t dev_caps;

    eripc_device_caps_t er_dev_caps;

    eripc_sysd_get_device_capabilities( eripcClient, &er_dev_caps );

    dev_caps.has_stylus = er_dev_caps.has_stylus;
    dev_caps.has_wifi = er_dev_caps.has_wifi;
    dev_caps.has_bluetooth = er_dev_caps.has_bluetooth;
    dev_caps.has_3g = er_dev_caps.has_3g;

    return &dev_caps;
}


//----------------------------------------------------------------------------
// Popup menu
//----------------------------------------------------------------------------

gboolean ipc_menu_add_menu( const char *name,
                            const char *group1,
                            const char *group2,
                            const char *group3 )
{
    return eripc_menu_add_menu(eripcClient, name, group1, group2, group3, "");
}


gboolean ipc_menu_add_group( const char *name,
                             const char *parent, 
                             const char *image )
{
    return eripc_menu_add_group(eripcClient, name, parent, image);
}


gboolean ipc_menu_add_item( const char *name,
                            const char *parent, 
                            const char *image  )
{
    return eripc_menu_add_item(eripcClient, name, parent, image);
}


gboolean ipc_menu_set_menu_label ( const char *name,
                                   const char *label )
{
    return eripc_menu_set_menu_label(eripcClient, name, label);
}


gboolean ipc_menu_set_group_label ( const char *name,
                                    const char *label )
{
    return eripc_menu_set_group_label(eripcClient, name, label);
}


gboolean ipc_menu_set_item_label ( const char *name,
                                   const char *parent, 
                                   const char *label )
{
    return eripc_menu_set_item_label(eripcClient, name, parent, label);
}


gboolean ipc_menu_show_menu( const char *name )
{
    //u_int64_t thetime = getCurrentTime();
    
    //WARNPRINTF("%lld:%s", thetime/1000, name);
    return eripc_menu_show_menu(eripcClient, name);
}


gboolean ipc_menu_set_group_state( const char *name,
                                   const char *state )
{
    return eripc_menu_set_group_state(eripcClient, name, state);
}


gboolean ipc_menu_set_item_state( const char *name,
                                  const char *parent,
                                  const char *state  )
{
    return eripc_menu_set_item_state(eripcClient, name, parent, state);
}


gboolean ipc_menu_block( void )
{
    return eripc_sysd_set_menu_state(eripcClient, "block");
}


gboolean ipc_menu_unblock( void )
{
    return eripc_sysd_set_menu_state(eripcClient, "unblock");
}


void ipc_menu_updates_finished()
{
    eripc_send_varargs(eripcClient->context, NULL, NULL, ERIPC_BUS_SESSION, DBUS_SERVICE_POPUP_MENU, "updatesFinished", ERIPC_TYPE_INVALID);
}


void ipc_menu_set_pagecounter(int cur_page, int num_pages, gboolean boundary_check)
{
    eripc_menu_set_pagecounter(eripcClient, cur_page, num_pages, boundary_check);
}


gboolean ipc_sys_busy( gboolean look_busy )
{
    if (look_busy)
        return eripc_sysd_set_busy(eripcClient, "delaydialog", NULL);
    else
        return eripc_sysd_reset_busy(eripcClient);
}

gboolean ipc_status_set_stylus( const char *state )
{
    g_return_val_if_fail( (state != NULL), FALSE);
    return eripc_menu_set_statusitem_state(eripcClient, "statusbar_stylus", state);
}


gboolean ipc_sys_is_in_portrait_mode()
{
    return g_in_portrait;
}


//============================================================================
// Local Function Implementation
//============================================================================

static gboolean is_in_portrait_orientation()
{
    gboolean result = TRUE;

    eripc_event_info_t* info = NULL;
    eripc_error_t retval = eripc_send_varargs_and_wait(eripcClient->context,
            &info,
            ERIPC_BUS_SESSION,
            DBUS_SERVICE_SYSTEM_CONTROL,
            "sysGetOrientation",
            ERIPC_TYPE_INVALID);

    if (retval != ERIPC_ERROR_SUCCESS)
    {
        ERRORPRINTF("Error launching eripc handler");
    }
    else if (info == NULL || info->args == NULL)
    {
        ERRORPRINTF("sysd returns OK but no reply structure");
    }
    else
    {
        const eripc_arg_t *arg_array = info->args;

        if (arg_array[0].type == ERIPC_TYPE_STRING
            && strcmp("portrait", arg_array[0].value.s) == 0)
        {
            result = TRUE;
        }
        else
        {
            result = FALSE;
        }
    }
    eripc_event_info_free(eripcClient->context, info);
    return result;
}


//----------------------------------------------------------------------------
// Signal/message handlers
//----------------------------------------------------------------------------

// show specified location
static void on_goto_location ( eripc_context_t          *context,
                               const eripc_event_info_t *info,
                               void                     *user_data )
{
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *location  = arg_array[0].value.s;

    if (location==NULL || location[0] == '\0') return;

    menu_on_goto_location( location );
}


static void on_page_change ( eripc_context_t          *context,
                             const eripc_event_info_t *info,
                             void                     *user_data )
{
    const eripc_arg_t *arg_array = info->args;
    g_return_if_fail(arg_array[0].type == ERIPC_TYPE_STRING);
    gchar* direction = arg_array[0].value.s;

    if (strcmp(direction, "prev") == 0) fileview_show_prev_page();
    if (strcmp(direction, "next") == 0) fileview_show_next_page();
}


static void on_filesystem_change( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    fileview_refresh(FALSE);
}


static void on_window_activated( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    fileview_set_on_top(TRUE);
    fileview_stop_update_display();
    //WARNPRINTF("MH");
    menu_show();
    fileview_grab_focus();
    fileview_refresh(FALSE);  // needed for icons + author
}


static void on_window_deactivated( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    fileview_set_on_top(FALSE);
    menu_hide();
}


static void on_menu_item ( eripc_context_t          *context,
                           const eripc_event_info_t *info,
                           void                     *user_data )
{
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[1].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[2].type == ERIPC_TYPE_STRING);
    g_return_if_fail(info->args[3].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *item      = arg_array[0].value.s;
    const char        *group     = arg_array[1].value.s;
    const char        *menu      = arg_array[2].value.s;
    const char        *state     = arg_array[3].value.s;

    if (item && group && menu && state)
    {
        menu_on_item_activated( item, group, menu, state );
    }
}


// storage device has been mounted
static void on_mounted ( eripc_context_t          *context,
                         const eripc_event_info_t *info,
                         void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    if (mountpoint && *mountpoint)
    {
        storage_media_mounted(mountpoint);
    }

    // Start the last-apps after a card has been mounted, this will thus be called
    // during startup and when usb is disconnected
    lastapps_start(LASTAPPS_TRIGGER_MOUNTED);
    
}


static void on_unmounted ( eripc_context_t          *context,
                           const eripc_event_info_t *info,
                           void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array  = info->args;
    const char        *mountpoint = arg_array[0].value.s;

    if (!mountpoint || !*mountpoint) return;

    if (g_storage_media) {
        if (strcmp(mountpoint, g_storage_media) == 0) {
            g_free(g_storage_media);
            g_storage_media = NULL;
            g_unmount_cb();
        } else {
            ERRORPRINTF("media mismatch, mountpoint=%s  need to unmount %s", g_storage_media, mountpoint);
        }
    } else {
        WARNPRINTF("storage not present [%s]", mountpoint);
    }
}


// system will be in standby mode
static void on_prepare_standby ( eripc_context_t          *context,
                                 const eripc_event_info_t *info,
                                 void                     *user_data )
{
    LOGPRINTF("entry");

    fileview_save_view_type();
}


// LOCALE setting (language) has changed
static void on_changed_locale ( eripc_context_t          *context,
                                const eripc_event_info_t *info,
                                void                     *user_data )
{
    LOGPRINTF("entry");
    g_return_if_fail(info->args);
    g_return_if_fail(info->args[0].type == ERIPC_TYPE_STRING);

    const eripc_arg_t *arg_array = info->args;
    const char        *locale    = arg_array[0].value.s;

    if (locale)
    {
        g_set_locale_callback(locale);
    }
}

/* Called when the display's orientation has changed
*
* Application may need to adapt its screen size, coordinates and/or origin.
* Possible values: "portrait", "landscape_clockwise", "landscape_anticlockwise"
*/
static void on_changed_orientation ( eripc_context_t          *context,
                                     const eripc_event_info_t *info,
                                     void                     *user_data )
{
    LOGPRINTF("entry");
    const eripc_arg_t *arg_array = info->args;

    if (arg_array[0].type == ERIPC_TYPE_STRING)
    {   
        const char *orientation = arg_array[0].value.s;
        if (orientation)
        {
            if (strcasecmp("portrait", orientation) == 0)
            {
                g_in_portrait = TRUE;
                fileview_move_undo();
            }
            else 
            {
                g_in_portrait = FALSE;
                // call orientation change code
                fileview_move_up( 5 ); // moveup 5 pixels to compensate for higher popupmenu in landscape
            }
        }
    }
}


const gchar* ipc_get_media()
{
    return g_storage_media;
}


void storage_media_mounted ( const gchar *mountpoint )
{
    LOGPRINTF("entry: mountpoint [%s]", mountpoint);
    g_assert(mountpoint && *mountpoint);

    if (g_storage_media) {
        ERRORPRINTF("storage already present %s (new %s)", g_storage_media, mountpoint);
    } else {
        g_storage_media = g_strdup(mountpoint);
        g_mount_cb(mountpoint);
    }
}



